#include "UppGL.h"

#define MAGIC 0xDEADBEEF

void EnableClientStatePNT() 
{
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_NORMAL_ARRAY);
	if (UseBumpMapping())
		glEnableVertexAttribArray(VERTEX_TANGENT_ATTRIB);
	
	glClientActiveTextureARB(GL_TEXTURE0_ARB + UNIT0_UNIT);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glClientActiveTextureARB(GL_TEXTURE0_ARB + UNIT1_UNIT);	
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
}
void DisableClientStatePNT()
{
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableVertexAttribArray(VERTEX_TANGENT_ATTRIB);		
}

void MeshBuffer::Activate() const
{
	ASSERT(vbo && ibo);
	// Vertex buffer
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, vbo);
	glVertexPointer(3, GL_FLOAT, sizeof(Vertex), MESH_BUFFER_OFFSET(position));
	glNormalPointer(GL_FLOAT, sizeof(Vertex), MESH_BUFFER_OFFSET(normal));
	glClientActiveTextureARB(GL_TEXTURE0_ARB + UNIT0_UNIT);
	glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), MESH_BUFFER_OFFSET(texture0));	
	glClientActiveTextureARB(GL_TEXTURE0_ARB + UNIT1_UNIT);
	glTexCoordPointer(2, GL_FLOAT, sizeof(Vertex), MESH_BUFFER_OFFSET(texture0));	
	
	if (UseBumpMapping() && HasTangents()) {
		glBindBufferARB(GL_ARRAY_BUFFER_ARB, tbo);
		glVertexAttribPointerARB(VERTEX_TANGENT_ATTRIB, 3, GL_FLOAT, GL_FALSE, sizeof(Vec3f), 0); 
	}

	// Index buffer
	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, ibo);		
}

void MeshBuffer::Deactivate()
{
	// Disable VBOs
	glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
	glBindBufferARB(GL_ELEMENT_ARRAY_BUFFER_ARB, 0);
}

void MeshBuffer::Clear() 
{
	if (vbo) {
		DestroyVBO(vbo);
		vbo = 0;
	}
	if (ibo) {
		DestroyVBO(ibo);
		ibo = 0;
	}
}

void MeshBuffer::QRender(const SubMesh *begin, const SubMesh *end) const
{
	while (begin <= end) {
		begin->material.Activate();	
 		glDrawElements(GL_TRIANGLES, begin->count, GL_UNSIGNED_INT, (GLvoid *)begin->byteoffset);
 		begin++;
	}	
}

void MeshBuffer::QRenderNoMat(const SubMesh *begin, const SubMesh *end) const
{
	int cnt = 0;	
	while (begin <= end) {
		cnt += begin->count;
		begin++;
	}		
	glDrawElements(GL_TRIANGLES, cnt, GL_UNSIGNED_INT, (GLvoid *)0);
}

void MeshData::Clear()
{
	buffer.Clear();
	submesh.Clear();
	vertex.Clear();
	index.Clear();
}

void MeshData::CalcBounds()
{
	Vec3f bounds_min, bounds_max;
	bounds_min.x = bounds_min.y = bounds_min.z = 1000000.0f;
	bounds_max.x = bounds_max.y = bounds_max.z = -1000000.0f;
	for (int i = 0; i < vertex.GetCount(); i++) {
		Vec3f &pos = vertex[i].position;
		bounds_min.x = min(bounds_min.x, pos.x);
		bounds_min.y = min(bounds_min.y, pos.y);
		bounds_min.z = min(bounds_min.z, pos.z);		
		
		bounds_max.x = max(bounds_max.x, pos.x);
		bounds_max.y = max(bounds_max.y, pos.y);
		bounds_max.z = max(bounds_max.z, pos.z);		
	}
	bounds.Create(bounds_min, bounds_max);
}

void MeshData::CreateTangents()
{
	ASSERT(HasMirror());
	tangents = CalculateTriangleTangents(vertex, index);
}

void MeshData::ReverseWinding()
{
	ASSERT(HasMirror());
	for (int i = 0; i < index.GetCount(); i+=3)
		Swap(index[i], index[i+2]);
	CreateNormals(true);
}

void MeshData::Scale(float x, float y, float z)
{
	for (int i = 0; i < vertex.GetCount(); i++) {
		Vec3f &v = vertex[i].position;
		v.x *= x;
		v.y *= y;
		v.z *= z;
	}
	CreateNormals(true);
	CalcBounds();
}

void MeshData::TextureOffset(float u, float v)
{
	Vec2f uv(u, v);
	for (int i = 0; i < vertex.GetCount(); i++) {
		Vertex &v = vertex[i];
		vertex[i].texture0 += uv;
	}
}

void MeshData::TextureScale(float u, float v)
{
	for (int i = 0; i < vertex.GetCount(); i++) {
		vertex[i].texture0.x *= u;
		vertex[i].texture0.y *= v;
	}
}



int MeshData::TriangleCount() const
{
	if (!submesh.GetCount()) return 0;
	const SubMesh *q = &submesh[0];
	const SubMesh *eos = &submesh.Top()+1;
	int cnt = 0;
	while (q < eos) {
		cnt += q->count;
		++q;	
	}
	return cnt/3;
}

void MeshData::CreateNormals(bool initialise)
{
	int i = 0;
	if (initialise)
		for (i = 0; i < vertex.GetCount(); i++)
			vertex[i].normal = Vec3f(0.0f, 0.0f, 0.0f);
	i = 0;
	while (i < index.GetCount()) {
		ASSERT(index[i] < (GLuint)vertex.GetCount());	
		ASSERT(index[i+1] < (GLuint)vertex.GetCount());
		ASSERT(index[i+2] < (GLuint)vertex.GetCount());
		VertexPNT &a = vertex[(int)index[i]];
		VertexPNT &b = vertex[(int)index[i+1]];
		VertexPNT &c = vertex[(int)index[i+2]];
		Vec3f n = GetNormal(a.position, b.position, c.position);
		a.normal += n;
		b.normal += n;
		c.normal += n;
		i += 3;
	}
	for (i = 0; i < vertex.GetCount(); i++)
		vertex[i].normal = glm::normalize(vertex[i].normal);
}

void _RemoveUselessTriangles(const Vector<Vertex> &vertex, Vector<GLuint> &in)
{
	GLuint *ptr = &in.Top();
	GLuint *start = &in[0];
	
	ptr -= 2;
	while (ptr >= start) {
		const Vec3f &v1 = vertex[(int)ptr[0]].position;	
		const Vec3f &v2 = vertex[(int)ptr[1]].position;
		const Vec3f &v3 = vertex[(int)ptr[2]].position;
		
		if (glm::length(v1 - v2) < 0.000001f
		|| glm::length(v3 - v2) < 0.000001f
		|| glm::length(v3 - v1) < 0.000001f) {
			in.Remove(ptr-start, 3);
		}
		
		ptr-=3;
	}
}

void MeshData::RemoveUselessTriangles()
{
	ASSERT(HasMirror());

	int before = index.GetCount()/3;
	if (submesh.GetCount() == 1) {
		_RemoveUselessTriangles(vertex, index);
		submesh[0].count = index.GetCount();
		RLOG("Before: " << before << " After: " << index.GetCount()/3);
		return;	
	}
	return;
	Vector<Vector<GLuint> > subindex;
	subindex.SetCount(submesh.GetCount());
	
	for (int i = 0; i < subindex.GetCount(); i++) {
		subindex[i].Insert(0, index, submesh[i].byteoffset/sizeof(GLuint), submesh[i].count);
		_RemoveUselessTriangles(vertex, subindex[i]);
	}

	index.Clear();
	for (int i = 0; i < subindex.GetCount(); i++) {
		submesh[i].byteoffset = index.GetCount()*sizeof(GLuint);
		submesh[i].count = subindex[i].GetCount();
		index.Append(subindex[i]);
	}

	RLOG("Before: " << before << " After: " << index.GetCount()/3);
}

void MeshData::Serialize(Stream &s)
{
	ASSERT(s.IsLoading() || HasMirror());
	int version = 2;
	s % version;
	s % vertex % index % bounds % submesh;
	if (version >= 2)
		s % tangents;
}

bool Mesh::Load(const String &name, bool cache, bool mirror)
{
	String file = GetFullPath(MESH_PATH, name);
	ASSERT(cache || mirror);
	if (SetResource(file)) {
		if (cache && !Cache())
			GLError(Format("VBO creation failed for mesh %s", file));
		if (mirror && HasMirror())
			LoadMesh(file);
		submesh <<= Get()->submesh;
		return true;
	}
	if (!LoadMesh(file))
		return false;
	MeshData *d = Get();
	if (cache && !Cache())
		GLError(Format("VBO creation failed for mesh %s", file));
	if (!mirror) {
		d->vertex.Clear();
		d->index.Clear();
	}
	submesh <<= Get()->submesh;
	return true;
}

bool Mesh::Cache(bool force)
{
	Chk();
	MeshData *d = Get();
	if (d->HasBuffer()) {
		if (!force)	return true;
		d->buffer.Clear();	
	}
	d->buffer.vbo = CreateVBO(d->vertex);
	d->buffer.ibo = CreateIBO(d->index);
	if (d->HasTangents())
		d->buffer.tbo = CreateVBO(d->tangents);
	return d->HasBuffer();	
}

bool Mesh::LoadMesh(const String &file)
{
	MeshData *d = Get();
	if (d) {
		d->vertex.Clear();
		d->index.Clear();
		d->submesh.Clear();
	}
	if (!FileExists(file))
		return GLErrorF(Format("Mesh file %s does not exist", file));
	if (!d)
		d = &CreateResource(file);	
	
	String ext = ToLower(GetFileExt(file));
	bool result = false;
	if (ext == ".bin")
		result = LoadBIN(file);
	else if (ext == ".obj")
		result = LoadOBJ(file);
	
	return result;
}

struct FaceV : public Moveable<FaceV> {
	int p, t, n;
	int ix;
	virtual bool operator==(const FaceV &f) { return p == f.p && t == f.t && n == f.n; }
};

bool Mesh::LoadOBJ(String file)
{
	String path = GetFullPath(MESH_PATH, file);
	FileIn fin(path);
	if (!fin.IsOpen()) return false;
	
	MeshData &d = *Get();
		
	int indexoff = d.vertex.GetCount();
	SubMesh *sub = &d.submesh.Add();
	sub->count = 0;
	sub->byteoffset = 0;

	Vector<Vec3f> v1;
	Vector<Vec2f> t1;
	Vector<Vec3f> n1;
	Vector<FaceV> f1;
	Vector<GLuint> poly;
	bool texcoords = false;
	bool normals = false;
	try {
		while (!fin.IsEof()) {
			String s = fin.GetLine();
			CParser cp(s);
			if (cp.IsChar('#') || !cp.IsId())
				continue;
			String id = cp.ReadId();
			if (id == "v") {
				// Z/Y swapped
				Vec3f &p = v1.Add();
				if (!cp.IsDouble()) return false;
				p.x = (float)cp.ReadDouble();
				if (!cp.IsDouble()) return false;
				p.z = (float)cp.ReadDouble();
				if (!cp.IsDouble()) return false;
				p.y = (float)cp.ReadDouble();
//				ASSERT(p.Length() > 0.0005);
			}			
			else if (id == "vt") {
				Vec2f &t = t1.Add();
				if (!cp.IsDouble()) return false;
				t.x = (float)cp.ReadDouble();
				if (!cp.IsDouble()) return false;
				t.y = (float)cp.ReadDouble();
				texcoords = true;
			}
			else if (id == "vn") {
				// Z/Y swapped
				Vec3f &n = n1.Add();
				if (!cp.IsDouble()) return false;
				n.x = (float)cp.ReadDouble();
				if (!cp.IsDouble()) return false;
				n.z = (float)cp.ReadDouble();
				if (!cp.IsDouble()) return false;
				n.y = (float)cp.ReadDouble();
				normals = true;
			}
			else if (id == "f") {
				ASSERT(sub);
				if (!cp.IsInt()) return false;				
				poly.SetCount(0);
				while (cp) {
					FaceV f;
					f.p = cp.ReadInt();
					if (texcoords && cp.Char('/')) {
						if (!cp.IsInt()) return false;
						f.t = cp.ReadInt();
					}
					if (normals && cp.Char('/')) {
						if (!cp.IsInt()) return false;
						f.n = cp.ReadInt();
					}
					
					bool found = false;
					for (int i = 0; i < f1.GetCount(); i++) {
						if (f == f1[i]) {
							f.ix = f1[i].ix;
							found = true;
							break;	
						}
					}
					if (!found) {
						f.ix = d.vertex.GetCount();
						VertexPNT &v = d.vertex.Add();
						v.position 	= v1[f.p-1];	
						v.normal = normals ? n1[f.n-1] : Vec3f(0.0f, 0.0f, 0.0f);
						v.texture0 	= texcoords ? t1[f.t-1] : Vec2f(0.0f, 0.0f);
						f1.Add(f);						
					}
					ASSERT(f.ix < d.vertex.GetCount());
					poly.Add(f.ix);
				}
				if (poly.GetCount() == 3) {
					d.index.Add(poly[0]);	
					d.index.Add(poly[1]);
					d.index.Add(poly[2]);
					sub->count += 3;
				}
				else {
					// Polygon - create two triangles
					// Ensure correct winding
					Vec3f a = d.vertex[(int)poly[1]].position - d.vertex[(int)poly[0]].position;
					Vec3f b = d.vertex[(int)poly[2]].position - d.vertex[(int)poly[0]].position;
					float dot = glm::dot(a, b);
					if (dot < 0.0f) {
						for (int i = 2; i < poly.GetCount(); i++) {
							d.index.Add(poly[0]);	
							d.index.Add(poly[i]);
							d.index.Add(poly[i-1]);								
						}
					}
					else {
						for (int i = 2; i < poly.GetCount(); i++) {
							d.index.Add(poly[0]);	
							d.index.Add(poly[i]);
							d.index.Add(poly[i-1]);								
						}						
					}						
					sub->count += (poly.GetCount() - 2)*3;						
				}
			}	
			else if (id == "usemap") {
				ASSERT(sub);
				String map = cp.GetPtr();
				try {
					sub->material.SetUnit0(map);
				}
				catch (Exc e) {
					NEVER(); // Add sys logging
				}
			}
			else if (id == "g") {
				int index = sub->byteoffset + sub->count*sizeof(int);
				if (index)
					sub = &d.submesh.Add();
				f1.Clear();
				sub->count = 0;
				sub->byteoffset = index;
			}			
		}
	} catch (Exc e) {
		NEVER();
		return false;
	}
	if (!normals)
		d.CreateNormals(false);
	d.index.Shrink();
	d.vertex.Shrink();
	d.CalcBounds();	
	return d.index.GetCount() && d.vertex.GetCount();
}

bool Mesh::LoadBIN(String file)
{
	String path = GetFullPath(MESH_PATH, file);
	FileIn fin(path);
	if (!fin.IsOpen()) return false;	
	
	fin.LoadThrowing();
	try {
		fin.Magic(MAGIC);
		fin % *Get();
		fin.Magic(MAGIC);	
	} catch (Exc e) {
		return false;
	}
	return true;	
}

bool Mesh::SaveBIN(String file)
{
	Chk();
	FileOut fout(GetFullPath(MESH_PATH, file));
	if (!fout.IsOpen()) return false;

	try {
		fout.Magic(MAGIC);
		fout % *Get();
		fout.Magic(MAGIC);
	} catch (Exc e) {
		return false;
	}
	return true;
}

Vector3f _CalculateTriangleTangents(const Vector<Vertex> &vertex, const int *ptr, const int *end)
{
	Vector3f out;
	Vector<float> cnt;
	
	out.SetCount(vertex.GetCount());
	memset(&out[0], 0, sizeof(float)*3*out.GetCount());
	cnt.SetCount(vertex.GetCount(), 0.0f);

	++end;	
	while (ptr < end) {
		int a = *ptr++;
		int b = *ptr++;
		int c = *ptr++;
		Vec3f tangent = CalculateTriangleTangent(vertex[a], vertex[b], vertex[c]);
		
		out[a] += tangent;
		++cnt[a];
		out[b] += tangent;
		++cnt[b];
		out[c] += tangent;
		++cnt[c];
	}
	for (int i = 0; i < out.GetCount(); i++)
		out[i] /= cnt[i];		
	return out;
}

Vector3f CalculateTriangleTangents(const Vector<Vertex> &vertex, const Vector<GLuint> &index)
{
	if (!index.GetCount()) return Vector3f();
	ASSERT(index.GetCount() % 3 == 0);
	return _CalculateTriangleTangents(vertex, (int*)&index[0], (int*)&index.Top());
}

Vector3f CalculateTriangleTangents(const Vector<Vertex> &vertex, const Vector<Triangle32> &index)
{
	if (!index.GetCount()) return Vector3f();
	return _CalculateTriangleTangents(vertex, &index[0].a, &index.Top().c);
}


Vec3f CalculateTriangleTangent(const Vertex &v1, const Vertex &v2, const Vertex &v3)
{
   	Vec3f edge1 = v2.position - v1.position;
   	Vec3f edge2 = v3.position - v1.position;
   	Vec2f edge1uv = v2.texture0 - v1.texture0;
   	Vec2f edge2uv = v3.texture0 - v1.texture0;

//   	ASSERT(edge1uv.y * edge2uv.x - edge1uv.x * edge2uv.y != 0.0f);
   	return glm::normalize(edge1 * -edge2uv.y + edge2 * edge1uv.y);
    // bitangent = (edge1 * -edge2uv.x + edge2 * edge1uv.x) * mul;
}

