#include "UppGL.h"

bool FrameBuffer::Create(Size _sz, bool depth_buffer)
{	
	Size maxsz = Texture::GetMaxSize();
	sz.cx = min(_sz.cx, maxsz.cx);
	sz.cy = min(_sz.cy, maxsz.cy);	
	
	glGenFramebuffersEXT(1, &fbo);
	Activate();
	if (!fbo) return GLErrorF(Format("FBO creation failed (%d, %d, %s)", sz.cx, sz.cy, depth_buffer ? "true" : "false"));
	GLLog(Format("FBO created (%d, %d, %s)", sz.cx, sz.cy, depth_buffer ? "depth_rbo" : "depth_texture"));
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo);	

	if (depth_buffer) {
        CreateDepthRBO();                     
        AttachDepthRBO();
        if (!rbo)
            return GLErrorF(Format("RBO creation failed (%d, %d, %s)", sz.cx, sz.cy, depth_buffer ? true : false));
	}
	Deactivate();

	GLLogErrors();

	return fbo && (!depth_buffer || rbo);
}


bool FrameBuffer::CreateDepthRBO()
{
	glGenRenderbuffersEXT(1, &rbo);
	if (!rbo) return false;
	glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, rbo);
	glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT, sz.cx, sz.cy);
	glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, 0);	
	
	GLLogErrors();
	return rbo;	
//  glRenderbufferStorageMultisampleEXT(GL_RENDERBUFFER_EXT, 1, GL_RGBA, fbo_width, fbo_height);
}

void FrameBuffer::DestroyDepthRBO()
{
	if (rbo)
		glDeleteRenderbuffersEXT(1, &rbo);
	rbo = 0;	
}

bool FrameBuffer::AttachDepthRBO()
{
	ASSERT(fbo && rbo);
	if (!fbo || !rbo) return false;
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo);
	glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, rbo);
	
	GLLogErrors();
	return true;
}

bool FrameBuffer::DetachDepthRBO()
{
	ASSERT(fbo);
	if (!fbo) return false;
	glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, 0);
	return true;
}

bool FrameBuffer::Activate() const
{
	ASSERT(fbo);
	if (!fbo) return false;
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo);
	return true;
}

void FrameBuffer::Deactivate() const
{
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
}

bool FrameBuffer::SetColorTexture(const RenderTexture &t)
{
	ASSERT(fbo && t.Id());
	if (!fbo) return false;
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fbo);
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT,
	                          GL_TEXTURE_2D, t.Id(), 0);
	colorattached = true;	                          
	return true;
}

bool FrameBuffer::SetDepthTexture(const RenderTexture &t)
{
	ASSERT(fbo && t.Id());
	if (!fbo) return false;
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT,
	                          GL_TEXTURE_2D, t.Id(), 0);
	depthattached = true;
	return true;	
}

bool FrameBuffer::DetachColor()
{
	ASSERT(fbo);
	if (!fbo) return false;
	if (!colorattached) return true;
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT,
	                          GL_TEXTURE_2D, 0, 0);
	colorattached = false;
	return true;
}

bool FrameBuffer::DetachDepth()
{
	ASSERT(fbo);
	if (!fbo) return false;
	if (!depthattached) return true;
	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT,
	                          GL_TEXTURE_2D, 0, 0);
	DetachDepthRBO();
	depthattached = false;
	return true;
}

bool FrameBuffer::IsComplete() const
{
	GLenum r = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if (r == GL_FRAMEBUFFER_COMPLETE_EXT) return true;
	String s;
	switch (r) {
        case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
            s = "FrameBuffer incoplete attachments";
            break;
        case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
            s = "FrameBuffer incoplete missing attachment";
            break;
        case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
            s = "FrameBuffer incoplete dimensions";
            break;
        case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
            s = "FrameBuffer incoplete formats";
            break;
        case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
            s = "FrameBuffer incoplete draw buffer";
            break;
        case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
            s = "FrameBuffer incoplete read buffer";
            break;
        case GL_FRAMEBUFFER_COMPLETE_EXT:
            s = "OK";
            break;
        case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
            s = "FrameBuffer unsupported";
            break;
        default:
            s = "FrameBuffer unreckognized error";

	};	
	return GLErrorF(Format("FrameBuffer incomplete: %s", s));
	
}

void FrameBuffer::Clear()
{
	if (fbo) {
		glDeleteFramebuffersEXT(1, &fbo);
		GLLog("FBO destroyed");
	}
	if (rbo)
		glDeleteRenderbuffersEXT(1, &rbo);
	fbo = rbo = 0;
	sz = Null;
}

bool RenderTexture::Create(const String &name, FrameBuffer &_fbo, int _type)
{
	ASSERT(!IsNull(_fbo));
	ASSERT(_type > NULL_TYPE && _type < TYPE_COUNT);
	ASSERT(!ResourceExists(name));
	if (IsNull(_fbo)) return false;
	fbo = &_fbo;	
	type = _type;
	Size sz = fbo->GetSize();
	
	TextureData &t = CreateResource(name);
	glGenTextures(1, &t.id);
	if (!t.id) return GLErrorF(Format("RenderTexture '%s' creation failed", name));
	GLLog(Format("RenderTexture '%s' created", name));
	t.Activate();
	if (type == COLOR_32) {
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);	
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, sz.cx, sz.cy, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL);		
	}
	else if (type == DEPTH_8) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_LUMINANCE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);		
		glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, sz.cx, sz.cy, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	}	
	else if (type == DEPTH_16) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_LUMINANCE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);		
		glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT16, sz.cx, sz.cy, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	}
	else if (type == DEPTH_24) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_LUMINANCE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);		
		glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24, sz.cx, sz.cy, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	}
	else if (type == DEPTH_32) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_DEPTH_TEXTURE_MODE, GL_LUMINANCE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);		
		glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT32, sz.cx, sz.cy, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
		
	}
	if (mipmaps) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glEnable(GL_TEXTURE_2D);
		glGenerateMipmapEXT(GL_TEXTURE_2D);
	}
	GLLogErrors();
	return true;
}

void RenderTexture::GenerateMipMaps()
{
	mipmaps = true;	
}

bool RenderTexture::StartRender(const Rect &r, bool clear)
{
	if (!fbo->Activate()) return false;
	bool color = false;
	
	switch (type) {
		case COLOR_32:	
			if (!fbo->SetColorTexture(*this)) return false;
			color = true;
			break;
		case DEPTH_8:
		case DEPTH_16:
		case DEPTH_24:
		case DEPTH_32:
			glDrawBuffer(GL_NONE);
			if (!fbo->SetDepthTexture(*this)) return false;
			break;
		default:
			return false;
	}
    glReadBuffer(GL_NONE);	
	if (!fbo->IsComplete()) return false;
		
	glPushAttrib(GL_VIEWPORT_BIT | GL_ENABLE_BIT);
	glViewport(r.left, r.top, r.right-r.left, r.bottom-r.top);

	if (color && clear) {
		glClearColor(0.9f, 0.0f, 0.9f, 0.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);	
	}
	else if (clear)
		glClear(GL_DEPTH_BUFFER_BIT);	

	GLLogErrors();
	return fbo->IsComplete();
}

void RenderTexture::EndRender()
{
	glPopAttrib();
	
	fbo->Deactivate();	
	if (type != COLOR_32)
		glDrawBuffer(GL_BACK);
	glReadBuffer(GL_BACK);
	
	if (mipmaps) {
		Activate();
		glEnable(GL_TEXTURE_2D);
		glGenerateMipmapEXT(GL_TEXTURE_2D);
	}
}
