#include "UppGL.h"

#ifndef flagDUMMY_SHADERPROGRAM
GLuint 							ShaderProgram::currentid = -1;
VectorMap<String, ShaderHandle>	ShaderProgram::vertshader;
VectorMap<String, ShaderHandle>	ShaderProgram::fragshader;
Vector<String>					ShaderProgram::defines;
bool							ShaderProgram::definesdirty = true;
String 							ShaderProgram::definestring;	

bool ShaderProgram::ReloadAll()
{
	GLLog("Reloading Shaders");
	for (int i = 0; i < vertshader.GetCount(); i++)
		glDeleteObjectARB(vertshader[i]);
	vertshader.Clear();
	for (int i = 0; i < fragshader.GetCount(); i++)
		glDeleteObjectARB(fragshader[i]);
	fragshader.Clear();

	ShaderProgram dummy;
	for (int i = 0; i < ResourceCount(); i++) {
		ShaderProgramData &p = GetResource(i);
		p.id = dummy.LoadShaderProgram(p.vertexfile, p.fragmentfile);
		if (p.id <= 0)
			return GLErrorF(Format("Unable to load shader (%s, %s)", p.vertexfile, p.fragmentfile));
		dummy.Set(&p);
		dummy.Activate();
		dummy.ResetUniforms();
	}
	GLLog("Reload Complete");
	return true;
}

bool ShaderProgram::Load(const String &vert, const String &frag)
{
	// Have we already created this program?
	String progname = CreateProgramName(vert, frag);
	if (SetResource(progname)) 
		return true;

	bool success = false;
	if (ShaderSupport()) {
		success = CreateShaderProgram(vert, frag);
		if (!success)
			GLError(Format("Failed to load shader %s:%s", vert, frag));
	}
	else
		GLError("Shaders not supported");
	return success;
}

String ShaderProgram::CreateProgramName(const String &vert, const String &frag)
{
	return vert + ';' + frag;
}

bool ShaderProgram::CreateShaderProgram(const String &vert, const String &frag)
{
	ShaderHandle id = LoadShaderProgram(vert, frag);
	if (id > 0) {
		ShaderProgramData &p = CreateResource(CreateProgramName(vert, frag));
		p.vertexfile = vert;
		p.fragmentfile = frag;
		p.id = id;
		
		// Setup vertex attribs
		int cnt = 0;
		glGetIntegerv(GL_MAX_VERTEX_ATTRIBS, &cnt);
		if (VERTEX_TANGENT_ATTRIB < cnt)
			glBindAttribLocationARB(p.id, VERTEX_TANGENT_ATTRIB, "vTangent");
		
		Set(&p);
		return true;
	}
	return false;
}

ShaderHandle ShaderProgram::LoadShaderProgram(const String &vert, const String &frag)
{
	int ix;

	// What about this vertex shader?
	ShaderHandle verthandle;
	ix = vertshader.Find(vert);
	if (ix < 0) {
		verthandle = CreateVertexShader(vert);
		if (verthandle <= 0) return false;	
		vertshader.Add(vert, verthandle);
	}
	else
		verthandle = vertshader[ix];
	// Fragment shader?
	ShaderHandle fraghandle;
	ix = fragshader.Find(frag);
	if (ix < 0) {
		fraghandle = CreateFragmentShader(frag);
		if (fraghandle <= 0) return false;	
		fragshader.Add(frag, fraghandle);	
	}
	else
		fraghandle = fragshader[ix];
	// Create a new program entry
	return CreateShaderProgram(verthandle, fraghandle);
}

ShaderHandle ShaderProgram::CreateVertexShader(const String &name)
{
	GLLog(Format("Loading vertex shader '%s'", name));
	return CreateShaderPart(name, GL_VERTEX_SHADER_ARB);	
}

ShaderHandle ShaderProgram::CreateFragmentShader(const String &name)
{
	GLLog(Format("Loading fragment shader '%s'", name));
	return CreateShaderPart(name, GL_FRAGMENT_SHADER_ARB);
}

ShaderHandle ShaderProgram::CreateShaderPart(const String &name, GLenum type)
{
	GLint compiled = false;
	// Read file
	String s = LoadFile(GetFullPath(SHADER_PATH, name));	
	if (IsNull(s)) {
		GLError(Format("Could not read shader file %s", name));
		return 0;
	}
	if (!PreProcess(s, name)) {
		GLError("Shader preprocessing failed");
		return 0;
	}
	// Create shader object
    ShaderHandle handle = glCreateShaderObjectARB(type);
	// Set #defines
	if (definesdirty) {
		StringStream out;
		definestring.Clear();
		if (defines.GetCount()) {
			for (int i = 0; i < defines.GetCount(); i++)
				out << "\n#define " << defines[i];
			out.PutEol();
			definestring = out;
		}
	}
	// Set shader code and compile
	const char *ptr[2];
	GLint		len[2];
	if (definestring.GetLength()) {
		ptr[0] = definestring.Begin();
		len[0] = definestring.GetLength();
		ptr[1] = s.Begin();
		len[1] = s.GetLength();
	    glShaderSourceARB(handle, 2, ptr, len);
	}
	else {
		ptr[0] = s.Begin();
		len[0] = s.GetLength();
		ptr[1] = NULL;
		len[1] = 0;
	    glShaderSourceARB(handle, 1, ptr, len);			
	}
	GLRLogErrors();
    glCompileShaderARB(handle);
    GLRLogErrors();
	// Check compilation success
    glGetObjectParameterivARB(handle, GL_OBJECT_COMPILE_STATUS_ARB, 
                               &compiled);
    if(!compiled) {
		char str[4096];
		glGetInfoLogARB(handle, sizeof(str), NULL, str);
		glDeleteObjectARB(handle);
		GLError(Format("Shader compilation failed for %s: %s", name, IsAlpha(str[0]) ? str : "Uknown reason"));
		GLRLogErrors();
		return 0;
	}
	else if (handle <= 0)
		GLError("Shader part creation failed, but no compilation error raised");
	return handle;
}

ShaderHandle ShaderProgram::CreateShaderProgram(ShaderHandle vert, ShaderHandle frag)
{
    GLint linked = false;
    // Create program object
    ShaderHandle handle = glCreateProgramObjectARB();
 	// Attach shaders
	glAttachObjectARB(handle, vert);
    glAttachObjectARB(handle, frag);
	// Link prgram
    glLinkProgramARB(handle);
    // Check link success
    glGetObjectParameterivARB(handle, GL_OBJECT_LINK_STATUS_ARB, &linked);
    if(!linked) {
		char str[4096];
		glGetInfoLogARB( handle, sizeof(str), NULL, str );
		glDeleteObjectARB(handle);
		GLError(Format("Shader program creation failed: %s", str));
	}
	else if (handle <= 0)
		GLError("Shader program creation failed, but no link error raised");
	return handle;
}

struct ShaderProcessor {
	VectorMap<String, String> inlines;
	String rootdir;
	
	ShaderProcessor(String _rootdir) : rootdir(_rootdir) {}
	String PreProcess(String prog, String name);
};

String ShaderProcessor::PreProcess(String prog, String name)
{
	StringStream in(prog);
	StringStream out;
	while (!in.IsEof()) {
		String line = in.GetLine();
		CParser cp(line);
		if (cp.IsChar('#')) {
			cp.GetChar();
			String macro = cp.ReadId();
			if (macro == "include") {
				if (!cp.IsString())
					return GLErrorS(Format("Missing filename after '#include' in '%s'", name));
				String file = cp.ReadString();
				String path = GetFullPath(SHADER_PATH, file);
				String include = LoadFile(path);
				if (include.GetLength() == 0)
					return GLErrorS(Format("Unable to open include file '%s' used in '%s'", file, name));
				include = PreProcess(include, file);
				if (IsNull(include)) return String();
				out.Put(include);
				out.PutCrLf();
			}
			else if (macro == "inline") {
				if (!cp.IsId())
					return GLErrorS(Format("Missing name after '#inline in '%s'", name));
				String name = cp.ReadId();
				StringStream iin;
				bool complete = false;
				while (!in.IsEof()) {
					line = in.GetLine();
					CParser cp1(line);
					if (cp1.IsChar('#')) {
						cp1.GetChar();
						if (cp1.IsId() && cp1.ReadId() == "end_inline") {
							complete = true;
							break;
						}
					}
					iin.PutLine(line);
				}
				if (!complete)
					return GLErrorS(Format("'#inline %s' without '#end_inline'", name));
				inlines.GetAdd(name) = iin;
			}
			else if (macro == "end_inline") {
				return GLErrorS(Format("'#end_inline' without '#inline' in '%s'", name));
			}
			else if (macro == "use_inline") {
				if (!cp.IsId())
					return GLErrorS(Format("Missing name after '#use_inline in '%s'", name));
				String n = cp.ReadId();
				int ix = inlines.Find(n);
				if (ix < 0)
					return GLErrorS(Format("Cannot find inline '%s' in '%s'", n, name));
				String iin = PreProcess(inlines[ix], name);
				if (IsNull(iin)) return String();
				out.Put(iin);
			}
			else
				out.PutLine(line);			
		}
		else
			out.PutLine(line);
	}
	return out;
	
}

bool ShaderProgram::PreProcess(String &prog, String name)
{
	ShaderProcessor s(GetFullPath(SHADER_PATH, ""));
	prog = s.PreProcess(prog, name);
	if (prog.IsEmpty())
		return GLErrorF(Format("Error parsing shader component '%s'", name));
	return true;
}

void ShaderProgram::RemoveDefine(String def)
{
	definesdirty = true; 
	int ix = FindIndex(defines, def);
	if (ix >= 0)
		defines.Remove(ix);
}

bool ShaderProgram::SetUniform(String name, int val) const
{
	ASSERT(!IsEmpty());
	ShaderProgramData &d = *Get();
	int ix = d.uniform.Find(name);
	if (ix < 0) {
		ShaderProgramData::IntUniform &u = d.uniform.Create<ShaderProgramData::IntUniform>(name);
		u.SetAddr(d.id, name);
		ix = d.uniform.GetCount()-1;
	}
	d.uniform[ix].Set(val);
	return (d.uniform[ix].addr >= 0);
}

bool ShaderProgram::SetUniform(String name, float val) const
{
	ASSERT(!IsEmpty());
	ShaderProgramData &d = *Get();
	int ix = d.uniform.Find(name);
	if (ix < 0) {
		ShaderProgramData::FloatUniform &u = d.uniform.Create<ShaderProgramData::FloatUniform>(name);
		u.SetAddr(d.id, name);
		ix = d.uniform.GetCount()-1;
	}
	d.uniform[ix].Set(val);
	return (d.uniform[ix].addr >= 0);
}

bool ShaderProgram::SetUniform(String name, const Vec3f &val) const
{
	ASSERT(!IsEmpty());
	ShaderProgramData &d = *Get();	
	int ix = d.uniform.Find(name);
	if (ix < 0) {
		ShaderProgramData::Vec3Uniform &u = d.uniform.Create<ShaderProgramData::Vec3Uniform>(name);
		u.SetAddr(d.id, name);
		ix = d.uniform.GetCount()-1;
	}
	d.uniform[ix].Set(val);
	return (d.uniform[ix].addr >= 0);
}
#endif

void ShaderProgramData::Uniform::SetAddr(ShaderHandle id, const String &name)
{
	addr = (id >= 0) ? glGetUniformLocation(id, name)
				 : GLErrorI(Format("Uniform address not found: %s", name), -1);
}

void ShaderProgramData::IntUniform::Set(int val)
{
	v = val;
	if (addr >= 0)
		glUniform1iARB(addr, val);
}

void ShaderProgramData::FloatUniform::Set(float val)
{
	v = val;
	if (addr >= 0)
		glUniform1fARB(addr, val);
}

void ShaderProgramData::Vec3Uniform::Set(const Vec3f &val)
{
	v = val;
	if (addr >= 0)
		glUniform3fv(addr, 1, &val.x);
}

void ShaderProgram::ResetUniforms() const
{
	ShaderProgramData &d = *Get();	
	if (d.id >= 0)
		for (int i = 0; i < d.uniform.GetCount(); i++) {
			d.uniform[i].SetAddr(d.id, d.uniform.GetKey(i));
			d.uniform[i].Reset();
		}
}