#include "UppGL.h"

Subdivider::Subdivider(Vec3f *vdata, Vec2f *texdata, Triangle32 *tdata, int vcnt, int tcnt)
{
	Init(vdata, texdata, tdata, vcnt, tcnt);
}

Subdivider::Subdivider(float *vdata, float *texdata, int *idata, int vcnt, int icnt)
{	
	Init((Vec3f *)vdata, (Vec2f *)texdata, (Triangle32 *)idata, vcnt, icnt);
}

void Subdivider::Init(Vec3f *vdata, Vec2f *texdata, Triangle32 *tdata, int vcnt, int tcnt)
{
	vertex.SetCount(vcnt);
	texture.SetCount(vcnt);
	tri.SetCount(tcnt);

	memcpy(&vertex[0], vdata, sizeof(Vec3f)*vcnt);
	memcpy(&texture[0], texdata, sizeof(Vec2f)*vcnt);
	memcpy(&tri[0], tdata, sizeof(Triangle32)*tcnt);	
}

void Subdivider::Divide(int level)
{
	while (level > 0) {
		DoDivide();
		level--;	
	}
}

void Subdivider::DoDivide()
{
	VectorMap<Point, int> newpt;
	int tcnt = tri.GetCount();
	for (int i = 0; i < tcnt; i++) {
		Triangle32 &t = tri[i];
		int d = HalfPt(newpt, t.a, t.b);
		int e = HalfPt(newpt, t.b, t.c);
		int f = HalfPt(newpt, t.c, t.a);
		Triangle32 t1(d, t.b, e);
		Triangle32 t2(f, e, t.c);
		Triangle32 t3(e, f, d);
		t.b = d;
		t.c = f;		
		tri.Add(t1);
		tri.Add(t2);
		tri.Add(t3);
	}
}

int Subdivider::HalfPt(VectorMap<Point, int> &pt, int a, int b)
{
	Point ref = (a < b) ? Point(a, b) : Point(b, a);
	int ix = pt.Find(ref);
	if (ix < 0) {
		Vec3f n = vertex[a] + (vertex[b] - vertex[a])/2.0f;
		Vec2f t = texture[a] + (texture[b] - texture[a])/2.0f;
		ix = vertex.GetCount();
		pt.Add(ref, vertex.GetCount());
		vertex.Add(n);
		texture.Add(t);
		return ix;
	}
	return pt[ix];
}

void Subdivider::ToMesh(MeshData &m)
{
	m.Clear();
	
	m.vertex.SetCount(vertex.GetCount());
	for (int i = 0; i < m.vertex.GetCount(); i++) {
		m.vertex[i].position = vertex[i];
		m.vertex[i].texture0 = texture[i];
	}
	
	m.index.SetCount(tri.GetCount()*3);
	Copy(&m.index[0], (unsigned int*)&tri[0].a, m.index.GetCount());
	
	SubMesh &s = m.submesh.Add();
	s.byteoffset = 0;
	s.count = m.index.GetCount();
	
	m.CreateNormals(true);
}

#define ROOT_HALF 0.707106f

void SetRadius(float radius, Subdivider &s)
{
	Vector<Vec3f> &v = s.Verticies();
	for (int i = 0; i < v.GetCount(); i++)
		v[i] = glm::normalize(v[i]) * radius;
}

Subdivider DivideSphere(float radius, int level)
{
	float vertex[] 	= {	ROOT_HALF, ROOT_HALF, 0.0f,
						ROOT_HALF, -ROOT_HALF, 0.0f,
						-ROOT_HALF, -ROOT_HALF, 0.0f,
						-ROOT_HALF, ROOT_HALF, 0.0f,
						0.0f, 0.0f, ROOT_HALF,
						0.0f, 0.0f, -ROOT_HALF };
	float texture[] = {	0.0f, 0.5f,
						0.25f, 0.5f,
						0.5f, 0.5f,
						0.75f, 0.5f,
						0.5f, 1.0f,
						0.5f, 0.0f };						
	int tri[] 		= { 0, 1, 4,
						1, 2, 4,
						2, 3, 4,
						3, 0, 4,
 						1, 0, 5,
						2, 1, 5,
						3, 2, 5,
						0, 3, 5 };
	Subdivider div(vertex, texture, tri, 6, 8);

	div.Divide(level);
	SetRadius(radius, div);
	return div;			
}

Subdivider DivideDemiSphere(float radius, int level)
{
	float vertex[] 	= {	ROOT_HALF, ROOT_HALF, 0.0f,
						ROOT_HALF, -ROOT_HALF, 0.0f,
						-ROOT_HALF, -ROOT_HALF, 0.0f,
						ROOT_HALF, ROOT_HALF, 0.0f,
						0.0f, 0.0f, ROOT_HALF };
	float texture[] = {	0.0f, 0.0f,
						0.25f, 0.0f,
						0.5f, 0.0f,
						0.75f, 0.0f,
						0.5f, 1.0f };
	int tri[] 		= { 0, 1, 4,
						1, 2, 4,
						2, 3, 4,
						3, 0, 4 };
	Subdivider div(vertex, texture, tri, 5, 4);
	div.Divide(level);
	SetRadius(radius, div);
	return div;				
}
