#ifdef MULTITHREADED
#error NOT THREAD SAFE
#endif

#include <Core/Core.h>
using namespace Upp;

struct ManagedResourceData 
{
private:
	int refcount;
	String name;
protected:
	ManagedResourceData() : refcount(0) { }
public:	
	virtual void Free() 	{ }
	void Retain()			{ ++refcount; }
	void Release()			{ --refcount; }
	int  RefCount() const	{ return refcount; }
	String Name() const		{ return name; }
	
	// Note: Init name should only be called ONCE and only by ManagedResource
	void InitName(const String &_name) { ASSERT(name.IsEmpty()); name = _name; }
};

// T must be derived from ManagedResourceData
template <class T>
class ManagedResource
{
	static ArrayMap<String, T> managed;
	T *data;
protected:
	ManagedResource() : data(NULL)					{ }
	virtual ~ManagedResource() 						{ Release(); }
	
	void 	Chk() const								{ ASSERT(data); }
	void 	Set(const ManagedResource<T> &m)		{ Set(m.Get()); }
	void 	Set(T *t)								{ if (t) t->Retain(); Release(); data = t; }
	void 	Set(const T *t)							{ Set(const_cast<T*>(t)); }
	T *		Get() const								{ return data; }
	
	T &		CreateResource(const String &name); 	
	bool	SetResource(const String &name); 
	bool 	ResourceExists(const String &name) const{ return FindResource(name) >= 0; }
	
	static T &	GetResource(int i) 					{ return managed[i]; }
	static int	ResourceCount()						{ return managed.GetCount(); }
	
	virtual T*	 GetNull() const					{ return NULL; }	
public:
	ManagedResource<T> &operator=(const ManagedResource<T> &m)			{ Set(m); return *this; }
	bool 			 	operator==(const ManagedResource<T> &m) const	{ return data == m.data; }
	bool 			 	operator!=(const ManagedResource<T> &m) const	{ return data != m.data; }
	
	String	 Name() const							{ Chk(); return data->Name(); }
	
	void 	 Clear()								{ Set(GetNull()); }	
	bool	 IsEmpty() const						{ return data == GetNull(); }
	bool	 IsNullInstance() const					{ return data == GetNull(); }
	unsigned GetHashValue() const					{ return Upp::GetHashValue((unsigned)(uintptr_t)data); }
private:
	void Release();
	int	 FindResource(const String &name) const		{ return managed.Find(name); }
};
template <class T>
ArrayMap<String, T> ManagedResource<T>::managed;

template <class T>
class ManagedResourceNeverNull : public ManagedResource<T>
{
	static T null_object;
public:
	ManagedResourceNeverNull() 						{ null_object.Retain(); Set(&null_object); }
protected:
	virtual T*	 GetNull() const					{ return &null_object; }		
};
template <class T>
T ManagedResourceNeverNull<T>::null_object;


template <class T>
T & ManagedResource<T>::CreateResource(const String &name)
{ 
	ASSERT(!ResourceExists(name)); 
	ASSERT(name.GetLength());
	T &t = managed.Add(name);
	t.InitName(name); 
	Set(&t);
	return t;
}

template <class T>
bool ManagedResource<T>::SetResource(const String &name)
{
	Clear();
	int ix = FindResource(name); 
	if (ix >= 0) {
		Set(&managed[ix]);
		return true;
	}
	return false;
}

bool IsExiting();

template <class T>
void ManagedResource<T>::Release()
{
	if (data && !IsExiting()) {
		data->Release(); 
		if (data->RefCount() <= 0) {
			data->Free();
			int ix = FindResource(data->Name());
			if (ix >= 0)
				managed.Remove(ix);
		}
	}
}