#include "UppGL.h"

#define HASH_VAL(x) memcpy(ptr, &x, sizeof(x)); ptr += sizeof(x)
#define HASH_PTR(x) { int i = (int)x; memcpy(ptr, &i, sizeof(int)); ptr += sizeof(int); }
#define HASH_L(x) 	{ unsigned i = x; memcpy(ptr, &i, sizeof(unsigned)); ptr += sizeof(unsigned); }

GLMaterial::GLMaterial()
{
	alpha = 1.0f;
	texture_blend0 = 0;
	shininess = 0.5f;
	UpdateHash();
}

void GLMaterial::Activate() const
{
	static int _texture_blend0 = -1;
	culling.Activate();
	shader.Activate();
	unit0.Activate(UNIT0_UNIT);
	if (!UseShaders()) {
		if (texture_blend0 != _texture_blend0) {
			switch (texture_blend0) {
				case 1:
					// Blending src colour and imput texture based on input texture's alpha
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_INTERPOLATE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_PREVIOUS);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_TEXTURE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE2_RGB, GL_TEXTURE);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND2_RGB, GL_ONE_MINUS_SRC_ALPHA);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PREVIOUS);
					break;
				case 2:
					// Blend src color and input texture retaining original alpha
					// Used for translucent materials
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_MODULATE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_PREVIOUS);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_TEXTURE);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_PRIMARY_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);			
					break;			
				case 3:
					// Used for alpha masking. Blend RGB, replace alpha
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_MODULATE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB, GL_PREVIOUS);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE1_RGB, GL_TEXTURE);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_REPLACE);
					glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_ALPHA, GL_TEXTURE);
					glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);			
					break;
				case 4:
					// Full replace
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
					break;
				default:
				case 0:
					// Blend RGB
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
					break;
			}
			_texture_blend0 = texture_blend0;
		}
		alpha_test.Activate();
	}
	unit1.Activate(UNIT1_UNIT);
	
	color.Activate(alpha);
	glMaterialf(GL_FRONT, GL_SHININESS, shininess);
}

byte GLMaterial::GetStateFlags() const
{
	return (culling * CULLING_HASH + IsTranslucent() * TRANSLUCENT_HASH + alpha_test*ALPHA_TEST_HASH);
}

GLMaterial & GLMaterial::operator=(const GLMaterial &m)
{
	culling = m.culling;
	shader = m.shader;
	unit0 = m.unit0;
	unit1 = m.unit1;
	color = m.color;
	alpha = m.alpha;
	texture_blend0 = m.texture_blend0;
	shininess = m.shininess;
	alpha_test = m.alpha_test;
	memcpy(hash, m.hash, HASH_SIZE);
	qhash = m.qhash;
	return *this;
}

void GLMaterial::UpdateHash()
{
	byte *ptr = hash;
	byte state_flags = GetStateFlags();
	HASH_VAL(state_flags);
	HASH_L(shader.GetHashValue());
	HASH_L(unit0.GetHashValue());
	HASH_L(unit1.GetHashValue());
	qhash = memhash(hash, HASH_SIZE);
}

void GLMaterial::Clear()
{
	shader.Clear();
	unit0.Clear();
	unit1.Clear();
	culling = GLCullState();
	alpha_test = GLAlphaTestState();
	shininess = 0.5f;
	color = Colorf();
	UpdateHash();
}

String GLMaterial::TrimField(const String &in) const
{
	return (in[0] == '\"') ? String(in.Begin()+1, in.GetLength()-2) : in;
}

bool GLMaterial::Load(String file, bool overwrite_unit0, bool overwrite_unit1)
{
	const MaterialDef &def = GetMaterialDef(file);
	
	if (UseShaders()) {
		if (!def.vertexshader.GetLength() || !def.fragshader.GetLength())
			return false;
		else if (!shader.Load(def.fragshader, def.vertexshader))
			return false;
		for (int i = 0; i < def.params.GetCount(); i++) {
			const String &s = def.params.GetKey(i);
			const Value &v = def.params[i];	
			if (!IsNull(v)) {
				if (v.GetType() == INT_V)
					shader.SetUniform(s, (int)v);
				else if (v.GetType() == DOUBLE_V)
					shader.SetUniform(s, (int)v);
				else {
					NEVER();
				}
			}
		}
	}
	if (def.unit0.GetLength() && (overwrite_unit0 || unit0.IsNullInstance()) && !unit0.Load(def.unit0))
		return false;
	if (def.unit1.GetLength() && (overwrite_unit1 || unit1.IsNullInstance()) && !unit1.Load(def.unit1))
		return false;
		
	color = def.color;
	alpha = color.A();
	texture_blend0 = def.texture_blend0;
	culling = def.culling;
	
	UpdateHash();
	return true;	
}

const GLMaterial::MaterialDef & GLMaterial::GetMaterialDef(const String &mat)
{
	String path = GetFullPath(MATERIAL_PATH, mat);
	int cacheix = cache.Find(path);
	if (cacheix < 0) {
		VectorMap<String, String> ini = LoadIniFile(path);
		if (!ini.GetCount())
			GLError(Format("Unable to open material %s", path));
		MaterialDef &md = cache.Add(path);
		
		String t; 
		// Shader
		md.fragshader = TrimField(ini.Get("VERTEX_SHADER", String()));
		md.vertexshader = TrimField(ini.Get("FRAGMENT_SHADER", String()));
		if (md.fragshader.GetLength() && md.vertexshader.GetLength()) {
			for (int i = 0; i < ini.GetCount(); i++) {
				if (ini.GetKey(i) == "UNIFORM_NAME") {
					if (i >= ini.GetCount()-1)
						GLError(Format("Bad uniform definition in %s: No type specified", path));
					else {
						String name = TrimField(ini[i]);
						String type = TrimField(ini.GetKey(i+1));
						bool nullerr = false;
						Value v;
						bool fail = false;
						if (type == "UNIFORM_INT")
							v = ScanInt(ini[i+1]);
						else if (type == "UNIFORM_FLOAT")
							v = ScanDouble(ini[i+1]);
						else if (type == "UNIFORM_VEC3")
						    v = ini[i+1];
						else {
							GLError(Format("Bad uniform definition in %s: Unrecognised type", path));
							fail = true;
					 	}
						if (!fail && v.IsNull()) {
							GLError(Format("Bad uniform definition in %s: Unparsable value", path));
							fail = true;
						}
						if (!fail)
							md.params.Add(name, v);
					}
				}
			}
		}		
		// Texture units
		md.unit0 = TrimField(ini.Get("UNIT_0", String()));
		if (UseBumpMapping())
			md.unit1 = TrimField(ini.Get("BUMP_MAP", String()));
		else
			md.unit1 = TrimField(ini.Get("UNIT_1", String()));
		// Color
		t = ini.Get("COLOR", String());
		if (t.GetLength()) {
			int raw = StrColorInt(t);
			if (!IsNull(raw))
				md.color.SetRaw(raw);
		}		
		// Shininess
		t = ini.Get("SHININESS", String());
		if (t.GetLength())
			md.shininess = (float)Nvl(StrDbl(t), 0.5);
		// Texture blend mode 0 (non-shader path)
		t = ini.Get("TEXTURE_BLEND_0", String());
		if (t.GetLength())
			md.texture_blend0 = Nvl(ScanInt(t), 0);		
		// Texture blend mode 0 (non-shader path)
		t = ini.Get("CULLING", String());
		if (t.GetLength())
			md.culling = Nvl(ScanInt(t), 1);
		// Alpha testing (non-shader path)
		t = ini.Get("ALPHA_TEST", String());
		if (t.GetLength())
			md.alpha_test = Nvl(ScanInt(t), 0);
		return md;
	}
	return cache[cacheix];
}

bool GLMaterial::Save(String file)
{
	String path = GetFullPath(MATERIAL_PATH, file);
	FileOut fout(file);
	if (!fout.IsOpen()) return false;
	return Save(fout);
}

bool GLMaterial::Save(Stream &s)
{
	if (!IsNull(shader)) {
		s << "VERTEX_SHADER=" << shader.VertexFile();
		s << "FRAGMENT_SHADER=" << shader.FragmentFile();
	}	
	if (!IsNull(unit0))
		s << "UNIT_0=" << unit0.Name();
	if (!IsNull(unit1))
		s << "UNIT_1=" << unit1.Name();
	if (!IsNull(color))
		s << "COLOR=0x" << FormatIntHex(color.Raw(), 10);
	if (!culling)
		s << "CULL_FACE=0";
	return true;
}

VectorMap<String, GLMaterial::MaterialDef > GLMaterial::cache;