#ifndef UNIT0_UNIT
#define UNIT0_UNIT 0
#endif

#ifndef UNIT1_UNIT
#define UNIT1_UNIT 1
#endif

#ifndef SHADOW_MAP_UNIT
#define SHADOW_MAP_UNIT 2
#endif

class GLMaterial : public Moveable<GLMaterial>
{
public:
	struct MaterialDef : public Moveable<MaterialDef> {
		MaterialDef() :
			/* color(1.0f, 1.0f, 1.0f, 1.0f), */
			shininess(0.5f),
			texture_blend0(0),
			culling(1)
			{}
		String fragshader;
		String vertexshader;
		VectorMap<String, Value> params;
		String unit0;
		String unit1;
		Colorf color;
		float shininess;
		int texture_blend0; // Fixed function only
		bool culling;
		bool alpha_test;	// Fixed function only
	};
private:
	ShaderProgram 	shader;
	GLCullState		culling;
	GLAlphaTestState alpha_test;
	Texture			unit0;
	Texture			unit1;
	Colorf			color;
	float 			alpha;
	float 			shininess;
	byte 			texture_blend0;
	enum { 
		HASH_SIZE = 4+1+4+4, 
		CULLING_HASH = 0x2,
		ALPHA_TEST_HASH = 0x1,
		TRANSLUCENT_HASH = 1 << 7, 
		UNIT_COUNT = 2 
	};
	byte 			hash[HASH_SIZE];
	unsigned		qhash;
	static VectorMap<String, MaterialDef> cache;
	
	byte 	GetStateFlags() const;
	String 	TrimField(const String &in) const;
	
	const MaterialDef &GetMaterial(const String &mat);
public:
	GLMaterial();
	GLMaterial(const GLMaterial &m) { *this = m; }
	
	void Activate() const;
	
	bool  Load(String file, bool overwrite_unit0 = true, bool overwrite_unit1 = true);
	bool  Save(String file);
	bool  Save(Stream &s);
	
	const ShaderProgram &	GetShader() const 		{ return shader; }
	const Texture &			GetUnit0() const		{ return unit0; }
	const Texture &			GetUnit1() const		{ return unit1; }
	bool 					GetCulling() const 		{ return culling; }
	const Colorf &			GetColor() const		{ return color; }
	float					GetShininess() const	{ return shininess; }
	bool					IsTranslucent() const	{ return color.A() < 1.0f; }
		
	void	SetShader(const ShaderProgram &s) 		{ shader = s; UpdateHash(); }
	void	SetShader(const String &vert, const String &frag) { shader.Load(vert, frag); UpdateHash(); }

	void	SetUnit0(const Texture &t)				{ unit0 = t; UpdateHash(); }
	void	SetUnit0(const String &name)			{ unit0.Load(name); UpdateHash(); }
	void	ClearUnit0()							{ unit0.Clear(); UpdateHash(); }

	void	SetUnit1(const Texture &t)				{ unit1 = t; UpdateHash(); }
	void	SetUnit1(const String &name)			{ unit1.Load(name); UpdateHash(); }
	void	ClearUnit1()							{ unit1.Clear(); UpdateHash(); }

	const MaterialDef &GetMaterialDef(const String &mat);

	void	SetBlendMode0(int mode)					{ texture_blend0 = mode; }

	void 	SetCulling(bool c) 						{ culling = c; UpdateHash(); }
	void	SetColor(const Colorf &c)				{ color = c; alpha = color.A(); UpdateHash(); }	
	void	SetAlpha(float _alpha)					{ alpha = color.A() * _alpha; }
	void	SetShininess(float v)					{ shininess = v; }
		
	bool	operator<(const GLMaterial &m) const	{ return memcmp(hash, m.hash, HASH_SIZE) < 0; }
	bool	operator>(const GLMaterial &m) const	{ return memcmp(hash, m.hash, HASH_SIZE) > 0; }
	bool	operator==(const GLMaterial &m) const	{ return memcmp(hash, m.hash, HASH_SIZE) == 0 && m.color == color && shininess == m.shininess && alpha == m.alpha; }
	bool	operator!=(const GLMaterial &m) const	{ return memcmp(hash, m.hash, HASH_SIZE) != 0 || m.color != color || shininess != m.shininess; }

	GLMaterial &operator=(const GLMaterial &m);

	void		Clear()	;

	unsigned 	GetHashValue() const				{ return qhash; }
	void 		UpdateHash();
	
	friend class MaterialLoader;	
};
