///////////////////////////////////////////////////////////////////////////////////////////////////
// OpenGL Mathematics Copyright (c) 2005 - 2009 G-Truc Creation (www.g-truc.net)
///////////////////////////////////////////////////////////////////////////////////////////////////
// Created : 2005-12-21
// Updated : 2008-11-27
// Licence : This source is under MIT License
// File    : glm/gtx/quaternion.h
///////////////////////////////////////////////////////////////////////////////////////////////////
// Dependency:
// - GLM core
///////////////////////////////////////////////////////////////////////////////////////////////////
// ToDo:
// - Study constructors with angles and axis
// - Study constructors with vec3 that are the imaginary component of quaternion
///////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef glm_gtx_quaternion
#define glm_gtx_quaternion

// Dependency:
#include "../glm.hpp"

namespace glm
{
	namespace detail
	{
		//! \brief Template for quaternion. (From GLM_GTX_quaternion extension)
		template <typename valType> 
		class tquat
		{
		public:
			valType x, y, z, w;

			// Constructors
			tquat();
			explicit tquat(valType const & s, tvec3<valType> const & v);
			explicit tquat(valType const & w, valType const & x, valType const & y, valType const & z);

			// Convertions
			explicit tquat(valType const & pitch, valType const & yaw, valType const & roll);
			explicit tquat(tvec3<valType> const & eulerAngles);
			explicit tquat(tmat3x3<valType> const & m);
			explicit tquat(tmat4x4<valType> const & m);

			// Accesses
			valType& operator[](int i);
			valType operator[](int i) const;

			// Operators
			tquat<valType>& operator*=(valType const & s);
			tquat<valType>& operator/=(valType const & s);
		};

		template <typename valType> 
		detail::tquat<valType> operator- (
			detail::tquat<valType> const & q);

		template <typename valType> 
		detail::tvec3<valType> operator* (
			detail::tquat<valType> const & q, 
			detail::tvec3<valType> const & v);

		template <typename valType> 
		detail::tvec4<valType> operator* (
			detail::tquat<valType> const & q, 
			detail::tvec4<valType> const & v);

		template <typename valType> 
		detail::tquat<valType> operator* (
			detail::tquat<valType> const & q, 
			valType const & s);

		template <typename valType> 
		detail::tquat<valType> operator* (
			valType const & s,
			detail::tquat<valType> const & q);

		template <typename valType> 
		detail::tquat<valType> operator/ (
			detail::tquat<valType> const & q, 
			valType const & s);

	} //namespace detail

	namespace gtx{
	//! GLM_GTX_quaternion extension: Quaternion types and functions
    namespace quaternion
    {
		//! Returns the length of the quaternion x. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		valType length(
			detail::tquat<valType> const & q);

        //! Returns the normalized quaternion of from x. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> normalize(
			detail::tquat<valType> const & q);
		
        //! Returns dot product of q1 and q2, i.e., q1[0] * q2[0] + q1[1] * q2[1] + ... 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		valType dot(
			detail::tquat<valType> const & q1, 
			detail::tquat<valType> const & q2);

        //! Returns the cross product of q1 and q2. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> cross(
			detail::tquat<valType> const & q1, 
			detail::tquat<valType> const & q2);
		
        //! Compute a cross product between a quaternion and a vector. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tvec3<valType> cross(
			detail::tquat<valType> const & q, 
			detail::tvec3<valType> const & v);

		//! Compute a cross product between a vector and a quaternion.
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tvec3<valType> cross(
			detail::tvec3<valType> const & v, 
			detail::tquat<valType> const & q);

		//! Returns a LERP interpolated quaternion of x and y according a. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> mix(
			detail::tquat<valType> const & x, 
			detail::tquat<valType> const & y, 
			valType const & a);
		
		//! Compute a point on a path according squad equation. 
		//! q1 and q2 are control points; s1 and s2 are intermediate control points.
		//! From GLM_GTX_transform extension.
		template <typename valType> 
		detail::tquat<valType> squad(
			detail::tquat<valType> const & q1, 
			detail::tquat<valType> const & q2, 
			detail::tquat<valType> const & s1, 
			detail::tquat<valType> const & s2, 
			valType const & h);

		//! Returns an intermediate control point for squad interpolation.
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> intermediate(
			detail::tquat<valType> const & prev, 
			detail::tquat<valType> const & curr, 
			detail::tquat<valType> const & next);

		//! Returns a exp of a quaternion. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tquat<valType> exp(
			detail::tquat<valType> const & q, 
			valType const & exponent);

		//! Returns a log of a quaternion. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tquat<valType> log(
			detail::tquat<valType> const & q);

		//! Returns x raised to the y power.
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tquat<valType> pow(
			detail::tquat<valType> const & x, 
			valType const & y);

		//! Returns quarternion square root.
		//! From GLM_GTX_quaternion extension.
		//template <typename valType> 
		//detail::tquat<valType> sqrt(
		//	detail::tquat<valType> const & q);
		
		//! Returns the q conjugate. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tquat<valType> conjugate(
			detail::tquat<valType> const & q);

		//! Returns the q inverse. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tquat<valType> inverse(
			detail::tquat<valType> const & q);

		//! Rotates a quaternion from an vector of 3 components axis and an angle expressed in degrees.
		//! From GLM_GTX_transform extension.
		template <typename valType> 
		detail::tquat<valType> rotate(
			detail::tquat<valType> const & q, 
			valType const & angle, 
			detail::tvec3<valType> const & v);

		//! Rotates a 3 components vector by a quaternion. 
		//! From GLM_GTX_transform extension.
		template <typename valType> 
		detail::tvec3<valType> rotate(
			detail::tquat<valType> const & q, 
			detail::tvec3<valType> const & v);

        //! Rotates a 4 components vector by a quaternion.
		//! From GLM_GTX_transform extension.
		template <typename valType> 
		detail::tvec4<valType> rotate(
			detail::tquat<valType> const & q, 
			detail::tvec4<valType> const & v);
		
        //! Returns the q rotation angle. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		valType angle(
			detail::tquat<valType> const & x);

		//! Returns the q rotation axis. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tvec3<valType> axis(
			detail::tquat<valType> const & x);

		//! Build a quaternion from an angle and an axis. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> angleAxis(
			valType const & angle, 
			valType const & x, 
			valType const & y, 
			valType const & z);

        //! Build a quaternion from an angle and an axis.
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> angleAxis(
			valType const & angle, 
			detail::tvec3<valType> const & v);

		//! Extract the real component of a quaternion.
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		valType extractRealComponent(
			detail::tquat<valType> const & q);

        //! Returns roll value of euler angles. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		valType roll(
			detail::tquat<valType> const & x);

		//! Returns pitch value of euler angles. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		valType pitch(
			detail::tquat<valType> const & x);

        //! Returns yaw value of euler angles. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		valType yaw(
			detail::tquat<valType> const & x);
		
		//! Returns euler angles, yitch as x, yaw as y, roll as z. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tvec3<valType> eularAngles(
			detail::tquat<valType> const & x);

		//! Converts a quaternion to a 3 * 3 matrix. 
		//! From GLM_GTX_quaternion extension.
        template <typename valType> 
		detail::tmat3x3<valType> toMat3(
			detail::tquat<valType> const & x);

		//! Converts a quaternion to a 4 * 4 matrix. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tmat4x4<valType> toMat4(
			detail::tquat<valType> const & x);

		//! Converts a 3 * 3 matrix to a quaternion. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> toQuat(
			detail::tmat3x3<valType> const & x);

		//! Converts a 4 * 4 matrix to a quaternion. 
		//! From GLM_GTX_quaternion extension.
		template <typename valType> 
		detail::tquat<valType> toQuat(
			detail::tmat4x4<valType> const & x);

		//! Quaternion of floating-point numbers. 
		//! From GLM_GTX_quaternion extension.
        typedef detail::tquat<float> quat;

    }//namespace quaternion
    }//namespace gtx
} //namespace glm

#define GLM_GTX_quaternion namespace gtx::quaternion
#ifndef GLM_GTX_GLOBAL
namespace glm {using GLM_GTX_quaternion;}
#endif//GLM_GTX_GLOBAL

#include "quaternion.inl"

#endif//glm_gtx_quaternion
