#include "RepDesigner.h"

#define IMAGECLASS RepDesignerImg
#define IMAGEFILE  <RepDesigner/RepDesigner.iml>
#include <Draw/iml_source.h>

FileSel& RepDesignerFs()
{
  static FileSel fs;
  return fs;
}

FileSel& PdfFs()
{
  static FileSel fs;
  return fs;
}

GUI_APP_MAIN
{
  SetLanguage(LNG_ENGLISH);
  SetDefaultCharset(CHARSET_UTF8);

  RepDesignerFs().Type("REP files", "*.rep")
           .AllFilesType()
           .DefaultExt("rep");
  PdfFs().Type("PDF files", "*.pdf")
         .AllFilesType()
         .DefaultExt("pdf");

  LoadFromFile(callback(RepDesigner::SerializeApp));
  new RepDesigner;
  Ctrl::EventLoop();
  StoreToFile(callback(RepDesigner::SerializeApp));
}

RepDesigner::RepDesigner()
{
  AddFrame(menubar);
  AddFrame(TopSeparatorFrame());
  AddFrame(toolbar);
  AddFrame(statusbar);
  //Add(editor.SizePos());
  sf0.Vert(header, sf1);
  sf1.Vert(editor, footer);
  sf0.SetPos(2000);
  sf1.SetPos(8000);
  Add(sf0.SizePos());
  
  menubar.Set(THISBACK(MainMenu));
  Sizeable().Zoomable();
  WhenClose = THISBACK(Destroy);
  menubar.WhenHelp = toolbar.WhenHelp = statusbar;
  static int doc;
  Title(Format("Document%d", ++doc));
  Icon(CtrlImg::File());
  header.ClearModify();
  editor.ClearModify();
  footer.ClearModify();
  SetBar();
  header.WhenRefreshBar = THISBACK(SetBar);
  editor.WhenRefreshBar = THISBACK(SetBar);
  footer.WhenRefreshBar = THISBACK(SetBar);
  OpenMain();
  ActiveFocus(editor);
  
	/* Program settings */
	CtrlLayoutOKCancel(fProgSetup, t_("Program setup"));
	CtrlLayout(fPageSetup);

  /* default values */
  ps.format = "A4";
  ps.oSize = 210;
  ps.vSize = 297;
  ps.tMarg = 10;
  ps.lMarg = 10;
  ps.bMarg = 10;
  ps.rMarg = 10;
  ps.hSize = 0;
  ps.fSize = 0;
  
  /* Open file from command line */
  // file name is passed without any prefix
  const Vector<String>& arg = CommandLine();
  for(int i=0; i<arg.GetCount(); i++) {
    if(arg[i][0] != '-' && arg[i][0] != '/') {
      OpenFile(arg[i]);
      i = arg.GetCount();
    }
  }  
}

void RepDesigner::SerializeApp(Stream& s)
{
  int version = 1;
  s / version;
  s % RepDesignerFs()
    % PdfFs();
  if(version >= 1)
    s % lrufile();  
  //if(version >= 2)
  //  s % sf0 % sf1;
  
}

void RepDesigner::FileBar(Bar& bar)
{
  bar.Add("New", CtrlImg::new_doc(), THISBACK(New))
     .Key(K_CTRL_N)
     .Help("Open new window");
  bar.Add("Open..", CtrlImg::open(), THISBACK(Open))
     .Key(K_CTRL_O)
     .Help("Open existing document");
  bar.Add(editor.IsModified() || header.IsModified() || footer.IsModified()|| modified, "Save", CtrlImg::save(), THISBACK(Save))
     .Key(K_CTRL_S)
     .Help("Save current document");
  bar.Add("SaveAs", CtrlImg::save_as(), THISBACK(SaveAs))
     .Help("Save current document with a new name");
  bar.Separator();
  bar.Add("Settings", RepDesignerImg::page_1_16(), THISBACK(PrgSetup))
     .Help("Report page settings");
  bar.ToolGap();
  bar.MenuSeparator();
  bar.Add("Print..", CtrlImg::print(), THISBACK(Print))
     .Key(K_CTRL_P)
     .Help("Print document");
  bar.Add("Export to PDF..", RepDesignerImg::pdf(), THISBACK(Pdf))
     .Help("Export document to PDF file");
  if(bar.IsMenuBar()) {
    if(lrufile().GetCount())
      lrufile()(bar, THISBACK(OpenFile));
    bar.Separator();
    bar.Add("Exit", THISBACK(Destroy));
  }
}

void RepDesigner::AboutMenu(Bar& bar)
{
  bar.Add("About..", THISBACK(About));
}

void RepDesigner::MainMenu(Bar& bar)
{
  bar.Add("File", THISBACK(FileBar));
  bar.Add("Window", callback(WindowsMenu));
  bar.Add("Help", THISBACK(AboutMenu));
}

void RepDesigner::New()
{
  new RepDesigner;
}

void RepDesigner::Load(const String& name)
{
  rep = ParseXML(LoadFile(name));
  if((String)rep.GetAdd("info").Attr("fileType") != "QTF report")
  {
    Exclamation("The selected file is not a valid report file.");
    return;
  }

  lrufile().NewEntry(name);
  ps.format = rep.GetAdd("page").Attr("format");
  ps.oSize = ScanInt(rep.GetAdd("page").Attr("oSize"));
  ps.vSize = ScanInt(rep.GetAdd("page").Attr("vSize"));
  ps.tMarg = ScanInt(rep.GetAdd("page").Attr("tMarg"));
  ps.lMarg = ScanInt(rep.GetAdd("page").Attr("lMarg"));
  ps.bMarg = ScanInt(rep.GetAdd("page").Attr("bMarg"));
  ps.rMarg = ScanInt(rep.GetAdd("page").Attr("rMarg"));
  ps.hSize = ScanInt(rep.GetAdd("page").Attr("hSize"));
  ps.fSize = ScanInt(rep.GetAdd("page").Attr("fSize"));
  AdjPageSize();
  editor.SetQTF(rep.GetAdd("body").At(0).GetText());
  header.SetQTF(rep.GetAdd("header").At(0).GetText());
  footer.SetQTF(rep.GetAdd("footer").At(0).GetText());
  
  filename = name;
  editor.ClearModify();
  header.ClearModify();
  footer.ClearModify();
  Title(filename);
  modified = false;
}

void RepDesigner::OpenFile(const String& fn)
{
  if(filename.IsEmpty() && !editor.IsModified() && !header.IsModified() && !footer.IsModified() && !modified)
    Load(fn);
  else
    (new RepDesigner)->Load(fn);
}

void RepDesigner::Open()
{
  FileSel& fs = RepDesignerFs();
  if(fs.ExecuteOpen())
    OpenFile(fs);
  else
    statusbar.Temporary("Loading aborted.");
}

void RepDesigner::DragAndDrop(Point, PasteClip& d)
{
  if(AcceptFiles(d)) {
    Vector<String> fn = GetFiles(d);
    for(int i = 0; i < fn.GetCount(); i++)
      if(FileExists(fn[i]))
        OpenFile(fn[i]);
  }
}

void RepDesigner::FrameDragAndDrop(Point p, PasteClip& d)
{
  DragAndDrop(p, d);
}

void RepDesigner::Save0()
{
  lrufile().NewEntry(filename);
  if(filename.IsEmpty())
    SaveAs();
  else
  {
    XmlNode r;
    r.GetAdd("info").SetAttr("fileType", "QTF report").SetAttr("version", "1");
    r.GetAdd("page").SetAttr("format", ps.format);
    r.GetAdd("page").SetAttr("oSize", AsString(ps.oSize));
    r.GetAdd("page").SetAttr("vSize", AsString(ps.vSize));
    r.GetAdd("page").SetAttr("tMarg", AsString(ps.tMarg));
    r.GetAdd("page").SetAttr("lMarg", AsString(ps.lMarg));
    r.GetAdd("page").SetAttr("bMarg", AsString(ps.bMarg));
    r.GetAdd("page").SetAttr("rMarg", AsString(ps.rMarg));
    r.GetAdd("page").SetAttr("hSize", AsString(ps.hSize));
    r.GetAdd("page").SetAttr("fSize", AsString(ps.fSize));
    r.GetAdd("body").At(0).CreateText(editor.GetQTF());
    r.GetAdd("header").At(0).CreateText(header.GetQTF());
    r.GetAdd("footer").At(0).CreateText(footer.GetQTF());

    if(SaveFile(filename, AsXML(r, XML_HEADER))) 
    {
      statusbar.Temporary("File " + filename + " was saved.");
      ClearModify();
      modified = false;
    }
    else
      Exclamation("Error saving the file [* " + DeQtf(filename) + "]!");
  }
}

void RepDesigner::Save()
{
  if(!editor.IsModified() && !header.IsModified() && !footer.IsModified() && !modified) return;
  Save0();
}

void RepDesigner::SaveAs()
{
  FileSel& fs = RepDesignerFs();
  if(fs.ExecuteSaveAs()) {
    filename = fs;
    Title(filename);
    Save0();
  }
}

void RepDesigner::Print()
{
  //editor.Print();
}

void RepDesigner::Pdf()
{
  FileSel& fs = PdfFs();
  if(!fs.ExecuteSaveAs("Output PDF file"))
    return;
  
  Report r(Size(MMsToPoints(ps.oSize), MMsToPoints(ps.vSize)), Margins(MMsToPoints(ps.tMarg), MMsToPoints(ps.lMarg), MMsToPoints(ps.bMarg), MMsToPoints(ps.rMarg)));
  r.Header(AsQTF(header.Get(), CHARSET_UTF8, QTF_NOSTYLES|QTF_BODY), MMsToPoints(ps.hSize));
  r.Footer(AsQTF(footer.Get(), CHARSET_UTF8, QTF_NOSTYLES|QTF_BODY), MMsToPoints(ps.fSize));
  r.Put(editor.Get());
  SaveFile(~fs, MyRepToPdf(r));
  
}

void RepDesigner::About()
{
  PromptOK("[A5 uWord]&Using [*^www://upp.sf.net^ Ultimate`+`+] technology.");
}

void RepDesigner::Destroy()
{
  if(editor.IsModified() || header.IsModified() || footer.IsModified() || modified) {
    switch(PromptYesNoCancel("Do you want to save the changes to the document?")) {
    case 1:
      Save();
      break;
    case -1:
      return;
    }
  }
  delete this;
}

void RepDesigner::MainBar(Bar& bar)
{
  FileBar(bar);
  bar.Separator();
  if(editor.HasFocus()) editor.DefaultBar(bar);
  else if(header.HasFocus()) header.DefaultBar(bar);
  else if(footer.HasFocus()) footer.DefaultBar(bar);
    
}

void RepDesigner::SetBar()
{
  toolbar.Set(THISBACK(MainBar));
}

void RepDesigner::AdjPageSize()
{
  header.SetPage(Size(MMsToPoints(ps.oSize - ps.lMarg - ps.rMarg), MMsToPoints(ps.hSize)));
  editor.SetPage(Size(MMsToPoints(ps.oSize - ps.lMarg - ps.rMarg), MMsToPoints(ps.vSize - ps.tMarg - ps.bMarg - ps.hSize - ps.fSize)));
  footer.SetPage(Size(MMsToPoints(ps.oSize - ps.lMarg - ps.rMarg), MMsToPoints(ps.fSize)));
  return;
}