#include "assert.h"

#include "Arrow.h"
#include "Connector.h"
#include "ConnectorEnd.h"
#include "Images.h"
#include "Label.h"
#include "Rect.h"
#include "UppCanvas.h"
#include "WayPoint.h"
#include "macros.h"

namespace cv{

	//##ModelId=4D605633025B
	Connector::Connector(Canvas* owner)
		: Canvas(owner)
		, dashed(false)
		, activePoint(NULL)
	{
		tail=new ConnectorEnd(this); 
		head=new ConnectorEnd(this);
		tail->beFree();
		head->beFree();
	}

	//##ModelId=4D605633025D
	Connector::Connector(Canvas* owner, Canvas* tailCanvas)
		: Canvas(owner)
		, dashed(false)
		, activePoint(NULL)
	{
		tail=new ConnectorEnd(this); 
		head=new ConnectorEnd(this); 
		tail->beCentered(tailCanvas);
		head->beFree();
	}

	//##ModelId=4D6056330260
	Connector::Connector(Canvas* owner,Canvas* tailCanvas,Canvas* headCanvas)
		: Canvas(owner)
		, dashed(false)
		, activePoint(NULL)
		{
			tail=new ConnectorEnd(this); tail->beCentered(tailCanvas);
			head=new ConnectorEnd(this); head->beCentered(headCanvas);
		}

	//##ModelId=4D66CDC4037A
	// affects bounds()
	Point& Connector::get_origin(){ 
		return tail->cached_point(); 
	}
	
	//##ModelId=4D66CDC4038A
	// affects bounds()
	Point& Connector::get_extent(){ 
		return head->cached_point(); 
	}


	//##ModelId=4D6056330265
	void Connector::connect(Canvas* c)
	{
		if (!tail->connected()) 
			tail->beCentered(c);
		else
			if (!head->connected())
				head->beCentered(c);
			else
				throw "already connected";
	}
	
	//##ModelId=4D72BAC3007D
	void Connector::connect(Canvas* tailSide, Canvas* headSide)
	{
		tail->beCentered(tailSide);
		head->beCentered(headSide);
	}

	//##ModelId=4D72C07F01F4
	bool Connector::contains(Point pt)
	{
		Rect r=bounds();
		r += 10;
		if (!r.contains(pt)) return false;
		
		Line l;
		double distance;

		if (!view()->isDragging())
			activePoint=NULL;
		
		if (hasPoints()){
			Point p1=origin;
			for(int i=0; i<waypoints.GetCount(); i++){
				
				if (!view()->isDragging())
					if (waypoints[i]->distance(pt) < 15)
						activePoint = waypoints[i];
				
				l.start=p1;
				l.end=*waypoints[i];
				distance = l.distance(pt);
				if (distance<8) 
					return true;
				p1 = *waypoints[i];
			}
			l.start=p1;
			l.end=extent;
			return l.distance(pt) < 8;
		}else{
			l.start=origin;
			l.end=extent;
			return l.distance(pt) < 8;
		}
	}

	//##ModelId=4D609571032C
	Point Connector::center() {
		return relative(0.5);
	}

	//##ModelId=4D6056330269
	bool Connector::isConnector(){ return true; }
	
	//##ModelId=4D613EBB0251
	bool Connector::isConnectedTo(Canvas* c)
	{
		if (tail->canvas){
			if (tail->canvas==c) return true;
			if (tail->canvas->isConnectedTo(c)) return true;
		}
		if (head->canvas){
			if (head->canvas==c) return true;
			if (head->canvas->isConnectedTo(c)) return true;
		}
		return false;
	}
	
	//##ModelId=4D691F2400AB
  void Connector::moveRelative(Point p){
    
    if (activePoint){
      activePoint->operator+=(p);
    	compactWayPoints();
    }else{
	   	head->moveRelative(p);
	    tail->moveRelative(p);
	    for(int i=0; i<waypoints.GetCount(); i++)
	      waypoints[i]->operator+=(p);
    }
  }

	//##ModelId=4D72F40C0109
	void Connector::compactWayPoints() {
		Point p1,p2;
		p1=extent;
		Line l;
		for(int i=waypoints.GetCount()-1; i>=0; i--){
			l.start=p1;
			l.end=*waypoints[i];
			if (l.length()<10)
				waypoints.Remove(i);
			p1=l.end;
		}
		if (hasPoints()){
			l.start=origin;
			l.end=*waypoints[0];
			if (l.length()<10)
				waypoints.Remove(0);
		}
	}

	//##ModelId=4D6328840196
	Label* Connector::newLabel(std::string text, Position position)
	{
		Label* label=new Label(this,text);
		labels.push_back(label);
		label->end->beRelative(this,position);

		
		Vect v;
		v.origin = label->end->point();
		v.radius = 22;
		v.set_alpha(M_PI / 2);
		label->end->beVector(v);
		
		return label;
	}

	//##ModelId=4D6056330264
	void Connector::paint(){
		
		origin = tail->point();
		extent = head->point();
		if (origin == extent) 
			return;
		
		Point p1 = origin;
		Point p2 = origin;
		Upp::String s;

#if 0
		// DEBUG
		s = Upp::Format("[origin] %s" ,origin.ToString());
		root()->_draw->DrawText( origin.x , origin.y + 10, s , Upp::StdFont() , Upp::Red() );
		s = Upp::Format("[extent] %s" ,extent.ToString());
		root()->_draw->DrawText( extent.x , extent.y + 10, s , Upp::StdFont() , Upp::Red() );
#endif


#if 1
		length();
#else
		// DEBUG
		s = Upp::Format("[connector] %d" ,length());
		root()->_draw->DrawText( (origin.x+extent.x)/2 , (origin.y+extent.y)/2 + 20, s , Upp::StdFont() , Upp::Red() );
#endif
		if (hasPoints()){

			for(int i=0; i<waypoints.GetCount(); i++){
				p2 = *waypoints[i];

#if 0
				// DEBUG
				s = Upp::Format("[%d] %s" ,i ,waypoints[i]->ToString());
				root()->_draw->DrawText( p2.x , p2.y + 10, s , Upp::StdFont() , Upp::Red() );
				s = Upp::Format("[%d] %d" ,i ,waypoints[i]->length);
				root()->_draw->DrawText( (p1.x+p2.x)/2 , (p1.y+p2.y)/2 + 20, s , Upp::StdFont() , Upp::Red() );
#endif
			
				root()->drawLine(p1,p2, dashed ? Upp::PEN_DASH : 1);
				p1 = p2;
				
				if (selected and !activePoint)
					root()->drawCircle(p2,6,2,Upp::Cyan);
			}
#if 0
			// DEBUG
			s = Upp::Format("[extent] %d",extent.distance(p2));
			root()->_draw->DrawText( (extent.x+p2.x)/2 , (extent.y+p2.y)/2 + 20 , s , Upp::StdFont() , Upp::Red() );
#endif
		}

		root()->drawLine(p2,extent, dashed ? Upp::PEN_DASH : 1);

		if (activePoint)
			root()->drawCircle(*activePoint,6,2,Upp::Red());
		
		if (tail->arrow->visible){
			tail->arrow->set_origin(origin);
			tail->arrow->paint(tail->alpha);
		}
		if (head->arrow->visible){
			head->arrow->set_origin(extent);
			head->arrow->paint(head->alpha);
		}
		
	}
	
	//##ModelId=4D64434B00CB
	void Connector::showArrows(bool state){
		tail->arrow->visible=state;
		head->arrow->visible=state;
	}
	
	//##ModelId=4D645A94000F
	double Connector::length(){
		Point& p1=get_origin();
		Point& p2=get_extent();
		if (!hasPoints()){
			return p1.distance(p2);
		}else{
			double r=0;
			double d;
			for(int i=0; i<waypoints.GetCount(); i++){
				d = waypoints[i]->distance(p1);
				r += d;
				waypoints[i]->length = d;
				p1 = *waypoints[i];
			}
			r += p1.distance(p2);
			return r;
		}
	}
	
	//##ModelId=4D6457000167
	Point Connector::relative(double quote) {
		double dist_from_origin = length() * quote;
		Point p1 = get_origin();
		Point p2 = get_extent();
		
		if (!hasPoints()){
			
			Vect v(p1 , head->alpha + M_PI, dist_from_origin);
			return v.asAbsolutePoint();
			
		} else {

			WayPoint* wp;
			
			for(int i=0; i<waypoints.GetCount(); i++){
				wp = waypoints[i];
				
				if (dist_from_origin < wp->length)
					return Point( (wp->x + p1.x)/2 , (wp->y + p1.y)/2 );
				dist_from_origin = dist_from_origin - wp->length;
				p1 = *wp;
			}
			return Point( (p2.x+p1.x)/2 , (p2.y+p1.y)/2 );
			
		}
	}

	//##ModelId=4D7159C5007D
	Point Connector::relative(Position pos){
		switch(pos){
			case NearHead:
				return relative(0.85);
			case NearCenter:
				return relative(0.50);
			case NearTail:
				return relative(0.15);
		}
	}
	
	//##ModelId=4D6462CB0128
	double Connector::relative(Point& p){
		static Line l;
		l.start = tail->point(); //get_origin();
		l.end = head->point(); //get_extent();
		return l.relative(p);
	}
	
	//##ModelId=4D72C1600196
	WayPoint& Connector::addPoint(Point p){
		
		WayPoint* wp;
		Point p1 = origin;
		Point p2;
		Line l;
		for(int i=0; i<waypoints.GetCount(); i++){
			p2=*waypoints[i];
			l.start=p1;
			l.end=p2;
			if (l.distance(p)<8){
				wp=new WayPoint(p);
				waypoints.Insert(i,wp);
				return *wp;
			}
			p1=p2;
		}
		l.start=p1;
		l.end=extent;
		if (l.distance(p)<8){
			wp=new WayPoint(p);
			if (hasPoints())
				waypoints.Insert(waypoints.GetCount(),wp);
			else
				waypoints.Add(wp);
			return *wp;
		}
		throw "this point is not mine...";
	}

	//##ModelId=4D72D24000DA
	Rect& Connector::bounds() {
		static Rect r;
		r.Set(tail->point(),head->point());
		r.Normalize();
		for(int i=0; i<waypoints.GetCount(); i++)
			r += *waypoints[i];
		return r;
	}

	//##ModelId=4D7413BB0232
	bool Connector::hasPoints() { return waypoints.GetCount() > 0; }

	//##ModelId=4D74400E02AF
	void Connector::beMinimal() {
		head->arrow->visible = false;
		tail->arrow->visible = false;
		each(Label,labels)
			(*i)->visible = false;
	}
	
	//##ModelId=4D744011008C
	void Connector::beNormal() {
		head->arrow->visible = true;
		tail->arrow->visible = true;
		each(Label,labels)
			(*i)->visible = true;
	}

	//##ModelId=4D74432D0399
	void Connector::stopMoving() {
		beNormal();
	}

	//##ModelId=4D74C74D030D
	void Connector::remove() {
		if (head->canvas)
			head->canvas->removeIncoming(this);
		if (tail->canvas)
			tail->canvas->removeOutcoming(this);
		each(Label,labels)
			(*i)->remove();
		labels.clear();
		head->canvas = NULL;
		tail->canvas = NULL;
		Canvas::remove();
	}

	//##ModelId=4D75815E029F
	Canvas* Connector::asCanvas() {
		return dynamic_cast<Canvas*>(this);
	}

  //##ModelId=4D7772AF003E
  void Connector::move(float x, float y){/**/}
  //##ModelId=4D7773870213
  void Connector::move(Point p){/**/}
	
}

