#include "ConnectorEnd.h"

#include "Arrow.h"
#include "connector.h"
#include "WayPoint.h"

#include "assert.h" // debug

namespace cv {

	//##ModelId=4D6056330288
	ConnectorEnd::ConnectorEnd(Connector* cn)
		: canvas(NULL)
		, connector(cn)
		, arrow(new Arrow(connector,None))
		{/**/}

	//##ModelId=4D6056330285
	ConnectorEnd::ConnectorEnd(Connector* cn, Canvas* c)
		: canvas(c)
		, connector(cn)
		, arrow(new Arrow(connector,None))
		{/**/}

	//##ModelId=4D6056330284
	ConnectorEnd* ConnectorEnd::counterPart(){
		return (this==connector->head) ? connector->tail : connector->head;
	}

	//##ModelId=4D73C92E0000
	bool ConnectorEnd::isTail() { return this == connector->tail; }
	//##ModelId=4D73C92E002E
	bool ConnectorEnd::isHead() { return !isTail(); }

	//##ModelId=4D5F11740138
	void ConnectorEnd::beFree() {
		Point p(0,0);
		beFree(p);
	}

	//##ModelId=4D6056330290
	void ConnectorEnd::beFree(Point& pt) {
		if (canvas){
			if (isTail())
				canvas->removeOutcoming(connector);
			else
				canvas->removeIncoming(connector);
			canvas = NULL;
		}
		Point p1,p2;
		attachment.freePoint = pt;
		attachment.policy = att_free;
		if (connector->tail == this)
			connector->set_origin(pt);
		else
			connector->set_extent(pt);
		Line l( connector->get_origin(), connector->get_extent());
		alpha = l.alpha();
	}

	//##ModelId=4D6056330291
	bool ConnectorEnd::isFree() {
		return attachment.policy == att_free;
	}


	//##ModelId=4D6705CD006D
	void ConnectorEnd::beVector(Vect v){
		attachment.policy = att_vector;
		attachment.vector = v;
	}

	//##ModelId=4D605633028E
	void ConnectorEnd::beCentered(Canvas* c){
		attachment.policy = att_center;
		canvas = c;
		if (isHead())
			c->addIncoming(connector);
		else
			c->addOutcoming(connector);
	}

	//##ModelId=4D690D730196
	void ConnectorEnd::beRelative(Canvas* c, Position pos){
		canvas = c;
		attachment.policy = att_positional;
		attachment.position = pos;
	}

	//##ModelId=4D646A6B0128
	void ConnectorEnd::beRelative(Canvas* c, Point& p){
		canvas = c;
		assert(c->isConnector());
		Connector* cn = dynamic_cast<Connector*>(c);
		attachment.policy = att_relative;
		attachment.quote = cn->relative(p);
	}

	//##ModelId=4D679850009C
	void ConnectorEnd::moveRelative(Point delta){
		
		switch(attachment.policy){

			case att_relative:
			case att_center:
				canvas->moveRelative(delta);
				break;
			case att_free:
				attachment.freePoint += delta;
				break;
			
			case att_positional:
				break;
			case att_vector:
				assert(canvas);
				{
					Vect v=attachment.vector;
					// attachment.vector.origin is updated in paint()
					v.rotate(connector->head->alpha);
					
					Point p;
					p = attachment.vector.asAbsolutePoint();
					
					Vect vd;
					vd.set(Point(0,0), delta);
					vd.rotate(connector->head->alpha,true);
					v.set(p+vd.asAbsolutePoint()); // -> alpha,radius
					attachment.vector = v;
				}
		}
	}

	//##ModelId=4D605633028A
	Point ConnectorEnd::point(bool exact){
		Point this_pt;
		Point other_pt;
		this_pt = pt;
		double rotate_alpha=0;
		{
			if (connector->waypoints.GetCount()){
				if (isTail())
					other_pt = *connector->waypoints[0];
				else
					other_pt = *connector->waypoints[connector->waypoints.GetCount()-1];
					
			}else{
				ConnectorEnd* other;
				other=counterPart();
				other_pt = isTail() ? connector->get_extent() : connector->get_origin();
			}
		}

		switch(attachment.policy){
			case att_absolute:
				break;
				
			case att_positional:
				this_pt = connector->relative(attachment.position);
				break;

			case att_relative:
				{
					Connector* c = (Connector*) canvas;
					this_pt = c->relative(attachment.quote);
				}
				break;
			
			case att_vector:
				{
					this_pt = connector->relative(attachment.position);
					
					Vect v=attachment.vector;
					v.origin = this_pt;
					attachment.vector.origin = this_pt;
					v.rotate(connector->head->alpha);
					this_pt = v.asAbsolutePoint();
				}
				break;
			
			case att_free:
				this_pt = attachment.freePoint;
				//rotate_alpha = M_PI;
				break;
				
			case att_center:
			
				this_pt = canvas->center();
				if (canvas->isShape() && exact){
					canvas->bounds().intersectionFrom(other_pt,/*out*/this_pt);
				}
				if (canvas->isConnector() && isHead()) {
					rotate_alpha = M_PI;
				}
		}
		
		if (exact) 
			pt = this_pt;
		
		Line l(this_pt,other_pt);
		alpha = l.alpha() + rotate_alpha;
		
		return this_pt;
	}

	//##ModelId=4D605633028B
	bool ConnectorEnd::connected(){
		return attachment.policy != att_free;
	}

	//##ModelId=4D66D40C037A
	Point& ConnectorEnd::cached_point(){
		return pt;
	}

}

