#include "connector.h"
#include "tools.h"
//#include "Images.h"

#include "assert.h"

namespace cv{

	//##ModelId=4D605633025B
	Connector::Connector(Canvas* owner)
		: Canvas(owner)
		, dashed(false)
		, activeSegment(NULL)
		, isWiring(false)
	{
		segments.Add(new ConnectorSegment(this));
		tail=new ConnectorEnd(this); 
		head=new ConnectorEnd(this);
		tail->beFree();
		head->beFree();
	}

	//##ModelId=4D605633025D
	Connector::Connector(Canvas* owner, Canvas* tailCanvas)
		: Canvas(owner)
		, dashed(false)
		, activeSegment(NULL)
		, isWiring(false)
	{
		segments.Add(new ConnectorSegment(this));
		tail=new ConnectorEnd(this); 
		head=new ConnectorEnd(this); 
		tail->beCentered(tailCanvas);
		head->beFree();
	}

	//##ModelId=4D6056330260
	Connector::Connector(Canvas* owner,Canvas* tailCanvas,Canvas* headCanvas)
		: Canvas(owner)
		, dashed(false)
		, activeSegment(NULL)
		, isWiring(false)
	{
		segments.Add(new ConnectorSegment(this));
		tail=new ConnectorEnd(this); tail->beCentered(tailCanvas);
		head=new ConnectorEnd(this); head->beCentered(headCanvas);
	}

	//##ModelId=4D66CDC4037A
	// affects bounds()
	Point& Connector::get_origin(){
		static Point _pt;
		_pt = tail->point(); // cached
		return _pt;
	}
	
	//##ModelId=4D66CDC4038A
	// affects bounds()
	Point& Connector::get_extent(){ 
		static Point _pt;
		_pt = head->point(); // cached
		return _pt;
	}

	//##ModelId=4D7DA44002AF
	void Connector::set_extent(Point pt) {
		lastSegment().end = pt;
	}
	
	//##ModelId=4D7DA4460000
	void Connector::set_origin(Point pt) {
		firstSegment().start = pt;
	}


	//##ModelId=4D6056330265
	void Connector::connect(Canvas* c)
	{
		if (!tail->connected()) 
			tail->beCentered(c);
		else
			if (!head->connected())
				head->beCentered(c);
			else
				throw "already connected";
		
	}
	
	//##ModelId=4D72BAC3007D
	void Connector::connect(Canvas* tailSide, Canvas* headSide)
	{
		tail->beCentered(tailSide);
		head->beCentered(headSide);
	}

	//##ModelId=4D72C07F01F4
	bool Connector::contains(Point pt)
	{
		Rect r=bounds();
		r += 10;
		if (!r.contains(pt)) return false;
		
		Line l;

		if (!view()->isDragging())
			activeSegment=NULL;
		
		if (!view()->isDragging())
			setActiveSegment(pt);
		
		for(int i=0; i<segments.GetCount(); i++)
			if (segments[i]->distance(pt) < 8)
				return true;
		return false;
	}

	//##ModelId=4D609571032C
	Point Connector::center() {
		return relative(0.5).asAbsolutePoint();
	}

	//##ModelId=4D6056330269
	bool Connector::isConnector(){ return true; }
	
	//##ModelId=4D613EBB0251
	bool Connector::isConnectedTo(Canvas* c)
	{
		if (tail->canvas){
			if (tail->canvas==c) return true;
			if (tail->canvas->isConnectedTo(c)) return true;
		}
		if (head->canvas){
			if (head->canvas==c) return true;
			if (head->canvas->isConnectedTo(c)) return true;
		}
		return false;
	}
	
	//##ModelId=4D691F2400AB
  void Connector::moveRelative(Point p){
    
    if (activeSegment){
      activeSegment->moveRelative(p);
    	compactSegments();
    }else{
	   	head->moveRelative(p);
	    tail->moveRelative(p);
	    for(int i=0; i<segments.GetCount(); i++){
	      segments[i]->movingPoint = false;
	      segments[i]->moveRelative(p);
	    }
    }
  }

	//##ModelId=4D72F40C0109
	void Connector::compactSegments() {
		int found;
		ConnectorSegment* s;
	
		found = -1;
		if (segments.GetCount() > 1){
			for(int i=0; found==-1 && i<segments.GetCount(); i++){
				if (segments[i]->_length < 10)
					found = i;
			}
			if (found>=0){
				s = segments[found];
				
				activeSegment = s->prevSegment;
				if (!activeSegment)
					diagram()->dragger->finished();
				
				if (s->prevSegment){
					s->prevSegment->nextSegment = s->nextSegment;
					if (s->nextSegment)
						s->prevSegment->end = s->nextSegment->start;
				}
				if (s->nextSegment){
					s->nextSegment->prevSegment = s->prevSegment;
					if (s->prevSegment)
						s->nextSegment->start = s->prevSegment->end;
				}
				delete s;
				segments.Remove(found);
			}
		}
		/**/
	}

	//##ModelId=4D6328840196
	Label* Connector::newLabel(std::string text, Position position)
	{
		Label* label=new Label(this,text);
		labels.push_back(label);
		label->end->beRelative(this,position);
		
		Vect v;
		v.origin = label->end->point();
		v.radius = 22;
		v.set_alpha(M_PI / 2);
		label->end->beVector(v);
		
		return label;
	}

	//##ModelId=4D6056330264
	void Connector::paint(){
		
		origin = tail->point(false);
		extent = head->point(false);
		if (origin == extent) 
			return;
		
		ConnectorSegment* s;
		Upp::String str;
		Point cursor;

#if 0
		// DEBUG
		str = Upp::Format("[origin] %s" ,origin.ToString());
		root()->_draw->DrawText( origin.x , origin.y + 10, str , Upp::StdFont() , Upp::Red() );
		str = Upp::Format("[extent] %s" ,extent.ToString());
		root()->_draw->DrawText( extent.x , extent.y + 10, str , Upp::StdFont() , Upp::Red() );
#endif


#if 1
		length();
#else
		// DEBUG
		str = Upp::Format("[connector] %d" ,length());
		root()->_draw->DrawText( (origin.x+extent.x)/2 , (origin.y+extent.y)/2 + 20, str , Upp::StdFont() , Upp::Red() );
#endif

		
		for(int i=0; i<segments.GetCount(); i++){
			s = segments[i];

#if 0
			// DEBUG
			cursor = s->end + Point(5,5);
			str = Upp::Format("[%d] %n" ,i ,s->_alpha);
			root()->_draw->DrawText( cursor.x , cursor.y, str , Upp::StdFont() , Upp::Red() );
#endif
		
			root()->drawLine(s->start, s->end, dashed ? Upp::PEN_DASH : 1);
			
			if (selected && !activeSegment && (s->nextSegment!=NULL))
				root()->drawCircle(s->end ,6,2,Upp::Cyan);
		}
#if 0
		// DEBUG
		s = Upp::Format("[extent] %d",extent.distance(p2));
		root()->_draw->DrawText( (extent.x+p2.x)/2 , (extent.y+p2.y)/2 + 20 , s , Upp::StdFont() , Upp::Red() );
#endif

		// activeSegment should not be the last segment
		if (activeSegment && (activeSegment->nextSegment!=NULL))
			root()->drawCircle(activeSegment->end,6,2,Upp::Red());
		
		if (tail->arrow->visible){
			tail->arrow->set_origin(origin);
			tail->arrow->paint(firstSegment()._alpha);
		}
		if (head->arrow->visible){
			head->arrow->set_origin(extent);
			head->arrow->paint(lastSegment()._alpha + M_PI);
		}
	}
	
	//##ModelId=4D64434B00CB
	void Connector::showArrows(bool state){
		tail->arrow->visible=state;
		head->arrow->visible=state;
	}
	
	//##ModelId=4D645A94000F
	double Connector::length(){
		
		double r=0;
		for(int i=0; i<segments.GetCount(); i++){
			segments[i]->_length = segments[i]->length();
			segments[i]->_alpha = segments[i]->alpha();
			r += segments[i]->_length;
		}
		return r;
	}

//##ModelId=4D7E16E80399
	void Connector::relative(Attachment* att){
		double quote;
		switch(att->position){
			case NearHead:
				quote = 0.85;
				break;
			case NearCenter:
				quote = 0.50;
				break;
			case NearTail:
				quote = 0.15;
				break;
		}
		double dist_from_origin = length() * quote;
		
		ConnectorSegment* seg;
		for(int i=0; i<segments.GetCount(); i++){
			seg = segments[i];
			if (dist_from_origin < seg->_length){
				Vect v;
				v.origin = seg->start;
				v.set(seg->end);
				v.radius = dist_from_origin;
				
				att->vector.set_origin(v.asAbsolutePoint());
				att->segment = seg;
				return;
			}
			dist_from_origin = dist_from_origin - seg->_length;
		}
	}

	//##ModelId=4D6457000167
	Vect Connector::relative(double quote){
		double dist_from_origin = length() * quote;
		
		ConnectorSegment* seg;
		for(int i=0; i<segments.GetCount(); i++){
			seg = segments[i];
			if (dist_from_origin < seg->_length){
				Vect v;
				v.origin = seg->start;
				v.set(seg->end);
				v.radius = dist_from_origin;
				
				return v;
			}
			dist_from_origin = dist_from_origin - seg->_length;
		}
		throw "...should not happen";
	}
	
	//##ModelId=4D7159C5007D
	Vect Connector::relative(Position pos){
		switch(pos){
			case NearHead:
				return relative(0.85);
			case NearCenter:
				return relative(0.50);
			case NearTail:
				return relative(0.15);
		}
		throw "unexpected";
	}
	
	//##ModelId=4D6462CB0128
	double Connector::relative(Point& p){
		static Line l;
		l.start = tail->point(); //get_origin();
		l.end = head->point(); //get_extent();
		return l.relative(p);
	}

	//##ModelId=4D83104203A9
	void Connector::appendSegmentAt(Point p){
	}

	//##ModelId=4D8310AC0271
	ConnectorSegment* Connector::insertSegmentBefore(int index, Point p){
		ConnectorSegment* snew;
		ConnectorSegment* s = segments[index];
		snew = new ConnectorSegment(this);
		snew->start = s->start;
		snew->end = p;
		s->start = p;
		snew->nextSegment = s;
		snew->prevSegment = s->prevSegment;
		if (s->prevSegment)
			s->prevSegment->nextSegment = snew;
		s->prevSegment = snew;
		segments.Insert(index,snew);
		return snew;
	}

	//##ModelId=4D831360007D
	ConnectorSegment* Connector::insertSegmentAfter(int index, Point p){
		ConnectorSegment* snew;
		ConnectorSegment* s = segments[index];
		snew = new ConnectorSegment(this);
		snew->start = p;
		s->end = p;
		s->nextSegment = snew;
		snew->prevSegment = s;
		segments.Insert(index+1,snew);
		return snew;
	}
	
	//##ModelId=4D72C1600196
	ConnectorSegment* Connector::addPoint(Point p){
		if (isWiring){
			return insertSegmentAfter(segments.GetCount()-1,p);
		}else{
			/*
				<---------p------------<>
				|----s-------------------|
				|----s----|----snew------|
				
				
			*/
			for(int i=0; i<segments.GetCount(); i++){
				if (segments[i]->distance(p) < 8){
					insertSegmentBefore(i,p);
					return segments[i];
				}
			}
			throw "this point is not mine...";
		}
	}

	//##ModelId=4D72D24000DA
	Rect& Connector::bounds() {
		static Rect r;
		r.Set(tail->point(),head->point());
		r.Normalize();
		for(int i=0; i<segments.GetCount(); i++)
			r += segments[i]->end;
		return r;
	}
	

	//##ModelId=4D74400E02AF
	void Connector::beMinimal() {
		head->arrow->visible = false;
		tail->arrow->visible = false;
		for(int i=0; i<labels.GetCount(); i++)
			labels[i]->visible = false;
	}
	
	//##ModelId=4D744011008C
	void Connector::beNormal() {
		head->arrow->visible = true;
		tail->arrow->visible = true;
		for(int i=0; i<labels.GetCount(); i++)
			labels[i]->visible = true;
	}

	//##ModelId=4D74432D0399
	void Connector::stopMoving() {
		beNormal();
	}

	//##ModelId=4D74C74D030D
	void Connector::remove() {
		if (head->canvas)
			head->canvas->removeIncoming(this);
		if (tail->canvas)
			tail->canvas->removeOutcoming(this);
		for(int i=0; i<labels.GetCount(); i++)
			labels[i]->remove();
		labels.clear();
		head->canvas = NULL;
		tail->canvas = NULL;
		Canvas::remove();
	}

	//##ModelId=4D75815E029F
	Canvas* Connector::asCanvas() {
		return dynamic_cast<Canvas*>(this);
	}

  //##ModelId=4D7772AF003E
  void Connector::move(float x, float y){/**/}
  //##ModelId=4D7773870213
  void Connector::move(Point p){/**/}

	//##ModelId=4D7AB89F03C8
	void Connector::setActiveSegment(int index) {
		activeSegment = segments[index];
		activeSegment->movingPoint = true;
	}
	
	//##ModelId=4D7AB96702EE
	void Connector::setActiveSegment(Point pt) {
		activeSegment = NULL;
		for(int i=0; i<segments.GetCount(); i++)
			if (segments[i]->end.distance(pt) < 15)
				setActiveSegment(i);
	}

	//##ModelId=4D7D9548030D
	ConnectorSegment& Connector::firstSegment() {return *segments[0];}
	
	//##ModelId=4D7D956303B9
	ConnectorSegment& Connector::lastSegment() {
		assert(segments.GetCount() >= 1);
		return *segments[segments.GetCount()-1];
	}

	//##ModelId=4D7E1E84003E
	void Connector::removeSegments() {
		ConnectorSegment* s;
		while(segments.GetCount() >1){
			s = segments[1];
			delete s;
			segments.Remove(1);
		}
		segments[0]->nextSegment = NULL;
	}

	//##ModelId=4D60563301D3
	Attachment::Attachment()
	 	:quote(0), policy(att_center)
	 	, segment(NULL)
	{
	}

	//##ModelId=4D6056330288
	ConnectorEnd::ConnectorEnd(Connector* cn)
		: canvas(NULL)
		, connector(cn)
		, arrow(new Arrow(connector,None))
		{/**/}

	//##ModelId=4D6056330285
	ConnectorEnd::ConnectorEnd(Connector* cn, Canvas* c)
		: canvas(c)
		, connector(cn)
		, arrow(new Arrow(connector,None))
		{/**/}

	//##ModelId=4D6056330284
	ConnectorEnd* ConnectorEnd::counterPart(){
		return (this==connector->head) ? connector->tail : connector->head;
	}

	//##ModelId=4D73C92E0000
	bool ConnectorEnd::isTail() { return this == connector->tail; }
	//##ModelId=4D73C92E002E
	bool ConnectorEnd::isHead() { return !isTail(); }

	//##ModelId=4D5F11740138
	void ConnectorEnd::beFree() {
		Point p(0,0);
		beFree(p);
	}

	//##ModelId=4D6056330290
	void ConnectorEnd::beFree(Point& pt) {
		if (canvas){
			if (isTail())
				canvas->removeOutcoming(connector);
			else
				canvas->removeIncoming(connector);
			canvas = NULL;
		}
		Point p1,p2;
		attachment.freePoint = pt;
		attachment.policy = att_free;
		if (connector->tail == this)
			connector->set_origin(pt);
		else
			connector->set_extent(pt);
	}

	//##ModelId=4D6056330291
	bool ConnectorEnd::isFree() {
		return attachment.policy == att_free;
	}


	//##ModelId=4D6705CD006D
	void ConnectorEnd::beVector(Vect v){
		attachment.policy = att_vector;
		attachment.vector = v;
	}

	//##ModelId=4D605633028E
	void ConnectorEnd::beCentered(Canvas* c){
		attachment.policy = att_center;
		canvas = c;
		registerConnector();
	}

	//##ModelId=4D690D730196
	void ConnectorEnd::beRelative(Canvas* c, Position pos){
		canvas = c;
		attachment.policy = att_positional;
		attachment.position = pos;
		registerConnector();
	}

	//##ModelId=4D646A6B0128
	void ConnectorEnd::beRelative(Canvas* c, Point& p){
		canvas = c;
		assert(c->isConnector());
		attachment.policy = att_relative;
		attachment.quote = c->asConnector()->relative(p); // todo assume c is ALWAYS a connector
		registerConnector();
	}

	//##ModelId=4D679850009C
	void ConnectorEnd::moveRelative(Point delta){
		
		switch(attachment.policy){

			case att_relative:
			case att_center:
				canvas->moveRelative(delta);
				break;
			case att_free:
				attachment.freePoint += delta;
				break;
			
			case att_positional:
				break;
			case att_vector:
				assert(canvas);
				{
					Point label_pos;
					label_pos = attachment.vector.asAbsolutePoint();

					Vect vd;
					vd.set(Point(0,0), delta);
					vd.rotate(attachment.segment->_alpha,true );
					attachment.vector.set(label_pos + vd.asAbsolutePoint());
					
					
				}
		}
	}

	//##ModelId=4D605633028A
	Point ConnectorEnd::point(bool cached){
		Point other_pt;
		Point this_pt;
		double rotate_alpha=0;

		if (attachment.policy == att_vector){
			this_pt = vectorPoint;
		}else{
			if (isTail()) {
				this_pt = connector->firstSegment().start;
				other_pt = connector->firstSegment().end;
			}else{
				this_pt = connector->lastSegment().end;
				other_pt = connector->lastSegment().start;
			}
		}
		
		if (cached)
			return this_pt;
	
		// not cached? ok let's work...

		switch(attachment.policy){
			case att_absolute:
				break;
				
			case att_positional:
				this_pt = connector->relative(attachment.position).asAbsolutePoint();
				break;

			case att_relative:
				{
					Connector* c = (Connector*) canvas;
					this_pt = c->relative(attachment.quote).asAbsolutePoint();
				}
				break;
			
			case att_vector:
				{
					// set (1) vector origin (2) segment where it is attached to
					connector->relative(&attachment);
					Vect v = attachment.vector;
					v.rotate(attachment.segment->_alpha);
					vectorPoint = v.asAbsolutePoint();
					return vectorPoint;
				}
				break;
			
			case att_free:
				this_pt = attachment.freePoint;
				//rotate_alpha = M_PI;
				break;
				
			case att_center:
			
				this_pt = canvas->center();
				if (canvas->isShape() && !cached){
					canvas->bounds().intersectionFrom(other_pt,/*out*/this_pt);
				}
				
		}
		
		if (isTail()){
			connector->firstSegment().start = this_pt;
		}else{
			connector->lastSegment().end = this_pt;
		}
		
		return this_pt;
	}

	//##ModelId=4D605633028B
	bool ConnectorEnd::connected(){
		return attachment.policy != att_free;
	}

	

	//##ModelId=4D7D438F009C
	void ConnectorEnd::registerConnector() {
		if (isHead())
			canvas->addIncoming(connector);
		else
			canvas->addOutcoming(connector);
	}

	//##ModelId=4D7DAEE802EE
	ConnectorSegment::ConnectorSegment(Connector* owner) 
		: movingPoint(false)
		, nextSegment(NULL)
		, prevSegment(NULL)
		, connector(owner)
		, kind(free)
	{/**/}

	//##ModelId=4D7DA8E003C8
	void ConnectorSegment::moveRelative(Point p) {
		
		if (movingPoint){
			end += p;
			if (nextSegment)
				nextSegment->start += p;
		}
		else{ // --> movingConnector
			if (nextSegment) end += p;
			if (prevSegment) start += p;
		}
	}

	//##ModelId=4D632A3203B9
	Label::Label(Connector* owner, std::string text)
		: CText(owner->asCanvas()->owner,0,0)
		, end(NULL)
		{
			((CDiagram*) owner->asCanvas()->owner)->add(this);
			end = new ConnectorEnd(owner,this);
			set_contents(text);
		}

//##ModelId=4D83E437007D
	Rect& Label::bounds(){
		static Rect r;
		r = Visual::bounds();
		r += 3;
		return r;
	}

	//##ModelId=4D7D643801C5
	Point& Label::get_origin() {
		static Point p;
		p = end->point(false);
		return p;
	}
	
	//##ModelId=4D7D65B20213
	void Label::set_origin(Point p) {
		// I get moved by Canvas just after creation
		if (end)
			end->vectorPoint = p;
	}

	//##ModelId=4D632D41038A
	void Label::paint()
	{
		CText::paint();
		if (selected)
		{
			Point fromConnector = attachment();
			Point fromLabel = center();
			Rect r=bounds();
			r += 4;
			r.intersectionFrom(fromConnector,/*out*/fromLabel);
			root()->drawLine(fromLabel,fromConnector,1);
		}
	}
	
	//##ModelId=4D691F240167
	Point Label::attachment(){
		return end->connector->relative(end->attachment.position).asAbsolutePoint();
	}
	
	//##ModelId=4D6711530000
	void Label::moveRelative(Point delta){
		end->moveRelative(delta);
	}

	//##ModelId=4D60563301B8
	Arrow::Arrow(Canvas* c,ArrowType t)
		:Canvas(c)
		,type(t)
		{}

	//##ModelId=4D60563301BD
	void Arrow::paint(){ paint(0); }
	
	//##ModelId=4D60563301BB
	void Arrow::paint(float rotation){
		UppCanvas* d=root();
		int x1,y1, x2,y2, x3,y3, x4,y4;
		switch(type){
			case None:
				break;
			case ClosedFilled:
			case ClosedEmpty:
				x1=origin.x ;
				y1=origin.y ;
				x2=origin.x + 12;
				y2=origin.y - 5;
				x3=origin.x + 12;
				y3=origin.y + 5; 
				d->drawLine(x1,y1,x2,y2,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x1,y1,x3,y3,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x3,y3,x2,y2,1,Upp::Black , rotation ,x1,y1);
				break;
			case DiamondEmpty:
  		case DiamondFilled:
			/*                                    
		        /\p2                         \p2
		       /  \                           \ 
		    p1/    \p3   ---------------------- p1
		      \    /                          /
		       \  /                         /p3
		        \/p4   
			*/
				x1=origin.x ;
				y1=origin.y ;
				x2=origin.x + 10;
				y2=origin.y - 5;
				x3=origin.x + 20;
				y3=origin.y ; 
				x4=origin.x + 10;
				y4=origin.y + 5;
				d->drawLine(x1,y1,x2,y2,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x2,y2,x3,y3,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x3,y3,x4,y4,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x4,y4,x1,y1,1,Upp::Black , rotation ,x1,y1);
				break;
			case OpenArrow:
				x1=origin.x;
				y1=origin.y;
				x2=origin.x - 12;
				y2=origin.y - 5;
				x3=origin.x - 12;
				y3=origin.y + 5;
				d->drawLine(x1,y1,x2,y2,1,Upp::Black , rotation ,x1,y1);
				d->drawLine(x1,y1,x3,y3,1,Upp::Black , rotation ,x1,y1);
		}

	}

}




