/*
*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_DICT_H
#define PYTHONPP_DICT_H

#include "PythonPP_object.h"

NAMESPACE_UPP

namespace pythonpp
{

template<class T> class CDictHelper;

template<class T>
class CDictProxy
{
protected:
	CDictHelper<T>& s;  //< the dictionary
	Object key;        //< item key
	T value;            //< value

public:
	CDictProxy<T> (CDictHelper<T>& map, const std::string& k)
	: s(map), value()
	{
		key = CString(k);
		if(map.HasKey(key)) value = map.GetItem(key);
	};

	CDictProxy<T> (CDictHelper<T>& map, const Object& k)
		: s(map), key(k), value()
	{
		if(map.HasKey(key)) value = map.GetItem(key);
	};

	~CDictProxy<T>(void)
	{
	}

	// CDictHelper<T> stuff
	// lvalue
	CDictProxy<T>& operator=(const CDictProxy<T>& other)
	{
		if (this != &other) {
			value = other.value;
			s.SetItem(key, other.value);
		}
		return *this;
	};

	CDictProxy<T>& operator= (const T& ob)
	{
		value = ob;
		s.SetItem (key, ob);
		return *this;
	}

	operator T&(void)
	{
		return value;
	}

	operator const T&(void) const
	{
		return value;
	}
}; // end of CDictProxy


template<class T>
class CDictHelper : public Object
{
protected:
	CDictHelper(void)
	{
	}

public:
	typedef size_t size_type;
	typedef Object key_type;
	typedef CDictProxy<T> data_type;
	typedef std::pair< const T, T > value_type;
	typedef std::pair< const T, CDictProxy<T> > reference;
	typedef const std::pair< const T, const T > const_reference;
	typedef std::pair< const T, CDictProxy<T> > pointer;

public:
	// Constructors

	CDictHelper(PyObject *obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
		// !!! Somebody should do a type-check here !!!
	}

	CDictHelper(const Object& obj)
	: Object(obj)
	{
		// !!! Somebody should do a type-check here !!!
	}

	virtual ~CDictHelper(void)
	{
	}

	CDictHelper& operator= (const Object& rhs)
	{
		return (*this = *rhs);
	}

	CDictHelper& operator= (PyObject* rhsp)
	{
		if (Get() != rhsp) {
			Set (rhsp);
		}
		return *this;
	}

public:
	void clear (void)
	{
		List k = Keys();
		for (List::iterator i = k.begin(); i.neq(k.end()); ++i) {
			DelItem(*i);
		}
	}

	// !!!
	virtual size_type size() const
	{
		return PyMapping_Length (Get());
	}

	// Element Access
	T operator [] (const String& key) const
	{
		return GetItem(key);
	}

	T operator [] (const Object& key) const
	{
		return GetItem(key);
	}

	CDictProxy<T> operator [] (const String& key)
	{
		return CDictProxy<T>(*this, key);
	}

	CDictProxy<T> operator [] (const Object& key)
	{
		return CDictProxy<T>(*this, key);
	}

	int GetLength (void) const
	{
		return PyMapping_Length (Get());
	}

	bool HasKey (const String& s) const
	{
		return PyMapping_HasKeyString (Get(), ~s) != 0;
	}

	bool HasKey (const Object& s) const
	{
		return PyMapping_HasKey (Get(), s.Get()) != 0;
	}

	// !!!
	virtual T GetItem (const String& s) const
	{
		return T(
			PyMapping_GetItemString (Get(), const_cast<char*>(~s)),
			eTakeOwnership
		);
	}

	// !!!
	virtual T GetItem (const Object& s) const
	{
		return T(
			PyObject_GetItem (Get(), s.Get()),
			eTakeOwnership
		);
	}

	// !!!
	virtual void SetItem (const String& key, const Object& obj)
	{
		if (PyMapping_SetItemString (Get(), const_cast<char*>(~key), obj) == -1) {
			throw SystemError("SetItem");
		}
	}

	// !!!
	virtual void SetItem (const Object& key, const Object& obj)
	{
		if (PyObject_SetItem (Get(), key, obj) == -1) {
			throw SystemError("SetItem");
		}
	}

	// !!!
	virtual void DelItem (const String& s)
	{
		if (PyMapping_DelItemString (Get(), const_cast<char*>(~s)) == -1) {
			throw SystemError("DelItem");
		}
	}

	// !!!
	virtual void DelItem (const Object& obj)
	{
		if (PyMapping_DelItem (Get(), obj) == -1) {
			throw SystemError("DelItem");
		}
	}

	// Queries
	List Keys (void) const
	{
		return List(PyMapping_Keys(Get()), eTakeOwnership);
	}

	List Values (void) const
	{
		return List(PyMapping_Values(Get()), eTakeOwnership);
	}

	List Items (void) const
	{
		return List(PyMapping_Items(Get()), eTakeOwnership);
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyMapping_Check (obj) != 0;
	}

public:
	// Iterator ...
	// Future development ...

};  // end of CDictHelper<T>

typedef CDictHelper<Object> CDictBase;

// PyDict_Type
class CDict : public CDictBase
{
// PyAPI_FUNC(PyObject *) PyDictProxy_New(PyObject *);
//PyAPI_FUNC(void) PyDict_Clear(PyObject *mp);
//PyAPI_FUNC(CInt) PyDict_Contains(PyObject *mp, PyObject *key);
//PyAPI_FUNC(PyObject *) PyDict_Copy(PyObject *mp);


//PyAPI_FUNC(CInt) PyDict_Next(PyObject *mp, int *pos, PyObject **key, PyObject **value);
//PyAPI_FUNC(CInt) PyDict_Update(PyObject *mp, PyObject *other);
//PyAPI_FUNC(CInt) PyDict_MergeFromSeq2(PyObject *d, PyObject *seq2, int override);

public:
	CDict(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: CDictBase(obj, ownership)
	{
	}
	CDict(const Object& obj)
	{
		if (!HasExactSameType(obj)) {
			throw TypeError("Invalid conversion");
		}
		Set(obj);
	}
	CDict(const CDict& obj)
	: CDictBase(obj)
	{
	}
	CDict(void)
	: CDictBase(PyDict_New(), eTakeOwnership)
	{
	}

public:
	// Assign operators ...
	CDict& operator= (const Object& obj)
	{
		if (this != &obj) {
			if (!HasExactSameType(obj)) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	CDict& operator= (PyObject* obj)
	{
		if (Get() != obj) {
			if (!HasExactSameType(obj)) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

public:
	// Function name is made STL compatible ...
	size_type size(void) const
	{
		return PyDict_Size(Get());
	}
	template<class O>
	void Merge(const CDictHelper<O>& other, bool override = true)
	{
		if(PyDict_Merge (Get(), other, (override ? 1 : 0)) == -1) {
			throw SystemError("Merge");
		}
	}

public:
	// Queries
	List Keys (void) const
	{
		return List(PyDict_Keys(Get()), eTakeOwnership);
	}

	List Values (void) const
	{ // each returned item is a (key, value) pair
		return List(PyDict_Values(Get()), eTakeOwnership);
	}

	List Items (void) const
	{
		return List(PyDict_Items(Get()), eTakeOwnership);
	}

	Object GetItem (const String& s) const
	{
		return Object(
			PyDict_GetItemString (Get(), ~s),
			eTakeOwnership
		);
	}
	Object GetItem (const Object& obj) const
	{
		return Object(
			PyDict_GetItem (Get(), obj),
			eTakeOwnership
		);
	}
	void SetItem (const String& key, const Object& obj)
	{
		if (PyDict_SetItemString (Get(), ~key, obj) == -1) {
			throw SystemError("SetItem");
		}
	}
	void SetItem (const Object& key, const Object& obj)
	{
		if (PyDict_SetItem (Get(), key, obj) == -1) {
			throw SystemError("SetItem");
		}
	}

	void DelItem (const String& key)
	{
		if (PyDict_DelItemString (Get(), ~key) == -1) {
			throw SystemError("DelItem");
		}
	}
	void DelItem (const Object& key)
	{
		if (PyDict_DelItem (Get(), key) == -1) {
			throw SystemError("DelItem");
		}
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyDict_Check(obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
#ifdef PyDict_CheckExact
		return PyDict_CheckExact(obj);
#else
		return obj->ob_type == &PyDict_Type;
#endif
	}
};

}                                       // namespace pythonpp

END_UPP_NAMESPACE

#endif                                  // PYTHONPP_DICT_H

