/*
*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_ERROR_H
#define PYTHONPP_ERROR_H

#include <Core/Core.h>

#ifdef _MSC_VER
// disable warning C4005: macro redefinition.
#pragma warning(disable: 4005)
#endif

#ifdef HAVE_GETHOSTBYNAME_R
#undef HAVE_GETHOSTBYNAME_R
#endif

#ifdef _POSIX_C_SOURCE
#undef _POSIX_C_SOURCE
#endif

#include <Python.h>

#ifdef HAVE_GETHOSTBYNAME_R
#undef HAVE_GETHOSTBYNAME_R
#endif

#ifdef _POSIX_C_SOURCE
#undef _POSIX_C_SOURCE
#endif

NAMESPACE_UPP

namespace pythonpp
{

class Error
{
///* Error testing and normalization */
//PyAPI_FUNC(int) PyErr_GivenExceptionMatches(PyObject *, PyObject *);
//PyAPI_FUNC(int) PyErr_ExceptionMatches(PyObject *);
//PyAPI_FUNC(void) PyErr_NormalizeException(PyObject**, PyObject**, PyObject**);



//PyAPI_DATA(PyObject *) PyExc_MemoryErrorInst;


/* Convenience functions */

//PyAPI_FUNC(int) PyErr_BadArgument(void);
//PyAPI_FUNC(PyObject *) PyErr_NoMemory(void);
//PyAPI_FUNC(PyObject *) PyErr_SetFromErrno(PyObject *);
//PyAPI_FUNC(PyObject *) PyErr_SetFromErrnoWithFilenameObject(
//    PyObject *, PyObject *);
//PyAPI_FUNC(PyObject *) PyErr_SetFromErrnoWithFilename(PyObject *, char *);
//#ifdef Py_WIN_WIDE_FILENAMES
//PyAPI_FUNC(PyObject *) PyErr_SetFromErrnoWithUnicodeFilename(
//    PyObject *, Py_UNICODE *);
//#endif /* Py_WIN_WIDE_FILENAMES */

//PyAPI_FUNC(PyObject *) PyErr_Format(PyObject *, const char *, ...)
//            Py_GCC_ATTRIBUTE((format(printf, 2, 3)));

//#ifdef MS_WINDOWS
//PyAPI_FUNC(PyObject *) PyErr_SetFromWindowsErrWithFilenameObject(
//    int, const char *);
//PyAPI_FUNC(PyObject *) PyErr_SetFromWindowsErrWithFilename(
//    int, const char *);
//#ifdef Py_WIN_WIDE_FILENAMES
//PyAPI_FUNC(PyObject *) PyErr_SetFromWindowsErrWithUnicodeFilename(
//    int, const Py_UNICODE *);
//#endif /* Py_WIN_WIDE_FILENAMES */
//PyAPI_FUNC(PyObject *) PyErr_SetFromWindowsErr(int);
//PyAPI_FUNC(PyObject *) PyErr_SetExcFromWindowsErrWithFilenameObject(
//    PyObject *,int, PyObject *);
//PyAPI_FUNC(PyObject *) PyErr_SetExcFromWindowsErrWithFilename(
//    PyObject *,int, const char *);
//#ifdef Py_WIN_WIDE_FILENAMES
//PyAPI_FUNC(PyObject *) PyErr_SetExcFromWindowsErrWithUnicodeFilename(
//    PyObject *,int, const Py_UNICODE *);
//#endif /* Py_WIN_WIDE_FILENAMES */
//PyAPI_FUNC(PyObject *) PyErr_SetExcFromWindowsErr(PyObject *, int);
//#endif /* MS_WINDOWS */

/* Function to create a new exception */
//PyAPI_FUNC(PyObject *) PyErr_NewException(char *name, PyObject *base,
//                                         PyObject *dict);
//PyAPI_FUNC(void) PyErr_WriteUnraisable(PyObject *);

/* Issue a warning or exception */
//PyAPI_FUNC(int) PyErr_Warn(PyObject *, char *);
//PyAPI_FUNC(int) PyErr_WarnExplicit(PyObject *, const char *,
//                   const char *, int,
//                   const char *, PyObject *);

/* In sigcheck.c or signalmodule.c */
//PyAPI_FUNC(int) PyErr_CheckSignals(void);
//PyAPI_FUNC(void) PyErr_SetInterrupt(void);

/* Support for adding program text to SyntaxErrors */
//PyAPI_FUNC(void) PyErr_SyntaxLocation(const char *, int);
//PyAPI_FUNC(PyObject *) PyErr_ProgramText(const char *, int);


public:
	Error(const String& msg)
	{
		SetString(msg);
	}
	Error(PyObject* value)
	{
		SetObject(value);
	}

protected:
	Error(const String& msg, PyObject* err_type)
	{
		SetString(err_type, msg);
	}

public:
	static void Check(void)
	{
		PyObject* err_type = PyErr_Occurred();
		if (err_type) {
			throw Error();
		}
	}
	static void Check(const char* value)
	{
		if (!value) {
			Check();
		}
	}
	static void Check(PyObject* obj)
	{
		if (!obj) {
			Check();
		}
	}
	static void Clear(void)
	{
		PyErr_Clear();
	}

public:
	static void SetString(const String& msg)
	{
		SetString(GetPyException(), ~msg);
	}
	static void SetObject(PyObject* value)
	{
		SetObject(GetPyException(), value);
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_Exception;
	}

protected:
	// Error without parameters should be thrown only in case
	// of system/python error. User should always provide
	// a message.
	Error(void)
	{
	}

protected:
	static void SetString(PyObject* excp_obj, const String& msg)
	{
		PyErr_SetString(excp_obj, ~msg);
	}
	static void SetObject(PyObject* excp_obj, PyObject* value)
	{
		PyErr_SetObject(excp_obj, value);
	}
	static void SetNone(PyObject* excp_obj)
	{
		PyErr_SetNone(excp_obj);
	}
};


///////////////////////////////////
// Predefined exceptions
///////////////////////////////////

class ZeroDivisionError : public Error
{
public:
	ZeroDivisionError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_ZeroDivisionError;
	}
};

class ValueError : public Error
{
public:
	ValueError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_ValueError;
	}
};

class UnicodeError : public Error
{
public:
	UnicodeError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	UnicodeError(const String& msg, PyObject* err_type)
	: Error(msg, err_type)
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_UnicodeError;
	}
};

class UnicodeEncodeError : public UnicodeError
{
public:
	UnicodeEncodeError(const String& msg)
	: UnicodeError(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_UnicodeEncodeError;
	}
};

class UnicodeDecodeError : public UnicodeError
{
public:
	UnicodeDecodeError(const String& msg)
	: UnicodeError(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_UnicodeDecodeError;
	}
};

class UnicodeTranslateError : public UnicodeError
{
public:
	UnicodeTranslateError(const String& msg)
	: UnicodeError(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_UnicodeTranslateError;
	}
};

class TypeError : public Error
{
public:
	TypeError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_TypeError;
	}
};

class SystemError : public Error
{
public:
	SystemError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_SystemError;
	}
};

class UnboundLocalError : public Error
{
public:
	UnboundLocalError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_UnboundLocalError;
	}
};

class SystemExit : public Error
{
public:
	SystemExit(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_SystemExit;
	}
};

class ReferenceError : public Error
{
public:
	ReferenceError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_ReferenceError;
	}
};

class TabError : public Error
{
public:
	TabError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_TabError;
	}
};

class IndentationError : public Error
{
public:
	IndentationError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_IndentationError;
	}
};

class SyntaxError : public Error
{
public:
	SyntaxError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_SyntaxError;
	}
};

class NotImplementedError : public Error
{
public:
	NotImplementedError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	NotImplementedError(const String& msg, PyObject* err_type)
	: Error(msg, err_type)
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_NotImplementedError;
	}
};

class RuntimeError : public Error
{
public:
	RuntimeError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_RuntimeError;
	}
};

class OverflowError : public Error
{
public:
	OverflowError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_OverflowError;
	}
};

class NameError : public Error
{
public:
	NameError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_NameError;
	}
};

class MemoryError : public Error
{
public:
	MemoryError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_MemoryError;
	}
};

class KeyboardInterrupt : public Error
{
public:
	KeyboardInterrupt(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_KeyboardInterrupt;
	}
};

class KeyError : public Error
{
public:
	KeyError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_KeyError;
	}
};

class IndexError : public Error
{
public:
	IndexError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_IndexError;
	}
};

class ImportError : public Error
{
public:
	ImportError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_ImportError;
	}
};

class OSError : public Error
{
public:
	OSError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	OSError(const String& msg, PyObject* err_type)
	: Error(msg, err_type)
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_OSError;
	}
};

class IOError : public Error
{
public:
	IOError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_IOError;
	}
};

class EnvironmentError : public Error
{
public:
	EnvironmentError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_EnvironmentError;
	}
};

class FloatingPointError : public Error
{
public:
	FloatingPointError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_FloatingPointError;
	}
};

class EOFError : public Error
{
public:
	EOFError(const String& msg)
	: Error(msg, PyExc_EOFError)
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_EOFError;
	}
};

class AttributeError : public Error
{
public:
	AttributeError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_AttributeError;
	}
};

class AssertionError : public Error
{
public:
	AssertionError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_AssertionError;
	}
};

class LookupError : public Error
{
public:
	LookupError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_LookupError;
	}
};

class ArithmeticError : public Error
{
public:
	ArithmeticError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_ArithmeticError;
	}
};

class StandardError : public Error
{
public:
	StandardError(void)
	{
	}
	StandardError(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	StandardError(const String& msg, PyObject* err_type)
	: Error(msg, err_type)
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_StandardError;
	}
};

class StopIteration : public Error
{
public:
	StopIteration(const String& msg)
	: Error(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_StopIteration;
	}
};

#ifdef MS_WINDOWS
class WindowsError : public OSError
{
public:
	WindowsError(const String& msg)
	: OSError(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_WindowsError;
	}
};
#endif
#ifdef __VMS
class VMSError : public OSError
{
public:
	VMSError(const String& msg)
	: OSError(msg, GetPyException())
	{
	}

protected:
	static PyObject* GetPyException(void)
	{
		return PyExc_VMSError;
	}
};
#endif


// This class is supposed to retrieve and store a python error code
class ErrorStackElem
{
public:
	ErrorStackElem(void)
	: m_ExceptionType(NULL),
		m_Value(NULL),
		m_Traceback(NULL)
	{
	}

public:
	void Fetch(void)
	{
		ASSERT(!m_ExceptionType && !m_Value && !m_Traceback);

		PyErr_Fetch(&m_ExceptionType, &m_Value, &m_Traceback);

		ASSERT(
			(m_ExceptionType) ||
			(!m_ExceptionType && !m_Value && !m_Traceback)
		);
	}
	void Restore(void)
	{
		// Do not pass a NULL type and non-NULL value or traceback.
		ASSERT(
			(m_ExceptionType) ||
			(!m_ExceptionType && !m_Value && !m_Traceback)
		);

		//  This call takes away a reference to each object: you must own
		// a reference to each object before the call and after the call
		// you no longer own these references.
		PyErr_Restore(m_ExceptionType, m_Value, m_Traceback);

		m_ExceptionType = NULL;
		m_Value = NULL;
		m_Traceback = NULL;
	}

private:
	PyObject* m_ExceptionType;
	PyObject* m_Value;
	PyObject* m_Traceback;
};

}                                       // namespace pythonpp

END_UPP_NAMESPACE

#endif                                  // PYTHONPP_ERROR_H

