/*
*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_OBJECT_H
#define PYTHONPP_OBJECT_H

#include "PythonPP_error.h"

NAMESPACE_UPP

namespace pythonpp
{

#if PY_VERSION_HEX >= 0x02050000 
	typedef Py_ssize_t py_ssize_t; 
#else
	typedef int py_ssize_t; 
#endif 

enum EOwnership {eTakeOwnership, eAcquireOwnership};
enum EOwnershipFuture {eOwned, eAcquired, eBorrowed};

class Type;
class CString;

// Strong-typed operation ...
inline PyObject* IncRefCount(PyObject* obj)
{
	Py_INCREF(obj);
	return obj;
}
inline PyObject* DecRefCount(PyObject* obj)
{
	Py_DECREF(obj);
	return obj;
}

// PyObject
// PyVarObject
class Object
{
// ???
///* Generic operations on objects */
//PyAPI_FUNC(CInt) PyObject_Print(PyObject *, FILE *, int);
//PyAPI_FUNC(void) _PyObject_Dump(PyObject *);
//#ifdef Py_USING_UNICODE
//PyAPI_FUNC(PyObject *) PyObject_Unicode(PyObject *);
//#endif
//PyAPI_FUNC(PyObject *) PyObject_RichCompare(PyObject *, PyObject *, int);
//PyAPI_FUNC(CInt) PyObject_RichCompareBool(PyObject *, PyObject *, int);
//PyAPI_FUNC(PyObject *) PyObject_GetAttr(PyObject *, PyObject *);
//PyAPI_FUNC(CInt) PyObject_HasAttr(PyObject *, PyObject *);
//PyAPI_FUNC(PyObject **) _PyObject_GetDictPtr(PyObject *);
//PyAPI_FUNC(PyObject *) PyObject_SelfIter(PyObject *);
//PyAPI_FUNC(PyObject *) PyObject_GenericGetAttr(PyObject *, PyObject *);
//PyAPI_FUNC(CInt) PyObject_GenericSetAttr(PyObject *,
//                          PyObject *, PyObject *);
//PyAPI_FUNC(CInt) PyObject_Not(PyObject *);
//PyAPI_FUNC(CInt) PyCallable_Check(PyObject *);
//PyAPI_FUNC(CInt) PyNumber_Coerce(PyObject **, PyObject **);
//PyAPI_FUNC(CInt) PyNumber_CoerceEx(PyObject **, PyObject **);

//PyAPI_FUNC(void) PyObject_ClearWeakRefs(PyObject *);

///* A slot function whose address we need to compare */
//extern int _PyObject_SlotCompare(PyObject *, PyObject *);

public:

public:
	/// Creates a python None object
	Object(void)
	: m_PyObject(Py_None)
	// , m_Ownership(eOwned)
	{
		IncRefCount(Get());
	}
	explicit Object(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: m_PyObject(obj)
	// , m_Ownership(eAcquireOwnership ? eAcquired : eOwned)  // !!! Currently this parameter does not much value of "ownership"
	{
		ASSERT(Get());
		if ( ownership == eAcquireOwnership ) {
			IncRefCount(Get());
		}
	}
	Object(const Object& obj)
	: m_PyObject(obj)
	// , m_Ownership(eOwned)
	{
		ASSERT(Get());
		IncRefCount(Get());
	}
	~Object(void)
	{
		Release();
	}

	Object& operator= (const Object& obj)
	{
		if (this != &obj)
		{
			Set(obj);
		}
		return *this;
	}

	Object& operator= (PyObject* obj)
	{
		if (Get() != obj) {
			Set (obj);
		}
		return *this;
	}

public:
	// Implicit conversion to the PyObject* type
	operator PyObject* (void) const
	{
		return m_PyObject;
	}
	PyObject* Get(void) const
	{
		return m_PyObject;
	}
	/// Not exception-safe this time
	void Set(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	{
		ASSERT(obj);

		Release();
		m_PyObject = obj;
		if ( ownership == eAcquireOwnership )
		{
			IncRefCount(*this);
		}
	}
	void Release(void)
	{
		// PyAPI_FUNC(void) IncRefCount(PyObject *);
		// PyAPI_FUNC(void) Py_DecRef(PyObject *);
		Py_XDECREF(m_PyObject);
		m_PyObject = NULL;
	}

public:
	// Atributes ...
	Object GetAttr (const String& name) const
	{
		PyObject* obj = PyObject_GetAttrString(Get(), const_cast<char*>(~name));
		if ( !obj ) {
			throw AttributeError("Attribute does not exist");
		}
		return Object(obj, eTakeOwnership);
	}
	void SetAttr(const String& name, const Object& value)
	{
		if(PyObject_SetAttrString (Get(), const_cast<char*>(~name), value.Get()) == -1) {
			throw AttributeError("SetAttr failed");
		}
	}
	void DelAttr (const String& name)
	{
		if(PyObject_DelAttrString (Get(), const_cast<char*>(~name)) == -1) {
			throw AttributeError("DelAttr failed");
		}
	}
	bool HasAttr (const String& name) const
	{
		return PyObject_HasAttrString (Get(), const_cast<char*>(~name)) != 0;
	}

public:
	// Itemss ...
	Object GetItem (const Object& key) const
	{
		PyObject* obj = PyObject_GetItem(Get(), key.Get());
		if ( !obj ) {
			throw KeyError("Item does not exist");
		}
		return Object(obj, eTakeOwnership);
	}
	/* Do not delete this ...
	void SetItem(const Object& key, const Object& value)
	{
		if(PyObject_SetItem (Get(), key.Get(), value.Get()) == -1) {
			throw CKeyError ("SetItem failed");
		}
	}
	*/
	void DelItem (const Object& key)
	{
		if(PyObject_DelItem (Get(), key.Get()) == -1) {
			throw KeyError ("DelItem failed");
		}
	}

public:
	long GetHashValue (void) const
	{
		long value = PyObject_Hash (Get());

		if ( value == -1) {
			throw SystemError("Invalid hash value");
		}
		return value;
	}

public:
	// There's really no reason to use this function instead of the common
	// expression m_PyObject->ob_type, which returns a pointer of type PyTypeObject*,
	// except when the incremented reference count is needed.
	Type GetType(void) const;
	PyTypeObject* GetObjType(void) const
	{
		return m_PyObject->ob_type;
	}
	// CString GetString(void) const;
	// CString GetRepresentation() const;
	// String as_string(void) const;

public:
	// Equality and comparison based on PyObject_Compare
	bool operator==(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result == 0;
	}
	bool operator!=(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result != 0;
	}
	bool operator>=(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result >= 0;
	}
	bool operator<=(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result <= 0;
	}
	bool operator<(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result < 0;
	}
	bool operator>(const Object& obj) const
	{
		int result = PyObject_Compare (Get(), obj);
		Error::Check();
		return result > 0;
	}

protected:
	bool IsNumeric (void) const
	{
		return PyNumber_Check (Get()) != 0;
	}
	bool IsSequence (void) const
	{
		return PySequence_Check (Get()) != 0;
	}
	bool IsTrue (void) const
	{
		return PyObject_IsTrue (Get()) != 0;
	}
	bool IsObjectType (const Type& t) const;

private:
	PyObject*   m_PyObject;
	// EOwnershipFuture  m_Ownership;
};

class None : public Object
{
public:
	None(void)
	: Object(Py_None)
	{
	}
	None(const Object& obj)
	: Object(obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}

	None& operator= (const None& obj)
	{
		if ( this != &obj) {
			Set(obj);
		}
		return *this;
	}
	None& operator= (const Object& obj)
	{
		if ( this != &obj) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		//  Py_None is an object of undefined type ...
		// So, we can compare only objects themelf ...
		return obj == Py_None;
	}
	static bool HasExactSameType(PyObject* obj)
	{
		//  Py_None is an object of undefined type ...
		// So, we can compare only objects themelf ...
		return obj == Py_None;
	}
};

// PyTypeObject
// PyHeapTypeObject
class Type : public Object
{
// int PyType_HasFeature(  PyObject *o, int feature)
// int PyType_IS_GC(   PyObject *o)
// int PyType_IsSubtype(   PyTypeObject *a, PyTypeObject *b)
// PyAPI_FUNC(PyObject *) PyType_GenericAlloc(PyTypeObject *, int);
// PyAPI_FUNC(PyObject *) PyType_GenericNew(PyTypeObject *, PyObject *, PyObject *);
// PyAPI_FUNC(CInt) PyType_Ready(PyTypeObject *);

public:
	Type(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
	}
	Type(const Object& obj)
	: Object(obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	Type(const Type& obj)
	: Object(obj)
	{
	}

	Type& operator= (const Object& obj)
	{
		if ( this != &obj) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

	Type& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			Set(obj);
		}
		return *this;
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyType_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyType_CheckExact (obj);
	}
};

inline
bool operator ==(const Type& l, const Type& r)
{
	return l.Get() == r.Get();
}

///////////////////////////////////////////////////////////////////////////
// Numeric interface
inline Object operator+ (const Object& a)
{
	PyObject* tmp_obj = PyNumber_Positive(a.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Positive");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator- (const Object& a)
{
	PyObject* tmp_obj = PyNumber_Negative(a.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Negative");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline Object abs(const Object& a)
{
	PyObject* tmp_obj = PyNumber_Absolute(a.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Absolute");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline std::pair<Object, Object> coerce(const Object& a, const Object& b)
{
	PyObject* p1;
	PyObject* p2;
	p1 = a.Get();
	p2 = b.Get();
	if(PyNumber_Coerce(&p1, &p2) == -1) {
		throw ArithmeticError("PyNumber_Coerce");
	}
	return std::pair<Object, Object>(Object(p1, eTakeOwnership), Object(p2, eTakeOwnership));
}

inline Object operator+ (const Object& a, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Add(a.Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Add");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline Object operator- (const Object& a, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Subtract(a.Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Subtract");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline Object operator* (const Object& a, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Multiply(a.Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Multiply");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline Object operator/ (const Object& a, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Divide(a.Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Divide");
	}
	return Object(tmp_obj, eTakeOwnership);
}

inline Object operator% (const Object& a, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Remainder(a.Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Remainder");
	}
	return Object(tmp_obj, eTakeOwnership);
}

//////////////////////////////////////////////////////////////////////////
inline
Type
Object::GetType(void) const
{
	// ???
	PyObject* obj = PyObject_Type (Get());
	if ( !obj ) {
		throw TypeError("Type does not exist");
	}
	return Type(obj, eTakeOwnership);
}

/* Do not delete this code ...
inline
CString
Object::GetString(void) const
{
	// ???
	PyObject* obj = PyObject_Str (Get());
	if ( !obj ) {
		throw CTypeError("Unable to convert an object to a string");
	}
	return CString(obj, eTakeOwnership);
}

inline
CString
Object::GetRepresentation() const
{
	// ???
	PyObject* obj = PyObject_Repr(Get());
	if ( !obj ) {
		throw CTypeError("Unable to convert an object to a representation");
	}
	return CString(obj, eTakeOwnership);
}

inline
String
Object::as_string(void) const
{
	return static_cast<String>(GetString());
}
*/

inline
bool
Object::IsObjectType (const Type& t) const
{
	return GetType().Get() == t.Get();
}

}
									// namespace pythonpp
END_UPP_NAMESPACE

#endif                                  // PYTHONPP_OBJECT_H

