/*
*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_PDT_H
#define PYTHONPP_PDT_H

#include "PythonPP_object.h"

NAMESPACE_UPP

namespace pythonpp
{

// PyBoolObject
class Bool : public Object
{
public:
	Bool(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	Bool(const Object& obj)
	: Object(obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	Bool(const Bool& obj)
	: Object(obj)
	{
	}
	Bool(bool value)
	: Object(value ? Py_True : Py_False)
	{
	}
	Bool(long value)
	{
		PyObject* tmp_obj = PyBool_FromLong(value);
		if ( !tmp_obj ) {
		throw TypeError("Invalid conversion");
		}
		Set(tmp_obj, eTakeOwnership);
	}

public:
	// Assign operators ...
	Bool& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	Bool& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

public:
	// Type conversion operators ...
	operator bool(void) const
	{
		return Get() == Py_True;
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyBool_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyBool_Check (obj);
	}
};

// PyInt_Type
class Int : public Object
{
//PyAPI_FUNC(PyObject *) PyInt_FromString(char*, char**, int);
//PyAPI_FUNC(unsigned long) PyInt_AsUnsignedLongMask(PyObject *);
//PyAPI_FUNC(long) PyInt_GetMax(void);
// long PyInt_AS_LONG( PyObject *io)
// unsigned long long PyInt_AsUnsignedLongLongMask(    PyObject *io)

public:
	Int(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
	}
	Int(const Object& obj)
	{
		PyObject* tmp_obj = PyNumber_Int(obj);
		if ( !tmp_obj ) {
			throw TypeError("Invalid conversion");
		}
		Set(tmp_obj, eTakeOwnership);
	}
	Int(const Int& obj)
	: Object(obj)
	{
	}
	Int(int value)
	: Object(PyInt_FromLong(long(value)), eTakeOwnership)
	{
	}
	Int(long value = 0L)
	: Object(PyInt_FromLong(value), eTakeOwnership)
	{
	}

public:
	// Assign operators ...
	Int& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			PyObject* tmp_obj = PyNumber_Int(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Int& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			PyObject* tmp_obj = PyNumber_Int(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Int& operator= (Int value)
	{
		Set(PyInt_FromLong(long(value)), eTakeOwnership);
		return *this;
	}
	Int& operator= (long value)
	{
		Set(PyInt_FromLong(value),  eTakeOwnership);
		return *this;
	}

public:
	// Type conversion operators ...
	operator long() const
	{
		return PyInt_AsLong(Get());
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyInt_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyInt_CheckExact (obj);
	}
};

// PyLong_Type
class Long : public Object
{
//PyAPI_FUNC(unsigned long) PyLong_AsUnsignedLongMask(PyObject *);
//PyAPI_FUNC(double) _PyLong_AsScaledDouble(PyObject *vv, int *e);

//PyAPI_FUNC(PyObject *) PyLong_FromVoidPtr(void *);
//PyAPI_FUNC(void *) PyLong_AsVoidPtr(PyObject *);
//PyAPI_FUNC(PyObject *) PyLong_FromString(char *, char **, int);

// ???
//PyAPI_FUNC(Int) _PyLong_Sign(PyObject *v);
//PyAPI_FUNC(size_t) _PyLong_NumBits(PyObject *v);
//PyAPI_FUNC(PyObject *) _PyLong_FromByteArray(
//    const unsigned char* bytes, size_t n,
//    int little_endian, int is_signed);
//PyAPI_FUNC(Int) _PyLong_AsByteArray(PyLongObject* v,
//    unsigned char* bytes, size_t n,
//    int little_endian, int is_signed);

// PyObject* PyLong_FromUnicode(   Py_UNICODE *u, int length, int base)

public:
	Long(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
	}
	Long(const Object& obj)
	{
		PyObject* tmp_obj = PyNumber_Long(obj);
		if ( !tmp_obj ) {
			throw TypeError("Invalid conversion");
		}
		Set(tmp_obj, eTakeOwnership);
	}
	Long(const Long& obj)
	: Object(obj)
	{
	}
	Long(Int value)
	: Object(PyLong_FromLong(static_cast<long>(value)), eTakeOwnership)
	{
	}
	Long(long value = 0L)
	: Object(PyLong_FromLong(value), eTakeOwnership)
	{
	}
	Long(unsigned long value)
	: Object(PyLong_FromUnsignedLong(value), eTakeOwnership)
	{
	}
	Long(long long value)
	: Object(PyLong_FromLongLong(value), eTakeOwnership)
	{
	}
	Long(unsigned long long value)
	: Object(PyLong_FromUnsignedLongLong(value), eTakeOwnership)
	{
	}
	Long(double value)
	: Object(PyLong_FromDouble(value), eTakeOwnership)
	{
	}

public:
	// Assign operators ...
	Long& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			PyObject* tmp_obj = PyNumber_Long(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Long& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			PyObject* tmp_obj = PyNumber_Long(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Long& operator= (Int value)
	{
		Set(PyLong_FromLong(long(value)), eTakeOwnership);
		return *this;
	}
	Long& operator= (long value)
	{
		Set(PyLong_FromLong(value),  eTakeOwnership);
		return *this;
	}
	Long& operator= (unsigned long value)
	{
		Set(PyLong_FromUnsignedLong(value),  eTakeOwnership);
		return *this;
	}
	Long& operator= (long long value)
	{
		Set(PyLong_FromLongLong(value),  eTakeOwnership);
		return *this;
	}
	Long& operator= (unsigned long long value)
	{
		Set(PyLong_FromUnsignedLongLong(value),  eTakeOwnership);
		return *this;
	}
	Long& operator= (double value)
	{
		Set(PyLong_FromDouble(value),  eTakeOwnership);
		return *this;
	}

public:
	// Type conversion operators ...
	operator long() const
	{
		return PyLong_AsLong(Get());
	}
	operator unsigned long() const
	{
		return PyLong_AsUnsignedLong(Get());
	}
	operator long long() const
	{
		return PyLong_AsLongLong(Get());
	}
	operator unsigned long long() const
	{
		return PyLong_AsUnsignedLongLong(Get());
	}
	operator double() const
	{
		return PyLong_AsDouble(Get());
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyLong_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyLong_CheckExact (obj);
	}
};

// PyFloat_Type
class Float : public Object
{
/* Return Python Float from string PyObject.  Second argument ignored on
input, and, if non-NULL, NULL is stored into *junk (this tried to serve a
purpose once but can't be made to work as intended). */
//PyAPI_FUNC(PyObject *) PyFloat_FromString(PyObject*, char** junk);

/* Extract C double from Python Float.  The macro version trades safety for
speed. */
//PyAPI_FUNC(double) PyFloat_AsDouble(PyObject *);

/* Write repr(v) into the char buffer argument, followed by null byte.  The
buffer must be "big enough"; >= 100 is very safe.
PyFloat_AsReprString(buf, x) strives to print enough digits so that
PyFloat_FromString(buf) then reproduces x exactly. */
//PyAPI_FUNC(void) PyFloat_AsReprString(char*, PyFloatObject *v);

/* Write str(v) into the char buffer argument, followed by null byte.  The
buffer must be "big enough"; >= 100 is very safe.  Note that it's
unusual to be able to get back the Float you started with from
PyFloat_AsString's result -- use PyFloat_AsReprString() if you want to
preserve precision across conversions. */
//PyAPI_FUNC(void) PyFloat_AsString(char*, PyFloatObject *v);

// ???
//PyAPI_FUNC(Int) _PyFloat_Pack4(double x, unsigned char *p, int le);
//PyAPI_FUNC(Int) _PyFloat_Pack8(double x, unsigned char *p, int le);
//PyAPI_FUNC(double) _PyFloat_Unpack4(const unsigned char *p, int le);
//PyAPI_FUNC(double) _PyFloat_Unpack8(const unsigned char *p, int le);

// PyObject* PyFloat_FromDouble(   double v)
// double PyFloat_AS_DOUBLE(   PyObject *pyfloat)

public:
	Float(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
	}
	Float(const Object& obj)
	{
		PyObject* tmp_obj = PyNumber_Float(obj);
		if ( !tmp_obj ) {
			throw TypeError("Invalid conversion");
		}
		Set(tmp_obj, eTakeOwnership);
	}
	Float(const Float& obj)
	: Object(obj)
	{
	}
	Float(double value = 0.0)
	: Object(PyFloat_FromDouble(value), eTakeOwnership)
	{
	}

public:
	// Assign operators ...
	Float& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			PyObject* tmp_obj = PyNumber_Float(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Float& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			PyObject* tmp_obj = PyNumber_Float(obj);
			if ( !tmp_obj ) {
				throw TypeError("Invalid conversion");
			}
			Set(tmp_obj, eTakeOwnership);
		}
		return *this;
	}
	Float& operator= (double value)
	{
		Set(PyFloat_FromDouble(value),  eTakeOwnership);
		return *this;
	}

public:
	// Type conversion operators ...
	operator double() const
	{
		return PyFloat_AsDouble(Get());
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyFloat_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyFloat_CheckExact (obj);
	}
};

class Complex : public Object
{
// Complex Numbers as C Structures ...
//PyAPI_FUNC(Py_complex) c_sum(Py_complex, Py_complex);
//PyAPI_FUNC(Py_complex) c_diff(Py_complex, Py_complex);
//PyAPI_FUNC(Py_complex) c_neg(Py_complex);
//PyAPI_FUNC(Py_complex) c_prod(Py_complex, Py_complex);
//PyAPI_FUNC(Py_complex) c_quot(Py_complex, Py_complex);
//PyAPI_FUNC(Py_complex) c_pow(Py_complex, Py_complex);

//PyAPI_FUNC(PyObject *) PyComplex_FromCComplex(Py_complex);
//PyAPI_FUNC(PyObject *) PyComplex_FromDoubles(double real, double imag);
//PyAPI_FUNC(double) PyComplex_RealAsDouble(PyObject *op);
//PyAPI_FUNC(double) PyComplex_ImagAsDouble(PyObject *op);
//PyAPI_FUNC(Py_complex) PyComplex_AsCComplex(PyObject *op);

public:
	Complex(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
	}
	Complex(const Object& obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
		Set(obj);
	}
	Complex(const Complex& obj)
	: Object(obj)
	{
	}
	Complex(double v = 0.0, double w = 0.0)
	: Object(PyComplex_FromDoubles(v, w), eTakeOwnership)
	{
	}

public:
	// Assign operators ...
	Complex& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	Complex& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	Complex& operator= (double value)
	{
		Set(PyComplex_FromDoubles(value, 0.0),  eTakeOwnership);
		return *this;
	}

public:
	// Type conversion operators ...
	operator double() const
	{
		return PyFloat_AsDouble(Get());
	}

public:
	double GetReal() const
	{
		return PyComplex_RealAsDouble(Get());
	}
	double GetImag() const
	{
		return PyComplex_ImagAsDouble(Get());
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PyComplex_Check (obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyComplex_CheckExact (obj);
	}
};

// Not implemented yet ...
class Char : public Object
{
public:
protected:
private:
};

// PyString_Type
// PyBaseString_Type
class PString : public Object
{
//PyAPI_FUNC(PyObject *) PyString_FromFormatV(const char*, va_list)
//                Py_GCC_ATTRIBUTE((format(printf, 1, 0)));
//PyAPI_FUNC(PyObject *) PyString_FromFormat(const char*, ...)
//                Py_GCC_ATTRIBUTE((format(printf, 1, 2)));
//PyAPI_FUNC(PyObject *) PyString_Repr(PyObject *, int);
//PyAPI_FUNC(void) PyString_Concat(PyObject **, PyObject *);
//PyAPI_FUNC(void) PyString_ConcatAndDel(PyObject **, PyObject *);
//PyAPI_FUNC(Int) _PyString_Resize(PyObject **, int);
//PyAPI_FUNC(Int) _PyString_Eq(PyObject *, PyObject*);
//PyAPI_FUNC(PyObject *) PyString_Format(PyObject *, PyObject *);
//PyAPI_FUNC(PyObject *) _PyString_FormatLong(PyObject*, int, int,
//                          int, char**, int*);
//PyAPI_FUNC(PyObject *) PyString_DecodeEscape(const char *, int,
//                           const char *, int,
//                           const char *);

//PyAPI_FUNC(void) PyString_InternInPlace(PyObject **);
//PyAPI_FUNC(void) PyString_InternImmortal(PyObject **);
//PyAPI_FUNC(PyObject *) PyString_InternFromString(const char *);
//PyAPI_FUNC(void) _Py_ReleaseInternedStrings(void);
//PyAPI_FUNC(PyObject *) _PyString_Join(PyObject *sep, PyObject *x);
//PyAPI_FUNC(PyObject*) PyString_Decode(
//    const char *s,              /* encoded string */
//    int size,                   /* size of buffer */
//    const char *encoding,       /* encoding */
//    const char *errors          /* error handling */
//    );
//PyAPI_FUNC(PyObject*) PyString_Encode(
//    const char *s,              /* string char buffer */
//    int size,                   /* number of chars to encode */
//    const char *encoding,       /* encoding */
//    const char *errors          /* error handling */
//    );

//PyAPI_FUNC(PyObject*) PyString_AsEncodedObject(
//    PyObject *str,      /* string object */
//    const char *encoding,   /* encoding */
//    const char *errors      /* error handling */
//    );
//PyAPI_FUNC(PyObject*) PyString_AsDecodedObject(
//    PyObject *str,      /* string object */
//    const char *encoding,   /* encoding */
//    const char *errors      /* error handling */
//    );

//PyAPI_FUNC(Int) PyString_AsStringAndSize(
//    register PyObject *obj, /* string or Unicode object */
//    register char **s,      /* pointer to buffer variable */
//    register int *len       /* pointer to length variable or NULL
//                   (only possible for 0-terminated
//                   strings) */
//    );

// int PyString_GET_SIZE(  PyObject *string)
// char* PyString_AS_STRING(   PyObject *string)

// int PyUnicode_GET_SIZE( PyObject *o)
// int PyUnicode_GET_DATA_SIZE(    PyObject *o)
// Py_UNICODE* PyUnicode_AS_UNICODE(   PyObject *o)
// const char* PyUnicode_AS_DATA(  PyObject *o)
// int Py_UNICODE_ISSPACE( Py_UNICODE ch)
// int Py_UNICODE_ISLOWER( Py_UNICODE ch)
// int Py_UNICODE_ISUPPER( Py_UNICODE ch)
// int Py_UNICODE_ISTITLE( Py_UNICODE ch)
// int Py_UNICODE_ISLINEBREAK( Py_UNICODE ch)
// int Py_UNICODE_ISDECIMAL(   Py_UNICODE ch)
// int Py_UNICODE_ISDIGIT( Py_UNICODE ch)
// int Py_UNICODE_ISNUMERIC(   Py_UNICODE ch)
// int Py_UNICODE_ISALPHA( Py_UNICODE ch)
// int Py_UNICODE_ISALNUM( Py_UNICODE ch)
// Py_UNICODE Py_UNICODE_TOLOWER(  Py_UNICODE ch)
// Py_UNICODE Py_UNICODE_TOUPPER(  Py_UNICODE ch)
// Py_UNICODE Py_UNICODE_TOTITLE(  Py_UNICODE ch)
// int Py_UNICODE_TODECIMAL(   Py_UNICODE ch)
// int Py_UNICODE_TODIGIT( Py_UNICODE ch)
// double Py_UNICODE_TONUMERIC(    Py_UNICODE ch)
// PyObject* PyUnicode_FromUnicode(    const Py_UNICODE *u, int size)
// Py_UNICODE* PyUnicode_AsUnicode(    PyObject *unicode)
// int PyUnicode_GetSize(  PyObject *unicode)
// PyObject* PyUnicode_FromEncodedObject(  PyObject *obj, const char *encoding, const char *errors)
// PyObject* PyUnicode_FromObject( PyObject *obj)
// PyObject* PyUnicode_FromWideChar(   const wchar_t *w, int size)
// int PyUnicode_AsWideChar(   PyUnicodeObject *unicode, wchar_t *w, int size)

public:
	PString(void)
	: Object(PyString_FromStringAndSize("", 0), eTakeOwnership)
	{
	}
	PString(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	PString(const Object& obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
		Set(obj);
	}
	PString(const PString& obj)
	: Object(obj)
	{
	}
	PString(const String& str)
	{
		operator= (str);
	}
	PString(const char* str)
	{
		operator= (str);
	}
	PString(const char* str, size_t str_size)
	{
		operator= (String(str, str_size));
	}

public:
	// Assign operators ...
	PString& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	PString& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	PString& operator= (const String& str)
	{
		Set(PyString_FromStringAndSize(~str, str.GetCount()), eTakeOwnership);
		return *this;
	}
	PString& operator= (const char* str)
	{
		return operator= (String(str));
	}

public:
	size_t GetSize (void) const
	{
		if ( PyUnicode_Check(Get()) ) {
			return static_cast<size_t>( PyUnicode_GET_SIZE( Get() ) );
		} else {
			return static_cast<size_t>( PyString_Size ( Get() ) );
		}
	}
	operator String (void) const
	{
		return AsSring();
	}

	String AsSring(void) const
	{
		if( PyUnicode_Check(Get()) ) {
			return FromUnicodeBuffer((wchar*)PyUnicode_AS_UNICODE(Get()), PyUnicode_GET_SIZE(Get()));
		}
		
		return String(PyString_AsString(Get()), PyString_Size(Get()));
	}

public:
	static bool HasExactSameType(PyObject* obj)
	{
		return PyString_CheckExact(obj)  ||  PyUnicode_CheckExact(obj);
	}
	static bool HasSameType(PyObject* obj)
	{
		return PyString_Check(obj)  ||  PyUnicode_Check(obj);
	}
};

// PyFile_Type
class File : public Object
{
// ????
//PyAPI_FUNC(Int) PyFile_SetEncoding(PyObject *, const char *);
//PyAPI_FUNC(Int) PyObject_AsFileDescriptor(PyObject *);


// Documented ...
//PyAPI_FUNC(PyObject *) PyFile_FromString(char *, char *);
//PyAPI_FUNC(PyObject *) PyFile_FromFile(FILE *, char *, char *, (*)(FILE *));
//PyAPI_FUNC(FILE *) PyFile_AsFile(PyObject *);
//PyAPI_FUNC(PyObject *) PyFile_GetLine(PyObject *, int);
//PyAPI_FUNC(PyObject *) PyFile_Name(PyObject *);
//PyAPI_FUNC(void) PyFile_SetBufSize(PyObject *, int);
// int PyFile_Encoding(    PyFileObject *p, char *enc)
//PyAPI_FUNC(Int) PyFile_SoftSpace(PyObject *, int);
//PyAPI_FUNC(Int) PyFile_WriteObject(PyObject *, PyObject *, int);
//PyAPI_FUNC(Int) PyFile_WriteString(const char *, PyObject *);

public:
public:
	static bool HasExactSameType(PyObject* obj)
	{
		return PyFile_CheckExact(obj);
	}
	static bool HasSameType(PyObject* obj)
	{
		return PyFile_Check(obj);
	}
};

///////////////////////////////////////////////////////////////////////////
inline Object operator+ (const Object& a, int j)
{
	PyObject* tmp_obj = PyNumber_Add(a.Get(), Int(j).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Add");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator+ (const Object& a, double v)
{
	PyObject* tmp_obj = PyNumber_Add(a.Get(), Float(v).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Add");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator+ (Int j, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Add(Int(j).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Add");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator+ (double v, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Add(Float(v).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Add");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator- (const Object& a, int j)
{
	PyObject* tmp_obj = PyNumber_Subtract(a.Get(), Int(j).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Subtract");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator- (const Object& a, double v)
{
	PyObject* tmp_obj = PyNumber_Subtract(a.Get(), Float(v).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Subtract");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator- (Int j, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Subtract(Int(j).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Subtract");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator- (double v, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Subtract(Float(v).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Subtract");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator* (const Object& a, int j)
{
	PyObject* tmp_obj = PyNumber_Multiply(a.Get(), Int(j).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Multiply");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator* (const Object& a, double v)
{
	PyObject* tmp_obj = PyNumber_Multiply(a.Get(), Float(v).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Multiply");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator* (Int j, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Multiply(Int(j).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Multiply");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator* (double v, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Multiply(Float(v).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Multiply");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator/ (const Object& a, int j)
{
	PyObject* tmp_obj = PyNumber_Divide(a.Get(), Int(j).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Divide");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator/ (const Object& a, double v)
{
	PyObject* tmp_obj = PyNumber_Divide(a.Get(), Float(v).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Divide");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator/ (Int j, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Divide(Int(j).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Divide");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator/ (double v, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Divide(Float(v).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Divide");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator% (const Object& a, int j)
{
	PyObject* tmp_obj = PyNumber_Remainder(a.Get(), Int(j).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Remainder");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator% (const Object& a, double v)
{
	PyObject* tmp_obj = PyNumber_Remainder(a.Get(), Float(v).Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Remainder");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator% (Int j, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Remainder(Int(j).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Remainder");
	}
	return Object(tmp_obj, eTakeOwnership);
}
inline Object operator% (double v, const Object& b)
{
	PyObject* tmp_obj = PyNumber_Remainder(Float(v).Get(), b.Get());
	if ( !tmp_obj ) {
		throw ArithmeticError("PyNumber_Remainder");
	}
	return Object(tmp_obj, eTakeOwnership);
}

}                                       // namespace pythonpp

END_UPP_NAMESPACE

#endif                                  // PYTHONPP_PDT_H

