/*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_EXT_H
#define PYTHONPP_EXT_H

#include "PythonPP_seq.h"
#include "PythonPP_dict.h"

NAMESPACE_UPP

namespace pythonpp
{

extern "C"
{
	typedef PyObject* (*TMethodVarArgsHandler)(PyObject* self, PyObject* args);
	typedef PyObject* (*TMethodKeywordHandler)(PyObject* self, PyObject* args, PyObject* dict);
}

//////////////////////////////////////////////////////////////////////////
/// Introduces constructor methods for a python PyMethodDef structure ...
struct MethodDef : Moveable<MethodDef>, public PyMethodDef
{
public:
	MethodDef();
	MethodDef(const char* name, PyCFunction func, int flags = 1, const char* doc = NULL);
	MethodDef(const MethodDef& other);

	MethodDef& operator = (const MethodDef& other);
};

//extern "C" void DoNotDeallocate(void*)
//{
//}

//////////////////////////////////////////////////////////////////////////
extern "C"
void standard_dealloc(PyObject* obj);

//////////////////////////////////////////////////////////////////////////
// PyTypeObject is inherited from a PyVarObject (C-kind of inheritance) ...
class ExtType : public PyTypeObject
{
public:
	ExtType(
		size_t basic_size,
		destructor dr = standard_dealloc,
		PyTypeObject* base = &PyBaseObject_Type);

public:
	PyTypeObject* GetObjType(void)
	{
		return this;
	}
	const PyTypeObject* GetObjType(void) const
	{
		return this;
	}
	void SetName(const char* name)
	{
		tp_name = const_cast<char*>(name);
	}
	void SetDescription(const char* descr)
	{
		tp_doc = const_cast<char*>(descr);
	}

public:
	void SupportGetAttr(getattrfunc func)
	{
		tp_getattr = func;
	}

private:
	void BasicInit();
};

inline
bool operator ==(const Object& l, const ExtType& r)
{
	return l.GetObjType() == r.GetObjType();
}

inline
bool operator ==(const ExtType& l, const Object& r)
{
	return l.GetObjType() == r.GetObjType();
}

//////////////////////////////////////////////////////////////////////////
namespace bind
{

enum EBindType { eReadOnly, eReadWrite };

class Base
{
public:
	Base(EBindType type = eReadOnly)
	: m_Type(type)
	{
	}
	virtual ~Base()
	{
	}

public:
	bool IsReadOnly() const
	{
		return m_Type == eReadOnly;
	}
	virtual PyObject* Get() const = 0;
	void Set(PyObject* value) const
	{
		if (IsReadOnly()) {
			throw AttributeError("Read-only property");
		}
		SetInternal(value);
	}

protected:
	virtual void SetInternal(PyObject* value) const
	{
	}

private:
	EBindType   m_Type;
};

class Long : public Base
{
public:
	Long(long& value, EBindType type = eReadOnly)
	: Base(type)
	, m_Value(&value)
	{
	}
	virtual ~Long()
	{
	}

public:
	virtual PyObject* Get() const
	{
		return PyLong_FromLong(*m_Value);
	}

protected:
	virtual void SetInternal(PyObject* value) const
	{
		long tmp_value = PyLong_AsLong(value);
		Error::Check();
		*m_Value = tmp_value;
	}

private:
	long* const m_Value;
};

class PString : public Base
{
public:
	PString(String& value, EBindType type = eReadOnly)
	: Base(type)
	, m_Value(&value)
	{
	}
	virtual ~PString()
	{
	}

public:
	virtual PyObject* Get() const
	{
		return PyString_FromString(m_Value->Begin());
	}

protected:
	virtual void SetInternal(PyObject* value) const
	{
		String tmp_value = String(PyString_AsString(value), static_cast<size_t>(PyString_Size(value)));
		Error::Check();
		*m_Value = tmp_value;
	}

private:
	String* const m_Value;
};

template <class T>
class Object : public Base
{
public:
	typedef PyObject* (T::*TGetFunc)(void) const;
	typedef void (T::*TSetFunc)(PyObject* value);

public:
	Object(T& obj, TGetFunc get, TSetFunc set = NULL)
	: Base(set == NULL ? eReadOnly : eReadWrite)
	, m_Obj(&obj)
	, m_GetFunc(get)
	, m_SetFunc(set)
	{
	}
	virtual ~Object()
	{
	}

public:
	virtual PyObject* Get() const
	{
		PyObject* obj = (m_Obj->*m_GetFunc)();
		Py_INCREF(obj);
		return obj;
	}

protected:
	virtual void SetInternal(PyObject* value) const
	{
		(m_Obj->*m_SetFunc)(value);
	}

private:
	T* const    m_Obj;
	TGetFunc    m_GetFunc;
	TSetFunc    m_SetFunc;
};

}

//////////////////////////////////////////////////////////////////////////
template<size_t N>
inline
void resize(Vector<MethodDef>& container)
{
	if (container.GetCount() < N) {
		container.SetCount(N);
	}
}

template<>
inline
void resize<0>(Vector<MethodDef>& /*container*/)
{
	;
}

//////////////////////////////////////////////////////////////////////////
template <class T>
class ExtObject : public PyObject
{
public:
	ExtObject()
	{
		// This is not an appropriate place for initialization ....
//            PyObject_INIT(this, GetType().GetPyTypeObject());
	}

	~ExtObject()
	{
	}

public:
	typedef Object (T::*TMethodVarArgsFunc)(const Tuple& args);
	/// Workaround for GCC 2.95
	struct SFunct : Moveable<SFunct>
	{
		SFunct(const TMethodVarArgsFunc& funct)
		: m_Funct(funct)
		{
		}

		operator TMethodVarArgsFunc() const
		{
			return m_Funct;
		}

		TMethodVarArgsFunc m_Funct;
	};

	// A helper class for generating functions ...
	template<size_t N = 0>
	class CClass
	{
	public:
		CClass()
		{
			// Finalyze a method definition ...
			GetMethodHndlList().push_back(MethodDef());
		}

	public:
		CClass<N + 1> Def(const char* name, TMethodVarArgsFunc func, const char* doc = 0)
		{
			TMethodHndlList& hndl_list = GetMethodHndlList();

			// Prepare data for python ...
			resize<N>(hndl_list);
			hndl_list[ePosition] = MethodDef(name, (PyCFunction)HandleMethodVarArgs, METH_VARARGS, doc);

			// Prepare data for our handler ...
			GetMethodList().push_back(func);

			return CClass<N + 1>();
		}

		static PyObject* HandleMethodVarArgs(PyObject* self, PyObject* args)
		{
			const TMethodVarArgsFunc func = GetMethodList()[ePosition];
			T* obj = static_cast<T*>(self);

			try {
				const Tuple args_tuple(args);

				return IncRefCount((obj->*func)(args_tuple));
			}
			/* CException is not defined here. We must translate all CException
			* to Error in user's code.
			catch (const CException& e) {
				Error::SetString(e.what());
			}
			*/
			catch(const Error&) {
				// An error message is already set by a previosly raised exception ...
				return NULL;
			}
			catch(...) {
				Error::SetString("Unknown error during executing of a method");
			}

			// NULL means "error".
			return NULL;
		}

	private:
		enum EPosition {ePosition = N};
	};

	template <size_t N> friend class CClass;

	static void Declare(
		const char* name, 
		const char* descr = 0,
		PyTypeObject* base = &PyBaseObject_Type
		)
	{ 
		ASSERT(sm_Base == NULL);
		sm_Base = base;

		ExtType& type = GetType();

		type.SetName(name);
		if (descr) {
			type.SetDescription(descr);
		}
		type.SupportGetAttr(GetAttrImpl);
	}

	static CClass<1> Def(const char* name, TMethodVarArgsFunc func, const char* doc = 0)
	{
		return CClass<0>().Def(name, func, doc);
	}

public:
	// Return a python object type.
	static ExtType& GetType()
	{
		ASSERT(sm_Base != NULL);
		static ExtType obj_type(sizeof(T), deallocator, sm_Base);

		return obj_type;
	}

	static Object& GetTypeObject()
	{
		static Object obj((PyObject*)&GetType(), pythonpp::eAcquireOwnership);

		return obj;
	}

	// Just delete an object ...
	static void deallocator (PyObject* obj)
	{
		delete static_cast<T*>(obj);
	}

protected:
	static void PrepareForPython(ExtObject<T>* self)
	{
		// Borrowed reference.
		PyObject_Init(self, GetType().GetObjType());
	}

protected:
	typedef ArrayMap<String, bind::Base> TAttrList;
	TAttrList m_AttrList;

	void ROAttr(const String& name, long& value)
	{
		m_AttrList[name] = new bind::Long(value);
	}
	void ROAttr(const String& name, String& value)
	{
		m_AttrList[name] = new bind::PString(value);
	}
	void ROAttr(const String& name, Object& value)
	{
		m_AttrList.Add(name, new bind::Object<Object>(value, &Object::Get));
	}

	void RWAttr(const String& name, long& value)
	{
		m_AttrList[name] = new bind::Long(value, bind::eReadWrite);
	}
	void RWAttr(const String& name, String& value)
	{
		m_AttrList[name] = new bind::PString(value, bind::eReadWrite);
	}

private:
	static PyTypeObject* sm_Base;

private:
	typedef Vector<MethodDef>  TMethodHndlList;
	static TMethodHndlList      sm_MethodHndlList;

	static TMethodHndlList& GetMethodHndlList()
	{
		return sm_MethodHndlList;
	}

	static PyObject* GetAttrImpl(PyObject* self, char* name)
	{
		ASSERT(self != NULL);
		ExtObject<T>* obj_ptr = static_cast<ExtObject<T>* >(self);
		int ind = obj_ptr->m_AttrList.Find(name);

		if (ind >= 0) {
			// Return an attribute value ...
			return obj_ptr->m_AttrList[ind].Get();
		}

		// Classic python implementation ...
		// It will do a linear search within the m_MethodHndlList table ...
		return Py_FindMethod(GetMethodHndlList().Begin(), self, name);
	}

private:
	typedef Vector<SFunct>  TMethodList;
	static TMethodList      sm_MethodList;

	static TMethodList& GetMethodList()
	{
		return sm_MethodList;
	}
};

template <class T> PyTypeObject* ExtObject<T>::sm_Base = NULL;
template <class T> typename ExtObject<T>::TMethodHndlList ExtObject<T>::sm_MethodHndlList;
template <class T> typename ExtObject<T>::TMethodList ExtObject<T>::sm_MethodList;

//////////////////////////////////////////////////////////////////////////
template <class T>
class ExtModule : public PyObject
{
public:
	ExtModule(const char* name, const char* descr = 0)
	: m_Module(NULL)
	{
		m_Module = Py_InitModule4(
			const_cast<char*>(name),
			&GetMethodHndlList().front(),
			const_cast<char*>(descr),
			this,
			PYTHON_API_VERSION);
		if (!m_Module) {
			throw SystemError("Cannot initialize module");
		}
	}
	~ExtModule()
	{
	}

public:
	typedef Object (T::*TMethodVarArgsFunc)(const Tuple& args);
	/// Workaround for GCC 2.95
	struct SFunct
	{
		SFunct(const TMethodVarArgsFunc& funct)
		: m_Funct(funct)
		{
		}

		operator TMethodVarArgsFunc() const
		{
			return m_Funct;
		}

		TMethodVarArgsFunc m_Funct;
	};

	// A helper class for generating functions ...
	template<size_t N = 0>
	class CClass
	{
	public:
		CClass()
		{
			// Finalyze method definition ...
			GetMethodHndlList().push_back(MethodDef());
		}

	public:
		CClass<N + 1> Def(const char* name, TMethodVarArgsFunc func, const char* doc = 0)
		{
			// Prepare data for python ...
			GetMethodHndlList()[ePosition] = MethodDef(name, HandleMethodVarArgs, METH_VARARGS, doc);

			// Prepare data for our handler ...
			GetMethodList().push_back(func);

			return CClass<N + 1>();
		}
		static PyObject* HandleMethodVarArgs(PyObject* self, PyObject* args)
		{
			const TMethodVarArgsFunc func = GetMethodList()[ePosition];
			T* obj = static_cast<T*>(self);

			try {
				const Tuple args_tuple(args);

				return IncRefCount((obj->*func)(args_tuple));
			}
			catch(const Error&) {
				// An error message is already set by an exception ...
				return NULL;
			}
			catch(...) {
				Error::SetString("Unknown error during executing of a method");
			}
			// NULL means "error".
			return NULL;
		}

	private:
		enum {ePosition = N};
	};

	static CClass<1> Def(const char* name, TMethodVarArgsFunc func, const char* doc = 0)
	{
		return CClass<0>().Def(name, func, doc);
	}

public:
	// Return a python object type.
	static ExtType& GetType()
	{
		static ExtType obj_type(sizeof(T), deallocator);

		return obj_type;
	}

	// Just delete an object ...
	static void deallocator (PyObject* obj)
	{
		delete static_cast<T*>(obj);
	}

protected:
	static void PrepareForPython(ExtModule<T>* self)
	{
		// Borrowed reference.
		PyObject_Init(self, GetType().GetPyTypeObject());
	}

private:
	typedef Vector<MethodDef>  TMethodHndlList;
	static TMethodHndlList      sm_MethodHndlList;

	static TMethodHndlList& GetMethodHndlList()
	{
		return sm_MethodHndlList;
	}

	static PyObject* GetAttrImpl(PyObject* self, char* name)
	{
		// Classic python implementation ...
		// It will do a linear search within the m_MethodHndlList table ...
		return Py_FindMethod(&GetMethodHndlList().front(), self, name);
	}

private:
	typedef Vector<SFunct>  TMethodList;
	static TMethodList      sm_MethodList;

	static TMethodList& GetMethodList()
	{
		return sm_MethodList;
	}

private:
	PyObject* m_Module;
};

template <class T> typename ExtModule<T>::TMethodHndlList ExtModule<T>::sm_MethodHndlList;
template <class T> typename ExtModule<T>::TMethodList ExtModule<T>::sm_MethodList;


///////////////////////////////////////////////////////////////////////////////
// New development
///////////////////////////////////////////////////////////////////////////////

// An attempt to wrap a module ...
class ModuleExt
{
public:
	static void Declare(const String& name, PyMethodDef* methods);

public:
	static const String& GetName()
	{
		return m_Name;
	}
	static PyObject* GetPyModule()
	{
		return m_Module;
	}
	static void AddConstValue(const String& name, PyObject* value);
	static void AddConst(const String& name, const String& value);
	static void AddConst(const String& name, long value);

private:
	static String m_Name;
	static PyObject* m_Module;
};

///////////////////////////////////
// User-defined exceptions
///////////////////////////////////

template <class T, class B = StandardError>
class UserError : public B
{
public:
	UserError()
	{
	}
	UserError(const String& msg)
	: B(msg, GetPyException())
	{
	}

protected:
	UserError(const String& msg, PyObject* err_type)
	: B(msg, err_type)
	{
	}

public:
	static void Declare(const String& name)
	{
		ASSERT(m_Exception == NULL);
		ASSERT(ModuleExt::GetPyModule());
		const String full_name = ModuleExt::GetName() + "." + name;
		m_Exception = PyErr_NewException(const_cast<char*>(~full_name), B::GetPyException(), NULL);
		Error::Check(m_Exception);
		if (PyModule_AddObject(ModuleExt::GetPyModule(), const_cast<char*>(~name), m_Exception) == -1) {
			throw SystemError("Unable to add an object to a module");
		}
	}
	static void Declare(const String& name, const pythonpp::CDict& dict)
	{
		ASSERT(m_Exception == NULL);
		ASSERT(ModuleExt::GetPyModule());
		const String full_name = ModuleExt::GetName() + "." + name;
		m_Exception = PyErr_NewException(const_cast<char*>(~full_name), B::GetPyException(), dict);
		Error::Check(m_Exception);
		if (PyModule_AddObject(ModuleExt::GetPyModule(), const_cast<char*>(~name), m_Exception) == -1) {
			throw SystemError("Unable to add an object to a module");
		}
	}

public:
	static PyObject* GetPyException()
	{
		ASSERT(m_Exception);
		return m_Exception;
	}

private:
	static PyObject* m_Exception;
};

template <class T, class B> PyObject* UserError<T, B>::m_Exception(NULL);


} // namespace pythonpp

END_UPP_NAMESPACE

#endif                                  // PYTHONPP_EXT_H

