/*
*
* License: BSD
* Author: Sergey Sikorskiy
*
*/

#ifndef PYTHONPP_SEQ_H
#define PYTHONPP_SEQ_H

#include "PythonPP_object.h"

NAMESPACE_UPP

namespace pythonpp
{

template<class T> class SequnceHelper;

template<class T>
class SequnceProxy
{
protected:
	SequnceHelper<T>& s;   //< the sequence
	int offset;             //< item number
	T value;                //< value

public:

	SequnceProxy (SequnceHelper<T>& seq, int j)
	: s(seq), offset(j), value (s.GetItem(j))
	{
	}

	SequnceProxy (const SequnceProxy<T>& range)
	: s(range.s), offset(range.offset), value(range.value)
	{
	}

	SequnceProxy (Object& obj)
	: s(dynamic_cast< SequnceHelper<T>&>(obj))
	, offset( NULL )
	, value(s.getItem(offset))
	{
	}

	~SequnceProxy(void)
	{
	}

	operator const T&(void) const
	{
		return value;
	}

	operator T&(void)
	{
		return value;
	}

	SequnceProxy<T>& operator=(const SequnceProxy<T>& rhs)
	{
		value = rhs.value;
		s.SetItem(offset, value);
		return *this;
	}

	SequnceProxy<T>& operator=(const T& obj)
	{
		value = obj;
		s.SetItem(offset, value);
		return *this;
	}

}; // end of SequnceProxy


template<class T>
class SequnceHelper : public Object
{
public:
	typedef size_t size_type;
	typedef SequnceProxy<T> reference;
	typedef T const_reference;
	typedef SequnceProxy<T>* pointer;
	typedef int difference_type;
	typedef T value_type;

public:
//        virtual size_type max_size() const
//        {
//            return std::string::npos; // ?
//        }

//        virtual size_type capacity() const
//        {
//            return size();
//        }

//        virtual void swap(SequnceHelper<T>& c)
//        {
//            SequnceHelper<T> temp = c;
//            c = ptr();
//            set(temp.ptr());
//        }

	// Function name made STL compatible ...
	// !!! Temporary hack. This function should not be virdual. 1/12/2005 2:43PM ...
	virtual size_type size () const
	{
		return PySequence_Length (Get());
	}

public:
		/// ???
//        explicit SequnceHelper<T> ()
//            :Object(PyTuple_New(0), true)
//        {
//            validate();
//        }

	SequnceHelper(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Object(obj, ownership)
	{
		// !!! Somebody should do a type-check here !!!
	}

	SequnceHelper (const Object& obj)
	: Object(obj)
	{
		// !!! Somebody should do a type-check here !!!
	}

	// Assignment acquires new ownership of pointer

	SequnceHelper& operator= (const Object& rhs)
	{
		return (*this = *rhs);
	}
	virtual ~SequnceHelper(void)
	{
	}

//        SequnceHelper<T>& operator= (PyObject* obj)
//        {
//            if ( Get() != obj) {
//                Set (obj);
//            }
//            return *this;
//        }

	size_type GetLength (void) const
	{
		return PySequence_Length (Get());
	}

	// Element access
	const T operator[] (int index) const
	{
		return GetItem(index);
	}

	SequnceProxy<T> operator [] (int index)
	{
		return SequnceProxy<T>(*this, index);
	}

	T GetItem (int i) const
	{
		PyObject* obj = PySequence_GetItem (Get(), i);
		if (obj == NULL) {
			Error::Check();
		}
		return T(obj, eTakeOwnership);
	}

	// !!! Temporary hack. This function should not be virdual. 1/12/2005 2:39PM ...
	virtual void SetItem (int i, const T& obj)
	{
		if (PySequence_SetItem (Get(), i, obj) == -1)
		{
			throw SystemError("Cannot set item with a sequence");
		}
		IncRefCount(obj);
	}

	SequnceHelper<T> Repeat (int count) const
	{
		return SequnceHelper<T> (PySequence_Repeat (Get(), count), eTakeOwnership);
	}

	SequnceHelper<T> Concat (const SequnceHelper<T>& other) const
	{
		return SequnceHelper<T> (PySequence_Concat(Get(), *other), eTakeOwnership);
	}

	// more STL compatability
	const T front (void) const
	{
		return GetItem(0);
	}

	SequnceProxy<T> front(void)
	{
		return SequnceProxy<T>(this, 0);
	}

	const T back (void) const
	{
		return GetItem(size() - 1);
	}

	SequnceProxy<T> back(void)
	{
		return SequnceProxy<T>(this, size() - 1);
	}

	void VerifyLength(size_type required_size) const
	{
		if ( size() != required_size )
		{
			throw IndexError ("Unexpected SequnceHelper<T> length.");
		}
	}

	void VerifyLength(size_type min_size, size_type max_size) const
	{
		size_type n = size();
		if (n < min_size || n > max_size)
		{
			throw IndexError ("Unexpected SequnceHelper<T> length.");
		}
	}

public:
	static bool HasSameType(PyObject* obj)
	{
		return PySequence_Check (obj) == 1;
	}

public:
	// Iterators ...

	class iterator
	// : public std::iterator<std::random_access_iterator_tag, SequnceProxy<T>, int>
	{
	protected:
		friend class SequnceHelper<T>;
		SequnceHelper<T>* seq;
		int position;

	public:
		~iterator (void)
		{
		}

		iterator (void)
		: seq( 0 )
		, position( 0 )
		{
		}

		iterator (SequnceHelper<T>* s, int where)
		: seq( s )
		, position( where )
		{
		}

		iterator (const iterator& other)
		: seq( other.seq )
		, position( other.position )
		{
		}

		bool eql (const iterator& other) const
		{
			return (*seq == *other.seq) && (position == other.position);
		}

		bool neq (const iterator& other) const
		{
			return (*seq != *other.seq) || (position != other.position);
		}

		bool lss (const iterator& other) const
		{
			return (position < other.position);
		}

		bool gtr (const iterator& other) const
		{
			return (position > other.position);
		}

		bool leq (const iterator& other) const
		{
			return (position <= other.position);
		}

		bool geq (const iterator& other) const
		{
			return (position >= other.position);
		}

		SequnceProxy<T> operator*(void)
		{
			return SequnceProxy<T>(*seq, position);
		}

		SequnceProxy<T> operator[] (int i)
		{
			return SequnceProxy<T>(*seq, position + i);
		}

		iterator& operator=(const iterator& other)
		{
			if ( this != &other ) {
				seq = other.seq;
				position = other.position;
			}
			return *this;
		}

		iterator operator+(int n) const
		{
			return iterator(seq, position + n);
		}

		iterator operator-(int n) const
		{
			return iterator(seq, position - n);
		}

		iterator& operator+=(int n)
		{
			position = position + n;
			return *this;
		}

		iterator& operator-=(int n)
		{
			position = position - n;
			return *this;
		}

		int operator-(const iterator& other) const
		{
			if (*seq != *other.seq) {
				throw SystemError("SequnceHelper<T>::iterator comparison error");
			}
			return position - other.position;
		}

		// prefix ++
		iterator& operator++ (void)
		{
			++position;
			return *this;
		}
		// postfix ++
		iterator operator++ (int)
		{
			return iterator(seq, position++);
		}
		// prefix --
		iterator& operator-- (void)
		{
			--position;
			return *this;
		}
		// postfix --
		iterator operator-- (int)
		{
			return iterator(seq, position--);
		}
	};    // end of class SequnceHelper<T>::iterator

	iterator begin (void)
	{
		return iterator(this, 0);
	}

	iterator end (void)
	{
		return iterator(this, GetLength());
	}

	class const_iterator // : public std::iterator<std::random_access_iterator_tag, Object, int>
	{
	protected:
		friend class SequnceHelper<T>;
		const SequnceHelper<T>* seq;
		int position;

	public:
		~const_iterator (void)
		{
		}

		const_iterator (void)
		: seq( 0 )
		, position( 0 )
		{
		}

		const_iterator (const SequnceHelper<T>* s, int where)
		: seq( s )
		, position( where )
		{
		}

		const_iterator(const const_iterator& other)
		: seq( other.seq )
		, position( other.position )
		{
		}

		const T operator*(void) const
		{
			return seq->GetItem(position);
		}

		const T operator[] (int i) const
		{
			return seq->GetItem(position + i);
		}

		const_iterator& operator=(const const_iterator& other)
		{
			if ( this != &other ) {
				seq = other.seq;
				position = other.position;
			}
			return *this;
		}

		const_iterator operator+(int n) const
		{
			return const_iterator(seq, position + n);
		}

		bool operator == (const const_iterator& other) const
		{
			return (*seq == *other.seq) && (position == other.position);
		}

		bool operator != (const const_iterator& other) const
		{
			return (*seq != *other.seq) || (position != other.position);
		}

		bool lss (const const_iterator& other) const
		{
			return (position < other.position);
		}

		bool gtr (const const_iterator& other) const
		{
			return (position > other.position);
		}

		bool leq (const const_iterator& other) const
		{
			return (position <= other.position);
		}

		bool geq (const const_iterator& other) const
		{
			return (position >= other.position);
		}

		const_iterator operator-(int n)
		{
			return const_iterator(seq, position - n);
		}

		const_iterator& operator+=(int n)
		{
			position = position + n;
			return *this;
		}

		const_iterator& operator-=(int n)
		{
			position = position - n;
			return *this;
		}

		int operator-(const const_iterator& other) const
		{
			if (*seq != *other.seq) {
				throw RuntimeError ("SequnceHelper<T>::const_iterator::- error");
			}
			return position - other.position;
		}

		// prefix ++
		const_iterator& operator++ (void)
		{
			++position;
			return *this;
		}
		// postfix ++
		const_iterator operator++ (int)
		{
			return const_iterator(seq, position++);
		}
		// prefix --
		const_iterator& operator-- ()
		{
			--position;
			return *this;
		}
		// postfix --
		const_iterator operator-- (int)
		{
			return const_iterator(seq, position--);
		}
	};    // end of class SequnceHelper<T>::const_iterator

	const_iterator begin (void) const
	{
		return const_iterator(this, 0);
	}

	const_iterator end (void) const
	{
		return const_iterator(this, GetLength());
	}
};

typedef SequnceHelper<Object> Sequence;

class List;

// PyTuple_Type
class Tuple : public Sequence
{
//PyAPI_FUNC(CInt) _PyTuple_Resize(PyObject **, int);
//PyAPI_FUNC(PyObject *) PyTuple_Pack(CInt, ...);
// int PyTuple_GET_SIZE(   PyObject *p)
// PyObject* PyTuple_GET_ITEM( PyObject *p, int pos)
// void PyTuple_SET_ITEM(  PyObject *p, int pos, PyObject *o)

public:
	Tuple(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Sequence(obj, ownership)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	Tuple(const Object& obj)
	: Sequence(obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	Tuple(const Tuple& obj)
	: Sequence(obj)
	{
	}
	Tuple(const List& obj);
	/// Create a Tuple of size "size" and initialize it with python None
	Tuple(size_t size = 0)
	: Sequence(PyTuple_New (size), eTakeOwnership)
	{
		// *** WARNING *** PyTuple_SetItem does not increment the new item's reference
		// count, but does decrement the reference count of the item it replaces,
		// if not nil.  It does *decrement* the reference count if it is *not*
		// inserted in the tuple.

		for ( size_t i = 0; i < size; ++i ) {
			if ( PyTuple_SetItem (Get(), i, Py_None) == -1 ) {
				IncRefCount(Py_None);
				throw SystemError("PyTuple_SetItem error");
			}
			IncRefCount(Py_None);
		}
	}

public:
	// Assign operators ...
	Tuple& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	Tuple& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

public:
	void SetItem (int offset, const Object& obj)
	{
		// PyTuple_SetItem does not increment the new item's reference
		// count, but does decrement the reference count of the item it
		// replaces.
		// It does *decrement* the reference count if it is *not*
		// inserted in the tuple.

		if ( PyTuple_SetItem (Get(), offset, obj) == -1 ) {
			IncRefCount(obj);
			throw SystemError("");
		}
		IncRefCount(obj);
	}
	Object GetItem(int offset)
	{
		PyObject* obj = PyTuple_GetItem(Get(), offset);
		if ( !obj ) {
			throw SystemError("");
		}
		return Object(obj);
	}

	// Fast version. Acquires ownership of obj ...
	void SetItemFast (int offset, PyObject* obj)
	{
		if(PyTuple_SetItem (Get(), offset, obj) == -1 ) {
			IncRefCount(obj);
			throw SystemError("");
		}
	}
	// Fast version. Does not increment a counter ...
	PyObject* GetItemFast(int offset)
	{
		return PyTuple_GetItem(Get(), offset);
	}

	Tuple GetSlice (int i, int j) const
	{
		return Tuple(PyTuple_GetSlice(Get(), i, j), eTakeOwnership);
	}
	size_type size(void) const
	{
		return PyTuple_Size(Get());
	}

public:
	static bool HasExactSameType(PyObject* obj)
	{
		return PyTuple_CheckExact(obj);
	}
	static bool HasSameType(PyObject* obj)
	{
		return PyTuple_Check (obj);
	}
};

// PyList_Type
class List : public Sequence
{
// PyAPI_FUNC(PyObject *) _PyList_Extend(PyListObject *, PyObject *);
// int PyList_GET_SIZE(    PyObject *list)
// PyObject* PyList_GET_ITEM(  PyObject *list, int i)
// void PyList_SET_ITEM(   PyObject *list, int i, PyObject *o)

public:
	List(PyObject* obj, EOwnership ownership = eAcquireOwnership)
	: Sequence(obj, ownership)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	List(const Object& obj)
	: Sequence(obj)
	{
		if ( !HasExactSameType(obj) ) {
			throw TypeError("Invalid conversion");
		}
	}
	List(const List& obj)
	: Sequence(obj)
	{
	}
	/// Create a Tuple of size "size" and initialize it with python None
	List(size_t size = 0)
	: Sequence(PyList_New (size), eTakeOwnership)
	{
		for ( size_t i = 0; i < size; ++i ) {
			if ( PyList_SetItem (Get(), i, Py_None) != 0 ) {
				throw SystemError("");
			}
			IncRefCount(Py_None);
		}
	}

public:
	// Assign operators ...
	List& operator= (const Object& obj)
	{
		if ( this != &obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}
	List& operator= (PyObject* obj)
	{
		if ( Get() != obj ) {
			if ( !HasExactSameType(obj) ) {
				throw TypeError("Invalid conversion");
			}
			Set(obj);
		}
		return *this;
	}

public:
	void SetItem (int offset, const Object& obj)
	{
		// PyList_SetItem does not increment the new item's reference
		// count, but does decrement the reference count of the item it replaces,
		// if not nil.  It does *decrement* the reference count if it is *not*
		// inserted in the list.

		if ( PyList_SetItem (Get(), offset, obj) == -1 ) {
			IncRefCount(obj);
			throw SystemError("");
		}
		IncRefCount(obj);
	}
	Object GetItem(int offset)
	{
		PyObject* obj = PyList_GetItem(Get(), offset);
		if ( !obj ) {
			throw SystemError("");
		}
		return Object(obj);
	}

	// Fast version. Acquires ownership of obj ...
	void SetItemFast (int offset, PyObject* obj)
	{
		if(PyList_SetItem (Get(), offset, obj) != 0) {
			throw SystemError("");
		}
	}
	// Fast version. Does not increment a counter ...
	PyObject* GetItemFast(int offset)
	{
		return PyList_GetItem(Get(), offset);
	}

public:
	List GetSlice (int i, int j) const
	{
		return List (PyList_GetSlice (Get(), i, j), eTakeOwnership);
	}
	void SetSlice (int i, int j, const Object& obj)
	{
		if(PyList_SetSlice (Get(), i, j, obj) != 0) {
			throw SystemError("");
		}
	}

public:
	void Append (const Object& obj)
	{
		Append(obj.Get());
	}
	void Append (PyObject* obj)
	{
		if(PyList_Append (Get(), obj) == -1) {
			throw SystemError("");
		}
	}
	void Insert (int i, const Object& obj)
	{
		if(PyList_Insert (Get(), i, obj) == -1) {
			throw SystemError("");
		}
	}
	void Sort (void)
	{
		if(PyList_Sort(Get()) == -1) {
			throw SystemError("");
		}
	}
	void Reverse (void)
	{
		if(PyList_Reverse(Get()) == -1) {
			throw SystemError("");
		}
	}

	void Clear (void)
	{
		Set(PyList_New(0));
		IncRefCount(Get());
	}

public:
	// Function name is made STL compatible ...
	size_type size(void) const
	{
		return PyList_Size(Get());
	}

public:
	static bool HasExactSameType(PyObject* obj)
	{
		return PyList_CheckExact (obj);
	}
	static bool HasSameType(PyObject* obj)
	{
		return PyList_Check (obj);
	}
};

#if PY_VERSION_HEX >= 0x02040000
// PySet_Type
class Set : public Object
{
public:
public:
	static bool HasSameType(PyObject* obj)
	{
		return PyAnySet_Check(obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyAnySet_Check(obj);
	}
};

// PyFrozenSet_Type
class FrozenSet : public Object
{
public:
public:
	static bool HasSameType(PyObject* obj)
	{
		return PyAnySet_Check(obj);
	}
	static bool HasExactSameType(PyObject* obj)
	{
		return PyFrozenSet_CheckExact(obj);
	}
};
#endif

//////////////////////////////////////////////////////////////////////////
inline
Tuple::Tuple(const List& obj)
: Sequence(PyList_AsTuple(obj), eTakeOwnership)
{
}

}                                       // namespace pythonpp

END_UPP_NAMESPACE

#endif                                  // PYTHONPP_SEQ_H

