#include <plugin/tif/tif.h>
#include "Image.h"

NAMESPACE_UPP

Image ImageAutolevel(const ImageBuffer& img){
	return ImageAutolevelDet<ImageBuffer, RGBA, 8>(img);
}

Image ImageAutolevel(const ImageBuffer16& img){
	return ImageAutolevelDet<ImageBuffer16, RGBA16, 16>(img);
}

template<class ImageBUffT, class T, int Bits>
Image ImageAutolevelDet(const ImageBUffT &img){
	uint32 max_range = uint32(pow(2.0f, Bits));
	
	const T *s = ~img;
	const T *e = s+img.GetLength();
	
	int64 v_lower = (s->r + s->g + s->b) / 3;
	int64 v_upper = (s->r + s->g + s->b) / 3;
	int64 t_val;
	
	while(s < e){
		t_val = (s->r + s->g + s->b) / 3;
		s++;
		if(t_val < v_lower)
			v_lower = t_val;
		if(t_val > v_upper)
			v_upper = t_val;
	}
	
	s = ~img;
	ImageBuffer w(img.GetSize());
	Unmultiply(w);
	
	w.SetDots(img.GetDots());
	RGBA *t = w;
	uint8 v_k = 0;
	
	while(s < e) {
		t->r = int((0.5 + (0.0 + s->r - v_lower)*255/(v_upper - v_lower)));;
		t->g = int((0.5 + (0.0 + s->g - v_lower)*255/(v_upper - v_lower)));;
		t->b = int((0.5 + (0.0 + s->b - v_lower)*255/(v_upper - v_lower)));;
		t->a = int(s->a * 255 / max_range);
		t++;
		s++;
	}
	
	Premultiply(w);
	
	return w;
}

template<class ImageBUffT, class T, int Bits>
Image ImageEequalizeDet(const ImageBUffT &img){
	uint32 max_range = uint32(pow(2.0f, Bits));
	
	Buffer<uint64> histogram;
	histogram.Alloc(max_range, 0);
	
	const T *start = *img;
	const T *s = start;
	const T *e = s+img.GetLength();
	
	uint64 *hist = histogram;
	
	int64 v_lower = (s->r + s->g + s->b) / 3;
	int64 v_upper = (s->r + s->g + s->b) / 3;
	int64 t_val;
	
	while(s < e){
		t_val = (s->r + s->g + s->b) / 3;
		(*(hist + t_val))++;
		s++;
		if(t_val < v_lower)
			v_lower = t_val;
		if(t_val > v_upper)
			v_upper = t_val;
	}
	
	for(uint32 i=0;i<max_range;++i){
		if(i>0)
			*(hist+i) += *(hist+i-1);
	}
	
	s = start;
	ImageBuffer w(img.GetSize());
	Unmultiply(w);
	w.SetDots(img.GetDots);
	RGBA *t = w;
	uint8 v_k = 0;
	uint64 v_cdf_min = *(hist+v_lower);
	int v_lenth = img.GetLength();
	
	while(s < e) {
		v_k = uint8(0.5+(((0.0+*(histogram + (s->r+s->g+s->b)/3))-v_cdf_min)*(256-1)/(v_lenth-v_cdf_min)));
		t->r = v_k;
		t->g = v_k;
		t->b = v_k;
		t->a = uint8(s->a*255/max_range);
		t++;
		s++;
	}
	
	Premultiply(w);
	//w.SetHotSpots(img);
	
	return w;
}

Image LoadImageAutolevel(const char *filename, const String tiff_ext){
	if(tiff_ext.Find(GetFileExtPos(filename))!=-1){
		tiff *tif_file = TIFFFileStreamOpen(filename, "r");
		if(0==tif_file)
			return Image();
		
		Array<TIFFPage> tiff_pages;
		
		int directories = TIFFNumberOfDirectories(tif_file);
		uint32 total_w = 0;
		uint32 total_h = 0;
		Size total_size;
		uint16 max_bits = 0;
		uint16 max_samples = 0;
		
		for(int d = 0; d<directories;++d){
			TIFFPage &page = tiff_pages.Add();
			
			TIFFSetDirectory(tif_file, d);
			TIFFGetField(tif_file, TIFFTAG_IMAGEWIDTH, &page.width);
			TIFFGetField(tif_file, TIFFTAG_IMAGELENGTH, &page.height);
			total_w+=page.width;
			total_h+=page.height;
			float xres, yres;
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_XRESOLUTION, &xres);
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_YRESOLUTION, &yres);
			uint16 resunit;
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_RESOLUTIONUNIT, &resunit);
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_BITSPERSAMPLE, &page.bits_per_sample);
			if(max_bits<page.bits_per_sample)
				max_bits = page.bits_per_sample;
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_SAMPLESPERPIXEL, &page.samples_per_pixel);
			
			if(max_samples<page.samples_per_pixel)
				max_samples = page.samples_per_pixel;
			
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_PHOTOMETRIC, &page.photometric);
			double dots_per_unit = (resunit == RESUNIT_INCH ? 600.0 : resunit == RESUNIT_CENTIMETER
				? 600.0 / 2.54 : 0);
			page.dot_size.cx = (xres ? fround(page.width * dots_per_unit / xres) : 0);
			page.dot_size.cy = (yres ? fround(page.height * dots_per_unit / yres) : 0);
			total_size+=page.dot_size;
			page.alpha = false;
			uint16 extrasamples, *sampletypes;
			TIFFGetFieldDefaulted(tif_file, TIFFTAG_EXTRASAMPLES, &extrasamples, &sampletypes);
			for(int e = 0; e < extrasamples; e++)
				if(sampletypes[e] == EXTRASAMPLE_ASSOCALPHA) {
					page.alpha = true;
					break;
				}
		}
		
		if(max_bits == 16){
			ImageBuffer16 img_res;
			img_res.Create(int(total_w), int(total_h));
			//img_res.SetDots(total_size);
			
			uint16* buffer_t = (uint16 *)_TIFFmalloc((uint32)(total_w * sizeof(uint16) * max_samples));
			RGBA16 *t = img_res;
			
			for(int d = 0; d<directories;++d){
				TIFFPage &page = tiff_pages[d];
				
				uint32 i,j;
				uint16 a_t = int(pow(2.0f,16))-1;
			
				TIFFSetDirectory(tif_file, d);
				for(j=0; j<page.height;++j){
					TIFFReadScanline(tif_file, (byte *)(buffer_t), j, 0);
					// copy line into data
					uint16 *s = buffer_t;
					for (i = 0; i < page.width; i++){
						if (page.samples_per_pixel == 3){
							t->r = *(s);
							t->g = *(s+1);
							t->b = *(s+2);
							t->a = a_t;
						}else{
							uint16 val = *s;
							t->r = val;
							t->g = val;
							t->b = val;
							t->a = a_t;
						}
						s+=page.samples_per_pixel;
						++t;
					}
				}
			}
			_TIFFfree((byte *)buffer_t);
			TIFFClose(tif_file);
			Image img_result = ImageAutolevel(img_res);
			return img_result;
		}
		TIFFClose(tif_file);
		delete tif_file;
	}
	
	FileIn in(filename);
	One<StreamRaster> r = StreamRaster::OpenAny(in);
	if(!r)
		return Image();
	Image t_img = r->GetImage();
	ImageBuffer t_img_buff(t_img);
	
	return ImageAutolevel(t_img_buff);
}

END_UPP_NAMESPACE