#include "main.h"

#include <CtrlLib/CtrlLib.h>
#include <Scatter/Scatter.h>

using namespace Upp;

class ImageView : public TopWindow {
public:
	virtual bool Key(dword key, int);

private:
	ImageCtrl             img;
	FileList              files;
	Splitter              splitter;
	Splitter              splitter1;
	Scatter				  histogram;
	
	String                dir;
	FrameTop<Button>      dirup;

	void Load(const char *filename);
	void Enter();
	void DoDir();
	void DirUp();
	void SetHistogram(const Image& img);
	uint64* GetHistogram(const Image& img);
public:
	typedef ImageView CLASSNAME;

	void Serialize(Stream& s);
	void LoadDir(const char *d);
	void LoadDir()                      { LoadDir(dir); }

	ImageView();
};

void ImageView::Load(const char *filename){
	img.SetImage(Null);
	
	Image m;
	
	if(GetCtrl()){
		m = LoadImageAutolevel(filename, ".img, .tif, .tiff");
	}else{
		FileIn in(filename);
		One<StreamRaster> r = StreamRaster::OpenAny(in);
		if(!r)
			return;
		m = r->GetImage();
	}
	
	if(!m)
		return;
	
	Size rsz = img.GetSize();
	Size isz = m.GetSize();
	if(isz.cx >= rsz.cx || isz.cy >= rsz.cy) {
		if(isz.cx * rsz.cx < rsz.cy * isz.cy)
			rsz.cx = isz.cx * rsz.cy / isz.cy;
		else
			rsz.cy = isz.cy * rsz.cx / isz.cx;
		Image m1 = Rescale(m, rsz);
		SetHistogram(m1);
		img.SetImage(m1);
	}
	else{
		SetHistogram(m);
		img.SetImage(m);
	}
}

Image LoadImageAutolevel(const char *filename){
	return LoadImageAutolevel(filename, ".tif, .tiff");
}

void ImageView::SetHistogram(const Image& img){
	double v_max_y = 0;
	int v_curr_value = 0;
	
	Vector<XY> hist_data;
	for(int i=0;i<256;++i)
		hist_data << XY(i, 0);
	
	const RGBA *s = img;
	const RGBA *e = s + img.GetLength();
	while(s < e) {
		v_curr_value = (s->r+s->g+s->b)/3;
		hist_data[v_curr_value].y++;
		if(hist_data[v_curr_value].y>v_max_y)
			v_max_y = hist_data[v_curr_value].y;
		s++;
	}
	histogram.SetData(0,hist_data);
	histogram.SetRange(255, v_max_y, histogram.GetY2Range());
}

void ImageView::LoadDir(const char *d){
	files.Clear();
	dir = d;
	Title(dir);
	::Load(files, dir, "*.*");
	SortByExt(files);
}

void ImageView::DirUp(){
	String n = DirectoryUp(dir);
	LoadDir(dir);
	files.FindSetCursor(n);
}

void ImageView::Enter(){
	if(!files.IsCursor()) {
		Title(dir);
		return;
	}
	const FileList::File& f = files.Get(files.GetCursor());
	if(f.name == "..") {
		Title(dir);
		return;
	}
	String p = AppendFileName(dir, f.name);
	Title(p);
	if(!f.isdir)
		Load(p);
}

void ImageView::DoDir(){
	if(!files.IsCursor())
		return;
	const FileList::File& f = files.Get(files.GetCursor());
	if(!f.isdir)
		return;
	LoadDir(AppendFileName(dir, f.name));
}

bool ImageView::Key(dword key, int){
	if(key == K_ENTER) {
		DoDir();
		return true;
	}
	return false;
}

void ImageView::Serialize(Stream& s){
	int version = 0;
	s / version;
	SerializePlacement(s);
	files.SerializeSettings(s);
	s % dir;
	s % splitter;
}

ImageView::ImageView(){
	Vector<XY> hist_data;
	for(int i=0;i<256;++i)
		hist_data << XY(i, 0);
	
	histogram.SetTitle(t_("Histogram"));
	histogram.AddSeries(hist_data, t_("data"), true);
	histogram.SetMarkStyle(1,Scatter::RECTANGLE);
	histogram.SetXYMin(0,0);
	histogram.SetRange(255, 100, histogram.GetY2Range());
	histogram.SetShowMark(0, false);
	
	splitter1.Vert(files, histogram);
	splitter.Horz(splitter1, img);
	splitter.SetPos(2700);
	Add(splitter.SizePos());

	files.WhenEnterItem = THISBACK(Enter);
	files.WhenLeftDouble = THISBACK(DoDir);
	dirup.Height(max(CtrlImg::DirUp().GetSize().cy, Draw::GetStdFontCy() + 6));
	dirup.SetImage(CtrlImg::DirUp());
	dirup.NormalStyle();
	dirup <<= THISBACK(DirUp);
	files.AddFrame(dirup);

	Sizeable().Zoomable();

	dir = GetCurrentDirectory();
}

GUI_APP_MAIN
{
	ImageView x;
	LoadFromFile(x);
	x.LoadDir();
	x.Run();
	StoreToFile(x);
}
