#ifndef _TimberCore_QTFStr_h_
#define _TimberCore_QTFStr_h_

#include <Core/Core.h>
#include <RichText/RichText.h>

using namespace Upp;

#define FREE_TABLE			0
#define KEEP_CELL_ON_PAGE	1
#define KEEP_TABLE_ON_PAGE	2
class QTFStr
{
	public:
		class QTFFormat : public Moveable<QTFFormat>
		{
			public:
				int		tableKeep;
				bool	tableBorder;
				bool	bold;
				bool	strike;
				bool	underline;
				bool	subscript;
				bool	superscript;
				bool	italic;
				int		align;
				int		charSize;
				char	font;
				int		bottomCellMargin;
				int		topCellMargin;
				int		leftCellMargin;
				int		rightCellMargin;
				String	foreColor;
				String	backColor;
				
				QTFFormat();
				void Clear(void);
				QTFFormat(const QTFFormat &fmt);
				QTFFormat const &operator=(QTFFormat const &fmt);
		};
		
	private:
	
		// QTF string container
		String str;
		
		// settings stack
		Vector<QTFFormat> formatStack;
		
		// table depth level
		int tableLevel;

		// table running values
		Vector<int>				tableColumns;
		Vector<int>				tableColumn;
		Vector<Vector<int> >	columnWidths;;
		
		// formatting running values
		QTFFormat format;
		
		// error flag -- stops any processing if set
		bool err;
	
		// outputs format codearound string
		String FormatString(const String &s);
		
	protected:
	
	public:
	
		// constructor
		QTFStr();
		
		// empty string and reset all formats to default
		QTFStr &Clear(void);
		
		// clears all formatting options -- reset them to defaults
		QTFStr &ClearFormats(void);
		
		// push/pop format values
		QTFStr &PushFormat(void);
		QTFStr &PopFormat(void);

		// set/get format
		QTFStr &GetFormat(QTFFormat &fmt) { fmt = format; return *this; }
		QTFStr &SetFormat(QTFFormat const &fmt) { format = fmt; return *this; }

		// checks condition, if true sets error value, logs the (first) error with its QTF
		// and stops any further processing. Returns err flag
		bool CheckError(bool cond, String const &msg);
		
		// format change functions
		QTFStr &TableOnPage(void)		{ format.tableKeep = KEEP_TABLE_ON_PAGE; return *this; }
		QTFStr &CellOnPage(void)		{ format.tableKeep = KEEP_CELL_ON_PAGE; return *this; }
		QTFStr &FreeTable(void)			{ format.tableKeep = FREE_TABLE; return *this; }
		QTFStr &TableBorder(void)		{ format.tableBorder = true; return *this; }
		QTFStr &NoTableBorder(void)		{ format.tableBorder = false; return *this; }
		QTFStr &Bold(void)				{ format.bold = true; return *this; }
		QTFStr &NoBold(void)			{ format.bold = false; return *this; }
		QTFStr &Strike(void)			{ format.strike = true; return *this; }
		QTFStr &NoStrike(void)			{ format.strike = false; return *this; }
		QTFStr &Underline(void)			{ format.underline = true; return *this; }
		QTFStr &NoUnderline(void)		{ format.underline = false; return *this; }
		QTFStr &Subscript(void)			{ format.subscript = true; return *this; }
		QTFStr &NoSubscript(void)		{ format.subscript = false; return *this; }
		QTFStr &SuperScript(void)		{ format.superscript = true; return *this; }
		QTFStr &NoSuperScript(void)		{ format.superscript = false; return *this; }
		QTFStr &Italic(void)			{ format.italic = true; return *this; }
		QTFStr &NoItalic(void)			{ format.italic = false; return *this; }
		QTFStr &AlignLeft(void)			{ format.align = ALIGN_LEFT; return *this; }
		QTFStr &AlignCenter(void)		{ format.align = ALIGN_CENTER; return *this; }
		QTFStr &AlignRight(void)		{ format.align = ALIGN_RIGHT; return *this; }
		QTFStr &AlignJustify(void)		{ format.align = ALIGN_JUSTIFY; return *this; }
		QTFStr &CharSize(int siz)		{ format.charSize = siz; return *this; }
		QTFStr &IncCharSize()			{ if (format.charSize < 9) format.charSize++; return *this; }
		QTFStr &DecCharSize()			{ if (format.charSize > 0) format.charSize--; return *this; }
		QTFStr &Font(char fnt)			{ format.font = fnt; return *this; }
		QTFStr &ForeColor(String col = "")		{ format.foreColor = (col == "" ? "0" : col); return *this; }
		QTFStr &BackColor(String col = "")		{ format.backColor = (col == "" ? "2" : col); return *this; }
		QTFStr &LeftCellMargin(int m = 25)		{ format.leftCellMargin = m; return *this; }
		QTFStr &RightCellMargin(int m = 25)		{ format.topCellMargin = m; return *this; }
		QTFStr &TopCellMargin(int m = 15)		{ format.rightCellMargin = m; return *this; }
		QTFStr &BottomCellMargin(int m = 15)	{ format.bottomCellMargin = m; return *this; }
		
		// output text
		QTFStr &Txt(const String &s, int merge = 0);
		QTFStr &Txt(RichObject const &obj, int merge = 0);
		QTFStr &operator()(const String &s, int merge = 0) { return Txt(s, merge); }
		QTFStr &operator()(RichObject const &obj, int merge = 0) { return Txt(obj, merge); }
		
	
		QTFStr &Iml(String const &Name, int cx, int cy, bool keepAspect = true, int merge = 0);
		QTFStr &CenterIml(String const &Name, int cx, int cy, bool keepAspect = true, int merge = 0);
		
		QTFStr &Object(RichObject const &obj, int merge = 0);
		QTFStr &CenterObject(RichObject const &obj, int merge = 0);
		
		// page title
		QTFStr &PageTitle(String const &s1, String const &s2);

		// outputs a double with given width and decimals
		// right justified
		QTFStr &Dbl(double d, int width, int decimals);
		
		/////////////////////////////////////////////////////////////////////////////////////////////
		//                                      TABLE HANDLING                                     //
		/////////////////////////////////////////////////////////////////////////////////////////////
		QTFStr &StartTable(Vector<int> const &colW);
		QTFStr &StartTable(
			int w1 = Null, int w2 = Null, int w3 = Null, int w4 = Null,
			int w5 = Null, int w6 = Null, int w7 = Null, int w8 = Null);
		QTFStr &TableRestart(void);
		QTFStr &Cell(String const &Txt, int merge = 0);
		QTFStr &TableTitle(String const &title);
		QTFStr &TableSubTitle(String const &title);
		QTFStr &TableNewLine(void);
		QTFStr &EndTable(void);

		/////////////////////////////////////////////////////////////////////////////////////////////

		// outputs a labeled, optionnally commented, double value
		QTFStr &OutValLine(String const &name, String const &val, String const &um, String const &comment);
		
		QTFStr &OutDbl(String const &name, double d, int prec, String const &um, String const &comment = ""); 
		QTFStr &OutLength(String const &name, double d, String const &comment = "");

		QTFStr &OutTimberDimension(String const &name, double d, String const &comment = "");
		QTFStr &OutTimberSurface(String const &name, double d, String const &comment = "");
		QTFStr &OutTimberResistanceModulus(String const &name, double d, String const &comment = "");
		QTFStr &OutTimberInertia(String const &name, double d, String const &comment = "");
		
		QTFStr &OutSteelDimension(String const &name, double d, String const &comment = "");
		QTFStr &OutSteelSurface(String const &name, double d, String const &comment = "");
		QTFStr &OutSteelResistanceModulus(String const &name, double d, String const &comment = "");
		QTFStr &OutSteelInertia(String const &name, double d, String const &comment = "");

		QTFStr &OutPointLoad(String const &name, double d, String const &comment = "");
		QTFStr &OutLinearLoad(String const &name, double d, String const &comment = "");
		QTFStr &OutSurfaceLoad(String const &name, double d, String const &comment = "");
		QTFStr &OutForce(String const &name, double d, String const &comment = "");
		QTFStr &OutBending(String const &name, double d, String const &comment = "");
		QTFStr &OutStress(String const &name, double d, String const &comment = "");
		QTFStr &OutLinearStiffness(String const &name, double d, String const &comment = "");
		QTFStr &OutDeformation(String const &name, double d, String const &comment = "");
		QTFStr &OutDeformationRate(String const &name, double lf, String const &comment = "");
		QTFStr &OutRotation(String const &name, double d, String const &comment = "");
		QTFStr &OutAngle(String const &name, double d, String const &comment = "");

		QTFStr &OutEmpty(void);
		double OutCheck(String const & fName, double f, String const &rName, double r);
		double OutCheckR(String const & fName, double f, String const &rName, double r);
		

		/////////////////////////////////////////////////////////////////////////////////////////////
		// TimberStruct specific tables
		// shortcuts to have all settings on a single place
		/////////////////////////////////////////////////////////////////////////////////////////////

		// starts a generic result table
		// used on most printouts in report
		QTFStr &StartResultTable(String const &desc);

		// table for internal forces printouts
		QTFStr &StartInternalForcesTable(double pos);
		QTFStr &OutInternalForcesLine(String const &loadCase, double Nsx, double Ndx, double Vsx, double Vdx, double Msx, double Mdx);
		
		QTFStr &StartMaterialsTable(String const &desc);
		QTFStr &OutMaterialsLine(String const &desc, String const &name);

		operator const char *() { return str; }
		operator const String&() { return str; }
		String const &ToString(void) const { return str; }
		
}; // END Class QTFStr

#endif
