#include "QTFStr.h"
#include "Settings.h"

//////////////////////////////////////////////////////////////////////////////////////////////
// QTFStr::QTFFormat class
//////////////////////////////////////////////////////////////////////////////////////////////
QTFStr::QTFFormat::QTFFormat()
{
	Clear();
}

void QTFStr::QTFFormat::Clear(void)
{
	tableKeep		= KEEP_TABLE_ON_PAGE;
	tableBorder		= false;
	
	bold			= false;
	strike			= false;
	underline		= false;
	subscript		= false;
	superscript		= false;
	italic			= false;
	align			= ALIGN_LEFT;
	charSize		= 3;
	font			= 'G';
	
	foreColor		= "0";	// black
	backColor		= "2";	// white
	
	leftCellMargin		= 25;
	rightCellMargin		= 25;
	topCellMargin		= 15;
	bottomCellMargin	= 15;
}

QTFStr::QTFFormat::QTFFormat(const QTFFormat &fmt)
{
	tableKeep		= fmt.tableKeep;
	tableBorder		= fmt.tableBorder;
	
	bold			= fmt.bold;
	strike			= fmt.strike;
	underline		= fmt.underline;
	subscript		= fmt.subscript;
	superscript		= fmt.superscript;
	italic			= fmt.italic;
	align			= fmt.align;
	charSize		= fmt.charSize;
	font			= fmt.font;
	
	foreColor		= fmt.foreColor;
	backColor		= fmt.backColor;

	leftCellMargin		= fmt.leftCellMargin;
	rightCellMargin		= fmt.rightCellMargin;
	topCellMargin		= fmt.topCellMargin;
	bottomCellMargin	= fmt.bottomCellMargin;
}

QTFStr::QTFFormat const &QTFStr::QTFFormat::operator=(QTFFormat const &fmt)
{
	tableKeep		= fmt.tableKeep;
	tableBorder		= fmt.tableBorder;
	
	bold			= fmt.bold;
	strike			= fmt.strike;
	underline		= fmt.underline;
	subscript		= fmt.subscript;
	superscript		= fmt.superscript;
	italic			= fmt.italic;
	align			= fmt.align;
	charSize		= fmt.charSize;
	font			= fmt.font;
	
	foreColor		= fmt.foreColor;
	backColor		= fmt.backColor;
	
	leftCellMargin		= fmt.leftCellMargin;
	rightCellMargin		= fmt.rightCellMargin;
	topCellMargin		= fmt.topCellMargin;
	bottomCellMargin	= fmt.bottomCellMargin;
	
	return *this;
}


//////////////////////////////////////////////////////////////////////////////////////////////
// QTFStr class
//////////////////////////////////////////////////////////////////////////////////////////////


QTFStr::QTFStr()
{
	Clear();
}

// clears all formatting options -- reset them to defaults
QTFStr &QTFStr::ClearFormats(void)
{
	format.Clear();
	
	return *this;
}

// empty string and reset all formats to default
QTFStr &QTFStr::Clear(void)
{
	str = "";
	ClearFormats();
	formatStack.Clear();
	
	tableLevel = 0;

	tableColumns.Clear();
	tableColumn.Clear();
	columnWidths.Clear();

	err = false;
	return *this;
}

// push/pop format values
QTFStr &QTFStr::PushFormat(void)
{
	formatStack.Add(format);
	return *this;
}

QTFStr &QTFStr::PopFormat(void)
{
	if(formatStack.GetCount())
		format = formatStack.Pop();
	return *this;
}

// checks condition, if true sets error value, logs the (first) error with its QTF
// and stops any further processing. Returns err flag
bool QTFStr::CheckError(bool cond, String const &msg)
{
	if(!cond && !err)
		return false;
	err = true;
	LOG(msg);
	return true;
}

// outputs format codea round string
String QTFStr::FormatString(const String &s)
{
	String res;
	
	// if all formats are default, just output the string
	if(
		format.bold			== false		&&
		format.strike		== false		&&
		format.underline	== false		&&
		format.subscript	== false		&&
		format.superscript	== false		&&
		format.italic		== false		&&
		format.align		== ALIGN_LEFT	&&
		format.charSize		== 3			&&
		format.font			== 'G'			&&
		format.foreColor	== "0"			&&
		format.backColor	== "2"
	)
		return s;
	res = "[";
	if(format.bold)		res += "*";
	if(format.strike)		res += "-";
	if(format.underline)	res += "_";
	if(format.subscript)	res += ",";
	if(format.superscript)	res += "`";
	if(format.italic)		res += "/";
	switch(format.align)
	{
		case ALIGN_CENTER:
			res += "=";
			break;
		case ALIGN_RIGHT:
			res += ">";
			break;
		case ALIGN_JUSTIFY:
			res += "#";
			break;
		default:
			break;
	}
	if(format.charSize != 3)
		res += FormatInt(format.charSize);
	if(format.font != 'G')
		res += format.font;
	if(format.foreColor != "0")
		res += "@" + format.foreColor;
	if(format.backColor != "2")
		res += "$" + format.backColor;
	
	res += " " + s + "]";
	return res;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//                                      TABLE HANDLING                                     //
/////////////////////////////////////////////////////////////////////////////////////////////
QTFStr &QTFStr::StartTable(Vector<int> const &colW)
{
	if(CheckError(colW.GetCount() == 0, "Internal error -- no columns in table")) return *this;

	// if nested table, treat it as an entry, so add
	// the :: separator if not on first column
	if(tableLevel > 0)
	{
		if(IsNull(tableColumn.Top()))
			tableColumn.Top() = 0;
		else
			str += ":: ";
		if(tableColumn.Top() >= tableColumns.Top())
			TableNewLine();
	}

	tableLevel++;
	Vector<int> &cWidths = columnWidths.Add();
	cWidths <<= colW;
	tableColumns.Add(cWidths.GetCount());
	tableColumn.Add() = Null;
	
	// output table format
	str += "{{" + FormatInt(columnWidths.Top()[0]);
	for(int iCol = 1; iCol < columnWidths.Top().GetCount(); iCol++)
		str += ":" + FormatInt(columnWidths.Top()[iCol]);
	if(!format.tableBorder)
		str += "~";
	if(format.tableKeep == KEEP_CELL_ON_PAGE)
		str += "k";
	else if(format.tableKeep == KEEP_TABLE_ON_PAGE)
		str += "K";
	return *this;
}

QTFStr &QTFStr::StartTable(int w1, int w2, int w3, int w4, int w5, int w6, int w7, int w8)
{
	// gather column widths and number of columns
	Vector<int>cw;
	while(true)
	{
		if(!IsNull(w1)) { cw.Add(w1); } else break;
		if(!IsNull(w2)) { cw.Add(w2); } else break;
		if(!IsNull(w3)) { cw.Add(w3); } else break;
		if(!IsNull(w4)) { cw.Add(w4); } else break;
		if(!IsNull(w5)) { cw.Add(w5); } else break;
		if(!IsNull(w6)) { cw.Add(w6); } else break;
		if(!IsNull(w7)) { cw.Add(w7); } else break;
		if(!IsNull(w8)) { cw.Add(w8); } else break;
		break;
	}
	return StartTable(cw);
}

QTFStr &QTFStr::TableRestart(void)
{
	Vector<int>cw = columnWidths.Top();
	EndTable();
	return StartTable(cw);
}

QTFStr &QTFStr::Cell(String const &s, int merge)
{
	if(CheckError(tableLevel <= 0, "Internal error -- no table in progress")) return *this;
	if(IsNull(tableColumn.Top()))
		tableColumn.Top() = 0;
	else
		str += "::";
	if(tableColumn.Top() >= tableColumns.Top())
		TableNewLine();

	if(merge > tableColumns.Top() - tableColumn.Top() - 1)
		merge = tableColumns.Top() - tableColumn.Top() - 1;
	if(merge)
		str += "-" + FormatInt(merge);

	if(format.leftCellMargin != 25)
		str += "l/" + FormatInt(format.leftCellMargin);
	if(format.rightCellMargin != 25)
		str += "r/" + FormatInt(format.rightCellMargin);
	if(format.topCellMargin != 15)
		str += "t/" + FormatInt(format.topCellMargin);
	if(format.bottomCellMargin != 15)
		str += "b/" + FormatInt(format.bottomCellMargin);

	str += " " + FormatString(s);
	for(int iCol = 0; iCol < merge; iCol++)
		str += ":: ";
	tableColumn.Top() += merge + 1;
	return *this;
}

QTFStr &QTFStr::TableNewLine(void)
{
	if(CheckError(tableLevel <= 0, "Internal error -- no table in progress")) return *this;
	if(IsNull(tableColumn.Top()))
		tableColumn.Top() = 0;
	for(int iCol = tableColumn.Top(); iCol < tableColumns.Top(); iCol++)
		str += ":: ";
	tableColumn.Top() = 0;
	return *this;
}

// bold, bigger (1 step) table title line
QTFStr &QTFStr::TableTitle(String const &title)
{
	if(CheckError(tableLevel <= 0, "Internal error -- no table in progress")) return *this;
	if(tableColumn.Top() > 0)
		TableNewLine();
	bool bold = format.bold;
	format.bold = true;
	IncCharSize();
	Cell(title, tableColumns.Top());
	DecCharSize();
	format.bold = bold;
	return *this;
}

// bold, table title line
QTFStr &QTFStr::TableSubTitle(String const &title)
{
	if(CheckError(tableLevel <= 0, "Internal error -- no table in progress")) return *this;
	if(tableColumn.Top() > 0)
		TableNewLine();
	bool bold = format.bold;
	format.bold = true;
	Cell(title, tableColumns.Top());
	format.bold = bold;
	return *this;
}

QTFStr &QTFStr::EndTable(void)
{
	if(CheckError(tableLevel <= 0, "Internal error -- no table in progress")) return *this;
	str += "}} ";
	tableLevel--;
	tableColumns.Pop();
	tableColumn.Pop();
	columnWidths.Pop();
	return *this;
}

/////////////////////////////////////////////////////////////////////////////////////////////
//                                      TAEXT OUTPUT HANDLING                              //
/////////////////////////////////////////////////////////////////////////////////////////////
QTFStr &QTFStr::Txt(const String &s, int merge)
{
	// if not in table, just output text with formatting
	// otherwise put it into current cell
	if(tableLevel <= 0)
	{
		str += FormatString(s);
		return *this;
	}
	else
		return Cell(s, merge);
}

QTFStr &QTFStr::Txt(RichObject const &obj, int merge)
{
/*
	QtfRichObject q(obj);
	String txt = AsQTF(ParseQTF(q.ToString()));
	int i = txt.Find("@@");
	if( i > 0)
		txt.Remove(0, i-1);
	i = txt.ReverseFind("]");
	if(i >= 0)
		txt.Trim(i);
*/
	String txt = AsQTF(obj);
	
	// if not in table, just output text with formatting
	// otherwise put it into current cell
	if(tableLevel <= 0)
	{
		str += FormatString(txt);
		return *this;
	}
	else
		return Cell(txt, merge);
}

QTFStr &QTFStr::Iml(String const &Name, int cx, int cy, bool keepAspect, int merge)
{
	// if not in table, just output text with formatting
	// otherwise put it into current cell
	String iml = "@@iml:" + FormatInt(cx) + (keepAspect ? "&" : "*") + FormatInt(cy) + "`" + Name + "`";
	if(tableLevel <= 0)
	{
		if(format.align != ALIGN_LEFT)
			iml += "&";
		str += FormatString(iml);
		return *this;
	}
	else
		return Cell(iml, merge);
}

QTFStr &QTFStr::CenterIml(String const &Name, int cx, int cy, bool keepAspect, int merge)
{
	int align = format.align;
	format.align = ALIGN_CENTER;
	Iml(Name, cx, cy, keepAspect, merge);
	format.align = align;
	return *this;
}

QTFStr &QTFStr::Object(RichObject const &obj, int merge)
{
	String txt = AsQTF(obj);
	
	// if not in table, just output text with formatting
	// otherwise put it into current cell
	if(tableLevel <= 0)
	{
		str += txt;
		return *this;
	}
	else
		return Cell(txt, merge);
}

QTFStr &QTFStr::CenterObject(RichObject const &obj, int merge)
{
	String txt = "[ [= " + AsQTF(obj) + "]]";
	
	// if not in table, just output text with formatting
	// otherwise put it into current cell
	if(tableLevel <= 0)
	{
		str += txt;
		return *this;
	}
	else
		return Cell(txt, merge);
}
		
// outputs a double with given width and decimals
// right justified
QTFStr &QTFStr::Dbl(double d, int width, int decimals)
{
	// null values are printed with some '*' chars
	if(IsNull(d))
		return Txt(Units::NULLVAL());

	String s1;
	if(d < 0)
	{
		s1 = "-";
		d = -d;
	}
	else
		s1 = " ";
	String s2 = FormatDouble(d, decimals, FD_ZERO);
	if(s2.GetCount() >= width || s2.Find('E') >= 0 || s2.Find('e') >= 0)
		return Txt(String('*', width));

	s1 += s2;
	if(s1.GetCount() < width)
		s1 = String(' ', width - s1.GetCount()) + s1;

	return Txt(s1);
}

// page title
QTFStr &QTFStr::PageTitle(String const &s1, String const &s2)
{
	String Sep = (s1 != "" && s2 != "" ? " - " : "");
	str += "[=*G4 " + DeQtf(s1 + Sep + s2) + "&]";
	return *this;
}

QTFStr &QTFStr::OutValLine(String const &name, String const &val, const String &um, String const &comment)
{
	Cell("").Cell(name);
	int align = format.align;
	AlignRight();
	Cell(val);
	format.align = align;
	Cell(um);
	Cell(comment);
	return *this;
}

QTFStr &QTFStr::OutDbl(String const &name, double d, int prec, String const &um, String const &comment)
{
	if(IsNull(d))
		return OutValLine(name, Units::NULLVAL(), um, comment);
	return OutValLine(name, FormatDouble(d, prec, FD_ZERO), um, comment);
}

QTFStr &QTFStr::OutLength(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.LengthToString(d);
	String ums = u.LengthsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutTimberDimension(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.TimberDimensionToString(d);
	String ums = u.TimberDimensionsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutTimberSurface(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.TimberSurfaceToString(d);
	String ums = u.TimberSurfacesUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutTimberResistanceModulus(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.TimberResistanceModulusToString(d);
	String ums = u.TimberResistanceModuliUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutTimberInertia(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.TimberInertiaToString(d);
	String ums = u.TimberInertiaeUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutSteelDimension(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.SteelDimensionToString(d);
	String ums = u.SteelDimensionsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutSteelSurface(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.SteelSurfaceToString(d);
	String ums = u.SteelSurfacesUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutSteelResistanceModulus(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.SteelResistanceModulusToString(d);
	String ums = u.SteelResistanceModuliUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutSteelInertia(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.SteelInertiaToString(d);
	String ums = u.SteelInertiaeUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutPointLoad(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.PointLoadToString(d);
	String ums = u.PointLoadsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutLinearLoad(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.LinearLoadToString(d);
	String ums = u.LinearLoadsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutSurfaceLoad(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.SurfaceLoadToString(d);
	String ums = u.SurfaceLoadsUmStr(true);
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutForce(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.ForceToString(d);
	String ums = u.ForcesUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutBending(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.BendingToString(d);
	String ums = u.BendingsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutStress(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.StressToString(d);
	String ums = u.StressesUmStr(true);
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutLinearStiffness(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.LinearStiffnessToString(d);
	String ums = u.LinearStiffnessesUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutDeformation(String const &name, double d, String const &comment)
{
	Units const &u = globalSettings().GetUnits();
	String s = u.DeformationToString(d);
	String ums = u.DeformationsUmStr();
	return OutValLine(name, s, ums, comment);
}

QTFStr &QTFStr::OutDeformationRate(String const &name, double lf, String const &comment)
{
	double d;
	lf = fabs(lf);
	if(lf == 0)
		d = 1e99;
	else
		d = 1/lf;
	return OutDbl(name, d, 0, "", comment);
}

QTFStr &QTFStr::OutRotation(String const &name, double d, String const &comment)
{
	if(IsNull(d))
		return OutValLine(name, Units::NULLVAL(), "rad", comment);
	return OutValLine(name, FormatDouble(d, 5, FD_ZERO), "rad", comment);
}

QTFStr &QTFStr::OutAngle(String const &name, double d, String const &comment)
{
	if(IsNull(d))
		return OutValLine(name, Units::NULLVAL(), t_("deg"), comment);
	return OutValLine(name, FormatDouble(d * 180.0 / M_PI, 1, FD_ZERO), t_("deg"), comment);
}

QTFStr &QTFStr::OutEmpty(void)
{
	str += " ::-4 :: :: :: :: ";
	return *this;
}

// starts a generic result table
// used on most printouts in report
QTFStr &QTFStr::StartResultTable(String const &desc)
{
	StartTable(1, 14, 8, 6, 18);
	TableTitle(desc);
	return *this;
}

QTFStr &QTFStr::StartInternalForcesTable(double pos)
{
	Units const &u = globalSettings().GetUnits();
	String lenUmStr = u.LengthsUmStr();
	String forcesUmStr = u.ForcesUmStr();
	String bendUmStr = u.BendingsUmStr();

	bool border = format.tableBorder;
	format.tableBorder = true;
	StartTable(1, 15, 5, 5, 5, 5, 5, 5);
	format.tableBorder = border;
	TableSubTitle(t_("Internal forces at x = ") + u.LengthToString(pos) + " " + lenUmStr + " - " + t_("characteristic values"));
	Txt
		("")
		(String("[*= ") + t_("Load case") + "]")
		(String("[*= ") + t_("InternalForces\vN") + "(" + forcesUmStr + ")]", 1)
		(String("[*= ") + t_("InternalForces\vV") + "(" + forcesUmStr + ")]", 1)
		(String("[*= ") + t_("InternalForces\vM") + "(" + bendUmStr + ")]", 1)
	;
	return *this;
}

QTFStr &QTFStr::OutInternalForcesLine(String const &loadCase, double Nsx, double Ndx, double Vsx, double Vdx, double Msx, double Mdx)
{
	Units u = globalSettings().GetUnits();
	Txt("")(loadCase);
	fabs(Nsx - Ndx) < 0.000001 ? Txt(u.ForceToString(Nsx), 1) : DecCharSize()(u.ForceToString(Nsx))(u.ForceToString(Ndx)).IncCharSize();
	fabs(Vsx - Vdx) < 0.000001 ? Txt(u.ForceToString(Vsx), 1) : DecCharSize()(u.ForceToString(Vsx))(u.ForceToString(Vdx)).IncCharSize();
	fabs(Msx - Mdx) < 0.000001 ? Txt(u.BendingToString(Msx), 1) : DecCharSize()(u.BendingToString(Msx))(u.BendingToString(Mdx)).IncCharSize();
	return *this; 
}

QTFStr &QTFStr::StartMaterialsTable(String const &desc)
{
	StartTable(1, 20, 80);
	TableTitle(desc);
		
	return *this;
}

QTFStr &QTFStr::OutMaterialsLine(String const &desc, String const &name)
{
	return Txt("")(desc)(name);
}

double QTFStr::OutCheck(String const & fName, double f, String const &rName, double r)
{
	if(f <= r)
		Txt(fName + " ≤ " + rName + " [_ " + t_("VERIFIED") + "]", 4);
	else
		Txt(fName + " > " + rName + " [@R_ " + t_("NOT VERIFIED") + "]", 4);
	if(r != 0)
		return fabs(f/r);
	else
		return 1e99;
}

double QTFStr::OutCheckR(String const & fName, double f, String const &rName, double r)
{
	if(f > r)
		Txt(fName + " > " + rName + " [_ " + t_("VERIFIED") + "]", 4);
	else
		Txt(fName + " ≤ " + rName + " [@R_ " + t_("NOT VERIFIED") + "]", 4);
	if(r != 0)
		return fabs(f/r);
	else
		return 1e99;
}
