#include "RegExp.h"

#include <Core/Core.h>
#include <RegExp/pcre/pcre.h>

#define _OPT 30

RegExp::RegExp()
		:pattern_options(0),
		current_position(0){
}

RegExp::RegExp(const String& s)
		:pattern_options(0),
		current_position(0)
{
	pattern = s;
	compile();	
}

RegExp::RegExp(const String& s, RegExpOptions o)
		:current_position(0)
{
	pattern_options = ~o;
	pattern = s;
	compile();	
}

RegExp::~RegExp(){
	pcre_free(re);
}

void RegExp::SetPattern(const String& s){
	pattern = s;
	compile();
}

void RegExp::SetOptions(RegExpOptions o){
	pattern_options = ~o;
	compile();
}

void RegExp::operator =(RegExp &r){
	pattern = r.GetPattern();
	pattern_options = r.GetOptions();
	compile();
}

void RegExp::ClearOptions(){
	pattern_options = 0;
	compile();
}

void RegExp::SetString(const String& txt){
	current_position = 0;
	text = txt;
	//text_len = text.GetLength();
}

bool RegExp::IsMatch(const String& txt){
	SetString(txt);
	if(!exec()) return false;
	return rc > 0;// != PCRE_ERROR_NOMATCH;
}

bool RegExp::Next(){
	if(!exec(current_position)){
		current_position = 0;
		return false;
	}
	current_position = ovector[1];
	return true;
}

Vector<Value> RegExp::Get(){
	const char **list;
	Vector<Value> results;
	int _tmp = pcre_get_substring_list(~text, ovector, rc, &list);
	if (_tmp < 0){
		Cout() << "Get substring list failed\n";
		pcre_free_substring_list(list);
	}
	else{
		int i;
		for(i = 0; i < rc; i++)
			results << list[i];
		/*if (list[i] != NULL) Cout() << "List not terminated by NULL\n";*/
		pcre_free_substring_list(list);
		return results;
	}
}

Vector<Value> RegExp::GetMatches(const String& txt){
	SetString(txt);
	Vector<Value> results;
	match_all(results);
	return results;
}

void RegExp::match_all(Vector<Value> &v){
	int offset = 0;
	while(exec(offset)){
		for (int i = 0; i < rc; i++)
			v << String(~text + ovector[2*i],
			                  ovector[2*i+1] - ovector[2*i]);
		offset = ovector[1];
	}
}

String RegExp::Replace(const String& txt, const String &str){
	SetString(txt);
	String s;
	replace_all(str, s);
	return s;
}

void RegExp::replace_all(const String &str, String &result){
	int offset = 0;
	while(exec(offset)){
		result.Cat(~text + offset, ovector[0] - offset);
		result.Cat(str);// ? + Get()[0] + ?
		offset = ovector[1];
	}
	//if(text.GetLength() - offset)
	result.Cat(~text + offset, text.GetLength() - offset); 
}

void RegExp::compile(){
	re = pcre_compile(
	~pattern,					/* the pattern */
	pattern_options,			/* default options */
	&error,						/* for error message */
	&erroffset,					/* for error offset */
	NULL);						/* use default character tables */
	current_position = 0;
#ifdef DEBUG_RegExp
	Cout() 	<< pattern << " - Pattern\n" 
			<< pattern_options << " - Options\n"
			<< "Error:" << GetError() << "\n";
#endif 
}

bool RegExp::exec(int offset, int options){
	rc = pcre_exec(
	re,							/* result of pcre_compile() */
	NULL,						/* we didn't study the pattern */
	~text,						/* the subject string */
	text.GetLength(),			/* the length of the subject string */
	offset,						/* start at offset 0 in the subject */
	options,					/* default options */
	ovector,					/* vector of integers for substring information */
	_OPT);						/* number of elements (NOT size in bytes) */

	if (rc < 0)
		return false;
	return true;
}

String RegExp::GetExecError(){
	if (rc < 0){
		switch(rc){
			case PCRE_ERROR_NOMATCH:{
				return "ERROR_NOMATCH";
				break;
			}
			case PCRE_ERROR_NULL:{
				return "ERROR_NULL";
				break;
			}
			case PCRE_ERROR_BADOPTION:{
				return "ERROR_BADOPTION";
				break;
			}
			case PCRE_ERROR_BADMAGIC:{
				return "ERROR_BADMAGIC";
				break;
			}
			case PCRE_ERROR_UNKNOWN_NODE:{
				return "ERROR_UNKNOWN_NODE";
				break;
			}
			case PCRE_ERROR_NOMEMORY:{
				return "ERROR_NOMEMORY";
				break;
			}
			case PCRE_ERROR_NOSUBSTRING:{
				return "ERROR_NOSUBSTRING";
				break;
			}
			case PCRE_ERROR_MATCHLIMIT:{
				return "ERROR_MATCHLIMIT";
				break;
			}
			case PCRE_ERROR_CALLOUT:{
				return "ERROR_CALLOUT";
				break;
			}
			case PCRE_ERROR_BADUTF8:{
				return "ERROR_BADUTF8";
				break;
			}
			case PCRE_ERROR_BADUTF8_OFFSET:{
				return "ERROR_BADUTF8_OFFSET";
				break;
			}
			case PCRE_ERROR_PARTIAL:{
				return "ERROR_PARTIAL";
				break;
			}
			case PCRE_ERROR_BADPARTIAL:{
				return "ERROR_BADPARTIAL";
				break;
			}
			case PCRE_ERROR_INTERNAL:{
				return "ERROR_INTERNAL";
				break;
			}
			case PCRE_ERROR_BADCOUNT:{
				return "ERROR_BADCOUNT";
				break;
			}
			case PCRE_ERROR_DFA_UITEM:{
				return "ERROR_DFA_UITEM";
				break;
			}
			case PCRE_ERROR_DFA_UCOND:{
				return "ERROR_DFA_UCOND";
				break;
			}
			case PCRE_ERROR_DFA_UMLIMIT:{
				return "ERROR_DFA_UMLIMIT";
				break;
			}
			case PCRE_ERROR_DFA_WSSIZE:{
				return "ERROR_DFA_WSSIZE";
				break;
			}
			case PCRE_ERROR_DFA_RECURSE:{
				return "ERROR_DFA_RECURSE";
				break;
			}
			case PCRE_ERROR_RECURSIONLIMIT:{
				return "ERROR_RECURSIONLIMIT";
				break;
			}
			default:
				return String("Error code: ")+rc;
		}
	}
	else
		return "MATCHED"; 
}
