#include <Core/Core.h>
#include <angelscript/angelscript.h>

using namespace Upp;

struct Oper {
	virtual double Execute() = 0;
	virtual ~Oper() {}
};

struct BinOper : Oper {
	One<Oper> a;
	One<Oper> b;
};

struct Add : BinOper {
	virtual double Execute() { return a->Execute() + b->Execute(); }
};

struct Sub : BinOper {
	virtual double Execute() { return a->Execute() - b->Execute(); }
};

struct Mul : BinOper {
	virtual double Execute() { return a->Execute() * b->Execute(); }
};

struct Div : BinOper {
	virtual double Execute() { return a->Execute() / b->Execute(); }
};

struct Const : Oper {
	double value;
	virtual double Execute() { return value; }
};

struct Var : Oper {
	double *var;
	virtual double Execute() { return *var; }
};

struct Compiler {
	VectorMap<String, double *> var;
	One<Oper> Term(CParser& p);
	One<Oper> Exp(CParser& p);
	One<Oper> Factor(CParser& p);
};

One<Oper> Compiler::Term(CParser& p)
{
	One<Oper> result;
	if(p.IsId()) {
		double *v = var.Get(p.ReadId(), NULL);
		if(!v)
			p.ThrowError("unknown variable");
		result.Create<Var>().var = v;
	}
	else
	if(p.Char('(')) {
		result = Exp(p);
		p.PassChar(')');
	}
	else
		result.Create<Const>().value = p.ReadDouble();
	return result;
}

One<Oper> Compiler::Factor(CParser& p)
{
	One<Oper> result = Term(p);
	for(;;)
		if(p.Char('*')) {
			One<Oper> rr;
			Mul& m = rr.Create<Mul>();
			m.a = result;
			m.b = Term(p);
			result = rr;
		}
		else
		if(p.Char('/')) {
			One<Oper> rr;
			Div& m = rr.Create<Div>();
			m.a = result;
			m.b = Term(p);
			result = rr;
		}
		else
			return result;
}

One<Oper> Compiler::Exp(CParser& p)
{
	One<Oper> result = Factor(p);
	for(;;)
		if(p.Char('+')) {
			One<Oper> rr;
			Add& m = rr.Create<Add>();
			m.a = result;
			m.b = Factor(p);
			result = rr;
		}
		else
		if(p.Char('-')) {
			One<Oper> rr;
			Sub& m = rr.Create<Sub>();
			m.a = result;
			m.b = Factor(p);
			result = rr;
		}
		else
			return result;
}

VectorMap<String, double> var;

double Exp(CParser& p);

double Term(CParser& p)
{
	if(p.Id("abs")) {
		p.PassChar('(');
		double x = Exp(p);
		p.PassChar(')');
		return fabs(x);
	}
	if(p.IsId())
		return var.Get(p.ReadId(), 0);
	if(p.Char('(')) {
		double x = Exp(p);
		p.PassChar(')');
		return x;
	}
	return p.ReadDouble();
}

double Mul(CParser& p)
{
	double x = Term(p);
	for(;;)
		if(p.Char('*'))
			x = x * Term(p);
		else
		if(p.Char('/')) {
			double y = Term(p);
			if(y == 0)
				p.ThrowError("Divide by zero");
			x = x / y;
		}
		else
			return x;
}

double Exp(CParser& p)
{
	double x = Mul(p);
	for(;;)
		if(p.Char('+'))
			x = x + Mul(p);
		else
		if(p.Char('-'))
			x = x - Mul(p);
		else
			return x;
}


CONSOLE_APP_MAIN
{
	double x, y;
	
	Compiler c;
	c.var.Add("x", &x);
	c.var.Add("y", &y);
	
	CParser p("1 / (1 - x * y + x - y)");
	One<Oper> fn = c.Exp(p);
	
	x = 5;
	y = 10;
	
	RDUMP(1 / (1 - x * y + x - y));
	RDUMP(fn->Execute());

	{
		RTIMING("Interpreted");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001) {
				var.GetAdd("x") = x;
				var.GetAdd("y") = y;
				CParser c = CParser("1 / (1 - x * y + x - y)");
				sum += Exp(c);
			}
		RDUMP(sum);
	}
	{
		RTIMING("Compiled");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001)
				sum += fn->Execute();
		RDUMP(sum);
	}
	{
		RTIMING("Direct");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001)
				sum += 1 / (1 - x * y + x - y);
		RDUMP(sum);
	}
	
	// Create the AngelScript engine
	asIScriptEngine *engine = asCreateScriptEngine(ANGELSCRIPT_VERSION);
	if(engine == 0)
	{
		Cout() << "Failed to create AngelScript engine.\n";
		SetExitCode(1);
		return;
	}

	const String script =
		"double calculate(double x, double y) { return 1 / (1 - x * y + x - y); }\n";

	asIScriptModule *mod = engine->GetModule("script", asGM_ALWAYS_CREATE);
	if(mod->AddScriptSection("script", ~script, script.GetCount()) < 0) {
		Cout() << "AddScriptSection() failed\n";
		engine->Release();
		SetExitCode(1);
		return;
	}

	if(mod->Build() < 0) {
		Cout() << "Build() failed\n";
		engine->Release();
		SetExitCode(1);
		return;
	}

	// Create a context that will execute the script.
	asIScriptContext *ctx = engine->CreateContext();
	if(ctx == 0) {
		Cout() << "Failed to create the context.\n";
		ctx->Release();
		engine->Release();
		SetExitCode(1);
		return;
	}

	// Find the function id for the function we want to execute.
	int funcId = engine->GetModule("script")->GetFunctionIdByDecl("double calculate(double, double)");
	if(funcId < 0) {
		Cout() << "The function \"calculate\" wasn't found.\n";
		ctx->Release();
		engine->Release();
		SetExitCode(1);
		return;
	}

	{
		RTIMING("AngelScript (interpreted)");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001) {
				if(ctx->Prepare(funcId) < 0) {
					Cout() << "Failed to prepare the context.\n";
					SetExitCode(1);
					break;
				}

				ctx->SetArgDouble(0, x);
				ctx->SetArgDouble(1, y);

				if(ctx->Execute() == asEXECUTION_FINISHED) {
					sum += ctx->GetReturnDouble();
					//ctx->Unprepare();
				}
				else break;
			}
		RDUMP(sum);
	}

	ctx->Unprepare();
	mod->RemoveFunction(funcId);

	const String script2 =
		"double calculate() {\n"
		"	double x, y, sum = 0;\n"
		"	for (x = 0; x < 1; x += 0.001)\n"
		"		for (y = 0; y < 1; y += 0.001)\n"
		"			sum += 1 / (1 - x * y + x - y);\n"
		"	return sum;\n"
		"}";

	mod = engine->GetModule("script2", asGM_ALWAYS_CREATE);
	if(mod->AddScriptSection("script2", ~script2, script2.GetCount()) < 0) {
		Cout() << "AddScriptSection() failed\n";
		engine->Release();
		SetExitCode(1);
		return;
	}

	if(mod->Build() < 0) {
		Cout() << "Build() failed\n";
		engine->Release();
		SetExitCode(1);
		return;
	}

	funcId = engine->GetModule("script2")->GetFunctionIdByDecl("double calculate()");
	if(funcId < 0) {
		Cout() << "The function \"calculate\" wasn't found.\n";
		ctx->Release();
		engine->Release();
		SetExitCode(1);
		return;
	}

	{
		RTIMING("AngelScript (fully interpreted)");
		double sum = 0;
		if(ctx->Prepare(funcId) < 0) {
			Cout() << "Failed to prepare the context.\n";
			SetExitCode(1);
		}

		if(ctx->Execute() == asEXECUTION_FINISHED)
			sum = ctx->GetReturnDouble();

		RDUMP(sum);
	}

	ctx->Release();
	engine->Release();
}
