#include "Jsonize.h"
using namespace Upp;

#define JSONIZE_LOG(a) //Cout() << a;

JsonIO::JsonIO(bool _isLoading)
	:isLoading(_isLoading)
	,parser(NULL)
	,type(JsonIO::TYPE_ENTRY)
{
}

bool ParseJsonStructure(JsonIO &json)
{
	if (!json.parser)
		return false;
	
	try
	{
		if (json.parser->Char('{'))
		{
			JSONIZE_LOG('{');
			json.SetType(JsonIO::TYPE_MAP);
			json.pos = json.parser->GetPos();
			if (!json.parser->IsChar('}'))
				do {
					String key = json.parser->ReadString();
					json.parser->PassChar(':');

					JsonIO *jsonElement = new JsonIO(true);
					jsonElement->parser = json.parser;
					jsonElement->pos = json.parser->GetPos();
					ParseJsonStructure(json.map.Add(key,jsonElement));
				}
				while(json.parser->Char(','));
			json.parser->Spaces();
			json.parser->PassChar('}');
			JSONIZE_LOG('}');
		}
		else
		if (json.parser->Char('['))
		{
			JSONIZE_LOG('[');
			json.SetType(JsonIO::TYPE_ARRAY);
			json.pos = json.parser->GetPos();
			if(!json.parser->IsChar(']'))
				do
				{
					JsonIO *jsonElement = new JsonIO(true);
					jsonElement->parser = json.parser;
					jsonElement->pos = json.parser->GetPos();
					ParseJsonStructure(json.array.Add(jsonElement));
				}
				while(json.parser->Char(','));
			json.parser->Spaces();
			json.parser->PassChar(']');
			JSONIZE_LOG(']');
		}
		else
		{
			json.type = JsonIO::TYPE_ENTRY;
			if (json.parser->IsString())
				json.parser->ReadString();
			else
			{
				char c;
				while (c = json.parser->PeekChar(), (IsAlNum(c) || c == '.' || c == '-' || c == '+'))
					json.parser->GetChar();
			}
			JSONIZE_LOG('.');
		}
	}
	catch (...)
	{
		return false;
	}
	
	return true;
}

String GenerateJsonRepresentation(JsonIO &json)
{
	String out;
	switch (json.type)
	{
	case JsonIO::TYPE_ENTRY:
		out = json.storingString;
		break;
	case JsonIO::TYPE_ARRAY:
		{
			String indent = json.isPretty ? ('\n'+String('\t',json.storingIndent)) : String();
			out << indent << '[';
			for (int i=0; i<json.array.GetCount(); ++i)
			{
				out << indent;
				if (json.isPretty)
					out << '\t';
				out << GenerateJsonRepresentation(json.array[i]);
				if (i < json.array.GetCount() - 1)
					out << ',';
			}
			out << indent << ']';
		}
		break;
	case JsonIO::TYPE_MAP:
		{
			String indent = json.isPretty ? (String('\t',json.storingIndent)) : String();
			if (json.storingIndent)
				out << '\n';
			out << indent << '{';
			for (int i=0; i<json.map.GetCount(); ++i)
			{
				if (json.isPretty)
					out << '\n' << indent << '\t';
				out << '"' << json.map.GetKey(i) << '"';
				out << ':';
				if (json.isPretty)
					out << ' ';
				out << GenerateJsonRepresentation(json.map[i]);
				if (i < json.map.GetCount() - 1)
					out << ',';
			}
			out << '\n' << indent << '}';
		}
		break;
	}
	return out;
}
void LoadFromJson(Callback1<JsonIO &> jsonize, const String &str)
{
	JsonIO json(true);
	CParser p(str);
	p.UnicodeEscape();
	json.parser = &p;
	json.pos = p.GetPos();
	JSONIZE_LOG("\nPARSE ");
	ParseJsonStructure(json);
	jsonize(json);
}

String StoreAsJson(Callback1<JsonIO &> jsonize)
{
	JsonIO json(false);
	json.SetType(JsonIO::TYPE_MAP);
	json.SetPretty(true);
	json.storingIndent = 0;
	jsonize(json);
	return GenerateJsonRepresentation(json);
}

template<class T> void LoadFromJson(T &v, const String &str)
{
	LoadFromJson(callback(&v, &T::Jsonize),str);
}

template<class T> String StoreAsJson(T &v)
{
	return StoreAsJson(callback(&v, &T::Jsonize));
}

template <class T> JsonIO & JsonIO::operator()(const char *key, T& var)
{
	if (IsLoading())
	{
		if (type == JsonIO::TYPE_MAP)
		{
			int entryI = map.Find(key);
			if (entryI >= 0)
			{
				Jsonize(map[entryI],var);
			}
		}
	}
	else
	{
		switch (type)
		{
		case JsonIO::TYPE_ENTRY:
			Jsonize(*this,var);
			break;
		case JsonIO::TYPE_MAP:
			{
				JsonIO *jsonElement = new JsonIO(false);
				jsonElement->SetPretty(isPretty);
				jsonElement->SetType(JsonIO::TYPE_MAP);
				jsonElement->storingIndent = storingIndent + 1;
				Jsonize(*jsonElement,var);
				map.Add(key,jsonElement);
			}
			break;
		case JsonIO::TYPE_ARRAY:
			{
				JsonIO *jsonElement = new JsonIO(false);
				jsonElement->SetPretty(isPretty);
				jsonElement->SetType(JsonIO::TYPE_MAP);
				jsonElement->storingIndent = storingIndent + 1;
				Jsonize(*jsonElement,var);
				array.Add(jsonElement);
			}
			break;
		}
	}
	return *this;
}

template <class T> void Jsonize(JsonIO &json, Vector<T> &var, const T &newT/*= T()*/)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ARRAY)
			return;
		try
		{
			var.Clear();
			for (int i=0; i<json.array.GetCount(); ++i)
			{
				Jsonize(json.array[i], var.Add(newT));
			}
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ARRAY);
		for (int i=0; i<var.GetCount(); ++i)
		{
			JsonIO *jsonElement = new JsonIO(false);
			jsonElement->SetPretty(json.isPretty);
			jsonElement->storingIndent = json.storingIndent + 1;
			Jsonize(json.array.Add(jsonElement), var[i]);
		}
	}
}

template <class T> void Jsonize(JsonIO &json, Array<T> &var, const T &newT/* = T()*/)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ARRAY)
			return;
		try
		{
			var.Clear();
			for (int i=0; i<json.array.GetCount(); ++i)
			{
				Jsonize(json.array[i], var.Add(newT));
			}
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ARRAY);
		for (int i=0; i<var.GetCount(); ++i)
		{
			JsonIO *jsonElement = new JsonIO(false);
			jsonElement->SetPretty(json.isPretty);
			jsonElement->storingIndent = json.storingIndent + 1;
			Jsonize(json.array.Add(jsonElement), var[i]);
		}
	}
}

template <class T> void Jsonize(JsonIO &json, VectorMap<String,T> &var, const T &newT/* = T()*/)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_MAP)
			return;
		try
		{
			var.Clear();
			for (int i=0; i<json.map.GetCount(); ++i)
			{
				Jsonize(json.map[i], var.Add(json.map.GetKey(i), newT));
			}
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_MAP);
		for (int i=0; i<var.GetCount(); ++i)
		{
			JsonIO *jsonElement = new JsonIO(false);
			jsonElement->SetPretty(json.isPretty);
			jsonElement->storingIndent = json.storingIndent + 1;
			Jsonize(json.map.Add(var.GetKey(i),jsonElement), var[i]);
		}
	}
}

template <class T> void Jsonize(JsonIO &json, ArrayMap<String,T> &var, const T &newT)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_MAP)
			return;
		try
		{
			var.Clear();
			for (int i=0; i<json.map.GetCount(); ++i)
			{
				Jsonize(json.map[i], var.Add(json.map.GetKey(i), newT));
			}
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_MAP);
		for (int i=0; i<var.GetCount(); ++i)
		{
			JsonIO *jsonElement = new JsonIO(false);
			jsonElement->SetPretty(json.isPretty);
			jsonElement->storingIndent = json.storingIndent + 1;
			Jsonize(json.map.Add(var.GetKey(i),jsonElement), var[i]);
		}
	}
}

template <> void Jsonize(JsonIO &json, String &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			var = json.parser->ReadString();
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = AsCString(var);
	}
}

template <> void Jsonize(JsonIO &json, int &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			var = json.parser->ReadInt();
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = FormatInt(var);
	}
}

template <> void Jsonize(JsonIO &json, int16 &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			var = (int16) json.parser->ReadInt();
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = FormatInt((int) var);
	}
}

template <> void Jsonize(JsonIO &json, byte &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			var = (byte) json.parser->ReadInt();
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = FormatInt((int) var);
	}
}

template <> void Jsonize(JsonIO &json, double &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			var = json.parser->ReadDouble();
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = FormatDouble(var);
	}
}

template <> void Jsonize(JsonIO &json, bool &var)
{
	if (json.IsLoading())
	{
		if (json.type != JsonIO::TYPE_ENTRY)
			return;
		try
		{
			json.parser->SetPos(json.pos);
			String s = ToLower(json.parser->ReadId());
			
			static const String STR_TRUE = "true";
			var = (s == STR_TRUE) ? true : false;
		}
		catch (...)
		{}
	}
	else
	{
		json.SetType(JsonIO::TYPE_ENTRY);
		json.storingString = var ? "true" : "false";
	}
}

template <> void Jsonize(JsonIO &json, Time &var)
{
	json
		("year",var.year)
		("month",var.month)
		("day",var.day)
		("hour",var.hour)
		("minute",var.minute)
		("second",var.second)
	;
}

//----------------------------------------------------------------------------------------------
#ifdef flagMAIN
struct TestStruct2
{
	int c;
	bool d;
	double e;
	Time   t;
	TestStruct2()
		:c(-1)
		,d(false)
		,e(-1.)
		,t(GetSysTime())
	{
	}
	
	void Jsonize(JsonIO &json)
	{
		json
			("c", c)
			("d", d)
			("e", e)
			//("t", t)
		;
	}
	void Xmlize(XmlIO &xml)
	{
		xml
			("c", c)
			("d", d)
			("e", e)
		;
	}
	void Serialize(Stream &s)
	{
		s % c % d % e;
	}
};
struct TestStruct
{
	String u;
	String a;
	int    b;
	TestStruct2 c;
	VectorMap<String, Vector<int> > map1;
	
	TestStruct()
		:u("test U!")
		,a("A test!")
		,b(-1)
	{
	}
	
	void Jsonize(JsonIO &json)
	{
		json
			("a",a)
			("b",b)
			("c",c)
			("u",u)
			//("map1",map1)
		;
	}
	void Xmlize(XmlIO &xml)
	{
		xml
			("a",a)
			("b",b)
			("c",c)
			("u",u)
			//("map1",map1)
		;
	}
	void Serialize(Stream &s)
	{
		s % a % b % c % u;
	}
	
	void Dump()
	{
		Cout() << "\n--------- ---------- ----------\n" << a << ',' << b << ',' << u << '\n' << c.c << ',' << c.d << ',' << c.e << ';' << c.t << '\n';
		for (int i=0; i<map1.GetCount(); ++i)
		{
			Cout() << map1.GetKey(i) << ": ";
			for (int j=0; j<map1[i].GetCount(); ++j)
				Cout() << map1[i][j] << ' ';
			Cout() << '\n';
		}
	}
};

CONSOLE_APP_MAIN
{
	String out = "{\"map1\":{\"1\":[1,2,3,4], \"2\":[10,11]},\"u\":\",,,^-.-^,,,\",\"b\":123,\"c\":{\"e\":900.55,\"d\":True,\"c\":345,\"t\":{\"year\":2012}}, \"a\":\"test\"}";
	
	TestStruct s;
	Cout() << out;
	LoadFromJson(s, out);
	s.Dump();

	s.map1.GetAdd("1", Vector<int>() << 1 << 2 << 3 << 4);
	s.map1.GetAdd("2", Vector<int>() << 10 << 11);
	out = StoreAsJson(s);
	Cout() << '\n' << out;

	TestStruct s2;
	LoadFromJson(s2,out);
	Cout() << '\n';
	s2.Dump();
	
	static const int TEST_COUNT = 10000;
	dword t0 = GetTickCount();
	for (int i=0; i<TEST_COUNT; ++i)
	{
		String out = StoreAsJson(s);
		LoadFromJson(s,out);
	}
	Cout() << "\nDELTA JSON: " << (int) (GetTickCount()-t0);
	t0 = GetTickCount();
	for (int i=0; i<TEST_COUNT; ++i)
	{
		String out = StoreAsXML(s,"test");
		LoadFromXML(s,out);
	}
	Cout() << "\nDELTA XML: " << (int) (GetTickCount()-t0);
	t0 = GetTickCount();
	for (int i=0; i<TEST_COUNT; ++i)
	{
		String out = StoreAsString(s);
		LoadFromString(s,out);
	}
	Cout() << "\nDELTA BINARY: " << (int) (GetTickCount()-t0);
}
#endif