#include <Core/Core.h>
#include <MtAlt/MtAlt.h>
using namespace Upp;

#define TEST

const String FILE_STATE  = "VerUpdate.state";
const String FILE_CONFIG = "VerUpdate.config";

const String DELIM = "##";
StaticMutex _mutex;

struct ProjectVersion : Moveable<ProjectVersion>
{
	struct Global
	{
		Date date;
		int  counter;
		bool needReUpdate;
		
		Global()
			:date(GetSysDate())
			,counter(0)
			,needReUpdate(false)
		{}
		void Xmlize(XmlIO &xml)
		{
			xml
				("date",date)
				("counter",counter)
			;
		}
	} global;
	struct Version
	{
		Vector<String> entries;
		Date date;
		int  counter;
		
		Version()
			:date(GetSysDate())
			,counter(0)
		{}
		Version(const Version &op, int)
			:entries(op.entries, 0)
			,date(op.date)
			,counter(op.counter)
		{}
		void Xmlize(XmlIO &xml)
		{
			xml
				("date",date)
				("counter",counter)
				("entries",entries)
			;
		}
		bool operator== (const Version &op) const
		{
			if (entries.GetCount() != op.entries.GetCount())
				return false;
			for (int i=0; i<entries.GetCount(); ++i)
			{
				if (entries[i] != op.entries[i])
					return false;
			}
			return true;
		}
		String operator() ()
		{
			String out;
			for (int i=0; i<entries.GetCount(); ++i)
			{
				if (i)
					out << '.';
				out << entries[i];
			}
			return out;
		}
	} version, versionUpdate;
	
	void Xmlize(XmlIO &xml)
	{
		xml
			("global",global)
			("version",version)
		;
	}
};

struct ProjectConfig : Moveable<ProjectConfig>
{
	VectorMap<String, Vector<String> > files;
	
	void Xmlize(XmlIO &xml)
	{
		xml
			("files",files)
		;
	}
};

class FileChangerNotify
{
public:
	virtual void OnFileChangeFinished_(const String &project, const String &filename, bool isVersionChanged, ProjectVersion::Version) = 0;
};

class FileChanger : public CallbackThread
{
public:
	FileChanger(FileChangerNotify *_notify,
				const String &_project,
		        const String &_filename, 
		        const Vector<String> &_strings,
		        const Date _globalDate,
		        const int  _globalCounter,
		        const ProjectVersion::Version &_versionNew,
		        const ProjectVersion::Version &_versionOld,
				bool  _isReUpdate = false)
		:notify(_notify)
		,project(_project)
		,filename(_filename)
		,versionNew(_versionNew, 0)
		,versionOld(_versionOld, 0)
		,globalDate(_globalDate)
		,globalCounter(_globalCounter)
		,isReUpdate(_isReUpdate)
	{
		for (int i=0; i<_strings.GetCount(); ++i)
		{
			int delimI1 = _strings[i].Find(DELIM);
			int delimI2 = _strings[i].ReverseFind(DELIM);
			if (delimI1 < 0 || delimI2 < 0 || delimI1 == delimI2)
				continue;
			strings.Add(_strings[i].Left(delimI1), 
			MakeTuple
			(
				_strings[i].Mid(delimI1+DELIM.GetLength(), delimI2-delimI1-DELIM.GetLength()),
				_strings[i].Right(_strings[i].GetLength()-delimI2-DELIM.GetLength())
				
			));
		}
		Request(&FileChanger::Work);
		Start();
	}
	
	struct StringStructure : Moveable<StringStructure>
	{
		enum TYPE {TYPE_STRING, TYPE_VERSION, TYPE_INCREMENT} type;
		char incrementType;
		String string;
		
		static void SetVersion(Vector<StringStructure> &s, const ProjectVersion::Version &v)
		{
			int vI = 0;
			ProjectVersion::Version version;
			for (int i=0; i<s.GetCount(); ++i)
			{
				if (s[i].type != TYPE_VERSION)
					continue;
				s[i].string = v.entries[vI++];
				if (vI >= v.entries.GetCount())
					break;
			}
		}
		static ProjectVersion::Version GetVersion(const Vector<StringStructure> &s)
		{
			ProjectVersion::Version version;
			for (int i=0; i<s.GetCount(); ++i)
			{
				if (s[i].type != TYPE_VERSION)
					continue;
				version.entries << s[i].string; 
			}
			return version;
		}
		static char GetIncrementType(const Vector<StringStructure> &s)
		{
			for (int i=0; i<s.GetCount(); ++i)
			{
				if (s[i].type != TYPE_INCREMENT)
					continue;
				return s[i].incrementType;
			}
			return 0;
		}
		static String GetString(const Vector<StringStructure> &s, const VectorMap<char,String> &increment)
		{
			String out;
			for (int i=0; i<s.GetCount(); ++i)
			{
				switch (s[i].type)
				{
				case TYPE_STRING:
					out << s[i].string;
					break;
				case TYPE_VERSION:
					out << s[i].string;
					break;
				case TYPE_INCREMENT:
					{
						int incrementI = increment.Find(s[i].incrementType);
						if (incrementI >= 0)
							out << increment[incrementI];
					}
					break;
				}
			}
			return out;
		}
	};
	
private:
	void Work()
	{
		FileIn file(filename);
		FileOut fileOut(filename+".tmp");
		file.Seek(0L);
		Vector<bool> processed;
		processed.SetCount(strings.GetCount(), false);
		bool isVersionChanged = false;
		String out;
		ProjectVersion::Version fileVersion;
		while (!file.IsEof())
		{
			int64 oldPosition = file.GetPos();
			String s = file.GetLine();
			out = s;
			for (int i=0; i<strings.GetCount(); ++i)
			{
				if (processed[i])
					continue;
				int sI1 = s.Find(strings.GetKey(i));
				if (sI1 < 0)
					continue;
				
				/*
				_mutex.Enter();
				Cout() << "\nWORK: '" << strings.GetKey(i) << "' ... '" << strings[i].b << "'";
				_mutex.Leave();
				*/
				int sI2 = s.ReverseFind(strings[i].b);
				if (sI2 < 0)
					continue;
				
				processed[i] = true;
				Vector<StringStructure> stringStructure;
				int l;
				/*
				_mutex.Enter();
				Cout() 
					<< "\nPARSE: s=(" 
					<< s.Mid(sI1+strings.GetKey(i).GetLength(),sI2-sI1-strings.GetKey(i).GetLength())
					<< "), mask=("
					<< strings[i].a
					<< ")"
				;
				_mutex.Leave();
				*/
				if (!ParseVersion
					(
						s.Mid
						(
							sI1+strings.GetKey(i).GetLength(),
							sI2-sI1-strings.GetKey(i).GetLength()
						), 
						strings[i].a, 
						stringStructure, 
						l
					)
				)
					continue;
				
				//определяем изменилась ли версия по сравнению с новой (и старой)
				fileVersion = StringStructure::GetVersion(stringStructure);
				if (!(fileVersion == versionNew))
				{
					_mutex.Enter();
					Cout() << "\nVER NEW: file=(" << fileVersion() << ") new=(" << versionNew() << ")";
					_mutex.Leave();
					if (!(fileVersion == versionOld))
						isVersionChanged = true;
				}
				
				//генерируем новую версию в зависимости от типа инкремента
				VectorMap<char,String> increment;
				increment.Add('c', FormatInt(versionNew.counter));
				increment.Add('C', FormatInt(globalCounter));
				increment.Add('d', FormatInt(GetSysDate().Get() - versionNew.date.Get()));
				increment.Add('D', FormatInt(GetSysDate().Get() - globalDate.Get()));
				
				//пересобираем в строку и записываем
				out.Clear();
				if (isReUpdate)
					StringStructure::SetVersion(stringStructure, versionNew);
				out << s.Left(sI1+strings.GetKey(i).GetLength()) << StringStructure::GetString(stringStructure, increment) << s.Right(s.GetLength()-sI2);
				_mutex.Enter();
				Cout() << "\nOUT: " << out;
				_mutex.Leave();
				break;
			}
			fileOut.PutLine(out);
		}
		file.Close();
		fileOut.Close();
		FileCopy(filename+".tmp", filename);
		FileDelete(filename+".tmp");
		notify->OnFileChangeFinished_(project, filename, isVersionChanged, fileVersion);
	}
	
	#ifdef TEST
	friend class MainWorker;
	#endif
	static bool ParseVersion(const String &s, const String &mask, Vector<StringStructure> &stringStructure, int &l)
	{
		l = 0;
		stringStructure.Clear();
		if (mask.IsEmpty())
			return true;
		
		bool out = true;
		int maskOffset = 0;
		static const String INCREMENT_SYMBOLS = "cCdD";
		int stringStructureI = -1;
		enum PARSE_STATE {STATE_UNKNOWN, STATE_SYMBOL, STATE_VERSION, STATE_INCREMENT} state = STATE_UNKNOWN;
		for (int i=0; i<s.GetLength(); ++i, ++l)
		{
			if (maskOffset >= mask.GetLength())
				break;
			
			if (state == STATE_UNKNOWN)
			{
				stringStructureI = stringStructureI < 0 ? 0 : stringStructureI+1;
				stringStructure.Add();

				if (mask[maskOffset] == '*')
				{
					stringStructure[stringStructureI].type = StringStructure::TYPE_VERSION;
					state = STATE_VERSION;
				}
				else
				if (INCREMENT_SYMBOLS.Find(mask[maskOffset]) >= 0)
				{
					stringStructure[stringStructureI].type = StringStructure::TYPE_INCREMENT;
					stringStructure[stringStructureI].incrementType = mask[maskOffset];
					state = STATE_INCREMENT;
				}
				else
				{
					stringStructure[stringStructureI].type = StringStructure::TYPE_STRING;
					state = STATE_SYMBOL;
				}
			}
			
			if (state == STATE_VERSION)
			{
				stringStructure[stringStructureI].string << (char) s[i];
				if (maskOffset < mask.GetLength()-1)
				{
					if (s[i+1] == mask[maskOffset+1])
					{
						++maskOffset;
						state = STATE_UNKNOWN;
						continue;
					}
				}
			}
			else
			if (state == STATE_INCREMENT)
			{
				stringStructure[stringStructureI].string << (char) s[i];
				if (!IsDigit(s[i]))
				{
					++maskOffset;
					state = STATE_UNKNOWN;
					continue;
				}
				else
				if (s[i+1] == mask[maskOffset+1])
				{
					++maskOffset;
					state = STATE_UNKNOWN;
					continue;
				}
			}
			
			if (state == STATE_SYMBOL)
			{
				if (s[i] != mask[maskOffset++])
					return false;

				state = STATE_UNKNOWN;
				stringStructure[stringStructureI].string << (char) s[i];
			}
		}
		return true;
	}
	
	FileChangerNotify        *notify;
	VectorMap<String,Tuple2<String,String> >  strings;
	String                    project, filename;
	ProjectVersion::Version   versionNew, versionOld;
	Date globalDate;
	int  globalCounter;
	bool isReUpdate;
};

class MainWorker : public CallbackQueue, public FileChangerNotify
{
public:
	MainWorker()
	{
		LoadFromXMLFile(versions, FILE_STATE);
		LoadFromXMLFile(config, FILE_CONFIG);
		
		#ifdef TEST
		/*
		String s = " 3, 10, 1901, 5.5";
		Vector<FileChanger::StringStructure> stringStructure;
		int l = -1;
		FileChanger::ParseVersion(s, " *, *, d, *.*", stringStructure, l);
		for (int i=0; i<stringStructure.GetCount(); ++i)
		{
			const FileChanger::StringStructure &structure = stringStructure[i];
			switch (structure.type)
			{
			case FileChanger::StringStructure::TYPE_VERSION:
				Cout() << "\nVERSION: " << structure.string;
				break;
			case FileChanger::StringStructure::TYPE_STRING:
				Cout() << "\nSTRING: " << structure.string;
				break;
			case FileChanger::StringStructure::TYPE_INCREMENT:
				Cout() << "\nINCREMENT: " << structure.incrementType;
				break;
			}
		}
		Cout() << "\n'" << s << "' :: " << l;
		
		config.GetAdd("Silos-master").files.GetAdd("Silos.cpp", Vector<String>() 
			<< "_TITLE = \"Силос-мастер ##*.*.d##\";"
		);
		config.GetAdd("Silos-master").files.GetAdd("Silos.rc", Vector<String>() 
			<< "fileVERSION ##*, *, d##, 0"
			<< "PRODUCTVERSION ##*, *, d##, 0"
			<< "VALUE \"fileVersion\", \"##*.*.d##\\0\""
			<< "VALUE \"ProductVersion\", \"##*.*.d##\\0\""
		);
		*/
		#endif
	}
	
	~MainWorker()
	{
		StoreAsXMLFile(versions, "versions", FILE_STATE);
		#ifdef TEST
		StoreAsXMLFile(config, "config", FILE_CONFIG);
		#endif
	}
	
	void Run(const Vector<String> &_projects)
	{
		//если не указаны проекты, добавляем все что есть в конфиге
		if (_projects.IsEmpty())
		{
			for (int i=0; i<config.GetCount(); ++i)
				projects.Add(config.GetKey(i));
		}
		else
		{
			for (int i=0; i<_projects.GetCount(); ++i)
				projects.Add(_projects[i]);
		}
		
		//добавляем все необходимые обработчики
		for (int projectI=0; projectI<projects.GetCount(); ++projectI)
		{
			ProjectVersion &ver = versions.GetAdd(projects.GetKey(projectI));
			++ver.global.counter;
			++ver.version.counter;
			
			const ProjectConfig &projectConfig = config.GetAdd(projects.GetKey(projectI));
			for (int fileI=0; fileI<projectConfig.files.GetCount(); ++fileI)
			{
				projects[projectI] << projectConfig.files.GetKey(fileI);
				fileChangers.Add
				(
				 	projectConfig.files.GetKey(fileI),
				 	new FileChanger
				 	(
				 		this,
				 		projects.GetKey(projectI),
				 		projectConfig.files.GetKey(fileI),
				 		projectConfig.files[fileI],
				 		ver.global.date,
				 		ver.global.counter,
				 		ver.version,
				 		ProjectVersion::Version()
				 	)
				);
			}
		}
		
		//главный цикл ожидания
		while (!fileChangers.IsEmpty())
		{
			Sleep(50);
			CallbackQueue::DoTasks();
		}

		//обновляем проекты, где есть файлы с изменёнными версиями
		for (int projectI=0; projectI<projects.GetCount(); ++projectI)
		{
			ProjectVersion &ver = versions.GetAdd(projects.GetKey(projectI));
			if (!ver.global.needReUpdate)
				continue;
			
			const ProjectConfig &projectConfig = config.GetAdd(projects.GetKey(projectI));
			_mutex.Enter();
			Cout() << "\nPROJECT REUPDATE: " << projects.GetKey(projectI) << " new=(" << ver.versionUpdate() << ")";
			_mutex.Leave();
			for (int fileI=0; fileI<projectConfig.files.GetCount(); ++fileI)
			{
				projects[projectI] << projectConfig.files.GetKey(fileI);
				fileChangers.Add
				(
				 	projectConfig.files.GetKey(fileI),
				 	new FileChanger
				 	(
				 		this,
				 		projects.GetKey(projectI),
				 		projectConfig.files.GetKey(fileI),
				 		projectConfig.files[fileI],
				 		ver.global.date,
				 		ver.global.counter,
				 		ver.versionUpdate,
				 		ver.version,
				 		true //ReUpdate!
				 	)
				);
			}
			
			ver.version = ver.versionUpdate;
		}
		
		//цикл ожидания обновления
		while (!fileChangers.IsEmpty())
		{
			Sleep(50);
			CallbackQueue::DoTasks();
		}
	}
	
	virtual void OnFileChangeFinished_(const String &project, const String &filename, bool isVersionChanged, ProjectVersion::Version ver) { Request(&MainWorker::OnFileChangeFinished, project, filename, isVersionChanged, ver); }
	
private:
	void OnFileChangeFinished(String project, String filename, bool isVersionChanged, ProjectVersion::Version ver)
	{
		fileChangers.RemoveKey(filename);
		Cout() << "\n" << filename << " - OK";
		if (isVersionChanged)
		{
			versions.GetAdd(project).versionUpdate = ver;
			versions.GetAdd(project).global.needReUpdate  = true;
		}
	}
	
	ArrayMap <String, FileChanger>     fileChangers;
	VectorMap<String, ProjectVersion>  versions;
	VectorMap<String, Vector<String> > projects;
	VectorMap<String, ProjectConfig>   config;
};

CONSOLE_APP_MAIN
{
	MainWorker().Run(CommandLine());
}
