//
// Copyright  2008-2010 ReCode Inc. and AWS Truepower, LLC. All rights reserved.
//
// Licensed under the Open Software License version 3.0
//
//
// this is a general utility class for static functions
// which don't really belong anywhere in particular
// but which may be used in more than one place
//
// We never make an instance of this class 
// and we never store any data in here 


#ifndef __UTILS_H__
#define __UTILS_H__





#include <CtrlLib/CtrlLib.h>

using namespace Upp;


#define ABOVE_RANGE 999999999

#include "Geom/Geom.h"




typedef WithDeepCopy<Vector <double> > doubleArray;
typedef WithDeepCopy<Vector <Point> > PointArray;
typedef WithDeepCopy<Vector <int> > intArray;
typedef WithDeepCopy<Array <intArray> > i2Array;
typedef WithDeepCopy<Array <i2Array> > i3Array;
typedef WithDeepCopy<Array <PointArray> > LineArray;
typedef WithDeepCopy<Array <doubleArray> > d2Array;
typedef WithDeepCopy<Array <d2Array> > d3Array;
typedef WithDeepCopy<Array <String> > StringArray;
typedef WithDeepCopy<Vector <Pointf> > PointfArray;
typedef WithDeepCopy<Array <PointfArray> > LinefArray;
typedef WithDeepCopy<Vector <bool> > boolArray;
typedef WithDeepCopy<Array <boolArray> > b2Array;
typedef WithDeepCopy<Vector <Rectf> > RectfArray;



#define LILEN 2000
#define PI 3.1415926535897


#define EARTH_RADIUS 6378135
//#define EARTH_RADIUS_SQUARED 40680606078225.0

enum PROJ
{
	PRJ_UNKNOWN = 0,
	PRJ_GEO = 1,
	PRJ_UTM = 2	
	
};

class Projection
{
	// projection files are messy!!!
	// why not one strict format?
	
protected:	
	String sProjection;
	String sDatum;	
	String sSpheroid;
	bool bValid;
	bool bFeetZ; // false = meters
	bool bDeg; // false = meters
	bool bNorth;
	int nZone;
	PROJ proj;	
	
public:
	Projection(String path);

	
	virtual ~Projection(){}

	bool IsGeographic(){return ToUpper(sProjection).Find("GEO")>=0;}
	bool IsInFeet(){return bFeetZ;}
	String GetDatum(){return sDatum;}
	String GetProjectionString(){return sProjection;}
	String GetSpheroid(){return sSpheroid;}
	bool IsValid(){return bValid;}
	PROJ GetProjection(){return proj;}
	int GetZone(){return nZone;}
	bool IsNorth(){return bNorth;}
};

template <class T> class Grid3d;

class CUtils 
{
public:
	typedef CUtils CLASSNAME;


	CUtils();
	virtual ~CUtils();


	static void SetCurrentDirectory_(String path)
	{
	#ifdef PLATFORM_WIN32	
		SetCurrentDirectory(path);
	#endif
	#ifdef PLATFORM_POSIX
		int n = chdir(path);
	#endif
	}
	
	static void ReadProjection(void* pOW,String path);
	
	static void Atan2Test();

	static int comma(const char *ptr, int j);
	static String tocomma(const char *ptr,int j);
	static int space(const char* pBuf,int c);
	static int graph(const char *ptr, int j);
	static int number(const char *ptr, int j);
	
	static Pointf GetNearestNodePt(Pointf pt,Rectf rcGrid,double res);
	
	static bool IsGeo(Rectf& rc);
	
	static String TermInQuotes(char* pBuf,int& c);
	static String TermInQuotes(char* pBuf,int& c,char cQuote);

	static String SigFigs(double f,int n);

	static bool IsPtInRect(Point& pt, Rect& rc); // references are for speed only
	static bool IsPtInRect(Pointf& pt, Rectf& rc); // references are for speed only

	static bool PtInPolyPolygon(LineArray& lines, Point ptTest,Rect r);
	static bool PtInPolyPolygonf(LinefArray& lines, Pointf ptTest,Rectf r);

	static int GetFolderPath(char* folder,const char* path);

	static bool G_PtInPolygon(Point *rgpts, int wnumpts, Point ptTest,Rect r) ;
	static bool G_PtInPolygonf(Pointf *rgpts, int wnumpts, Pointf ptTest,Rectf r) ;

	static bool G_PtInPolyRect(Point *rgpts, int wnumpts, Point ptTest, Rect *prbound) ;
	static bool G_PtInPolyRectf(Pointf *rgpts, int wnumpts, Pointf ptTest, Rectf *prbound) ;

	static bool Intersect(Point& pA1, Point& pA2, Point& pB1, Point& pB2) ;
	static bool Intersectf(Pointf& pA1, Pointf& pA2, Pointf& pB1, Pointf& pB2) ;

	static Pointf Intersectionf(Pointf& p1, Pointf& p2,Rectf& rc);
	static Point Intersection(Point& p1, Point& p2, Point& p3, Point& p4);
	static Pointf Intersectionf(Pointf& p1, Pointf& p2, Pointf& p3, Pointf& p4);

	static int  CCW(Point& p0, Point& p1, Point& p2) ;
	static int  CCWf(Pointf& p0, Pointf& p1, Pointf& p2) ;

	static bool PtInMPRect(Rect&,Point&);
	static bool PtInMPRectf(Rectf,Pointf);
	static bool PtInflatedMPRectf(Rectf& rc,Pointf& pt,double d);
	static bool PtInflatedMPRectf(Pointf ptRC1,Pointf ptRC2,Pointf& pt,double d);
	static bool MPRectfInMPRectf(Rectf& rcInner,Rectf& rcOuter);

	static Rectf MPRect(Pointf pt1,Pointf pt2);
	static Rectf MPRect(Pointf pt1,Pointf pt2,Pointf pt3);
	static Rectf MPRect(Rectf rc,Pointf pt3);
	
	static Rectf Inflate(Rectf rc,double dx,double dy);

	static void CombinedMapRect(Rectf& rc1,const Rectf rc2);

	static String GetFileName(String path);

	static String GetFolder(const String path);

//	static double frand(); // use this
//	static uint32 RandomInt32();
	
	static bool IsRectfInRectf(Rectf rcOuter,Rectf rcInner);
	
//	static double LineFromPoint(Pointf ptA1, Pointf ptA2, Pointf ptC);

	static void GrowRectToIncludePt(Rectf& rc,Pointf pt);

	static String EatSpace(String s);

	static Pointf Rotate(double rotAngle,Pointf pt);

	static double Distance( Pointf P, Pointf P0, Pointf P1 );
	static Pointf DistanceManhattan( Pointf P, Pointf P0, Pointf P1 );
	static double DotProduct(Pointf v1, Pointf v2);
	static double Distance(Pointf p1,Pointf p2);
	static double Distance3D(Pointf p1,double z1, Pointf p2,double z2);
	
	static float Sqrt(float x);
	
	static double DistanceOfPointFromLine(Pointf3 pt3P,Pointf3 pt3A,Pointf3 pt3B);

	static double GetGaussianDistributedRandomNumber();
	static void GetGaussianDistributedRandomNumber(double& y1,double& y2);

	static double GetNormAreaUnderRose(double startRadians,double stopRadians,doubleArray& d,doubleArray& prob);
	static double GetNormAreaUnderRose(double startRadians,double stopRadians,doubleArray& d);

	static double Decimals(double val, int places);
	
	static double GetMagnitude(Pointf pt){return sqrt(pt.x*pt.x+pt.y*pt.y);} // vector scalar

	// interpolation functions
	static double BiLinear(Pointf pt,double res,double dBL,double dBR,double dTL,double dTR);
	static double BiLinear(Pointf pt,double resX,double resY,double dBL,double dBR,double dTL,double dTR);
	static double BiCubic(Pointf pt,Pointf ptOffset,int nX,int nY,double resx,double resy,const d2Array& d);
//	static double BiCubic(Pointf pt,Pointf ptOffset,int nX,int nY,double resx,double resy,const d3Array& d,const int sec);
	static double BiCubic(Pointf pt,Pointf ptOffset,int nX,int nY,double resx,double resy,const Grid3d<double>& d,const int sec);
	         
	template<class T> static void Fill(Vector<T>& a,T t);
             
private:
	static double BiCubic(Pointf pt,double resx,double resy,d2Array& z);

public:

	static String GetMonth(int m); 	// 1 based
	static int GetDaysInMonth(int m);		// 1 based
	static int GetMonthFromMinuteOfYear(int moy);
	static int GetMonthIndex(String month);

	static double GetCentralMeridian(int zone);

	static double COE(Rectf rc,Pointf pt); // curvature of earth adjustment
	

//	static Pointf GeographicToUTM(Pointf ptGeo,void* pMain,String MapDatum=String("WGS_1984"));
	static Pointf UTMtoGeographic(Pointf ptUtm, int ZoneNum, bool bSouth,String MapDatum="WGS_1984");
	static Rectf UTMtoGeographicMax(Rectf ptUtm, int ZoneNum, bool bSouth,String MapDatum="WGS_1984");
	static int GetZoneUTM(Pointf pt);
	
	static bool Intersects(const Rectf& rc1,const Rectf& rc2);
	static bool Intersects(const Rect& rc1,const Rect& rc2);
	
	static bool IsSame(Rectf rc1,Rectf rc2);
	
	// export these to utility file at some point
	static unsigned short BigEndianUnsignedShort(char* p);
	static short BigEndianShort(char* p);
	static unsigned int BigEndianUnsignedInt(char* p);
	static int BigEndianInt(char* p);
	static double BigEndianDouble(char* p);
	static int LittleEndianInt(char* p);
	static double LittleEndianDouble(char* p);
	static float LittleEndianFloat(char* p);
	static float BigEndianFloat(char* p);
	
	static Point GetSouthEastNodeIndices(const Pointf& pt,const Rectf& rcGrid,const double& res);
	static Pointf GetSouthEastNodePt(const Pointf& pt,const Rectf& rcGrid,const double& res);
	
	static void Fill(d2Array& ar,double val);
	
//	static String GetFileTitle(String path){return GetFileTitle(path);}
	
	static void SerializeString(Stream& s,String& ss);
	static void SerializeMultiLineString(Stream& s,String& ss);
	static void SerializeStringArray(Stream& s,StringArray& ss);
	
	template<class T> static bool SerializeArray1D(Stream& s,T& ar)
	{
		int n;
		
		if(s.IsStoring())
		{
			n = ar.GetCount();
			s % n;
		}
		else // IsLoading
		{
			s % n;
			if(n>=0)
				ar.SetCount(n);	
			else
				return false;	
		}	
	
		for(int i=0;i<n;i++)
		{
			s % ar[i];			
		}
		
		return true;		
	}	
	
	template<class T> static void SerializeArray2D(Stream& s,T& ar)
	{
		int n1,n2;
		
		if(s.IsStoring())
		{
			n1 = ar.GetCount();
			s % n1;
			
			for(int i=0;i<n1;i++)
			{
				n2 = ar[i].GetCount();
				
				s % n2;
				
				for(int j=0;j<n2;j++)
				{
					s % ar[i][j];
				}
			}			
		}
		else // IsLoading
		{
			s % n1;
			ar.SetCount(n1);		
	
			for(int i=0;i<n1;i++)
			{
				s % n2;
				ar[i].SetCount(n2);
				for(int j=0;j<n2;j++)
				{
					s % ar[i][j];
				}
			}			
		}	
	}
	
	template<class T> static void SerializeArray3D(Stream& s,T& ar)
	{
		int n1,n2,n3;
		
		if(s.IsStoring())
		{
			n1 = ar.GetCount();
			s % n1;
			
			for(int i=0;i<n1;i++)
			{
				n2 = ar[i].GetCount();
				
				s % n2;
				
				for(int j=0;j<n2;j++)
				{
					n3 = ar[i][j].GetCount();
					
					s % n3;
					
					for(int k=0;k<n3;k++)
					{
						s % ar[i][j][k];
					}
				}
			}			
		}
		else // IsLoading
		{
			s % n1;
			ar.SetCount(n1);		
	
			for(int i=0;i<n1;i++)
			{
				s % n2;
				ar[i].SetCount(n2);
				for(int j=0;j<n2;j++)
				{
					s % n3;
					ar[i][j].SetCount(n3);
					for(int k=0;k<n3;k++)
					{
						s % ar[i][j][k];
					}
				}
			}			
		}	
	}
	

	template<class T, class Less> static void BubbleSort(T& list,Less less)
	{
		bool swapped;
		int n = list.GetCount();
		do
		{
			swapped = false;
			n--;
			for(int i=0;i<n;i++)
			{
				if(less(list[i+1],list[i]))
				{
					list.Swap(i,i+1);
					swapped = true;
				}
			}
						
		}while(swapped);		
	}

protected:
	static Pointf GeographicToUTM(Pointf ptGeo,int zoneForce,String MapDatum=String("WGS_1984"));
//	static Pointf GeographicToUKtm(Pointf ptGeo,String MapDatum=String("OSGB36"));
//	static Pointf UKtmToGeographic(Pointf ptUKtm,String MapDatum=String("OSGB36"));

//	static Pointf GeographicToUTM(int degLat,int minLat,double secLat,int degLon,int minLon,double secLon)
//		{return GeographicToUTM(Pointf(secLat/3600.0+minLat/60.0+(double)degLat,secLon/3600.0+minLon/60.0+(double)degLon));}
//	static Pointf GeographicToUTM(int degLat,double minLat,int degLon,double minLon){return GeographicToUTM(Pointf(minLat/60.0+(double)degLat,minLon/60.0+(double)degLon));}
//	static Pointf GeographicToUTM(double degLat,double degLon){return GeographicToUTM(Pointf(degLat,degLon));}

private:
	static void GetABK(String MapDescription,double& a, double& b, double& k);
	static Pointf GeographicToUTM(Pointf ptGeo,String MapDatum=String("WGS_1984"));
		
};


// this class is for efficient 2d double-precision floating point data storage
//class Grid2d 
template <class T> class Grid2d
{
public:
	Grid2d(){m_nRows=m_nCols=0;m_ppData=NULL;}// default constructor
	Grid2d(const Grid2d& grid)							// copy constructor
	{
		m_nRows=m_nCols=0;
		m_ppData = NULL;

		if(grid.m_nRows==0 || grid.m_nCols==0)
		{
			return;			
		}
		
		if(Dimension(grid.m_nCols,grid.m_nRows,0))
		{
			// copy values		
			for(int i=0;i<grid.m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					this->Set(i,j,grid.m_ppData[i][j]);	
				}				
			}	
		}		
	}
	
	Grid2d& operator=(const Grid2d& grid)
	{
		if(this==&grid)return *this;
		
		Clear();
		
		if(grid.m_nRows==0 || grid.m_nCols==0)
		{
			return *this;			
		}
		
		if(Dimension(grid.m_nCols,grid.m_nRows,0))
		{
			// copy values		
			for(int i=0;i<grid.m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					this->Set(i,j,grid.m_ppData[i][j]);	
				}				
			}	
		}		
		
		return *this;
	}
	
	virtual ~Grid2d()
	{
		Clear();
	}
	
	void Clear()
	{
		if(m_nCols==0)
			return;
		
		for(int i=0;i<m_nCols;i++)
		{
			delete[] m_ppData[i];
//			m_ppData[i].Clear();
		}
		delete[] m_ppData;
//		m_ppData.Clear();
		m_ppData=NULL;
		m_nRows=m_nCols=0;
	}
	
	int GetNumX(){return m_nCols;}
	int GetNumY(){return m_nRows;}
	int GetRows(){return m_nRows;}
	int GetCols(){return m_nCols;}

	bool SetSize(int nCols,int nRows,T val=0)
		{return Dimension(nCols,nRows,val);}
	bool Dimension(int nCols,int nRows,T val=0)
	{
		if(m_ppData!=NULL)
		{
			if(nRows==m_nRows && nCols==m_nCols)
			{
//				if(val>-99999)
				{
					for(int i=0;i<nCols;i++)
					{
						for(int j=0;j<nRows;j++)
						{
							m_ppData[i][j] = val;
						}
					}
				}
				
				return true;
			}				

			Clear();
		}

		m_nCols = nCols;
		m_nRows = nRows;
		
		m_ppData = new T*[nCols];
//		m_ppData.Alloc(nCols);
		
		if(m_ppData==NULL)
		{
			m_nRows=m_nCols=0;
			return false; // failed memory allocation so return false
		}
		
		for(int i=0;i<m_nCols;i++)
		{
			m_ppData[i] = new T[nRows];
//			m_ppData[i].Alloc(nRows);

			if(m_ppData[i]==NULL)
			{
				// we failed to allocate memory so destroy everything and return false
				for(  ;i<=0;i--)
				{
					delete[] m_ppData[i];
				}
				delete[] m_ppData;
				m_ppData = NULL;
				m_nRows=m_nCols=0;
				return false;
			}
			
//			if(val>-99999) // init values
			{
				for(int j=0;j<m_nRows;j++)
				{
					m_ppData[i][j] = val;
				}
			}
		}
		
		return true;
	}
	
	void Fill(T val)
	{
		for(int i=0;i<m_nCols;i++)
		{
			for(int j=0;j<m_nRows;j++)
			{
				m_ppData[i][j] = val;
			}
		}		
	}
	
	T* operator[](int nCol)			{return GetCol(nCol);}
	
	inline T* GetCol(int nCol)	
	{
		if(nCol>=0 && nCol<m_nCols)
			return m_ppData[nCol];
		else
			return m_ppData[0];
	}
	
	T Get(int nCol,int nRow)			{if(nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols)return m_ppData[nCol][nRow];else return T();}
	void Set(int nCol,int nRow,T val)	{if(nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols)m_ppData[nCol][nRow]=val;}	
	
	virtual void Serialize(Stream& s)
	{
		int n = -1;
		s % n;
		
		int	version = VERSION_MAJOR*100000000 + VERSION_MINOR*1000000 /*+ VERSION_VAR*10000*/ + VERSION_COMMIT;
		n = version;
		s % n; // so you can check it and do logic on it
		version = n;

		String sType = "Grid2d";	
		if(version%10000<139)// october 2008 - delete in 1 year
		{
			s % sType;	
		}
		else
		{
			CUtils::SerializeString(s,sType);	
		}

		int nRows = m_nRows;
		int nCols = m_nCols;
				
		s % nCols % nRows;
		
		if(s.IsLoading())
		{
			if(!Dimension(nCols,nRows,0))
				return;	
		}

		for(int i=0;i<m_nCols;i++)
		{
			for(int j=0;j<m_nRows;j++)
			{
				s % m_ppData[i][j];	
			}				
		}			
	}
	
protected:
	
	// current size of grid
	int m_nRows;
	int m_nCols;	
	
	// data values
	T** m_ppData;	
	
//	WithDeepCopy<Buffer<WithDeepCopy<Buffer<T> > > >  m_ppData;	
};


//class Grid3d
template <class T> class Grid3d
{
public:
	Grid3d(){m_nRows=m_nCols=0;m_nDirs=0;m_pppData=NULL;}// default constructor
	Grid3d(const Grid3d& grid)							// copy constructor
	{
		m_nRows=m_nCols=m_nDirs=0;
		m_pppData = NULL;

		if(grid.m_nRows==0 || grid.m_nCols==0 || grid.m_nDirs==0)
		{
			return;			
		}
		
		if(Dimension(grid.m_nCols,grid.m_nRows,grid.m_nDirs))
		{
			// copy values		
			for(int i=0;i<grid.m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					for(int k=0;k<m_nDirs;k++)
					{
						this->Set(i,j,k,grid.m_pppData[i][j][k]);	
					}
				}				
			}	
		}
		else
			DUMP("error allocating memory in copy constructor");		
	}
	
	Grid3d& operator=(const Grid3d& grid)
	{
		if(this==&grid)return *this;
		
		Clear();
		
		if(grid.m_nRows==0 || grid.m_nCols==0 || grid.m_nDirs==0)
		{
			return *this;			
		}
		
		if(Dimension(grid.m_nCols,grid.m_nRows,grid.m_nDirs))
		{
			// copy values		
			for(int i=0;i<m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					for(int k=0;k<m_nDirs;k++)
					{
						this->Set(i,j,k,grid.m_pppData[i][j][k]);	
					}
				}				
			}	
		}		
		
		return *this;
	}
	
	virtual ~Grid3d()
	{
		Clear();
	}
	
	void Clear()
	{
		if(m_nCols==0)
			return;
		
		for(int i=0;i<m_nCols;i++)
		{
			for(int j=0;j<m_nRows;j++)
			{
				delete[] m_pppData[i][j];				
			}			
			delete[] m_pppData[i];
		}
		delete[] m_pppData;
		m_pppData=NULL;
		m_nRows=m_nCols=0;
	}
	
	int GetNumX(){return m_nCols;}
	int GetNumY(){return m_nRows;}
	int GetNumZ(){return m_nDirs;}
	int GetRows(){return m_nRows;}
	int GetCols(){return m_nCols;}
	int GetDirs(){return m_nDirs;}

	bool SetSize(int nCols,int nRows,int nDirs,T val=-99999)
		{return Dimension(nCols,nRows,nDirs,val);}
	bool Dimension(int nCols,int nRows,int nDirs,T val=-99999)
	{
		if(m_pppData!=NULL)
		{
			if(nRows==m_nRows && nCols==m_nCols && nDirs==m_nDirs)
			{
				if(val>-99999)
				{
					for(int i=0;i<nCols;i++)
					{
						for(int j=0;j<nRows;j++)
						{
							for(int k=0;k<m_nDirs;k++)
							{
								m_pppData[i][j][k] = val;
							}
						}
					}
				}
				
				return true;
			}				

			Clear();
		}

		m_nCols = nCols;
		m_nRows = nRows;
		m_nDirs = nDirs;
		
		m_pppData = new T**[nCols];
		
		if(m_pppData==NULL)
		{
			m_nRows=m_nCols=m_nDirs=0;
			return false; // failed memory allocation so return false
		}
		
		for(int i=0;i<m_nCols;i++)
		{
			m_pppData[i] = new T*[nRows];

			if(m_pppData[i]==NULL)
			{
				i--;
				// we failed to allocate memory so destroy everything and return false
				for(  ;i>=0;i--)
				{
					for(int j=0;j<m_nRows;j++)
					{
						delete[] m_pppData[i][j];
					}					
					delete[] m_pppData[i];
				}
				delete[] m_pppData;
				m_pppData = NULL;
				m_nRows=m_nCols=m_nDirs=0;
				return false;
			}
			
			for(int j=0;j<m_nRows;j++)
			{
				m_pppData[i][j] = new T[nDirs];
				
				if(m_pppData[i][j]==NULL)
				{
					for(  ;j>=0;j--)
					{
						delete m_pppData[i][j];
					}
					
					i--;
					// we failed to allocate memory so destroy everything and return false
					for(  ;i>=0;i--)
					{
						for(int j=0;j<m_nRows;j++)
						{
							delete[] m_pppData[i][j];
						}					
						delete[] m_pppData[i];
					}
					delete[] m_pppData;
					m_pppData = NULL;
					m_nRows=m_nCols=m_nDirs=0;
					return false;
				}			
			}
			
			if(val>-99999) // init values
			{
				for(int j=0;j<m_nRows;j++)
				{
					for(int k=0;k<m_nDirs;k++)
					{
						m_pppData[i][j][k] = val;
					}
				}
			}
		}
		
		return true;
	}
	
	T** operator[](int nCol)			{return GetCol(nCol);}
	
	inline T** GetCol(int nCol)	
	{
		if(nCol>=0 && nCol<m_nCols)
			return m_pppData[nCol];
		else
			return m_pppData[0];
	}
	
	template <class S> bool GetGrid2dAtDir(int z,Grid2d<S>& grid)
	{
		return GetGrid2dAtZ(z,grid);
	}

	template <class S> bool GetGrid2dAtZ(int z,Grid2d<S>& grid)
	{
		if(!grid.Dimension(m_nCols,m_nRows))
			return false;
		
		for(int x=0;x<m_nCols;x++)
		{
			for(int y=0;y<m_nRows;y++)
			{
				grid[x][y] = (S)m_pppData[x][y][z];
			}
		}	
		
		return true;
	}

	template <class S> bool GetGrid2dAtX(int x,Grid2d<S>& grid)
	{
		if(!grid.Dimension(m_nRows,m_nDirs))
			return false;
		
		for(int y=0;y<m_nRows;y++)
		{
			for(int z=0;z<m_nDirs;z++)
			{
				grid[y][z] = (S)m_pppData[x][y][z];
			}
		}	
		
		return true;
	}

	
	T Get(int nCol,int nRow,int nDir)const		{if(nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols && nDir>=0 && nDir<m_nDirs)return m_pppData[nCol][nRow][nDir];return 0.0;}
	void Set(int nCol,int nRow,int nDir,T val)	{if(nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols && nDir>=0 && nDir<m_nDirs)m_pppData[nCol][nRow][nDir]=val;}	
	T* Get(int nCol,int nRow){if(nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols)return m_pppData[nCol][nRow];return NULL;}
	
	virtual void Serialize(Stream& s)
	{
		int n = -1;
		s % n;
		
		int	version = VERSION_MAJOR*100000000 + VERSION_MINOR*1000000 /*+ VERSION_VAR*10000*/ + VERSION_COMMIT;
		n = version;
		s % n; // so you can check it and do logic on it
		version = n;

		String sType = "Grid3d";	
		if(version%10000<139)// october 2008 - delete in 1 year
		{
			s % sType;	
		}
		else
		{
			CUtils::SerializeString(s,sType);	
		}

		int nRows = m_nRows;
		int nCols = m_nCols;
		int nDirs = m_nDirs;
				
		s % nCols % nRows % nDirs;
		
		if(s.IsLoading())
		{
			if(!Dimension(nCols,nRows,nDirs))
				return;	
		}

		for(int i=0;i<m_nCols;i++)
		{
			for(int j=0;j<m_nRows;j++)
			{
				for(int k=0;k<m_nDirs;k++)
				{
					s % m_pppData[i][j][k];	
				}
			}				
		}			
	}
	
protected:
	
	// current size of grid
	int m_nRows;
	int m_nCols;
	int m_nDirs;	
	
	// data values
	T*** m_pppData;	
	

};

//class Grid4d
template <class T> class Grid4d
{
public:
	Grid4d(){m_nLayers=m_nRows=m_nCols=0;m_nDirs=0;m_ppppData=NULL;}// default constructor
	Grid4d(const Grid4d& grid)							// copy constructor
	{
		m_nLayers=m_nRows=m_nCols=m_nDirs=0;
		m_ppppData = NULL;

		if(grid.m_nLayers==0 || grid.m_nRows==0 || grid.m_nCols==0 || grid.m_nDirs==0)
		{
			return;			
		}
		
		if(Dimension(grid.m_nLayers,grid.m_nCols,grid.m_nRows,grid.m_nDirs))
		{
			// copy values		
			for(int z=0;z<grid.m_nLayers;z++)
			{
				for(int i=0;i<grid.m_nCols;i++)
				{
					for(int j=0;j<m_nRows;j++)
					{
						for(int k=0;k<m_nDirs;k++)
						{
							this->Set(z,i,j,k,grid.m_ppppData[z][i][j][k]);	
						}
					}				
				}
			}
		}		
	}
	
	Grid4d& operator=(const Grid4d& grid)
	{
		if(this==&grid)return *this;
		
		Clear();
		
		if(grid.m_nLayers=0 || grid.m_nRows==0 || grid.m_nCols==0 || grid.m_nDirs==0)
		{
			return *this;			
		}
		
		if(Dimension(grid.m_nCols,grid.m_nRows,grid.m_nDirs))
		{
			// copy values		
			for(int z=0;z<grid.m_nLayers;z++)
			{
				for(int i=0;i<grid.m_nCols;i++)
				{
					for(int j=0;j<m_nRows;j++)
					{
						for(int k=0;k<m_nDirs;k++)
						{
							this->Set(z,i,j,k,grid.m_ppppData[z][i][j][k]);	
						}
					}				
				}
			}
		}		
		
		return *this;
	}
	
	virtual ~Grid4d()
	{
		Clear();
	}
	
	void Clear(int z=-1)
	{
		if(m_nLayers==0)
			return;
		
		if(z<0)
			z=m_nLayers-1;
		
		for(  ;z>=0;z--)
		{
			for(int i=0;i<m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					delete[] m_ppppData[z][i][j];				
				}			
				delete[] m_ppppData[z][i];
			}
			delete[] m_ppppData[z];
		}
		delete[] m_ppppData;
		m_ppppData=NULL;
		m_nLayers=m_nRows=m_nCols=m_nDirs=0;
	}
	
	int GetLayers(){return m_nLayers;}
	int GetRows(){return m_nRows;}
	int GetCols(){return m_nCols;}
	int GetDirs(){return m_nDirs;}

	bool SetSize(int layers,int nCols,int nRows,int nDirs,T val=-99999)
		{return Dimension(layers,nCols,nRows,nDirs,val);}
	bool Dimension(int layers,int nCols,int nRows,int nDirs,T val=-99999)
	{
		if(m_ppppData!=NULL)
		{
			if(layers==m_nLayers && nRows==m_nRows && nCols==m_nCols && nDirs==m_nDirs)
			{
				if(val>-99999)
				{
					for(int z=0;z<layers;z++)
					{
						for(int i=0;i<nCols;i++)
						{
							for(int j=0;j<nRows;j++)
							{
								for(int k=0;k<m_nDirs;k++)
								{
									m_ppppData[z][i][j][k] = val;
								}
							}
						}
					}
				}
				
				return true;
			}				

			Clear();
		}

		m_nLayers = layers;
		m_nCols = nCols;
		m_nRows = nRows;
		m_nDirs = nDirs;
		
		m_ppppData = new T***[layers];
		
		if(m_ppppData==NULL)
		{
			m_nLayers=m_nRows=m_nCols=m_nDirs=0;
			return false; // failed memory allocation so return false
		}
		
		for(int z=0;z<m_nLayers;z++)
		{
			m_ppppData[z] = new T**[nCols];
			
			if(m_ppppData[z]==NULL)
			{
				z--;
				// we failed to allocate memory so destroy everything and return false
				Clear(z);
				return false;
			}

			for(int i=0;i<m_nCols;i++)
			{
				m_ppppData[z][i] = new T*[nRows];
	
				if(m_ppppData[z][i]==NULL)
				{
					i--;
					// we failed to allocate memory so destroy everything and return false
					for(  ;i>=0;i--)
					{
						for(int j=0;j<m_nRows;j++)
						{
							delete[] m_ppppData[z][i][j];
						}					
						delete[] m_ppppData[z][i];
					}
					Clear(--z);
					return false;
				}
				
				for(int j=0;j<m_nRows;j++)
				{
					m_ppppData[z][i][j] = new T[nDirs];
					
					if(m_ppppData[z][i][j]==NULL)
					{
						for(  ;j>=0;j--)
						{
							delete m_ppppData[z][i][j];
						}
						
						i--;
						// we failed to allocate memory so destroy everything and return false
						for(  ;i>=0;i--)
						{
							for(int j=0;j<m_nRows;j++)
							{
								delete[] m_ppppData[z][i][j];
							}					
							delete[] m_ppppData[i];
						}
						Clear(--z);
						return false;
					}			
				}
				
				if(val>-99999) // init values
				{
					for(int j=0;j<m_nRows;j++)
					{
						for(int k=0;k<m_nDirs;k++)
						{
							m_ppppData[z][i][j][k] = val;
						}
					}
				}
			}
		}
		return true;
	}
	
	T*** operator[](int layer)			{return GetLayer(layer);}
	
	inline T*** GetLayer(int layer)	
	{
		if(layer>=0 && layer<m_nLayers)
			return m_ppppData[layer];
		else
			return m_ppppData[0];
	}
	
	template <class S> bool GetGrid3dAtZ(int z,Grid3d<S>& grid)
	{
		if(!grid.Dimension(m_nCols,m_nRows,m_nDirs))
			return false;
		
		for(int x=0;x<m_nCols;x++)
		{
			for(int y=0;y<m_nRows;y++)
			{
				for(int d=0;d<m_nDirs;d++)
				{
					grid[x][y][d] = (S)m_ppppData[z][x][y][d];
				}
			}
		}	
		
		return true;
	}

	template <class S> bool GetGrid2dAtZAtDir(int z,int dir,Grid2d<S>& grid)
	{
		if(!grid.Dimension(m_nCols,m_nRows))
			return false;
		
		for(int x=0;x<m_nCols;x++)
		{
			for(int y=0;y<m_nRows;y++)
			{
				grid[x][y] = (S)m_ppppData[z][x][y][dir];
			}
		}	
		
		return true;
	}
	
	T Get(int layers,int nCol,int nRow,int nDir)const		
		{if(layers>=0 && nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols && nDir>=0 && nDir<m_nDirs && layers<m_nLayers)return m_ppppData[layers][nCol][nRow][nDir];return 0.0;}
	void Set(int layers,int nCol,int nRow,int nDir,T val)	
		{if(layers>=0 && nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols && nDir>=0 && nDir<m_nDirs && layers<m_nLayers)m_ppppData[layers][nCol][nRow][nDir]=val;}	
	T* Get(int layers,int nCol,int nRow)
		{if(layers>=0 && nRow>=0 && nCol>=0 && nRow<m_nRows && nCol<m_nCols && layers<m_nLayers)return m_ppppData[layers][nCol][nRow];return NULL;}
	
	virtual void Serialize(Stream& s)
	{
		int n = -1;
		s % n;
		
		int	version = VERSION_MAJOR*100000000 + VERSION_MINOR*1000000 /*+ VERSION_VAR*10000*/ + VERSION_COMMIT;
		n = version;
		s % n; // so you can check it and do logic on it
		version = n;

		String sType = "Grid4d";	
		if(version%10000<139)// october 2008 - delete in 1 year
		{
			s % sType;	
		}
		else
		{
			CUtils::SerializeString(s,sType);	
		}

		int layers = m_nLayers;
		int nRows = m_nRows;
		int nCols = m_nCols;
		int nDirs = m_nDirs;
				
		s % nCols % nRows % nDirs % layers;
		
		if(s.IsLoading())
		{
			if(!Dimension(layers,nCols,nRows,nDirs))
				return;	
		}

		for(int z=0;z<m_nLayers;z++)
		{
			for(int i=0;i<m_nCols;i++)
			{
				for(int j=0;j<m_nRows;j++)
				{
					for(int k=0;k<m_nDirs;k++)
					{
						s % m_ppppData[z][i][j][k];	
					}
				}				
			}			
		}
	}
	
protected:
	
	// current size of grid
	int m_nLayers;
	int m_nRows;
	int m_nCols;
	int m_nDirs;	
	
	// data values
	T**** m_ppppData;	
};


// this is the binary wind resource file format class
// this class would be a lot simpler if it were not for the fact that different blocks can have
// different data types.
class WRB
{
public:
	
	enum BLOCK_MEANING
	{
		BM_UNKNOWN				=0,
		BM_ELEVATION			=1,
		BM_MEANWINDSPEED		=2,
		BM_A					=3,
		BM_K					=4,
		BM_POWER				=5,
		BM_TI					=6,
		BM_INFLOWANGLE			=7,
		BM_P					=8,
		BM_DIRECTION			=9,
		BM_Z0					=10,
		BM_AIRDENSITY			=11,
		BM_VERTICALVELOCITY		=12,
		BM_WINDSHEAREXPONENT	=13,
		BM_GROUNDPOROSITY		=14,
		BM_VEGETATIONHEIGHT		=15,
		BM_ELOSS				=16,
		BM_UNCERTAINTY			=17
		// add more here
	};
	
	WRB()
	{
		header.resX=-1;
		header.resY=-1;
	}
	WRB(Rectf rcGrid,double res,short directions=12,short heights=0,short windSpeeds=0)
	{
		header.type=1001;
		header.version=1;
		header.horzUnits=1;
		header.vertUnits=1;
		strcpy(header.datumProjection,"UTM-unknown");
		header.directions=directions;
		header.heights=heights;
		header.windSpeeds=windSpeeds;
		header.minX=rcGrid.left;
		header.maxX=rcGrid.right;
		header.minY=rcGrid.bottom;
		header.maxY=rcGrid.top;
		header.resX=res;
		header.resY=res;
		header.numBlocks=0;
		blocks.SetCount(0);
	}
	
	virtual ~WRB() // called when local object goes out of scope
	{
		for(int i=blocks.GetCount()-1;i>=0;i--)
		{
			switch(blocks[i]->header.dataType)
			{
				case 1:
				{
					Block<double>* pB = (Block<double>*)blocks[i];
					delete pB;			
					break;
				}
				case 2:
				{
					Block<unsigned char>* pB = (Block<unsigned char>*)blocks[i];
					delete pB;			
					break;
				}
				case 3:
				{
					Block<short>* pB = (Block<short>*)blocks[i];
					delete pB;			
					break;
				}
				case 4:
				{
					Block<int32>* pB = (Block<int32>*)blocks[i];
					delete pB;			
					break;
				}
				case 0:
				default:
				{
					Block<float>* pB = (Block<float>*)blocks[i];			
					delete pB;	
				}
			}
		}	
	}
	
	struct Header
	{
		// fixed size = 100 bytes
		unsigned short type;
		unsigned short version;
		unsigned char horzUnits;
		unsigned char vertUnits;
		char datumProjection[30];
		short directions;
		short heights;
		short windSpeeds;
		double minX;
		double maxX;
		double minY;
		double maxY;
		double resX;
		double resY;
		unsigned short numBlocks;
		
		// this takes up no space
		Size GetGridSize()
		{
			Size size;
			
			size.cx = (int)((maxX-minX)/resX) + 1;
			size.cy = (int)((maxY-minY)/resY) + 1;

			return size;
		}
		
		Rectf GetGridRect(){return Rectf(minX,maxY,maxX,minY);}
	};
	
	struct BlockHead
	{
		// fixed size = 64bytes
		unsigned short meaning;
		float height;
		short direction;
		float windSpeed;
		double probability;
		int id;
		int64 offset;
		unsigned char dataType;	// type of T
		
		bool bFirst; // working flag to mark whether we extracted this yet (false when extracted)		

		BlockHead()				{	bFirst = true;	}
		virtual ~BlockHead()	{}
		
		bool IsSameLayer(BlockHead& bh)
		{
			if(meaning!=bh.meaning)				return false;
			if((height<0)^(bh.height<0))		return false;
			if((direction<0)^(bh.direction<0))	return false;
			if((windSpeed<0)^(bh.windSpeed<0))	return false;
			if(dataType!=bh.dataType)			return false;
			
			return true;
		}		

		bool IsSameLayerExceptHeight(BlockHead& bh)
		{
			if(meaning!=bh.meaning)				return false;
			if((direction<0)^(bh.direction<0))	return false;
			if((windSpeed<0)^(bh.windSpeed<0))	return false;
			if(dataType!=bh.dataType)			return false;
//			if((height<0)^(bh.height<0))		return false;
			
			return true;
		}		

	};
	
	class BlockBase
	{
		// need to abstract in order to allow different block data types within one WRB
		public:
		BlockHead header;

		bool IsSameLayer(BlockBase* pB)				{return header.IsSameLayer(pB->header);}
		bool IsSameLayerExceptHeight(BlockBase* pB)	{return header.IsSameLayerExceptHeight(pB->header);}
	};
		
	template <class T> class Block : public BlockBase
	{
		public:
		Grid2d<T> data;		
	};
		
	// data
	Header header;
	Array<BlockBase*> blocks;	
	
	int GetHeights(Vector<float>& heights)
	{
		heights.SetCount(0);
		
		for(int i=0;i<blocks.GetCount();i++)
		{
			float height = blocks[i]->header.height;
			
			bool b = false;
			for(int j=0;j<heights.GetCount();j++)
			{
				if(fabs(height-heights[j])<0.00001)
				{
					b = true;
					break;
				}
			}
			
			if(!b && height>0.00001)
				heights.Add(height);
		}
		
		return heights.GetCount();	
	}
	
	bool Uninitialised()		{return header.resX<=0.0 || header.resX<=0.0;}
	bool Initialised()			{return !Uninitialised();}
	
	int GetDirections(Vector<short>& directions)
	{
		directions.SetCount(0);
		
		for(int i=0;i<blocks.GetCount();i++)
		{
			short direction = blocks[i]->header.direction;
			
			bool b = false;
			for(int j=0;j<directions.GetCount();j++)
			{
				if(direction==directions[j])
				{
					b = true;
					break;
				}
			}
			
			if(!b)
				directions.Add(direction);
		}
		
		return directions.GetCount();	
	}

	int GetWindSpeeds(Vector<float>& windSpeeds)
	{
		windSpeeds.SetCount(0);
		
		for(int i=0;i<blocks.GetCount();i++)
		{
			float windSpeed = blocks[i]->header.windSpeed;
			
			bool b = false;
			for(int j=0;j<windSpeeds.GetCount();j++)
			{
				if(fabs(windSpeed-windSpeeds[j])<0.00001)
				{
					b = true;
					break;
				}
			}
			
			if(!b)
				windSpeeds.Add(windSpeed);
		}
		
		return windSpeeds.GetCount();	
	}
/*	
	bool Contains(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		unsigned char dataType = 0;
		
		return Contains(meaning,height,direction,windSpeed,dataType);
	}
*/
	bool Contains(short meaning, float height, 
					short direction, float windSpeed, unsigned char& dataType,int id=0)
	{
		dataType = 0;
		
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				dataType = blocks[i]->header.dataType;
				return true;
			}
		}
		
		return false;
	}


	double GetValue(int nBlock,int x,int y)
	{
		switch(blocks[nBlock]->header.dataType)
		{
			case 1:
			{
				return ((Block<double>*)blocks[nBlock])->data[x][y];
			}
			case 2:
			{
				return ((Block<unsigned char>*)blocks[nBlock])->data[x][y];
			}
			case 3:
			{
				return ((Block<short>*)blocks[nBlock])->data[x][y];
			}
			case 4:
			{
				return ((Block<int32>*)blocks[nBlock])->data[x][y];
			}
			case 0:
			default:
			{
				return ((Block<float>*)blocks[nBlock])->data[x][y];
			}
		}
		return 0.0;
	}	
	
	// overwrites values as original data type
	void SetValue(int nBlock,int x,int y,double val)
	{
		switch(blocks[nBlock]->header.dataType)
		{
			case 1:
			{
				((Block<double>*)blocks[nBlock])->data[x][y] = val;
			}
			case 2:
			{
				((Block<byte>*)blocks[nBlock])->data[x][y] = byte(val);
			}
			case 3:
			{
				((Block<short>*)blocks[nBlock])->data[x][y] = short(val);
			}
			case 4:
			{
				((Block<int32>*)blocks[nBlock])->data[x][y] = int32(val);
			}
			case 0:
			default:
			{
				((Block<float>*)blocks[nBlock])->data[x][y] = float(val);
			}
		}
	}

	String GetMeaning(int i,bool bIgD=false,bool bIgZ=false,bool bIgV=false)
	{
		String s;
		
		switch(blocks[i]->header.meaning)
		{
			case BM_ELEVATION:		
				s = "Elevation";
				break;
			case BM_MEANWINDSPEED:
				s = "Mean Wind Speed";	
				break;
			case BM_A:
				s = "Weibull A";				
				break;
			case BM_K:
				s = "Weibull K";				
				break;
			case BM_POWER:
				s = "Power";				
				break;
			case BM_TI:
				s = "Turbulence Intensity";					
				break;
			case BM_INFLOWANGLE:			
				s = "Inflow Angle";
				break;
			case BM_P:
				s = "Probability";					
				break;
			case BM_DIRECTION:
				s = "Direction";			
				break;
			case BM_Z0:
				s = "Roughness Length";					
				break;
			case BM_AIRDENSITY:
				s = "Air Density";
				break;
			case BM_VERTICALVELOCITY:
				s = "Vertical Velocity";
				break;
			case BM_WINDSHEAREXPONENT:
				s = "Wind Shear Exponent";
				break;
			default:
				s = "Unknown";		
		}
		
		if(blocks[i]->header.height>0 && !bIgZ)
		{
			s = Format("%s (height = %.2f)",s,blocks[i]->header.height);
		}

		if(blocks[i]->header.direction>=0 && !bIgD)
		{
			s = Format("%s (direction = %d)",s,blocks[i]->header.direction);
		}
		
		if(blocks[i]->header.windSpeed>=0 && !bIgV)
		{
			s = Format("%s (wind speed = %.2f)",s,blocks[i]->header.windSpeed);
		}
		
		return s;
	}

	bool BlockToGrid2d(Grid2d<double>& grid,int nBlock)
	{
		Size size = header.GetGridSize();
		for(int x=0;x<size.cx;x++)
		{
			for(int y=0;y<size.cy;y++)
			{
				grid[x][y] = GetValue(nBlock,x,y);
			}
		}
		return true;
	}
	
	// use this function with extreme care - its here for convenience but requires external range checking
	// returns the index of the added block
	template <class T> int AddBlockFromPtrPtr(T** ppT)
	{
		Size size = header.GetGridSize();
		
		if(size.cx<=0 || size.cy<=0)
			return -1;
		
		Block<T>* pB = new Block<T>;
		if(!pB->data.Dimension(size.cx,size.cy))
		{
			delete pB;
			return -1;
		}
		
		for(int x=0;x<size.cx;x++)
		{
			for(int y=0;y<size.cy;y++)
			{
				pB->data[x][y] = ppT[x][y];
			}
		}
		
		pB->header.bFirst=true;
		pB->header.dataType = GetDataType(pB->data);
		blocks.Add(pB);
		header.numBlocks = blocks.GetCount();

		return blocks.GetCount()-1;
	}
	
	bool Add2dArrayAsBlock(d2Array& grid,WRB::BLOCK_MEANING meaning=BM_UNKNOWN,short direction=-1,
						float height=0.0,float windSpeed=0.0,double probability=1.0,int id=0)
	{
		Size size = header.GetGridSize();
		
		if(size.cx<=0 || size.cy<=0)
			return false;
		
		if(grid.GetCount()!=size.cx)
			return false;
		
		if(grid[0].GetCount()!=size.cy)
			return false;
		
		Block<double>* pB = new Block<double>;
		if(!pB->data.Dimension(size.cx,size.cy,0))
		{
			delete pB;
			return false;
		}
		
		for(int x=0;x<size.cx;x++)
		{
			for(int y=0;y<size.cy;y++)
			{
				pB->data[x][y] = grid[x][y];
			}
		}
		
		pB->header.bFirst=true;
		pB->header.dataType = GetDataType(pB->data);
		pB->header.direction = direction;
		pB->header.meaning = meaning;
		pB->header.height = height;
		pB->header.windSpeed = windSpeed;
		pB->header.probability = probability;
		pB->header.id = id;
		blocks.Add(pB);
		header.numBlocks = blocks.GetCount();
		return true;	
	}

	template <class T> bool AddGrid2dAsBlock(Grid2d<T>& grid,WRB::BLOCK_MEANING meaning=BM_UNKNOWN,short direction=-1,
												float height=0.0,float windSpeed=0.0,double probability=1.0,int id=0)
	{
		Size size = header.GetGridSize();
		
		if(size.cx<=0 || size.cy<=0)
			return false;
		
		if(grid.GetNumX()!=size.cx)
			return false;
		
		if(grid.GetNumY()!=size.cy)
			return false;
		
		Block<T>* pB = new Block<T>;
		if(!pB->data.Dimension(size.cx,size.cy))
		{
			delete pB;
			return false;
		}
		pB->data = grid;
		pB->header.bFirst=true;
		pB->header.dataType = GetDataType(pB->data);
		pB->header.direction = direction;
		pB->header.meaning = meaning;
		pB->header.height = height;
		pB->header.windSpeed = windSpeed;
		pB->header.probability = probability;
		pB->header.id = id;
		blocks.Add(pB);
		header.numBlocks = blocks.GetCount();
		return true;	
	}
	
	int GetDataType(Grid2d<float>& data)			{return 0;}
	int GetDataType(Grid2d<double>& data)			{return 1;}
	int GetDataType(Grid2d<unsigned char>& data)	{return 2;}
	int GetDataType(Grid2d<short>& data)			{return 3;}
	int GetDataType(Grid2d<int32>& data)			{return 4;}	
	
	Grid2d<double>* GetGridOfDoubles(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				if(blocks[i]->header.dataType!=1)
				{
//					PromptOK("You asked for the wrong data type!");
					DUMP("You asked for the wrong data type!");
					return NULL;
				}
				blocks[i]->header.bFirst = false;
				return &((Block<double>*)blocks[i])->data;	
			}
		}
		
		return NULL;
	}

	Grid2d<float>* GetGridOfFloats(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				if(blocks[i]->header.dataType!=0)
				{
//					PromptOK("You asked for the wrong data type!");
					DUMP("You asked for the wrong data type!");
					return NULL;
				}
				blocks[i]->header.bFirst = false;
				return &((Block<float>*)blocks[i])->data;	
			}
		}
		
		return NULL;
	}

	Grid2d<short>* GetGridOfShorts(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				if(blocks[i]->header.dataType!=3)
				{
//					PromptOK("You asked for the wrong data type!");
					DUMP("You asked for the wrong data type!");
					return NULL;
				}
				blocks[i]->header.bFirst = false;
				return &((Block<short>*)blocks[i])->data;	
			}
		}
		
		return NULL;
	}

	Grid2d<unsigned char>* GetGridOfBytes(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				if(blocks[i]->header.dataType!=2)
				{
//					PromptOK("You asked for the wrong data type!");
					DUMP("You asked for the wrong data type!");
					return NULL;
				}
				blocks[i]->header.bFirst = false;
				return &((Block<unsigned char>*)blocks[i])->data;	
			}
		}
		
		return NULL;
	}

	Grid2d<int32>* GetGridOfInts(short meaning, float height, 
					short direction, float windSpeed, int id=0)
	{
		for(int i=0;i<blocks.GetCount();i++)
		{
			if(meaning==blocks[i]->header.meaning
			&& (height==blocks[i]->header.height || height<0)
			&& (direction==blocks[i]->header.direction || direction<0)
			&& (fabs(windSpeed-blocks[i]->header.windSpeed)<0.000001 || windSpeed<0)
			)
			{
				if(blocks[i]->header.dataType!=4)
				{
//					PromptOK("You asked for the wrong data type!");
					DUMP("You asked for the wrong data type!");
					return NULL;
				}
				blocks[i]->header.bFirst = false;
				return &((Block<int32>*)blocks[i])->data;	
			}
		}
		
		return NULL;
	}

	bool Read(FileIn& file)
	{
		char cBuf[1000];
		file.Get(cBuf,100);
		// unfortunately we have to do this ourselves to avoid possible problems with 
		// byte alignment on different compilers
		header.type = *(unsigned short*)cBuf;
		header.version = *(unsigned short*)(cBuf+2);
		header.horzUnits = cBuf[4];
		header.vertUnits = cBuf[5];
		strncpy(header.datumProjection,cBuf+6,30);
		header.directions = *(unsigned short*)(cBuf+36);
		header.heights = *(unsigned short*)(cBuf+38);
		header.windSpeeds = *(unsigned short*)(cBuf+40);
		header.minX = *(double*)(cBuf+42);
		header.maxX = *(double*)(cBuf+50);
		header.minY = *(double*)(cBuf+58);
		header.maxY = *(double*)(cBuf+66);
		header.resX = *(double*)(cBuf+74);
		header.resY = *(double*)(cBuf+82);
		header.numBlocks = *(unsigned short*)(cBuf+90);
		
		Size size = header.GetGridSize();
		
		blocks.SetCount(0);
		Array<BlockHead> headers;
		headers.SetCount(header.numBlocks);
		for(int i=0;i<header.numBlocks;i++)
		{
			file.Get(cBuf,64);
			
			headers[i].meaning = *(unsigned short*)cBuf;
			headers[i].height = *(float*)(cBuf+2);
			headers[i].direction = *(short*)(cBuf+6);
			headers[i].windSpeed = *(float*)(cBuf+8);
			headers[i].probability = *(double*)(cBuf+12);
			headers[i].id = *(int32*)(cBuf+20);
			headers[i].offset = *(int64*)(cBuf+24);
			headers[i].dataType = *(unsigned char*)(cBuf+32);					
		}
		
		bool b;
		
		for(int i=0;i<headers.GetCount();i++)
		{
			file.Seek(headers[i].offset);
			
			DUMP(headers[i].offset);
			
			switch(headers[i].dataType)
			{
				case 1:
				{
					BlockBase* pB = new Block<double>;
					b = ((Block<double>*)pB)->data.Dimension(size.cx,size.cy,0.0);
					pB->header = headers[i];
					blocks.Add(pB);
					break;
				}
				case 2:
				{
					BlockBase* pB = new Block<unsigned char>;
					b = ((Block<unsigned char>*)pB)->data.Dimension(size.cx,size.cy,0);
					pB->header = headers[i];
					blocks.Add(pB);
					break;
				}
				case 3:
				{
					BlockBase* pB = new Block<short>;
					b = ((Block<short>*)pB)->data.Dimension(size.cx,size.cy,0);
					pB->header = headers[i];
					blocks.Add(pB);
					break;
				}
				case 4:
				{
					BlockBase* pB = new Block<int32>;
					b = ((Block<int32>*)pB)->data.Dimension(size.cx,size.cy,0);
					pB->header = headers[i];
					blocks.Add(pB);
					break;
				}
				case 0:
				default:
				{
					BlockBase* pB = new Block<float>;
					b = ((Block<float>*)pB)->data.Dimension(size.cx,size.cy,0.0);
					pB->header = headers[i];
					blocks.Add(pB);
					break;
				}
			}		
			
			if(!b)
			{
				DUMP("failed to allocate memory in WRB::Read()");
				blocks.SetCount(0); // free all memory allocated so far
				return false;	
			}
			
			int n;
			switch(blocks[i]->header.dataType)
			{
				case 1:
					n = 8;
					break;
				case 2:
					n = 1;
					break;
				case 3:
					n = 2;
					break;
				case 0:
				case 4:
				default:
					n = 4;
			}

			int column = size.cx*n;
			
			Buffer<char> buf;
			buf.Alloc(column);

			for(int y=0;y<size.cy;y++)
			{
				int read = file.Get(buf,column);
				
				if(read!=column)
				{
					PromptOK("the file appears to be truncated - aborting...");
					DUMP("read error");
					blocks.SetCount(0); // free all memory allocated so far
					return false;
				}
				
				for(int x=0;x<size.cx;x++)
				{
					switch(blocks[i]->header.dataType)
					{
						case 1:
						{
							double* pD = (double*)(buf+x*n);	
							((Block<double>*)blocks[i])->data.Set(x,y,*pD);
							break;
						}
						case 2:
						{
							unsigned char* pB = (unsigned char*)(buf+x*n);
							((Block<unsigned char>*)blocks[i])->data.Set(x,y,*pB);
							break;
						}
						case 3:
						{
							short* pS = (short*)(buf+x*n);
							((Block<short>*)blocks[i])->data.Set(x,y,*pS);
							break;
						}
						case 4:
						{
							int32* pL = (int32*)(buf+x*n);
							((Block<int32>*)blocks[i])->data.Set(x,y,*pL);
							break;
						}
						case 0:
						default:
						{
							float* pF = (float*)(buf+x*n);
							((Block<float>*)blocks[i])->data.Set(x,y,*pF);
							break;
						}
					}
				}
			}
		}
		
		return true;
	}
	
	bool Write(FileOut& file)
	{
		int c=0;
		
//		file.Put(&header,100);
		file.Put((void*)&header.type,2);
		file.Put((void*)&header.version,2);
		file.Put((void*)&header.horzUnits,1);
		file.Put((void*)&header.vertUnits,1);
		file.Put((void*)&header.datumProjection,30);
		file.Put((void*)&header.directions,2);
		file.Put((void*)&header.heights,2);
		file.Put((void*)&header.windSpeeds,2);
		file.Put((void*)&header.minX,8);
		file.Put((void*)&header.maxX,8);
		file.Put((void*)&header.minY,8);
		file.Put((void*)&header.maxY,8);
		file.Put((void*)&header.resX,8);
		file.Put((void*)&header.resY,8);
		file.Put((void*)&header.numBlocks,2);
		file.Put(0,8); // filler
		
		c+=100;
		
		Size size = header.GetGridSize();

		size = header.GetGridSize();
		c += header.numBlocks * 64;
		
		for(int i=0;i<header.numBlocks;i++)
		{
			blocks[i]->header.offset = c;
			int n;
			switch(blocks[i]->header.dataType)
			{
				case 1:
					n = 8;
					c += ((Block<double>*)blocks[i])->data.GetRows()*((Block<double>*)blocks[i])->data.GetCols()*n;
					break;
				case 2:
					n = 1;
					c += ((Block<unsigned char>*)blocks[i])->data.GetRows()*((Block<unsigned char>*)blocks[i])->data.GetCols()*n;
					break;
				case 3:
					n = 2;
					c += ((Block<short>*)blocks[i])->data.GetRows()*((Block<short>*)blocks[i])->data.GetCols()*n;
					break;
				case 0:
				case 4:
				default:
					n = 4;
					c += ((Block<float>*)blocks[i])->data.GetRows()*((Block<float>*)blocks[i])->data.GetCols()*n;
			}
			
		}

		for(int i=0;i<header.numBlocks;i++)
		{
//			file.Put(&blocks[i]->header,64);	
			
			file.Put((void*)&blocks[i]->header.meaning,2);
			file.Put((void*)&blocks[i]->header.height,4);
			file.Put((void*)&blocks[i]->header.direction,2);
			file.Put((void*)&blocks[i]->header.windSpeed,4);
			file.Put((void*)&blocks[i]->header.probability,8);
			file.Put((void*)&blocks[i]->header.id,4);
			file.Put((void*)&blocks[i]->header.offset,8);
			file.Put((void*)&blocks[i]->header.dataType,1);		
			file.Put(0,31);//filler			
		}
		
		for(int i=0;i<header.numBlocks;i++)
		{
			int n;
			switch(blocks[i]->header.dataType)
			{
				case 1:
					n = 8;
					break;
				case 2:
					n = 1;
					break;
				case 3:
					n = 2;
					break;
				case 0:
				case 4:
				default:
					n = 4;
			}
			
			n *= size.cx;
			Buffer<char> buf;
			buf.Alloc(n);
			
			for(int k=0;k<size.cy;k++)
			{
				for(int j=0;j<size.cx;j++)
				{
					switch(blocks[i]->header.dataType)
					{
						case 1: 
						{
							double d = ((Block<double>*)blocks[i])->data.Get(j,k);
							memcpy((buf+j*8),&d,8);
							break;
						}
						case 2: 
						{
							unsigned char d = ((Block<unsigned char>*)blocks[i])->data.Get(j,k);
							memcpy((buf+j),&d,1);
							break;
						}
						case 3: 
						{
							short d = ((Block<short>*)blocks[i])->data.Get(j,k);
							memcpy((buf+j*2),&d,2);
							break;
						}
						case 4: 
						{
							int32 d = ((Block<int32>*)blocks[i])->data.Get(j,k);
							memcpy((buf+j*4),&d,4);
							break;
						}
						case 0:
						default: 
						{
							float d = ((Block<float>*)blocks[i])->data.Get(j,k);
							memcpy((buf+j*4),&d,4);
							break;
						}
					}
				}
				file.Put(buf,n);	
			}
		}
		return true;		
	}
		
	bool InterpretAsMeteodyn()
	{
		




		return true;
	}
	
	bool InterpretAsVentos()
	{
		
		
		
		return true;	
	}
	
	
};


#endif //__UTILS_H__
