#ifndef _GraphDraw_GrapDraw_h
#define _GraphDraw_GrapDraw_h

#include <iostream>

#include <CtrlLib/CtrlLib.h>

#include <Core/Core.h>

#include <ScatterDraw/ScatterDraw.h>

using namespace Upp;

#ifndef TRACE_INFO
#define TRACE_INFO(TXT) //{ std::ostringstream str; str <<  "\n" << TXT; LOG(str.str().c_str()); }
#endif

#include "GraphFunctions.h"
#include "SeriesConfig.h"
#include "SeriesGroup.h"
#include "CoordinateConverter.h"
#include "GridStepIterator.h"






namespace GraphDraw_ns
{
	typedef enum {
		LEFT_OF_GRAPH = 0,
		RIGHT_OF_GRAPH,
		TOP_OF_GRAPH,
		BOTTOM_OF_GRAPH,
		OVER_GRAPH
	} ElementPosition;

	class TickMark {

		public:
			int _tickLength;
			TickMark() : _tickLength(2) { UpdateTick(); }
			virtual ~TickMark() {}

			virtual void Paint(Draw &w, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {};
			inline void Paint(Draw &p, ElementPosition axisPos, const int scale, const Point& cp, const Color& markColor) const { Paint(p, axisPos, scale, cp.x, cp.y, markColor); }

			virtual void Paint(Painter &p, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {};
			inline void Paint(Painter &p, ElementPosition axisPos, const int scale, const Point& cp, const Color& markColor) const { Paint(p, axisPos, scale, cp.x, cp.y, markColor); }

			inline int GetTickLength()       { return _tickLength; }
			inline TickMark* SetTickLength(int v) { _tickLength = v; UpdateTick(); return this; }

			virtual void UpdateTick() {}; // called when tick drawing needs to be recalculated
	};

	class RoundTickMark  : public TickMark {
		public:
			RoundTickMark() { SetTickLength(5);	}
			virtual ~RoundTickMark() {}

			virtual void Paint(Draw &w, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {
				int radius = fround(scale*_tickLength*2);
				int radius2 = radius/2;
				w.DrawEllipse(x - radius2, y - radius2, radius, radius, markColor, 1, markColor);
			}

			virtual void Paint(Painter &p, ElementPosition axisPos, const int& scale, int x, int y, const Color& markColor) const {
			}
	};

	class TriangleTickMark  : public TickMark {
		public:
			TriangleTickMark() { SetTickLength(5);	}
			virtual ~TriangleTickMark() {}

			virtual void Paint(Draw &dw, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {
				Upp::Vector<Point> p;
				p << Point(x, y);
				if (axisPos==LEFT_OF_GRAPH)	{
					p << Point(x-_tickLength, y-2) << Point(x-_tickLength, y+2);
				} else if (axisPos==RIGHT_OF_GRAPH)	{
					p << Point(x+_tickLength, y-2) << Point(x+_tickLength, y+2);
				} else if (axisPos==BOTTOM_OF_GRAPH)	{
					p << Point(x-2, y+_tickLength) << Point(x+2, y+_tickLength);
				} else {
					p << Point(x-2, y-_tickLength) << Point(x+2, y-_tickLength);
				}
				p << Point(x, y);
				dw.DrawPolygon( p, markColor, scale/2, markColor);
			}

			virtual void Paint(Painter &p, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {
			}
	};

	class LineTickMark  : public TickMark {
		public:
			LineTickMark() { SetTickLength(6);	}
			virtual ~LineTickMark() {}

			virtual void Paint(Draw &dw, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {
				if ((axisPos==LEFT_OF_GRAPH) || ( axisPos==RIGHT_OF_GRAPH ))	{
					dw.DrawLine(x-_tickLength/2, y, x+_tickLength/2, y, 2, markColor);
				} else {
					dw.DrawLine(x, y-_tickLength/2, x, y+_tickLength/2, 2, markColor);
				}
			}

			virtual void Paint(Painter &p, ElementPosition axisPos, const int scale, int x, int y, const Color& markColor) const {
			}
	};


	// ============================
	//    GraphElementFrame   CLASS
	// ============================
	class GraphElementFrame {
		protected:
			Rect            _frame;  // Frame on which element is painted (absolute position in complete draw area)
			int             _width;  // width of GraphElement (in screen size)
			ElementPosition _pos;    // position to plot area

		public:
			Callback1<GraphElementFrame*> WhenLeftDown;
			Callback1<GraphElementFrame*> WhenLeftDouble;

			GraphElementFrame() : _width(5), _pos(LEFT_OF_GRAPH) {}
			GraphElementFrame(GraphElementFrame* p) : _width(p->_width), _pos(LEFT_OF_GRAPH), WhenLeftDown(p->WhenLeftDown), WhenLeftDouble(p->WhenLeftDouble)	{}

			virtual ~GraphElementFrame() {}

			inline bool Contains(Point p) const { return _frame.Contains(p); }
			inline const Rect& GetFrame() const { return _frame; }
			inline void SetFrame(Rect v) { _frame = v; }
			inline int GetElementWidth() { return _width; }
			inline ElementPosition GetElementPos() { return _pos; }

         // Paint element somewhere inside the graph area (legend, ...)
			// Offset and clipping are set with the '_frame' settings
			virtual void PaintOverGraph(Draw& dw, int scale) { };

			// Paint the element in his own area
			// There is no clipping ==> drawing can overlap plot or other elements
			virtual void PaintElement(Draw& dw, int scale) = 0;

			// Paint additionnal element stuff on PLOT AREA : grids, square zones,  anything you wan't
			// Painting zone is clipped so nothing can be drawn outside
			virtual void PaintOnPlot(Draw& dw, int otherWidth, int scale) {}
			virtual void Update() {}; // called when coordinates need update

			virtual GraphElementFrame* Clone() = 0; 

	};

	template<class DERIVED>
	class CRTPGraphElementFrame : public GraphElementFrame {
		public:
			CRTPGraphElementFrame() {}
			CRTPGraphElementFrame(CRTPGraphElementFrame& p) : GraphElementFrame(p) {}
			virtual ~CRTPGraphElementFrame() {}

			inline DERIVED&  SetElementWidth(int v) { _width = v; return *static_cast<DERIVED*>(this); }
			inline DERIVED&  SetElementPos(ElementPosition v) { _pos = v; return *static_cast<DERIVED*>(this); }
	};


	template<class COORDCONVERTER>
	class BlankAreaElement : public CRTPGraphElementFrame< BlankAreaElement<COORDCONVERTER> >
	{
		typedef BlankAreaElement<COORDCONVERTER> CLASSNAME;
		public:
			BlankAreaElement() {}
			BlankAreaElement( BlankAreaElement& p) : CRTPGraphElementFrame< BlankAreaElement<COORDCONVERTER> >(p) {}
			virtual ~BlankAreaElement() {}
			virtual void PaintElement(Draw& dw, int scale) { /* do noting */}
			virtual CLASSNAME* Clone() { return new CLASSNAME(*this); } 
	};


	// ============================
	//    LabelElement   CLASS
	// ============================
	template<class COORDCONVERTER>
	class LabelElement : public CRTPGraphElementFrame< LabelElement<COORDCONVERTER> >
	{
		private:
		Upp::String _label;
		Font        _font;
		Color       _color;
		Color       _bckGndcolor;
		typedef LabelElement<COORDCONVERTER> LABELCLASS;
		typedef CRTPGraphElementFrame< LabelElement<COORDCONVERTER> > _B;

		public:
		LabelElement() : _color( Blue() ), _bckGndcolor(Null) {}
		LabelElement(LabelElement& p) : _B(p), _color( p._color ), _bckGndcolor(p._bckGndcolor) {}
		virtual ~LabelElement() {}

		virtual LABELCLASS* Clone() { return new LABELCLASS(*this); }; 

		template<class T>	inline LABELCLASS& SetLabel(T& v) { _label = v; return *this; }
		template<class T>	inline LABELCLASS& SetFont(T& v) { _font = v; return *this; }
		template<class T>	inline LABELCLASS& SetTextColor(T v) { _color = v; return *this; }
		template<class T>	inline LABELCLASS& SetBckGndColor(T v) { _bckGndcolor = v; return *this; }

		inline const Upp::String& GetLabel() const { return _label; }

		virtual void PaintElement(Draw& dw, int scale)
		{
			if ( !_bckGndcolor.IsNullInstance() )
				dw.DrawRect( 0, 0, _B::GetFrame().GetWidth(), _B::GetFrame().GetHeight(), _bckGndcolor);

			Size sz = GetTextSize(_label, _font);
			switch(_B::GetElementPos()){
				case LEFT_OF_GRAPH:
					dw.DrawText( _B::GetElementWidth()/2-sz.cy/2, _B::GetFrame().GetHeight()/2+sz.cx/2 , 900, _label, _font, _color);
					break;
				case BOTTOM_OF_GRAPH:
					dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()/2-sz.cy/2 , _label, _font, _color);
					break;
				case TOP_OF_GRAPH:
					dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()/2-sz.cy/2 , _label, _font, _color);
					break;
				case RIGHT_OF_GRAPH:
					dw.DrawText( _B::GetElementWidth()/2+sz.cy/2, _B::GetFrame().GetHeight()/2-sz.cx/2 , 2700, _label, _font, _color);
					break;
			}
		}
	};

	// ============================
	//    LabelElement   CLASS
	// ============================
	template<class COORDCONVERTER>
	class RichLabelElement : public CRTPGraphElementFrame< RichLabelElement<COORDCONVERTER> >
	{
		private:
		RichText    _label;
		Color       _bckGndcolor;
		typedef RichLabelElement<COORDCONVERTER> CLASSANME;
		typedef CRTPGraphElementFrame< RichLabelElement<COORDCONVERTER> > _B;

		public:
		RichLabelElement() : _bckGndcolor(Null) {}
		RichLabelElement(RichLabelElement& p) : _B(p), _bckGndcolor(p._bckGndcolor) {}
		virtual ~RichLabelElement() {}

		virtual CLASSANME* Clone() { return new CLASSANME(*this); }; 
		
		template<class T>
		inline void SetLabel(T& v) { _label = ParseQTF(v); }
		inline const Upp::String& GetLabel() const { return _label; }

		virtual void PaintElement(Draw& dw, int scale)
		{
			if ( !_bckGndcolor.IsNullInstance() )
				dw.DrawRect( 0, 0, _B::GetFrame().GetWidth(), _B::GetFrame().GetHeight(), _bckGndcolor);

			//Size sz(0,0);
			switch(_B::GetElementPos()){
				case LEFT_OF_GRAPH:

					//dw.DrawText( _B::GetElementWidth()/2-sz.cy/2, _B::GetFrame().GetHeight()/2+sz.cx/2 , 900, _label, _font, _color);
					break;
				case BOTTOM_OF_GRAPH:
					//dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()/2-sz.cy/2 , _label, _font, _color);
					break;
				case TOP_OF_GRAPH:
					_label.Paint(dw, 0, 0, 1000);
					//dw.DrawText( _B::GetFrame().GetWidth()/2-sz.cx/2, _B::GetElementWidth()/2-sz.cy/2 , _label, _font, _color);
					break;
				case RIGHT_OF_GRAPH:
					//dw.DrawText( _B::GetElementWidth()/2+sz.cy/2, _B::GetFrame().GetHeight()/2-sz.cx/2 , 2700, _label, _font, _color);
					break;
			}
		}
	};



	// ============================
	//    RichLegendElement   CLASS
	// ============================
	template<class COORDCONVERTER>
	class RichLegendElement : public CRTPGraphElementFrame< RichLegendElement<COORDCONVERTER> >
	{
		private:
		RichText    _legend;
		Color       _bckGndcolor;
		typedef RichLegendElement<COORDCONVERTER> CLASSNAME;
		typedef CRTPGraphElementFrame< RichLegendElement<COORDCONVERTER> > _B;

		public:
		RichLegendElement() : _bckGndcolor(Null) {}
		RichLegendElement(RichLegendElement& p) : _B(p), _bckGndcolor(p._bckGndcolor) {}
		virtual ~RichLegendElement() {}

		virtual CLASSNAME* Clone() { return new CLASSNAME(*this); }; 

		template<class T>
		inline void SetLegend(T& v) { _legend = ParseQTF(v); }
		inline const Upp::String& GetLegend() const { return AsQTF(_legend); }

		inline void SetBackGndColor(Color v) { _bckGndcolor = v; }
		inline const Color& GetBackGndColor() const { return _bckGndcolor; }

		virtual void PaintElement(Draw& dw, int scale) {}
		virtual void PaintOverGraph(Draw& dw, int scale)
		{
			if ( !_bckGndcolor.IsNullInstance() )
				dw.DrawRect( 0, 0, _B::GetFrame().GetWidth(), _B::GetFrame().GetHeight(), _bckGndcolor);
			_legend.Paint(dw, 0, 0, 1000);
			String text = "This is the legend !!";
			dw.DrawText( 0,0 , text, StdFont(), Black());
		}
	};

	// ============================
	//    GridAxisDraw   CLASS
	// ============================
	template<class COORDCONVERTER, class GRIDSTEPMANAGER=GridStepManager<COORDCONVERTER> >
	class GridAxisDraw : public CRTPGraphElementFrame< GridAxisDraw<COORDCONVERTER> >
	{
		public:
		typedef GridAxisDraw<COORDCONVERTER, GRIDSTEPMANAGER>  CLASSNAME;
		typedef COORDCONVERTER TypeCoordConverter;
		typedef GRIDSTEPMANAGER TypeGridStepManager;
		typedef CRTPGraphElementFrame< GridAxisDraw<COORDCONVERTER> > _B;

//		protected:
		int       _axisWidth;
		Color     _axisColor;
		Font      _axisTextFont;
		Color     _axisTextColor;
		Color     _axisTickColor;
		Color     _gridColor;
		One<TickMark> _majorTickMark;
		One<TickMark> _minorTickMark;

		One< TypeGridStepManager > _gridStepManager;
		COORDCONVERTER& _coordConverter;

		/*
		/home/didier/MyApps/GraphDraw/GraphDraw.h:760: erreur: no matching function for call to

::GridAxisDraw(   GraphDraw_ns::GridAxisDraw<GraphDraw_ns::GenericCoordinateConverter<int, double>, GraphDraw_ns::GridStepManager<GraphDraw_ns::GenericCoordinateConverter<int, double>, 100> >&   )’
::GridAxisDraw(   GraphDraw_ns::GridAxisDraw<GraphDraw_ns::CoordinateConverter       <int, double>, GraphDraw_ns::GridStepManager<GraphDraw_ns::CoordinateConverter       <int, double>, 100> >
		*/
		public:
		GridAxisDraw(COORDCONVERTER& coordConv)
		: _gridStepManager( new TypeGridStepManager(coordConv) )
		, _axisWidth(2)
		, _axisColor( Blue())
		, _axisTextFont(StdFont())
		, _axisTextColor(Red())
		, _axisTickColor(Red())
		, _gridColor(LtGray())
		, _coordConverter( coordConv )
		, _majorTickMark(new TickMark())
		, _minorTickMark(new TickMark())
		{
		}

		GridAxisDraw(GridAxisDraw& p)
		: _B(p)
		, _gridStepManager( new TypeGridStepManager(p._coordConverter) )
		, _axisWidth(p._axisWidth)
		, _axisColor(p._axisColor)
		, _axisTextFont(p._axisTextFont)
		, _axisTextColor(p._axisTextColor)
		, _axisTickColor(p._axisTickColor)
		, _gridColor(Null)  // This is a slave Axis ==> so no need to draw grid
		, _coordConverter( p._coordConverter )
		, _majorTickMark(new TickMark())
		, _minorTickMark(new TickMark())
		{
		}

		virtual ~GridAxisDraw() {}


		virtual CLASSNAME* Clone() { return new CLASSNAME(*this); } // called when coordinates need update

		virtual void Update() { _gridStepManager->Update(); }

		TypeGridStepManager& GetGridStepManager() { return *_gridStepManager; }
		inline CLASSNAME& setAxisColor(Color v)                         { _axisColor = v; return *this;  }
		inline CLASSNAME& setAxisWidth(int v)                           { _axisWidth = v; return *this;  }
		inline CLASSNAME& setAxisTextFont(Font v)                       { _axisTextFont = v; return *this; }
		inline CLASSNAME& setAxisTextColor(Color v)                     { _axisTextColor = v; return *this; }
		inline CLASSNAME& setAxisTickColor(Color v)                     { _axisTickColor = v; return *this; }
		inline CLASSNAME& setGridColor(Color v)                         { _gridColor = v; return *this; }
		inline CLASSNAME& setMajorTickMark(TickMark* v)                 { _majorTickMark = v; return *this;  }
		inline CLASSNAME& setMinorTickMark(TickMark* v)                 { _minorTickMark = v; return *this;  }

		inline COORDCONVERTER& GetCoordConverter()                 { return _coordConverter;  }

/*		inline Color GetAxisColor(Color v)                        { _axisColor = v; return *this;  }
		inline int GetAxisWidth(int v)                            { _axisWidth = v; return *this;  }
		inline Font GetAxisTextFont(Font v)                       { _axisTextFont = v; return *this; }
		inline Color GetAxisTextColor(Color v)                    { _axisTextColor = v; return *this; }
		inline Color GetAxisTickColor(Color v)                    { _axisTickColor = v; return *this; }
		inline Color GetGridColor(Color v)                        { _gridColor = v; return *this; }
*/
		inline typename TypeGridStepManager::Iterator getMajorBeginIterator(void) { return  _gridStepManager->Begin(); }
		inline typename TypeGridStepManager::Iterator getMajorEndIterator(void)   { return  _gridStepManager->End(); }


		template<class T>
		T& CreateGridStepManager() {
				_gridStepManager =  new T(_coordConverter);
				_gridStepManager.Update();
				return (*_gridStepManager);
		}

		template<int GRAPH_SIDE>
		void PaintTickText(Draw& dw,  typename COORDCONVERTER::TypeGraphCoord v, typename COORDCONVERTER::TypeScreenCoord x, typename COORDCONVERTER::TypeScreenCoord y, Color& color, Font& font) {
			Upp::String text=FormatDouble(v, 3, FD_FIX);
			Size sz = GetTextSize(text, font);
			if (GRAPH_SIDE == LEFT_OF_GRAPH) {
				dw.DrawText( x-sz.cx, y-(sz.cy/2) , text, font, color);
			}
			else if (GRAPH_SIDE == BOTTOM_OF_GRAPH) {
				dw.DrawText( x-(sz.cx/2), y , text, font, color);
			}
			else if (GRAPH_SIDE == RIGHT_OF_GRAPH) {
				dw.DrawText( x, y-(sz.cy/2) , text, font, color);
			}
			else {
				dw.DrawText( x-(sz.cx/2), y-sz.cy , text, font, color);
			}
		}

		virtual void PaintAxisLeft(Draw& dw)
		{
			typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
			typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();

			DrawVArrowEnd(dw, _B::GetElementWidth(), _coordConverter.getScreenMin(), _B::GetElementWidth(), _coordConverter.getScreenMax(), 2, 6, 8, _axisColor);
			while ( iter != endIter)
			{
				if (_majorTickMark.IsEmpty()) {
					dw.DrawLineOp(_B::GetElementWidth()-8, *iter, _B::GetElementWidth(), *iter, 2, _axisTickColor);
					PaintTickText<LEFT_OF_GRAPH>(dw, iter.getValue(), _B::GetElementWidth()-8, *iter, _axisTextColor, _axisTextFont);
				} else {
					_majorTickMark->Paint(dw, LEFT_OF_GRAPH, 1, _B::GetElementWidth(), *iter, _axisTickColor );
					PaintTickText<LEFT_OF_GRAPH>(dw, iter.getValue(), _B::GetElementWidth()-_majorTickMark->GetTickLength(), *iter, _axisTextColor, _axisTextFont);
				}
				++iter;
			}
		}

		virtual void PaintAxisRight(Draw& dw)
		{
			typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
			typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();
			
			//				DrawVArrowEnd(dw, 0, _coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(), 2, 6, 8, _axisColor);
			dw.DrawLineOp(0, _coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(), 2, _axisColor);
			while ( iter != endIter)
			{
				if (_majorTickMark.IsEmpty())
				{
					dw.DrawLineOp(0, *iter, 8, *iter, 2, _axisTickColor);
					PaintTickText<RIGHT_OF_GRAPH>(dw, iter.getValue(), 8, *iter, _axisTextColor, _axisTextFont);
				} else {
					_majorTickMark->Paint(dw, RIGHT_OF_GRAPH, 1, 0, *iter, _axisTickColor );
					PaintTickText<RIGHT_OF_GRAPH>(dw, iter.getValue(), _majorTickMark->GetTickLength(), *iter, _axisTextColor, _axisTextFont);
				}
				++iter;
			}
		}

		virtual void PaintAxisBottom(Draw& dw)
		{
			typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
			typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();
			//DrawHArrowEnd(dw, _coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(), 0, 2, 6, 8, _axisColor );
			dw.DrawLine(_coordConverter.getScreenMin(), 0, _coordConverter.getScreenMax(), 0, 2, _axisColor );
			while ( iter != endIter)
			{
				if (_majorTickMark.IsEmpty()) {
					dw.DrawLineOp(*iter, 0, *iter, 4, 2, _axisTickColor);
					PaintTickText<BOTTOM_OF_GRAPH>(dw, iter.getValue(), *iter, 4, _axisTextColor, _axisTextFont);
				} else {
					_majorTickMark->Paint(dw, BOTTOM_OF_GRAPH, 1, *iter, 0, _axisTickColor );
					PaintTickText<BOTTOM_OF_GRAPH>(dw, iter.getValue(), *iter, _majorTickMark->GetTickLength(), _axisTextColor, _axisTextFont);
				}
				++iter;
			}
		}

		virtual void PaintAxisTop(Draw& dw)
		{
			typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
			typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();
			//DrawHArrowEnd(dw, _coordConverter.getScreenMin(), _B::GetElementWidth(), _coordConverter.getScreenMax(), _B::GetElementWidth(), 2, 6, 8, _axisColor );
			dw.DrawLine(_coordConverter.getScreenMin(), _B::GetElementWidth(), _coordConverter.getScreenMax(), _B::GetElementWidth(), 2, _axisColor );
			while ( iter != endIter)
			{
				if (_majorTickMark.IsEmpty()) {
					dw.DrawLineOp(*iter, _B::GetElementWidth(), *iter, _B::GetElementWidth()-4, 2, _axisTickColor);
					PaintTickText<TOP_OF_GRAPH>(dw, iter.getValue(), *iter, _B::GetElementWidth()-4, _axisTextColor, _axisTextFont);
				} else {
					_majorTickMark->Paint(dw, TOP_OF_GRAPH, 1, *iter, _B::GetElementWidth(), _axisTickColor );
					PaintTickText<TOP_OF_GRAPH>(dw, iter.getValue(), *iter, _B::GetElementWidth()-_majorTickMark->GetTickLength(), _axisTextColor, _axisTextFont);
				}
				++iter;
			}
		}

		virtual void PaintVGrid(Draw& dw, int xRange)
		{
			if ( !_gridColor.IsNullInstance())
			{
				typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
				typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();
				while( iter != endIter)
				{
					dw.DrawLineOp( 0, *iter, xRange, *iter, 1, _gridColor );
					++iter;
				}
			}
		}

		virtual void PaintHGrid(Draw& dw, int yRange)
		{
			if ( !_gridColor.IsNullInstance())
			{
				typename TypeGridStepManager::Iterator iter = getMajorBeginIterator();
				typename TypeGridStepManager::Iterator endIter = getMajorEndIterator();
				while( iter != endIter)
				{
					dw.DrawLineOp( *iter, 0, *iter, yRange, 1, _gridColor );
					++iter;
				}
			}
		}

		virtual void PaintOnPlot(Draw& dw, int otherWidth, int scale)
		{
			switch(_B::GetElementPos()){
				case LEFT_OF_GRAPH:
				case RIGHT_OF_GRAPH:
					PaintVGrid(dw, otherWidth);
					break;
				case BOTTOM_OF_GRAPH:
				case TOP_OF_GRAPH:
					PaintHGrid(dw, otherWidth);
					break;
			}
		}

		virtual void PaintElement(Draw& dw, int scale)
		{
			switch(_B::GetElementPos()){
				case LEFT_OF_GRAPH:
					PaintAxisLeft(dw);
					break;
				case BOTTOM_OF_GRAPH:
					PaintAxisBottom(dw);
					break;
				case TOP_OF_GRAPH:
					PaintAxisTop(dw);
					break;
				case RIGHT_OF_GRAPH:
					PaintAxisRight(dw);
					break;
			}
		}
	};


	// ------------------------------------------------------------------------------------------------------------------------------------------------------
	//
	//            G r a p h D r a  w    C l a s s e s
	//
	// ------------------------------------------------------------------------------------------------------------------------------------------------------

	// ============================
	//    EmptyGraphDraw   CLASS
	// ============================
	template<class DATASOURCE, class SERIESPLOT, class MARKPLOT, class COORDCONVERTER>
	class EmptyGraphDraw : public SeriesGroup< SeriesConfig<COORDCONVERTER, SERIESPLOT, MARKPLOT>, COORDCONVERTER, EmptyGraphDraw<DATASOURCE,SERIESPLOT,MARKPLOT,COORDCONVERTER> >
	{
		public:
		typedef EmptyGraphDraw<DATASOURCE, SERIESPLOT, MARKPLOT, COORDCONVERTER> CLASSNAME;

		typedef COORDCONVERTER                                              TypeCoordConverter;
		typedef DATASOURCE                                                  TypeDataSource;
		typedef SERIESPLOT                                                  TypeSeriesPlot;
		typedef GridAxisDraw<COORDCONVERTER>                                TypeGridAxisDraw;
		typedef GridStepManager<COORDCONVERTER>                             TypeGridStepManager;
		typedef SeriesConfig<COORDCONVERTER, SERIESPLOT, MARKPLOT>          TypeSeriesConfig;
		typedef SeriesGroup<TypeSeriesConfig,COORDCONVERTER,EmptyGraphDraw> TypeSeriesGroup;
		typedef TypeSeriesGroup                                             TSG;
		typedef typename COORDCONVERTER::TypeScreenCoord                    TypeScreenCoord;
		typedef typename COORDCONVERTER::TypeGraphCoord                     TypeGraphCoord;

		protected:
		// graph elements draw aound the graph
		Vector< GraphElementFrame* >  _createdElements;
		Vector< GraphElementFrame* >  _leftElements;
		Vector< GraphElementFrame* >  _topElements;
		Vector< GraphElementFrame* >  _rightElements;
		Vector< GraphElementFrame* >  _bottomElements;
		Vector< GraphElementFrame* >  _overElements;
		Vector< TypeCoordConverter* > _xConverters;
		Vector< TypeCoordConverter* > _yConverters;
#define _seriesList  TSG::series

		Size  _screenPlotSize;
		Rect  _screenRect;  // whole graph screen Rect
		Point _paintTopLeft;

		inline void updateSizes()
		{
			// --------------
			// GRAPH ELEMENTS
			// --------------
			TypeScreenCoord offsetLeft=0;
			TypeScreenCoord offsetTop=0;
			TypeScreenCoord offsetRight=0;
			TypeScreenCoord offsetBottom=0;
			
			// ======================
			//   calcul des offset
			for (int j = 0; j < _leftElements.GetCount(); j++)
			{
				offsetLeft += _leftElements[j]->GetElementWidth();
			}
			for (int j = 0; j < _topElements.GetCount(); j++)
			{
				offsetTop += _topElements[j]->GetElementWidth();
			}
			for (int j = 0; j < _bottomElements.GetCount(); j++)
			{
				offsetBottom += _bottomElements[j]->GetElementWidth();
			}
			for (int j = 0; j < _rightElements.GetCount(); j++)
			{
				offsetRight += _rightElements[j]->GetElementWidth();
			}
			
			_screenPlotSize.cx = _screenRect.GetWidth() - offsetLeft - offsetRight;
			_screenPlotSize.cy = _screenRect.GetHeight() - offsetTop - offsetBottom;

			_paintTopLeft.x = offsetLeft;
			_paintTopLeft.y = offsetTop;


			for (int j = 0; j < _xConverters.GetCount(); j++) {
				_xConverters[j]->updateScreenSize( 0, _screenPlotSize.cx );
			}

			for (int j = 0; j < _yConverters.GetCount(); j++) {
				_yConverters[j]->updateScreenSize( _screenPlotSize.cy, 0  );
			}

			// ======================
			//  Update element positions
			Rect r;
			
			r.Set( Point( offsetLeft, offsetTop ), Size(40,40) );
			for (int j = 0; j < _leftElements.GetCount(); j++)
			{
				r.OffsetHorz( -_leftElements[j]->GetElementWidth() );
				r.SetSize( _leftElements[j]->GetElementWidth() ,_screenPlotSize.cy);
				_leftElements[j]->Update();
				_leftElements[j]->SetFrame(r);
			}

			r.Set( Point( offsetLeft, offsetTop ), Size(40,40) );
			for (int j = 0; j < _topElements.GetCount(); j++)
			{
				r.OffsetVert( -_topElements[j]->GetElementWidth() );
				r.SetSize( _screenPlotSize.cx, _topElements[j]->GetElementWidth());
				_topElements[j]->SetFrame(r);
				_topElements[j]->Update();
			}

			r.Set( Point( offsetLeft, offsetTop+_screenPlotSize.cy ), Size(40,40) );
			for (int j = 0; j < _bottomElements.GetCount(); j++)
			{
				r.SetSize( _screenPlotSize.cx, _bottomElements[j]->GetElementWidth());
				_bottomElements[j]->SetFrame(r);
				_bottomElements[j]->Update();
				r.OffsetVert( _bottomElements[j]->GetElementWidth() );
			}

			r.Set( Point( offsetLeft+_screenPlotSize.cx, offsetTop ), Size(40,40) );
			for (int j = 0; j < _rightElements.GetCount(); j++)
			{
				r.SetSize( _rightElements[j]->GetElementWidth() ,_screenPlotSize.cy);
				_rightElements[j]->SetFrame(r);
				_rightElements[j]->Update();
				r.OffsetHorz( _rightElements[j]->GetElementWidth() );
			}
		}

		public:
		EmptyGraphDraw() {};

		virtual ~EmptyGraphDraw() {
			for (int j = 0; j < _createdElements.GetCount(); j++)
			{
				delete ( _createdElements[j] );
			}
			_createdElements.Clear();
		}

		TypeCoordConverter& GetXCoordConverter() { return *TypeSeriesGroup::_currentXConverter; }
		TypeCoordConverter& GetYCoordConverter() { return *TypeSeriesGroup::_currentYConverter; }

		CLASSNAME& setScreenSize(Rect r)
		{
			_screenRect = r;
			updateSizes();
			return *this;
		}


		TypeCoordConverter& AddXConverter(TypeCoordConverter& conv) {
			_xConverters << &conv;
			TypeSeriesGroup::_currentXConverter = &conv;
			return conv;
		}

		TypeCoordConverter& AddYConverter(TypeCoordConverter& conv) {
			_yConverters << &conv;
			TypeSeriesGroup::_currentYConverter = &conv;
			return conv;
		}


		template<class T, int POS_OF_GRAPH>
		CLASSNAME& AddElement(T& v) {
				switch(POS_OF_GRAPH){
					case LEFT_OF_GRAPH:
						_leftElements << &v.SetElementPos(LEFT_OF_GRAPH);
						break;
					case BOTTOM_OF_GRAPH:
						_bottomElements << &v.SetElementPos(BOTTOM_OF_GRAPH);
						break;
					case TOP_OF_GRAPH:
						_topElements << &v.SetElementPos(TOP_OF_GRAPH);
						break;
					case RIGHT_OF_GRAPH:
						_rightElements << &v.SetElementPos(RIGHT_OF_GRAPH);
						break;
					case OVER_GRAPH:
						_overElements << &v.SetElementPos(OVER_GRAPH);
						break;
				}
				return *this;
		}

		template<class T>  CLASSNAME& AddLeftElement(T& v)   { return AddElement<T, LEFT_OF_GRAPH>(v); }
		template<class T>  CLASSNAME& AddRightElement(T& v)  { return AddElement<T, RIGHT_OF_GRAPH>(v); }
		template<class T>  CLASSNAME& AddTopElement(T& v)    { return AddElement<T, TOP_OF_GRAPH>(v); }
		template<class T>  CLASSNAME& AddBottomElement(T& v) { return AddElement<T, BOTTOM_OF_GRAPH>(v); }
		template<class T>  CLASSNAME& AddOverElement(T& v)   { return AddElement<T, OVER_GRAPH>(v); }


		template<class T, int POS_OF_GRAPH>
		T& CloneElement(int elementWidth, T& p) {
			T* e = dynamic_cast<T*>( p.Clone() );
			e->SetElementWidth(elementWidth);
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e);
			return *e;
		}
		template<class T>	T& CloneLeftElement(int elementWidth, T& p)   { return CloneElement<T, LEFT_OF_GRAPH>(elementWidth, p); }
		template<class T>	T& CloneRightElement(int elementWidth, T& p)  { return CloneElement<T, RIGHT_OF_GRAPH>(elementWidth, p); }
		template<class T>	T& CloneTopElement(int elementWidth, T& p)    { return CloneElement<T, TOP_OF_GRAPH>(elementWidth, p); }
		template<class T>	T& CloneBottomElement(int elementWidth, T& p) { return CloneElement<T, BOTTOM_OF_GRAPH>(elementWidth, p); }
		template<class T>	T& CloneOverElement(T& p)   { return CloneElement<OVER_GRAPH>(0, p); }


		template<class T, int POS_OF_GRAPH>
		T& CreateElement(int elementWidth) {
			T* e = new T();
			e->SetElementWidth(elementWidth);
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e);
			return *e;
		}
		template<class T>	T& CreateLeftElement(int elementWidth)   { return CreateElement<T, LEFT_OF_GRAPH>(elementWidth); }
		template<class T>	T& CreateRightElement(int elementWidth)  { return CreateElement<T, RIGHT_OF_GRAPH>(elementWidth); }
		template<class T>	T& CreateTopElement(int elementWidth)    { return CreateElement<T, TOP_OF_GRAPH>(elementWidth); }
		template<class T>	T& CreateBottomElement(int elementWidth) { return CreateElement<T, BOTTOM_OF_GRAPH>(elementWidth); }
		template<class T>	T& CreateOverElement()   { return CreateElement<OVER_GRAPH>(0); }


		template<class T, int POS_OF_GRAPH, class P1>
		T& CreateElement(int elementWidth, P1& p1 ) {
			T* e = new T(p1);
			e->SetElementWidth(elementWidth);
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e);
			return *e;
		}
		template<class T, class P1>	T& CreateLeftElement(int elementWidth, P1& p1)   { return CreateElement<T, LEFT_OF_GRAPH>(elementWidth, p1); }
		template<class T, class P1>	T& CreateRightElement(int elementWidth, P1& p1)  { return CreateElement<T, RIGHT_OF_GRAPH>(elementWidth, p1); }
		template<class T, class P1>	T& CreateTopElement(int elementWidth, P1& p1)    { return CreateElement<T, TOP_OF_GRAPH>(elementWidth, p1); }
		template<class T, class P1>	T& CreateBottomElement(int elementWidth, P1& p1) { return CreateElement<T, BOTTOM_OF_GRAPH>(elementWidth, p1); }
		template<class T, class P1>	T& CreateOverElement(P1& p1) { return CreateElement<T, OVER_GRAPH>(0, p1); }


		template<class T, int POS_OF_GRAPH, class P1, class P2>
		T& CreateElement(int elementWidth, P1& p1, P2& p2 ) {
			T* e = new T(p1,p2);
			e->SetElementWidth(elementWidth);
			_createdElements << e; // to manage object destruction
			AddElement<T, POS_OF_GRAPH>(*e);
			return *e;
		}
		template<class T, class P1, class P2>	T& CreateLeftElement(int elementWidth, P1& p1, P2& p2)   { return CreateElement<T, LEFT_OF_GRAPH>(elementWidth, p1, p2); }
		template<class T, class P1, class P2>	T& CreateRightElement(int elementWidth, P1& p1, P2& p2)  { return CreateElement<T, RIGHT_OF_GRAPH>(elementWidth, p1, p2); }
		template<class T, class P1, class P2>	T& CreateTopElement(int elementWidth, P1& p1, P2& p2)    { return CreateElement<T, TOP_OF_GRAPH>(elementWidth, p1, p2); }
		template<class T, class P1, class P2>	T& CreateBottomElement(int elementWidth, P1& p1, P2& p2) { return CreateElement<T, BOTTOM_OF_GRAPH>(elementWidth, p1, p2); }
		template<class T, class P1, class P2>	T& CreateOverElement( P1& p1, P2& p2) { return CreateElement<T, OVER_GRAPH>(0, p1, p2); }




		virtual void Refresh() {};

		//			template<class T>
		//			void Paint(T& dw)
		void Paint(Draw& dw, int scale=1)
		{

			// ------------
			// paint graph area background
			// ------------
			dw.DrawRect(dw.GetPaintRect(), White());

			// --------------------------------------
			// BEGIN paint in PLOT AREA
			dw.Clipoff(_paintTopLeft.x, _paintTopLeft.y, _screenPlotSize.cx, _screenPlotSize.cy);
			dw.DrawRect(0, 0, _screenPlotSize.cx, _screenPlotSize.cy, LtGray());
			// --------------------------------------

			// --------------
			// GRAPH ELEMENTS on PLOT area   ( X/Y Grid, or anything else )
			// --------------
			for (int j = 0; j < _overElements.GetCount(); j++)
			{
				_overElements[j]->PaintOnPlot(dw, _screenPlotSize.cx, scale);
			}
			for (int j = 0; j < _leftElements.GetCount(); j++)
			{
				_leftElements[j]->PaintOnPlot(dw, _screenPlotSize.cx, scale);
			}

			for (int j = 0; j < _bottomElements.GetCount(); j++)
			{
				_bottomElements[j]->PaintOnPlot(dw, _screenPlotSize.cy, scale);
			}

			for (int j = 0; j < _rightElements.GetCount(); j++)
			{
				_rightElements[j]->PaintOnPlot(dw, _screenPlotSize.cx, scale);
			}

			for (int j = 0; j < _topElements.GetCount(); j++)
			{
				_topElements[j]->PaintOnPlot(dw, _screenPlotSize.cy, scale);
			}

			// ------------
			// paint DATA
			// ------------
			if (!_seriesList.IsEmpty())
			{
				for (int j = 0; j < _seriesList.GetCount(); j++)
				{
					if (_seriesList[j].opacity == 0 || (!_seriesList[j].seriesPlot && !_seriesList[j].markPlot))
						continue;

					{
						Vector<Point> p1;
						// Do coordinate conversion
						for (int c=0; c<_seriesList[j].PointsData()->GetCount(); ++c)
						{
//							_seriesList[j].xConverter;
//							_seriesList[j].yConverter;
							p1 << Point(_seriesList[j].xConverter->toScreen( _seriesList[j].PointsData()->x(c)), _seriesList[j].yConverter->toScreen( _seriesList[j].PointsData()->y(c)));
							//p1 << Point(_xConverter.toScreen( _seriesList[j].PointsData()->x(c)), _yConverter.toScreen( _seriesList[j].PointsData()->y(c)));
						}

						// Draw lines
						if ( !_seriesList[j].seriesPlot.IsEmpty() )
							_seriesList[j].seriesPlot->Paint(dw,
							                                 p1,
							                                 scale,
							                                 _seriesList[j].opacity,
							                                 fround(_seriesList[j].thickness),
							                                 _seriesList[j].color,
							                                 _seriesList[j].dash,
							                                 LtRed(),
							                                 _seriesList[j].fillColor,
							                                 1,//_xConverter.getGraphToScreenScale(),//l / xRange,
							                                 1,//_yConverter.getGraphToScreenScale(),//h / yRange,
							                                 0//int(h * (1 + yMin / yRange)));
							);

						// Draw marks
						if (_seriesList[j].markWidth >= 1 && _seriesList[j].markPlot)
						{
							if ( !_seriesList[j].markPlot.IsEmpty() )
								for (int c=0; c<_seriesList[j].PointsData()->GetCount(); ++c)
								{
									_seriesList[j].markPlot->Paint(dw,
									                               scale,
									                               p1[c],
									                               _seriesList[j].markWidth,
									                               _seriesList[j].markColor);
								}
						}
					}
				}
			}


			// --------------------------------------
			dw.End();
			// END of paint in PLOT AREA
			// --------------------------------------

			// --------------
			// GRAPH ELEMENTS
			// --------------
			for (int j = 0; j < _leftElements.GetCount(); j++)
			{
				dw.Offset(_leftElements[j]->GetFrame().TopLeft());
				_leftElements[j]->PaintElement(dw, scale);
				dw.End();
			}

			for (int j = 0; j < _bottomElements.GetCount(); j++)
			{
				dw.Offset(_bottomElements[j]->GetFrame().TopLeft());
				_bottomElements[j]->PaintElement(dw, scale);
				dw.End();
			}

			for (int j = 0; j < _rightElements.GetCount(); j++)
			{
				dw.Offset(_rightElements[j]->GetFrame().TopLeft());
				_rightElements[j]->PaintElement(dw, scale);
				dw.End();
			}

			for (int j = 0; j < _topElements.GetCount(); j++)
			{
				dw.Offset(_topElements[j]->GetFrame().TopLeft());
				_topElements[j]->PaintElement(dw, scale);
				dw.End();
			}

			// --------------
			// GRAPH ELEMENTS on ALL GRAPH area   ( legend, or anything else )
			// --------------
			for (int j = 0; j < _overElements.GetCount(); j++)
			{
				dw.Clipoff( _overElements[j]->GetFrame() );
				_overElements[j]->PaintOverGraph(dw, scale);
				dw.End();
			}

			TRACE_INFO("==================================" );
		}
	};


	// ============================
	//    EmptyGraphDraw   CLASS
	// ============================
	template<class DATASOURCE, class SERIESPLOT=SeriesPlot, class MARKPLOT=MarkPlot, class COORDCONVERTER = GenericCoordinateConverter<Upp::int32, double> >
	class StdGraphDraw : public EmptyGraphDraw<DATASOURCE,SERIESPLOT,MARKPLOT,COORDCONVERTER>
	{
		public:
		typedef StdGraphDraw<DATASOURCE, SERIESPLOT, MARKPLOT, COORDCONVERTER> CLASSNAME;
		typedef EmptyGraphDraw<DATASOURCE,SERIESPLOT,MARKPLOT,COORDCONVERTER> BASECLASS;
		typedef COORDCONVERTER                           TypeCoordConverter;
		typedef DATASOURCE                               TypeDataSource;
		typedef SERIESPLOT                               TypeSeriesPlot;
		typedef GridAxisDraw<COORDCONVERTER>             TypeGridAxisDraw;
		typedef GridStepManager<COORDCONVERTER>          TypeGridStepManager;
		typedef SeriesConfig<COORDCONVERTER, SERIESPLOT, MARKPLOT>       TypeSeriesConfig;
		typedef SeriesGroup<TypeSeriesConfig,COORDCONVERTER,StdGraphDraw>  TypeSeriesGroup;
		typedef TypeSeriesGroup                          TSG;
		typedef typename COORDCONVERTER::TypeScreenCoord ScreenCoord;
		typedef typename COORDCONVERTER::TypeGraphCoord  GraphCoord;

		protected:
		TypeCoordConverter        _xConverter;
		TypeCoordConverter        _yConverter;
		TypeGridAxisDraw          _xGridDraw;
		TypeGridAxisDraw          _yGridDraw;

#define _seriesList  TSG::series

		Rectf _graphRect;


		public:
		StdGraphDraw()
		: _xGridDraw(_xConverter)
		, _yGridDraw(_yConverter)
		{
			_xGridDraw.SetElementWidth(20);
			_xGridDraw.setAxisColor( Blue() ).setAxisTextFont(StdFont()).setAxisTextColor( Blue() ).setAxisTickColor( Red() );
			_xGridDraw.setGridColor( Gray() );
			_xGridDraw.setMajorTickMark( (new RoundTickMark())->SetTickLength( 4 ) );

			_yGridDraw.SetElementWidth(30);
			_yGridDraw.setAxisColor( Blue() ).setAxisTextFont(StdFont()).setAxisTextColor( Blue() ).setAxisTickColor( Red() );
			_yGridDraw.setGridColor( Gray() );
			_yGridDraw.setMajorTickMark( (new TriangleTickMark())->SetTickLength( 8 ) );

			BASECLASS::AddXConverter(_xConverter);
			BASECLASS::AddYConverter(_yConverter);
			BASECLASS::AddLeftElement(_yGridDraw);
			BASECLASS::AddBottomElement(_xGridDraw);

			setGraphSize(0, 100, 0, 100);
		};

		virtual ~StdGraphDraw() {}


		CLASSNAME& setGraphSize(Rectf r)
		{
			_xConverter.updateGraphSize(r.TopLeft().x, r.BottomRight().x);
			_yConverter.updateGraphSize(r.TopLeft().y, r.BottomRight().y);
			BASECLASS::updateSizes();
			return *this;
		}

//		Rectf& GetGraphSize(Rectf r) { return _graphRect; }

		CLASSNAME& setGraphSize(GraphCoord x0, GraphCoord x1, GraphCoord y0, GraphCoord y1)
		{
			_xConverter.updateGraphSize( x0, x1);
			_yConverter.updateGraphSize( y0, y1);
			BASECLASS::updateSizes();
			return *this;
		}

		TypeCoordConverter& GetXCoordConverter() { return _xConverter; }
		TypeCoordConverter& GetYCoordConverter() { return _yConverter; }

		TypeGridAxisDraw& GetXGridAxisDraw() { return _xGridDraw; }
		TypeGridAxisDraw& GetYGridAxisDraw() { return _yGridDraw; }

		CLASSNAME& setXAxisRectWidth(int v) { _xGridDraw.SetElementWidth(v); return *this; }
		CLASSNAME& setYAxisRectWidth(int v) { _yGridDraw.SetElementWidth(v); return *this; }


		CLASSNAME& setXNbGridSteps(int v) { _xGridDraw.GetGridStepManager().SetNbSteps(v); return *this; }
		CLASSNAME& setYNbGridSteps(int v) { _yGridDraw.GetGridStepManager().SetNbSteps(v); return *this; }
	};
};


template <class GRAPHCOORD, class SCREENCOORD, class DATASOURCE, class SERIESPLOT, class MARKPLOT>
struct GraphDrawTypes {
		typedef GRAPHCOORD                                                                GraphCoord;
		typedef SCREENCOORD                                                               ScreenCoord;
		typedef SERIESPLOT                                                                SeriesPLot;
		typedef MARKPLOT                                                                  MarkPLot;
		typedef typename GraphDraw_ns::CoordinateConverter<SCREENCOORD,GRAPHCOORD>        CoordConverter;
		typedef DATASOURCE                                                                DataSource;

		typedef typename GraphDraw_ns::EmptyGraphDraw<DATASOURCE, SERIESPLOT, MARKPLOT, CoordConverter> EmptyGraphDraw;
		typedef typename GraphDraw_ns::GenericCoordinateConverter<ScreenCoord,GraphCoord>               GenericCoordConv;
		typedef typename GraphDraw_ns::BlankAreaElement<CoordConverter>                                 BlankAreaElement;
		typedef typename GraphDraw_ns::LabelElement<CoordConverter>                                     LabelElement;
		typedef typename GraphDraw_ns::GridAxisDraw<GenericCoordConv>                                   GridAxisDraw;
};


#endif
