// Serial.cpp


#include "Serial.h"

#ifdef PLATFORM_WIN32

SerialPort::SerialPort()
{

	memset( &m_OverlappedRead, 0, sizeof( OVERLAPPED ) );
 	memset( &m_OverlappedWrite, 0, sizeof( OVERLAPPED ) );
	m_hIDComDev = NULL;
	m_bOpened = FALSE;

}

SerialPort::~SerialPort()
{
	Close();

}
/*
Notes picked from the microsoft site on asynch communications

After opening the file object for asynchronous I/O, an OVERLAPPED structure must be properly created, 
	initialized, and passed into each call to functions such as ReadFile and WriteFile. 
	Keep the following in mind when using the OVERLAPPED structure in asynchronous read and write operations:

    * Do not deallocate or modify the OVERLAPPED structure or the data buffer 
    	until all asynchronous I/O operations to the file object have been completed.
    * If you declare your pointer to the OVERLAPPED structure as a local variable, 
    	do not exit the local function until all asynchronous I/O operations to the file object have been completed. 
    	If the local function is exited prematurely, the OVERLAPPED structure will go out of scope 
    	and it will be inaccessible to any ReadFile or WriteFile functions it encounters outside of that function.

You can also create an event and put the handle in the OVERLAPPED structure; the wait functions can then be used
 to wait for the I/O operation to complete by waiting on the event handle.

As previously stated, when working with an asynchronous handle, applications should use care when making
 determinations about when to free resources associated with a specified I/O operation on that handle. 
 If the handle is deallocated prematurely, ReadFile or WriteFile may incorrectly report that the I/O operation 
 is complete. Further, the WriteFile function will sometimes return TRUE with a GetLastError value of ERROR_SUCCESS, 
 even though it is using an asynchronous handle (which can also return FALSE with ERROR_IO_PENDING). 
 Programmers accustomed to synchronous I/O design will usually release data buffer resources at this point 
 because TRUE and ERROR_SUCCESS signify the operation is complete. However, if I/O completion ports are being used
 with this asynchronous handle, a completion packet will also be sent even though the I/O operation completed 
 immediately. In other words, if the application frees resources after WriteFile returns TRUE with ERROR_SUCCESS 
 in addition to in the I/O completion port routine, it will have a double-free error condition. In this example, 
 the recommendation would be to allow the completion port routine to be solely responsible for all freeing operations
  for such resources.

*/
bool SerialPort::Open( int nPort, String Params )
{

	if( m_bOpened ) return( true );

	char szPort[15];
	DCB dcb;

	m_hIDComDev = CreateFile( "COM"+AsString(nPort), GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL );
	if( m_hIDComDev == NULL ) return( FALSE );

	memset( &m_OverlappedRead, 0, sizeof( OVERLAPPED ) );
 	memset( &m_OverlappedWrite, 0, sizeof( OVERLAPPED ) );

	COMMTIMEOUTS CommTimeOuts;
	CommTimeOuts.ReadIntervalTimeout = 0xFFFFFFFF;
	CommTimeOuts.ReadTotalTimeoutMultiplier = 0;
	CommTimeOuts.ReadTotalTimeoutConstant = 0;
	CommTimeOuts.WriteTotalTimeoutMultiplier = 0;
	CommTimeOuts.WriteTotalTimeoutConstant = 1000;
	SetCommTimeouts( m_hIDComDev, &CommTimeOuts );

	m_OverlappedRead.hEvent = CreateEvent( NULL, true, FALSE, NULL );
	m_OverlappedWrite.hEvent = CreateEvent( NULL, true, FALSE, NULL );

	dcb.DCBlength = sizeof( DCB );
	if (!GetCommState( m_hIDComDev, &dcb ))	return false;
	if (!BuildCommDCB(Params,&dcb)) return false;

//	unsigned char ucSet;
//	ucSet = (unsigned char) ( ( FC_RTSCTS & FC_DTRDSR ) != 0 );
//	ucSet = (unsigned char) ( ( FC_RTSCTS & FC_RTSCTS ) != 0 );
//	ucSet = (unsigned char) ( ( FC_RTSCTS & FC_XONXOFF ) != 0 );

	// SetCommState(handle, dcb)
	// SetupComm(Handle, word inqueue, word outqueue)
	if( !SetCommState( m_hIDComDev, &dcb ) ||
		!SetupComm( m_hIDComDev, 10000, 10000 ) ||
		m_OverlappedRead.hEvent == NULL ||
		m_OverlappedWrite.hEvent == NULL ){
		DWORD dwError = GetLastError();
		if( m_OverlappedRead.hEvent != NULL ) CloseHandle( m_OverlappedRead.hEvent );
		if( m_OverlappedWrite.hEvent != NULL ) CloseHandle( m_OverlappedWrite.hEvent );
		CloseHandle( m_hIDComDev );
		return( FALSE );
		}

	m_bOpened = true;

	return( m_bOpened );

}

bool SerialPort::Close( void )
{

	if( !m_bOpened || m_hIDComDev == NULL ) return( true );

	if( m_OverlappedRead.hEvent != NULL ) CloseHandle( m_OverlappedRead.hEvent );
	if( m_OverlappedWrite.hEvent != NULL ) CloseHandle( m_OverlappedWrite.hEvent );
	CloseHandle( m_hIDComDev );
	m_bOpened = FALSE;
	m_hIDComDev = NULL;

	return( true );

}

bool SerialPort::WriteCommByte( unsigned char ucByte )
{
	bool bWriteStat;
	DWORD dwBytesWritten;

	bWriteStat = WriteFile( m_hIDComDev, (LPSTR) &ucByte, 1, &dwBytesWritten, &m_OverlappedWrite );
	if( !bWriteStat && ( GetLastError() == ERROR_IO_PENDING ) ){
		if( WaitForSingleObject( m_OverlappedWrite.hEvent, 1000 ) ) dwBytesWritten = 0;
		else{
			GetOverlappedResult( m_hIDComDev, &m_OverlappedWrite, &dwBytesWritten, FALSE );
			m_OverlappedWrite.Offset += dwBytesWritten;
			}
		}

	return( true );

}

int SerialPort::SendData( const char *buffer, int size )
{

	if( !m_bOpened || m_hIDComDev == NULL ) return( 0 );

	DWORD dwBytesWritten = 0;
	int i;
	for( i=0; i<size; i++ ){
		WriteCommByte( buffer[i] );
		dwBytesWritten++;
		}

	return( (int) dwBytesWritten );

}

int SerialPort::ReadDataWaiting( void )
{

	if( !m_bOpened || m_hIDComDev == NULL ) return( 0 );

	DWORD dwErrorFlags;
	COMSTAT ComStat;

	ClearCommError( m_hIDComDev, &dwErrorFlags, &ComStat );

	return( (int) ComStat.cbInQue );

}

int SerialPort::ReadData( void *buffer, int limit )
{

	if( !m_bOpened || m_hIDComDev == NULL ) return( 0 );

	bool bReadStatus;
	DWORD dwBytesRead, dwErrorFlags;
	COMSTAT ComStat;

	ClearCommError( m_hIDComDev, &dwErrorFlags, &ComStat );
	if( !ComStat.cbInQue ) return( 0 );

//	dwBytesRead = (DWORD) ComStat.cbInQue;
//	if( limit < (int) dwBytesRead ) dwBytesRead = (DWORD) limit;
	dwBytesRead = limit;
	
	bReadStatus = ReadFile( m_hIDComDev, buffer, dwBytesRead, &dwBytesRead, &m_OverlappedRead );
	if( !bReadStatus ){
		if( GetLastError() == ERROR_IO_PENDING ){
			WaitForSingleObject( m_OverlappedRead.hEvent, 2000 );
			return( (int) dwBytesRead );
			}
		return( 0 );
		}

	return( (int) dwBytesRead );

}


#else

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h> 
#include <unistd.h> 
#include <errno.h>

#include "Serial.h"

using namespace Upp;

SerialPort::SerialPort()
: m_fileDesc(-1)
{
}


bool SerialPort::Open(int nPort, int nBaud)
{
	SerialPortSpeeds sp;
	
	char device[1000];
	
	switch(nBaud)
	{
		case 115200:
			sp = BDS_115200;
			break;
		case 57600:
			sp = BDS_57600;
			break;
		case 38400:
			sp = BDS_38400;
			break;
		case 19200:
			sp = BDS_19200;
			break;
		case 9600:
			sp = BDS_9600;
			break;
		case 4800:
			sp = BDS_4800;
			break;
		case 2400:
			sp = BDS_2400;
			break;
		case 1200:
			sp = BDS_1200;
			break;
		case 600:
			sp = BDS_600;
			break;
		case 300:
			sp = BDS_300;
			break;
		case 110:
			sp = BDS_110;
			break;
	}
	
	sprintf(device,"/dev/ttyS%d\0",nPort);	
	
	return Open(device,sp);
	
	return (m_fileDesc != -1);
}


bool SerialPort::Open( const char * const device, const SerialPortSpeeds sp )
{
	if (m_fileDesc != -1) Close();

	m_fileDesc = ::open(device, O_RDWR, 0644);
	if (m_fileDesc == -1)
	{
		return false;
		
		SerialException errExc("Serial device initialization error for device : ");
		errExc << m_deviceName;
		errExc << "\nError: " << strerror(errno);
		throw( errExc );
	}
	else
	{
		struct termios options;

		// Get the current options for the port...
		tcgetattr(m_fileDesc, &options);

		// Set the baud rates
		cfsetispeed(&options, sp);
		cfsetospeed(&options, sp);
		
		// Enable the receiver and set local mode...
		options.c_cflag |= (CLOCAL | CREAD);
		
		// Disable special caracters processing and set RAW mode
		cfmakeraw(&options);
		
		// Set the new options for the port...
		tcsetattr(m_fileDesc, TCSANOW, &options);
	}
	
	return true;
}

bool SerialPort::Close()
{
	if (m_fileDesc != -1)
	{
		::close(m_fileDesc);
		m_fileDesc = -1;
	}
	
	return m_fileDesc = -1;
}


SerialPort::~SerialPort()
{
	Close();
}


String SerialPort::BufToString(const char * const buf, const int nbBytes, const char * const prefixString)
{
	String res;
	res << "\n" << prefixString << "   0x";
	
	for(int c=0; c < nbBytes; c++)
	{
		res << FormatIntHex(buf[c]) << "  "; 
	}
	return res;
}

SerialPort::WriteReturnValues SerialPort::write( int nbBytesToWrite, const char* const buf )
{
	int l_nbBytesWritten= 0;
	
//	displayBuf( buf, nbBytesToWrite, "TX ");
	l_nbBytesWritten = ::write(m_fileDesc, buf, nbBytesToWrite);
	if (l_nbBytesWritten != nbBytesToWrite)
	{
		SerialException errMsg("Write error on serial device : ");
		errMsg << m_deviceName;
		throw( errMsg );
	}
	return WRITE_OK;
}


SerialPort::ReadReturnValues SerialPort::read(char* buf, int& nbBytesToRead, int timeOutus )
{
	if(m_fileDesc==-1)
		return READ_ERROR;
	
	int 	l_error		= 0;
	int 	l_nbBytesRead	= 0;
	
	timeval l_TimeOut;
	l_TimeOut.tv_sec=0;
	l_TimeOut.tv_usec = timeOutus;
	
	fd_set 	readFdSet;
	
	// INIT du readFdSet pour utilisation par le select
	FD_ZERO (&readFdSet);
	FD_SET( m_fileDesc, &readFdSet);
	
	if ( timeOutus < 0) // pas de TIMEOUT
	{
		l_error = select(m_fileDesc+1, &readFdSet, 0, 0, NULL);
	}
	else
	{
		l_error = select(m_fileDesc+1, &readFdSet, 0, 0, &l_TimeOut);
	}
	
	if (l_error == 0)
	{
		return READ_TIMEDOUT;
	}
	else if (l_error < 0)
	{
		SerialException errMsg("Read error on serial device: ");
		errMsg << m_deviceName << ",  error code:" << l_error;
		throw(errMsg);
	}
	else
	{
		l_nbBytesRead = ::read(m_fileDesc, buf, nbBytesToRead);
		if (l_nbBytesRead == 0)
		{
			SerialException errMsg("EOF reached on serial device: ");
			errMsg << m_deviceName;
			throw( errMsg );
		}
		else if (l_nbBytesRead == -1)
		{
			SerialException errMsg("Read ERROR  on serial device: ");
			errMsg << m_deviceName << strerror( errno );
			throw( errMsg );
		}
		nbBytesToRead = l_nbBytesRead;
	}
	
	return READ_OK;
}



#endif