#ifndef _Barcode_Barcode_h_
#define _Barcode_Barcode_h_

#include <Draw/Draw.h>

NAMESPACE_UPP

#define NO_UNIT_WIDTH

// define some common interfaces for linear barcode
//
//
class Barcode{
public:
	const Drawing& AsDrawing()const
	{
		if(draw.IsEmpty())
		{
			const_cast<Barcode&>(*this).Prepare();
		}
		return *draw;
	}
	operator const Drawing&()const{
		return AsDrawing();
	}
	
	Size GetSize()const{
		return AsDrawing().GetSize();
	}
	
	Barcode& Text(const char * text)
	{
		SetText(text);
		return *this;
	}
	
	Barcode& Text(char * text){
		return Text(const_cast<const char*>(text));
	}
	
	Barcode& DisplayText(const char * text)
	{
		this->text=text;
		return MarkAsDirty();
	}
	Barcode& DisplayText(char* text){
		return DisplayText(const_cast<const char*>(text));
	}
	
	Barcode& Font(Upp::Font font){
		this->font=font;
		return MarkAsDirty();
	}
	
	Barcode& Color(Upp::Color color){
		this->color=color;
		return MarkAsDirty();
	}
	
	Barcode& Height(int height){
		this->height=height;
		return MarkAsDirty();
	}
#ifndef NO_UNIT_WIDTH	
	Barcode& UnitWidth(int width){		
		unitWidth=width;
		return MarkAsDirty();
	}
#endif	
	Barcode& NoQuietZone(bool v=true)
	{
		noQuietZone=(int8)v;
		return MarkAsDirty();
	}

	virtual ~Barcode(){}
	virtual void Prepare()=0;
	virtual void SetText(const char * text)=0;
protected:
	One<Drawing>  draw;
	Upp::String text; // displayed text;
	Upp::Font	font; // for text display
	Upp::Color	color; // the bar color & display text
	int16		height;
#ifndef NO_UNIT_WIDTH
	int8		noQuietZone;	
	int8	    unitWidth;
#else
	int16	    noQuietZone;
	const static int unitWidth=1;
#endif


	virtual int  GetWidth()const=0;

	Barcode& MarkAsDirty(){
		draw.Clear();
		return *this;
	}

	void Init()
	{
		font=StdFont().Height(10);
		color=Black();
		noQuietZone=(int8)false;
#ifndef NO_UNIT_WIDTH
		unitWidth=1;
#endif
		height=40;
	}
};

class Code128 :public Moveable<Code128>, public Barcode
{
public:
	Code128(){Init();  }
	Code128(const char* text)
	{ Init();Text(text);}
	Code128(char * text)
	{ Init();Text(text);}

	virtual void Prepare();
	virtual void SetText(const char * text);
protected:
	Vector<uint8> data;

	const Vector<uint8>& GetData()const{ return data; }

	int BarCodeWidth()const{
		return (data.GetCount()*11+2+(noQuietZone?0:20))*unitWidth;
	}
	
	virtual int  GetWidth()const
	{
		return  max(BarCodeWidth(),
					GetTextSize(text,font).cx
				); 
	}
};


// 2013-3-6: Add EAN8, and UPCA preferred
//
class EAN : public Moveable<EAN>,public Barcode
{
public:
	EAN()
	{Init(); ClearData();}
	EAN(const char* text)
	{Init(); height=83; Text(text);}
	EAN(char * text)
	{Init(); height=83; Text(text);}
	
	EAN& Type(int upcvariant);	
	int    Type()const;

	virtual void Prepare();
	virtual void SetText(const char *);

	enum{EAN13, DEFAULT=EAN13, EAN8, UPCA,
		// virtual types, automatic choice
			EAN8UPCAEAN13, // if can be represented as EAN8
						   // do it; otherwise if can be
						   // represented as UPCA, do it;
						   // otherwise represented as EAN13
			EAN8EAN13		// similar to above
		};
protected:
	char ___[14];
	int		type; // one of EAN, UPCA, EAN8
	
	int BarcodeWidth()const{
		const int guard=2*3+5; //guard area width(stard,end and middle)
		int digits;
		switch(ActualType())
		{
		case EAN8:
			digits=8;
			break;
		default: // UPCA & EAN13
			digits=12;
		}
		// each digit has a total width of 7 x-dimensions
		ASSERT(noQuietZone<2);
		return ( (noQuietZone==1?guard+12:guard+18)
					+7*digits )*unitWidth;
	}

	virtual int  GetWidth()const{
		return max(BarcodeWidth(), GetTextSize(text,font).cx);
	}
	
	void ClearData()
	{
		*reinterpret_cast<int*>(___)=0;
	}

	// convert virtual type to actual type
	int ActualType()const;
	
	void Init()
	{
		Barcode::Init();
		type=EAN13;
	}
};

END_UPP_NAMESPACE

#endif
