#include "Barcode.h"

using namespace Upp;

#include <queue>
class Encoder128;
// There might be an O(n) or O(n^2)[DP?]
// algorithm to arrive at the shortest Code 128 encoding.
// but given the fact that texts to be
// encoded are generally of very limited length, A*
// search used here should be fast enough
//
class Encoder128
{

public:
#pragma warning(disable:4355)
	Encoder128(const char* text):
		//input(input),
		openset(CmpIndex(*const_cast<const Encoder128*>(this)))
#pragma warning(default:4355)
	{
		input=Unescape(text);
		Encode();
	}
	
	Vector<uint8>& GetEncoded(){ return output; }

	int f(int i)const{ 
		return states[i].cost+input.GetCount()-states[i].pos; }
	
	static void Render(Draw& draw, const Rect& r, const Code128& c128);
	


private:
	struct CmpIndex
	{
		CmpIndex(const Encoder128& enc): encoder(enc){}
		bool operator()(int i1, int i2)const;
		const Encoder128& encoder;
	};

	struct DrawInfo{
		Draw& draw;
		int   left;
		int   top;
		int   height;
		int   unitWidth;
		Color color;
		
		DrawInfo(Draw& _draw):draw(_draw){}
	};
	

	enum { CODE_A=1, CODE_B=2, CODE_C=4,
	     FNC1=256, FNC2, FNC3, FNC4 };

	struct State128 : Moveable<State128>
	{
		bool  inDblFNC4; // in double FNC4 escape sequence
		int      pos; // at most 256 bytes, more than enough for practical problem
		int     cost;// length of code128 values for the prefix processed 
		int   context; // codeset A B or C
		int   	parent; // index of the state that derives this state;
		uint8 	data[4]; // a input char requires at most 4 symbol
	};


private:
	void Encode();
	// derive a new state from a given state,
	// 
	// returns index of newly created state
	// in the states Vector. Should set the new
	// state properly before adding to the openset
	int Derive(int index);
	
	Vector<uint8> Collect(int index)const;

	int CheckDigit();
	void Add(int value){ output<<(char)value; }
	static int  AvailableSet(unsigned ascii);
	static int  EncodeOne(unsigned ascii, int codeset);
	static Vector<uint16> Unescape(const char * text);
	void InitialStates();
	void EncodeC(int indexOfCurState);
	void EncodeSingle(int indexOfCurState);
	void EncodeSingleAltAB(int indexOfCurState);
	
	static void DrawValue(DrawInfo& di, int code128value);
	static void DrawBar(DrawInfo& di, int cnt, bool blank=false);
	
	
	
	Vector<State128> states;
	std::priority_queue<int, std::vector<int>, CmpIndex> openset;
	Vector<uint16> input;
	Vector<uint8> output;
};

bool Encoder128::CmpIndex::operator()(int i1, int i2)const
{
	return encoder.f(i1) >
			 encoder.f(i2);
}

// derivate a child state from current state and add
// it to the states vector.
// returns the index of the newly derived state in 
// the states vector. 
//
int Encoder128::Derive(int index)
{
	State128* t;
	if(index>=0){
		//State128& s=states[index];
		t=&states.Add();
		*t=states[index];
	}else{
		t=&states.Add();
		t->cost=0;
		t->pos=0;
		t->context=-1; // unknown context
		t->inDblFNC4=false;
	}
	t->parent=index;
	*reinterpret_cast<int32*>(t->data)=0xFFFFFFFF;
	return t-&*states.Begin();
}

Vector<uint8> Encoder128::Collect(int index)const
{
	Vector<uint8> v;
	const State128& s=states[index];
	v.Reserve(s.cost+2); // extra ones for check digit and stop symbol
	v.At(s.cost-1);
	
	do{
		const State128& s=states[index];
		int i=s.cost;
		for(int j=3; j>=0; --j)
			if(s.data[j]!=0xFF){				
				v[--i]=s.data[j]; 
				ASSERT(s.data[j]<=107);
			}
		index=s.parent;
	}while(index!=-1);
	return v;
}

// do an A* search for the shortest encoding
//
void Encoder128::Encode()
{
	if(input.IsEmpty())
		return;
	
    InitialStates(); // add the inital states (states with no parents)

	while(true) // search loop
	{
		int i=openset.top(); // read, to be processed;
		openset.pop(); // remove from openset
		
		State128& s=states[i];
		if(s.pos==input.GetCount()){ // input processed & shortest encoding found
			output=Collect(i);  Add(CheckDigit());
			Add(106); /*stop symbol*/ return;
		}
		if( (input.GetCount()-s.pos>=4 && 
				IsDigit(input[s.pos]) && IsDigit(input[s.pos+1]) &&
				IsDigit(input[s.pos+2]) && IsDigit(input[s.pos+3]) ) || 
			(s.context==CODE_C && input.GetCount()-s.pos>=2 
			    && IsDigit(input[s.pos]) && IsDigit(input[s.pos+1]))
		){
			EncodeC(i); 
		}else if(s.inDblFNC4 && input.GetCount()-s.pos>=2 &&
			IsDigit(input[s.pos]) && IsDigit(input[s.pos+1])
		){  // note both path cost=4, their goodness depends on
			// input followed. We add both to openset.
			EncodeC(i);  // path 1
			
			int j=Derive(i); // yet another path
			State128& t=states[j];
			t.data[0]=EncodeOne(FNC4,t.context);
			t.data[1]=EncodeOne(input[t.pos++],t.context);
			t.data[2]=t.data[0];
			t.data[3]=EncodeOne(input[t.pos++],t.context);
			t.cost+=4;
			openset.push(j);
		}else// CODE C is out of question in this branch.
			EncodeSingle(i);
	}
//	#ifdef _DEBUG
//	for(int i=0; i<output.GetCount(); ++i)
//	{
//		LOG(String().Cat()<<(int)output[i]<<',');
//	}
//	#endif
}

// i is the index of the current 
// state to be derived.
//
// basically, no need to consider encoding 2 bytes together
// like in Code C in this function.
void Encoder128::EncodeSingle(int i)
{
	State128& s=states[i];
	unsigned c=input[s.pos];
	bool islatin=c>127&&c<256;
	unsigned d=islatin?c-128:c;
	int code_set=AvailableSet(c);

	if(code_set & s.context) // will remain in the current context
	{
		int j=Derive(i);
		State128& t=states[j];
		++t.pos;
		if( (t.inDblFNC4&&islatin) || (!t.inDblFNC4 && !islatin) ){
			t.data[0]=EncodeOne(d, t.context);
			++t.cost;
		}else{ // FNC4 escape required, 2 path, double escape or single escape;
			int k=Derive(i); // derive new state for dbl escape
			State128& u=states[k];

			u.data[0]=EncodeOne(FNC4, u.context);
			u.data[1]=u.data[0];
			u.data[2]=EncodeOne(d, u.context);
			u.inDblFNC4=!u.inDblFNC4;
			++u.pos;
			u.cost+=3;
			openset.push(k);
			
			State128& t=states[j]; //refresh reference to avoid vector expansion invalidate orig ref
			t.data[0]=u.data[0]; // use t for single escape path
			t.data[1]=u.data[2];
			t.cost+=2;
		}
		openset.push(j);
	}  // have to switch context
	   // CODE C will not be a target.
	   // both SwitchTo and ShiftTo should be considered when applicable.
	else if(s.context==CODE_C)// code_set will be either CODE_A or CODE_B
	{
		int j=Derive(i);
		State128& t=states[j];
		t.data[0]=101; // Switch To Code A from CODE c
		t.context=CODE_A;
		++t.cost;
		
		int k=Derive(i);
		State128& u=states[k];
		u.data[0]=100; // switch to Code B from CODE C
		u.context=CODE_B;
		++u.cost;
		openset.push(j);  openset.push(k);
	}else
		EncodeSingleAltAB(i);
}
// this function assumes the ascii can be represented in
// the requested codeset. codeset is used only 
// when a ascii has different representations in different
// code sets.
//
int Encoder128::EncodeOne(unsigned ascii, int codeset)
{
	ASSERT(AvailableSet(ascii) & codeset);
	switch(ascii)
	{
	case FNC1:
		return 102;
	case FNC2:
		return 97;
	case FNC3:
		return 96;
	case FNC4:
		return codeset==CODE_A ? 101 : 100;
	}
	return ((int)ascii+ (ascii<=0x1fu?0x40:-0x20));
}

int Encoder128::CheckDigit()
{
	ASSERT(!output.IsEmpty());
	int check=output[0];
	for(int i=1; i<output.GetCount();++i)
	{
		check+=i*output[i];
		check%=103;
	}
	return check;
}

void Encoder128::InitialStates()
{
	if(input.GetCount()>=2 && IsDigit(input[0]) && IsDigit(input[1]) )
	{
		int i=Derive(-1);
		State128& s=states[i];
		s.data[0]=105; // start c;
		s.data[1]=input[0]*10+input[1]-'0'*11;
		s.cost=2;
		s.pos=2;
		s.context=CODE_C;
		openset.push(i);
	}else{
		unsigned c=input[0];
		bool islatin=c>127&&c<256;
		unsigned d=islatin?c-128:c;
		int context_set=AvailableSet(c);
		if(context_set & CODE_A)
		{
			int i=Derive(-1);
			State128& s=states[i];
			s.data[0]=103; // start A;
			++s.pos;
			s.context=CODE_A;
			if(islatin){
				s.data[1]=EncodeOne(FNC4,CODE_A); // dbl FNC4 escape
				s.data[2]=s.data[1];
				s.data[3]=EncodeOne(d, CODE_A);
				s.inDblFNC4=true;
				s.cost=4;
				
				int j=Derive(-1); // single FNC4 escape
				{
					State128& s=states[i]; // refresh reference;
					State128& t=states[j];
					t.data[0]=103; // start A
					t.data[1]=s.data[1];
					t.data[2]=s.data[3];
					t.cost=3;
					t.context=CODE_A;
					++t.pos;
				}
				openset.push(j);
			}else{
				s.data[1]=EncodeOne(d, CODE_A);
				s.cost=2;
			}
			openset.push(i);
		}
		if(context_set & CODE_B)
		{
			int i=Derive(-1);
			State128& s=states[i];
			s.data[0]=104; // start B;
			++s.pos;
			s.context=CODE_B;
			if(islatin){
				s.data[1]=EncodeOne(FNC4,CODE_B); // dbl FNC4 escape
				s.data[2]=s.data[1];
				s.data[3]=EncodeOne(d, CODE_B);
				s.inDblFNC4=true;
				s.cost=4;
				
				int j=Derive(-1); // single FNC4 escape
				{
					State128& s=states[i];
					State128& t=states[j];
					t.data[0]=104; // start B
					t.data[1]=s.data[1];
					t.data[2]=s.data[3];
					t.cost=3;
					
					++t.pos;
					t.context=CODE_B;
				}					
				openset.push(j);
			}else{
				s.data[1]=EncodeOne(d, CODE_B);
				s.cost=2;
			}
			openset.push(i);
		}
		if( c==FNC1/*equivalent to context_set & CODE_C*/)
		{
			int i=Derive(-1);
			State128& s=states[i];
			s.data[0]=105; // start C;
			s.data[1]=102; 
			s.cost=2;
			++s.pos;
			s.context=CODE_C;
			openset.push(i);
		}
	}
}

void Encoder128::EncodeC(int i)
{
	int j=Derive(i);
	int k=0;

	State128& t=states[j];
	if(t.context!=CODE_C)
	{
		if(t.inDblFNC4)
		{
			t.data[0]=EncodeOne(FNC4, t.context);
			t.data[1]=t.data[0];
			k=2;
			t.cost+=2;
			t.inDblFNC4=false;
		}
		t.data[k++]=99; // move to code c;
		t.context=CODE_C;
		++t.cost;
	}
	ASSERT(IsDigit(input[t.pos]) && IsDigit(input[t.pos+1]));
	t.data[k]=input[t.pos]*10+input[t.pos+1]-'0'*11;
	t.pos+=2;
	++t.cost;
	openset.push(j);
}

// only encode 1 byte of input. If current context is CODE A,
// will need to switch/shift to CODE B, and vice versa.
//
void Encoder128::EncodeSingleAltAB(int i)
{
	// switch to A/B or shift to A/B, 
	int j=Derive(i);
	int k=Derive(i);
	State128& s=states[j];	
	State128& t=states[k];
	ASSERT(s.context!=CODE_C);
	
	s.data[0]=s.context==CODE_A?100:101; // switch to B/A;
	s.context=s.context==CODE_A?CODE_B:CODE_A; // and record the fact
	
	t.data[0]=98; // SHIFT BETWEEN A & B;
	
	unsigned c=input[s.pos];
	bool islatin=c>127&&c<256;
	unsigned d=islatin?c-128:c;
	if( (s.inDblFNC4 && islatin) || (!s.inDblFNC4 && !islatin) )
	{
		s.data[1]=EncodeOne(d, s.context);
		t.data[1]=s.data[1];
		s.cost+=2;
		t.cost+=2;
	}else{ // worst case, FNC4 escape is required
		// u & v, dbl FNC4 escape path;
		int l=Derive(i); State128& u=states[l]; 
		int m=Derive(i); State128& v=states[m];
		State128& s=states[j];	
		State128& t=states[k];
	
		u=s; v=t;
		u.data[1]=EncodeOne(FNC4, u.context);
		u.data[2]=u.data[1]; // FNC4
		u.data[3]=EncodeOne(d, u.context);
		u.cost+=4; ++u.pos; u.inDblFNC4=!u.inDblFNC4;
		
		v.data[1]=u.data[1];
		v.data[2]=u.data[2];
		v.data[3]=u.data[3];
		v.cost+=4; ++v.pos; v.inDblFNC4=u.inDblFNC4;
		
		openset.push(l); openset.push(m);		
		
		// s & t, single FNC4 escape path
		s.data[1]=u.data[1];
		s.data[2]=u.data[3];
		s.cost+=3;
		
		t.data[1]=v.data[1];
		t.data[2]=v.data[3];
		t.cost+=3;
	}
	++s.pos; ++t.pos;
	openset.push(j); openset.push(k);
}

// returns the combination of CODE_A, CODE_B, CODE_C
// in which @param v can be represented.
//
int  Encoder128::AvailableSet(unsigned v)
{
	ASSERT(v<=FNC4);
	if(v==FNC1) // FNC1 in A, B & C
		return CODE_A | CODE_B | CODE_C;
	if(v>=FNC2 && v<=FNC4)
		return CODE_A | CODE_B;
	if(v>127)
		v-=128;
	if(v<0x20)  // [0, 0x20) only in A
		return CODE_A;
	if(v>=0x60 && v<0x80) // [0x60,0x80) only in B
		return CODE_B;
	return CODE_A | CODE_B;  // The rest in A and B
}


Vector<uint16> Encoder128::Unescape(const char * text)
{
	Vector<uint16> v;
	v.Reserve(2*strlen(text));
	int c;
	while( (c=(unsigned char)*text++)!=0)
	{
		if(c=='&') // escape sequence
		{
			switch(*text++)
			{
			case '0':
				v<<0;
				continue;
			case '1':
				v<<FNC1;  // FNC 1
				continue;
			case '2':
				v<<FNC2;   // FNC 2
				continue;
			case '3':
				v<<FNC3;  // FNC 3
				continue;
			case '4': 
				v<<FNC4;  // FNC 4
				continue;
			case '&':
				v<<'&';
				continue;
			}
			v.Clear(); // invalid escape sequence
			break;
		}else
			v<<(unsigned char)c;
	}
	return v;
}

void Encoder128::Render(Draw& w, const Rect& r, const Code128& c128)
{
	class _:public Code128{
	public:
		using Code128::GetData;
		using Code128::BarCodeWidth;
		using Code128::color;
		using Code128::noQuietZone;
		using Code128::unitWidth;
		using Code128::text;
		using Code128::height;
		using Code128::font;
		using Code128::GetWidth;

	};
	const _&c=*reinterpret_cast<const _*>(&c128);
	int width=c.GetWidth();
	if(width==0)
		return;


	DrawInfo di(w);
	di.unitWidth=r.GetWidth()*c.unitWidth/width;
	if(di.unitWidth==0)
		return;
	Size textSize(0,0);
	if(!c.text.IsEmpty())
	{
		textSize=GetTextSize( c.text, c.font);
		textSize.cy+=2*di.unitWidth;
	}


	ASSERT(c.noQuietZone==0 || c.noQuietZone==1);
	di.left=(r.GetWidth()-c.BarCodeWidth())/2;
	if(c.noQuietZone==0)
		di.left+=10*di.unitWidth; // skip first quiet zone
	di.height=r.GetHeight()-textSize.cy;
	di.top=r.top;
	di.color=c.color;
	for(int i=0; i<c.GetData().GetCount(); ++i)
	{
		DrawValue(di, c.GetData()[i]);
	}
	DrawBar(di, 2); // last bar for stop symbol which we left out
	if(textSize.cy!=0)
	{
		di.draw.DrawText((r.GetWidth()-textSize.cx)/2, di.height+2*c.unitWidth,c.text,c.font,c.color);
	}
		
}

void Encoder128::DrawValue(DrawInfo& di, int v)
{
	static char stroke[][6]={
		{2,1,2,2,2,2}, {2,2,2,1,2,2}, {2,2,2,2,2,1}, {1,2,1,2,2,3},
		{1,2,1,3,2,2}, {1,3,1,2,2,2}, {1,2,2,2,1,3}, {1,2,2,3,1,2},
		{1,3,2,2,1,2}, {2,2,1,2,1,3}, {2,2,1,3,1,2}, {2,3,1,2,1,2},
		{1,1,2,2,3,2}, {1,2,2,1,3,2}, {1,2,2,2,3,1}, {1,1,3,2,2,2},
		{1,2,3,1,2,2}, {1,2,3,2,2,1}, {2,2,3,2,1,1}, {2,2,1,1,3,2},
		{2,2,1,2,3,1}, {2,1,3,2,1,2}, {2,2,3,1,1,2}, {3,1,2,1,3,1},
		{3,1,1,2,2,2}, {3,2,1,1,2,2}, {3,2,1,2,2,1},
		{3,1,2,2,1,2}, {3,2,2,1,1,2}, {3,2,2,2,1,1}, {2,1,2,1,2,3},
		{2,1,2,3,2,1}, {2,3,2,1,2,1}, {1,1,1,3,2,3}, {1,3,1,1,2,3},
		{1,3,1,3,2,1}, {1,1,2,3,1,3}, {1,3,2,1,1,3}, {1,3,2,3,1,1},
		{2,1,1,3,1,3}, {2,3,1,1,1,3}, {2,3,1,3,1,1}, {1,1,2,1,3,3},
		{1,1,2,3,3,1}, {1,3,2,1,3,1}, {1,1,3,1,2,3}, {1,1,3,3,2,1},
		{1,3,3,1,2,1}, {3,1,3,1,2,1}, {2,1,1,3,3,1}, {2,3,1,1,3,1},
		{2,1,3,1,1,3}, {2,1,3,3,1,1}, {2,1,3,1,3,1}, {3,1,1,1,2,3},
		{3,1,1,3,2,1}, {3,3,1,1,2,1}, {3,1,2,1,1,3}, {3,1,2,3,1,1},
		{3,3,2,1,1,1}, {3,1,4,1,1,1}, {2,2,1,4,1,1}, {4,3,1,1,1,1},
		{1,1,1,2,2,4}, {1,1,1,4,2,2}, {1,2,1,1,2,4}, {1,2,1,4,2,1},
		{1,4,1,1,2,2}, {1,4,1,2,2,1}, {1,1,2,2,1,4}, {1,1,2,4,1,2},
		{1,2,2,1,1,4}, {1,2,2,4,1,1}, {1,4,2,1,1,2}, {1,4,2,2,1,1},
		{2,4,1,2,1,1}, {2,2,1,1,1,4}, {4,1,3,1,1,1}, {2,4,1,1,1,2},
		{1,3,4,1,1,1}, {1,1,1,2,4,2}, {1,2,1,1,4,2}, {1,2,1,2,4,1},
		{1,1,4,2,1,2}, {1,2,4,1,1,2}, {1,2,4,2,1,1}, {4,1,1,2,1,2},
		{4,2,1,1,1,2}, {4,2,1,2,1,1}, {2,1,2,1,4,1}, {2,1,4,1,2,1},
		{4,1,2,1,2,1}, {1,1,1,1,4,3}, {1,1,1,3,4,1}, {1,3,1,1,4,1},
		{1,1,4,1,1,3}, {1,1,4,3,1,1}, {4,1,1,1,1,3}, {4,1,1,3,1,1},
		{1,1,3,1,4,1}, {1,1,4,1,3,1}, {3,1,1,1,4,1}, {4,1,1,1,3,1},
		{2,1,1,4,1,2}, {2,1,1,2,1,4}, {2,1,1,2,3,2}, {2,3,3,1,1,1}
	};
	ASSERT(sizeof(stroke)/sizeof(stroke[1])==107);
	ASSERT(v>=0 && v<107);
	DrawBar(di, stroke[v][0]);
	DrawBar(di, stroke[v][1], true);
	DrawBar(di, stroke[v][2]);
	DrawBar(di, stroke[v][3], true);
	DrawBar(di, stroke[v][4]);
	DrawBar(di, stroke[v][5], true);
}


void Encoder128::DrawBar(DrawInfo& di, int cnt, bool blank)
{
	int width=cnt*di.unitWidth;
	if(!blank)
	{
		di.draw.DrawRect(di.left, di.top, width, di.height, di.color);
	}
	di.left+=width;
}

void Code128::Prepare()
{
	DrawingDraw w(1,1);
	if(!data.IsEmpty())
	{
		int h=height;
		if(!text.IsEmpty())
		{
			h+=unitWidth*2+GetTextSize(text, font).cy;
		}
		w.Create(GetWidth(),h);
		Encoder128::Render(w, w.GetSize(), const_cast<const Code128&>(*this));
	}
	const_cast<Code128&>(*this).draw.Create<Drawing>()=w;	
}
static bool prep(const char *text, StringBuffer& sb)
{
	bool haslatin=false;
	const char * first;
	int c;
	
	for(first=text; (c=*first)!='\0' ; ++first)
	{
		if(c<0)
		{
			haslatin=true;
			sb.Cat(text,first);
			break;
		}
	}
	if(haslatin)
	{
		for(;(c=*first)!='\0';++first)
		{
			if(c<0)
			{
				sb.Cat("&4");
				//DUMP(int((unsigned char)(char)c)-128);
				c+=128;
				//DUMP(c);
				if(c==0)
					sb.Cat('&');
			}
			sb.Cat(c);
		}
		//DUMP(sb.Begin());
	}
	// fow now, do it the simple and stupid way
	//
	return haslatin;
}

void Code128::SetText(const char * text)
{
//	const char * t=text;
//	StringBuffer sb;
//	if( prep(text, sb) )
//	{
//		t=sb.Begin();
//	}
	Encoder128 enc(text);	
	data=enc.GetEncoded();
	this->text=text; // will set display text as text by default
	draw.Clear(); // MarkAsDirty
}
