#include "Barcode.h"
//#include <queue>
//#include <algorithm>

using namespace Upp;


typedef PDF417::CodeWord CW; // codeword type
// possible Code128 modes and submodes
enum { M_TEXT, M_TEXT_UP=M_TEXT, M_TEXT_LOW, M_TEXT_MIXED, M_TEXT_PUNC, M_BYTE, M_NUMERIC};
extern "C" extern char TextCodeValue[127][2];

inline int TextMixedValue(int ASCII)
{
	ASSERT(unsigned(ASCII)<128);
	return TextCodeValue[ASCII][0];
}

inline int TextPuncValue(int ASCII)
{
	ASSERT(unsigned(ASCII)<128);
	return TextCodeValue[ASCII][1];
}


namespace{ // a simple, adhoc big number class for PDF417 encoding only
struct BN{
	BN& operator*=(uint32 n);
	BN& operator+=(uint32 n);
	BN& operator/=(uint32 n);
	BN& operator=(uint32 n);
	BN& operator=(const BN& n);
	
	BN& divide(uint32 n, uint32& remainder);
	
	bool operator==(uint32 n)const;
	
	//BN& BN(){}
	//BN& BN(uint32 n){ *this=n; }
	
	String ToString()const;
	uint32 v[5]; // more than enough to hold 45-digit decimals. Highest DWORD on index 4.
	
	
};

BN& BN::operator*=(uint32 n)
{
	uint64 w;
	uint32 carry=0;
	for(int i=0; i<5; ++i)
	{
		w=(uint64)n*(uint64)v[i];
		w+=carry;
		v[i]=(uint32)w;
		carry=w>>32;
	}
	ASSERT(carry==0); // no overflow is expected here
	return *this;
}

BN& BN::divide(uint32 n, uint32& remainder)
{
	uint64 w;
	remainder=0;
	for(int i=4; i>=0; --i)
	{
		w=((uint64)remainder<<32)+(uint64)v[i];
		v[i]=uint32(w/n);
		remainder=w%n;
	}
	return *this;
}

BN& BN::operator/=(uint32 n)
{
	uint32 r;
	return this->divide(n,r);
}
BN& BN::operator+=(uint32 n)
{
	v[0]+=n;
	if(v[0]<n)
	{
		int i=1;
		while(i<5 && ++v[i]==0)
			++i;
		ASSERT(i!=5); // overflow not expected!
	}
	return *this;
}

BN& BN::operator=(uint32 n)
{
	v[0]=n;
	v[1]=v[2]=v[3]=v[4]=0;
	return *this;
}

BN& BN::operator=(const BN& n)
{
	v[0]=n.v[0]; v[1]=n.v[1]; v[2]=n.v[2]; v[3]=n.v[3]; v[4]=n.v[4];
	return *this;
}

bool BN::operator==(uint32 n)const
{
	return v[0]==n && v[1]==0 && v[2]==0
		                && v[3]==0 && v[4]==0;
}

String BN::ToString()const
{
	if(*this==0)
		return String().Cat()<<'0';
	BN n;
	n=*this;
	Vector<uint32> m;
	uint32 r;
	while(!(n==uint32(0)))
	{
		n.divide(1000000000,r);
		m<<r;
	}
	String s;
	s<<m.Top();
	for(int i=m.GetCount()-2; i>=0; --i)
		s<<Format("%09d",int32(m[i]));
	return s;
}


}// eons anomymous

struct Prefix
{
	struct Cost{
		short major;
		short minor;
		short mode ;
		
		void Normalize();
		unsigned CodeWordCount()const;
		Cost& AddMinor(int v){ minor+=v; Normalize(); return *this; }
		Cost& ZeroAsInfinity(){ if(major==0 && minor==0)major=30000; return *this; }
		bool operator <(const Cost& rhs)const;		
	};
	Vector<CW>    major;
	Vector<uint8>  minor;
	Cost GetCost(int mode)const; 
	bool IsEmpty()const{ return major.IsEmpty() && minor.IsEmpty(); }
	Prefix& operator=(const Prefix& p);
	Prefix& WrapText();
	Prefix& WrapByte();
	Prefix& WrapNumeric();
	void WrapBN(BN& n);
	//Prefix& operator=(Prefix&& p);	
	
};
unsigned Prefix::Cost::CodeWordCount()const
{
	ASSERT(mode>=M_TEXT && mode<=M_NUMERIC);
	unsigned m;
	switch(mode)
	{
	case M_BYTE:
		m=minor/6*5+minor%6;
		break;
	case M_NUMERIC:
		{
			unsigned r=minor%44;
			m=minor/44*15+(r==0?0:1+r/3);
		}
		break;
	default: // all text summodes
		m=(minor+1)/2;
	}
	return major+m;
}

// bost rhs and *this should have been Normalized!
//
bool Prefix::Cost::operator <(const Cost& rhs)const
{
	if( (mode<=M_TEXT_PUNC && rhs.mode<=M_TEXT_PUNC) || mode==rhs.mode)
		return major<rhs.major || major==rhs.major && minor<rhs.minor;
	return this->CodeWordCount()<rhs.CodeWordCount();
}


// normalize the returned Cost for easier comparison
void Prefix::Cost::Normalize()
{
	switch(mode)
	{
	case M_BYTE:
		major+=minor/6*5;
		minor%=6;
		break;
	case M_NUMERIC:
		major+=minor/44*15;
		minor%=44;
		break;
	default: // text
		major+=minor/2;
		minor%=2;
		break;
	}
}

// special case, when a prefix is empty, we think it's of
// infinity big cost, means even a bad prefix we arrive at 
// will be regarded as better and thus replace it.
//
// for this purpose, we assign a arbitary big cost for empty entries
//
// note in the start stage, the Text-UpperCase mode indeed have 0 cost
// fortunately we never need to check its cost, so we can safely ignore it.
//
Prefix::Cost Prefix::GetCost(int mode)const
{
	ASSERT(mode>=M_TEXT && mode<=M_NUMERIC);
	Cost c;
	c.mode=mode;
	c.major=major.GetCount();
	c.minor=minor.GetCount();
	
	if(c.minor!=0)
		c.Normalize();
//	else if(c.major==0)
//		c.major=30000; // our chosen number for infinity

	return c;
}

Prefix& Prefix::operator=(const Prefix& p)
{
	major.Clear();
	minor.Clear();
	major.Append(p.major);
	minor.Append(p.minor);
	return *this;
}
// Treat the minor prefix as text and wrap it up.
// caller must guarantee the validity of the treatment
// using it on a Byte or Numeric partial will of course have meaningless results
//
Prefix& Prefix::WrapText()
{
	int i;
	for(i=0; i+1<minor.GetCount(); i+=2)
	{
		ASSERT(minor[i]<30 && minor[i+1]<30);
		major<<minor[i]*30+minor[i+1];
	}
	if(i<minor.GetCount())
	{
		ASSERT(minor[i]<30);
		major<<minor[i]*30+29; // 29 is a special value for all 4 Text submodes 
	}
	minor.Clear();
	return *this;
}

Prefix& Prefix::WrapByte()
{
	if(minor.GetCount()%6==0)
	{
		ASSERT(major.Top()==901);
		major.Top()=924;
	}
	int i;
	for(i=0; i+5<minor.GetCount();i+=6)
	{
		uint64 v=(uint64(minor[i+0])<<40) + (uint64(minor[i+1])<<32) + (uint64(minor[i+2])<<24)
				+(uint64(minor[i+3])<<16) + (uint64(minor[i+4])<< 8) + (uint64(minor[i+5])    );
		int t[5];
		for(int j=0; j<5; ++j)
		{
			t[j]=v%900;	v/=900;
		}
		ASSERT(v==0);
		for(int j=4; j>=0; --j)
			major<<t[j];
	}
	while(i<minor.GetCount())
	{
		major<<minor[i];
		++i;
	}
	minor.Clear();
	return *this;
}


Prefix& Prefix::WrapNumeric()
{
	BN v;
	v=1;
	for(int i=0; i<minor.GetCount();++i)
	{
		v*=10;
		v+=minor[i];
		if(i%44==43){
			WrapBN(v);
			v=1;
		}
	}
	if(!(v==1))
		WrapBN(v);
	minor.Clear();
	return *this;
}

// n will be changed by this function
void Prefix::WrapBN(BN& n)
{
	Vector<uint16> m;
	//DUMP(n);
	uint32 r;
	while(!(n==0))
	{
		n.divide(900,r);
		m<<(uint16)r;
	}
	for(int i=m.GetCount()-1; i>=0;--i)
		major<<m[i];
}

namespace pdf417{

struct Stage
{
	void Process(int c, Stage& next);
	Stage& ProcessTextUp(int c, Stage& next);
	Stage& ProcessTextLow(int c, Stage& next);
	Stage& ProcessTextMixed(int c, Stage& next);
	Stage& ProcessTextPunc(int c, Stage& next);
	Stage& ProcessByte(int c, Stage& next);
	Stage& ProcessNumeric(int c, Stage& next);
	Stage& InitialStage();
	
	void DeriveTextsFromTextUp();
	
	Stage& operator= (Stage&& rhs);
	
	Prefix::Cost Cost(int mode)const{ return prefixes[mode].GetCost(mode);}
	
	Prefix prefixes[6];
	String ToString()const;
	const char * p; // points to next input character to be processed
};

Vector<uint16> ErrorCorrection(const Vector<uint16>& cws, unsigned eclevel, unsigned w, unsigned l);
void CodeWordBarWidths(int codeword, int tableno, int barwidths[8]);
int LeftCW(int row, int rowcnt, int colcnt, int eclevel);
int RightCW(int row, int rowcnt, int colcnt, int eclevel);
void InitTextCodeValueTable();

}// eons pdf417

Vector<CW> encode(const char * text, int len)
{
	static bool inited;
	if(!inited)
	{
		pdf417::InitTextCodeValueTable();
		inited=true;
		
	}
	
	pdf417::Stage cur, next;
	cur.p=text;
	next.p=text+1;
	cur.InitialStage();
	while(cur.p-text<len)
	{
		next.Process(*(uint8*)cur.p, cur);
		cur=std::move(next);
		next.p=cur.p+1;
	}
	int best=0;
	unsigned best_cost=cur.prefixes[0].GetCost(M_TEXT).CodeWordCount();
	for(int i=0; i<6; ++i)
	{
		unsigned c=cur.prefixes[i].GetCost(i).CodeWordCount();
		if(c<best_cost)
		{
			best_cost=c;
			best=i;
		}
	}
	auto& pref=cur.prefixes[best];
	switch(best)
	{
	case M_BYTE:
		pref.WrapByte();
		break;
	case M_NUMERIC:
		pref.WrapNumeric();
		break;
	default:
		pref.WrapText();
	}
	return pref.major;
}

namespace pdf417{
void Stage::Process(int c, Stage& prev)
{
	ProcessTextUp(c, prev).ProcessTextLow(c, prev)
		.ProcessTextMixed(c, prev).ProcessTextPunc(c, prev)
		.ProcessByte(c, prev).ProcessNumeric(c, prev);
}

void Stage::DeriveTextsFromTextUp()
{
	if(Cost(M_TEXT_UP).AddMinor(1)<Cost(M_TEXT_LOW).ZeroAsInfinity())// Text Lower
	{
		prefixes[M_TEXT_LOW]=prefixes[M_TEXT_UP];
		prefixes[M_TEXT_LOW].minor<<(char)27/* LOW */;
	}
	if(Cost(M_TEXT_UP).AddMinor(1)<Cost(M_TEXT_MIXED).ZeroAsInfinity())// Text Mixed
	{
		prefixes[M_TEXT_MIXED]=prefixes[M_TEXT_UP];
		prefixes[M_TEXT_MIXED].minor<<(char)28/* Mix */;
		if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_PUNC).ZeroAsInfinity()) // Text Mixed
		{
			prefixes[M_TEXT_PUNC]=prefixes[M_TEXT_MIXED];
			prefixes[M_TEXT_PUNC].minor<<(char)25/* PUN */;
		}
	}
}

Stage& Stage::ProcessTextUp(int c, Stage& prev)
{
	ASSERT((unsigned)c<256);
	char v;
	bool f=false;
	if( ((c>='A' && c<='Z') || c==' ') && // can be represented in Text-UpperCase mode
	    prev.Cost(M_TEXT_UP).AddMinor(1)<Cost(M_TEXT_UP).ZeroAsInfinity()
	){
		prefixes[M_TEXT_UP]=prev.prefixes[M_TEXT_UP];
		prefixes[M_TEXT_UP].minor<<char(c==' '?26:c-'A');
		f=true;
	}else if( c<127 && (v=TextPuncValue(c))!=-1 &&// can be represented with a shift to punc
		prev.Cost(M_TEXT_UP).AddMinor(2)<Cost(M_TEXT_UP).ZeroAsInfinity()
	){
		ASSERT(unsigned(v)<29);
		prefixes[M_TEXT_UP]=prev.prefixes[M_TEXT_UP];
		prefixes[M_TEXT_UP].minor<<(char)29/* Shift Punc */<<(char)v;
		f=true;
	}
	if(f){ // try to derive other prefixes from the prefix for TextUp we just created
		DeriveTextsFromTextUp();
		if(Cost(M_TEXT_UP).CodeWordCount()+1<Cost(M_BYTE).ZeroAsInfinity().CodeWordCount())// Byte
		{
			prefixes[M_BYTE]=prefixes[M_TEXT_UP];
			prefixes[M_BYTE].WrapText().major<<901;
			// 901 switch to byte sequence that's not a multiple of 6
			// 924 switch to byte sequence that is a multiple of 6
			// we will need to revisit to see if this value need to be change to
			// 924 instead. Note the last major codeword before the byte sequence
			// is wrapped up is always 901, so this work is quite easy.
		}
		if(Cost(M_TEXT_UP).CodeWordCount()+1<Cost(M_NUMERIC).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_NUMERIC]=prefixes[M_TEXT_UP];
			prefixes[M_NUMERIC].WrapText().major<<902/* Switch to Numeric */;
		}
		
	}
	return *this;
}

Stage& Stage::ProcessTextLow(int c, Stage& prev)
{
	ASSERT((unsigned)c<256);
	char v;
	bool f=false;
	if( ((c>='a' && c<='z') || c==' ') && // can be represented in Text-LowerCase mode
	    prev.Cost(M_TEXT_LOW).AddMinor(1)<Cost(M_TEXT_LOW).ZeroAsInfinity()
	){
		prefixes[M_TEXT_LOW]=prev.prefixes[M_TEXT_LOW];
		prefixes[M_TEXT_LOW].minor<<char(c==' '?26:c-'a');
		f=true;
	}else if(c>='A' && c<='Z' &&  // Shift to Upper for UpperCase
		prev.Cost(M_TEXT_LOW).AddMinor(2)<Cost(M_TEXT_LOW).ZeroAsInfinity()        
	){
		prefixes[M_TEXT_LOW]=prev.prefixes[M_TEXT_LOW];
		prefixes[M_TEXT_LOW].minor<<27/*Shift Upp*/<<c-'A';
		f=true;
		
	}else if( c<127 && (v=TextPuncValue(c))!=-1 &&// can be represented with a shift to Punc
		prev.Cost(M_TEXT_LOW).AddMinor(2)<Cost(M_TEXT_LOW).ZeroAsInfinity()
	){
		ASSERT(unsigned(v)<29);
		prefixes[M_TEXT_LOW]=prev.prefixes[M_TEXT_LOW];
		prefixes[M_TEXT_LOW].minor<<(char)29/* Shift Punc */<<(char)v;
		f=true;
	}
	if(f){ // try to derive other prefixes from the prefix for TextUp we just created
		if(Cost(M_TEXT_LOW).AddMinor(1)<Cost(M_TEXT_MIXED).ZeroAsInfinity())// Text Mixed
		{
			prefixes[M_TEXT_MIXED]=prefixes[M_TEXT_LOW];
			prefixes[M_TEXT_MIXED].minor<<(char)28/* Mix */;
			if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_PUNC).ZeroAsInfinity()) // Text Punc
			{
				prefixes[M_TEXT_PUNC]=prefixes[M_TEXT_MIXED];
				prefixes[M_TEXT_PUNC].minor<<(char)25/* PUN */;
			}
			if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_UP).ZeroAsInfinity())// Text Upper
			{
				prefixes[M_TEXT_UP]=prefixes[M_TEXT_MIXED];
				prefixes[M_TEXT_UP].minor<<(char)28/* UPPER */;
			}
		}
		if(Cost(M_TEXT_LOW).CodeWordCount()+1<Cost(M_BYTE).ZeroAsInfinity().CodeWordCount())// Byte
		{
			prefixes[M_BYTE]=prefixes[M_TEXT_LOW];
			prefixes[M_BYTE].WrapText().major<<901;
			// 901 switch to byte sequence that's not a multiple of 6
			// 924 switch to byte sequence that is a multiple of 6
			// we will need to revisit to see if this value need to be change to
			// 924 instead. Note the last major codeword before the byte sequence
			// is wrapped up is always 901, so this work is quite easy.
		}
		if(Cost(M_TEXT_LOW).CodeWordCount()+1<Cost(M_NUMERIC).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_NUMERIC]=prefixes[M_TEXT_UP];
			prefixes[M_NUMERIC].WrapText().major<<902/* Switch to Numeric */;
		}
		
	}
	return *this;
}
Stage& Stage::ProcessTextMixed(int c, Stage& prev)
{
	ASSERT((unsigned)c<256);
	char v;
	bool f=false;
	if( c<127 && (v=TextMixedValue(c))!=-1 && // can be represented in Mixed mode
	    prev.Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_MIXED).ZeroAsInfinity()
	){
		ASSERT(v<25 || v==26);
		prefixes[M_TEXT_MIXED]=prev.prefixes[M_TEXT_MIXED];
		prefixes[M_TEXT_MIXED].minor<<v;
		f=true;
	}else if( c<127 && (v=TextPuncValue(c))!=-1 &&// can be represented with a shift to punc
		prev.Cost(M_TEXT_MIXED).AddMinor(2)<Cost(M_TEXT_MIXED).ZeroAsInfinity()
	){
		ASSERT(unsigned(v)<29);
		prefixes[M_TEXT_MIXED]=prev.prefixes[M_TEXT_MIXED];
		prefixes[M_TEXT_MIXED].minor<<(char)29/* Shift Punc */<<v;
		f=true;
	}
	if(f){ // try to derive other prefixes from the prefix for TextMIXED we just created
		if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_UP).ZeroAsInfinity())// Text UPPER
		{
			prefixes[M_TEXT_UP]=prefixes[M_TEXT_MIXED];
			prefixes[M_TEXT_UP].minor<<(char)28/*UP*/;
		}
		if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_LOW).ZeroAsInfinity())// Text LOWER
		{
			prefixes[M_TEXT_LOW]=prefixes[M_TEXT_MIXED];
			prefixes[M_TEXT_LOW].minor<<(char)27/*LOW*/;
		}
		if(Cost(M_TEXT_MIXED).AddMinor(1)<Cost(M_TEXT_PUNC).ZeroAsInfinity())// Text Puntuation
		{
			prefixes[M_TEXT_PUNC]=prefixes[M_TEXT_MIXED];
			prefixes[M_TEXT_PUNC].minor<<(char)25/*PUNC*/;
		}
		if(Cost(M_TEXT_MIXED).CodeWordCount()+1<Cost(M_BYTE).ZeroAsInfinity().CodeWordCount())// Byte
		{
			prefixes[M_BYTE]=prefixes[M_TEXT_MIXED];
			prefixes[M_BYTE].WrapText().major<<901;
			// 901 switch to byte sequence that's not a multiple of 6
			// 924 switch to byte sequence that is a multiple of 6
			// we will need to revisit to see if this value need to be change to
			// 924 instead. Note the last major codeword before the byte sequence
			// is wrapped up is always 901, so this work is quite easy.
		}
		if(Cost(M_TEXT_MIXED).CodeWordCount()+1<Cost(M_NUMERIC).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_NUMERIC]=prefixes[M_TEXT_MIXED];
			prefixes[M_NUMERIC].WrapText().major<<902/* Switch to Numeric */;
		}
		
	}
	return *this;
}
Stage& Stage::ProcessTextPunc(int c, Stage& prev)
{
	ASSERT((unsigned)c<256);
	char v;
	//bool f=false;	
	if( c<127 && (v=TextPuncValue(c))!=-1 && // can be represented in Punc mode
	    prev.Cost(M_TEXT_PUNC).AddMinor(1)<Cost(M_TEXT_PUNC).ZeroAsInfinity()
	){
		ASSERT(v<29);
		prefixes[M_TEXT_PUNC]=prev.prefixes[M_TEXT_PUNC];
		prefixes[M_TEXT_PUNC].minor<<v;


		// try to derive other prefixes from the prefix for TextMIXED we just created
		if(Cost(M_TEXT_PUNC).AddMinor(1)<Cost(M_TEXT_UP).ZeroAsInfinity())// Text UPPER
		{
			prefixes[M_TEXT_UP]=prefixes[M_TEXT_PUNC];
			prefixes[M_TEXT_UP].minor<<(char)29/*UP*/;
			
			if(Cost(M_TEXT_UP).AddMinor(1)<Cost(M_TEXT_LOW).ZeroAsInfinity())// Text Lower
			{
				prefixes[M_TEXT_LOW]=prefixes[M_TEXT_UP];
				prefixes[M_TEXT_LOW].minor<<(char)27/* LOW */;
			}
			if(Cost(M_TEXT_UP).AddMinor(1)<Cost(M_TEXT_MIXED).ZeroAsInfinity())// Text Mixed
			{
				prefixes[M_TEXT_MIXED]=prefixes[M_TEXT_UP];
				prefixes[M_TEXT_MIXED].minor<<(char)28/* Mix */;
			}
		}
		if(Cost(M_TEXT_PUNC).CodeWordCount()+1<Cost(M_BYTE).ZeroAsInfinity().CodeWordCount())// Byte
		{
			prefixes[M_BYTE]=prefixes[M_TEXT_PUNC];
			prefixes[M_BYTE].WrapText().major<<901;
			// 901 switch to byte sequence that's not a multiple of 6
			// 924 switch to byte sequence that is a multiple of 6
			// we will need to revisit to see if this value need to be change to
			// 924 instead. Note the last major codeword before the byte sequence
			// is wrapped up is always 901, so this work is quite easy.
		}
		if(Cost(M_TEXT_PUNC).CodeWordCount()+1<Cost(M_NUMERIC).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_NUMERIC]=prefixes[M_TEXT_PUNC];
			prefixes[M_NUMERIC].WrapText().major<<902/* Switch to Numeric */;
		}
		
	}
	return *this;
}
Stage& Stage::ProcessByte(int c, Stage& prev)
{
	if(prev.Cost(M_BYTE).AddMinor(1)<Cost(M_BYTE).ZeroAsInfinity())
	{
		prefixes[M_BYTE]=prev.prefixes[M_BYTE];
		prefixes[M_BYTE].minor<<(char)c;
		if(Cost(M_BYTE).CodeWordCount()+1<Cost(M_TEXT_UP).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_TEXT_UP]=prefixes[M_BYTE];
			prefixes[M_TEXT_UP].WrapByte().major<<900/* switch to Text Up */;
			
			DeriveTextsFromTextUp();			
		}
		if(Cost(M_BYTE).CodeWordCount()+1<Cost(M_NUMERIC).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_NUMERIC]=prefixes[M_BYTE];
			prefixes[M_NUMERIC].WrapByte().major<<902/* switch to Numeric */;						
		}
	}
	return *this;
}
Stage& Stage::ProcessNumeric(int c, Stage& prev)
{
	int v=c-'0';
	if(v>=0 && v<=9 && prev.Cost(M_NUMERIC).AddMinor(1)<Cost(M_NUMERIC).ZeroAsInfinity())
	{
		prefixes[M_NUMERIC]=prev.prefixes[M_NUMERIC];
		prefixes[M_NUMERIC].minor<<(char)v;
		if(Cost(M_NUMERIC).CodeWordCount()+1<Cost(M_TEXT_UP).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_TEXT_UP]=prefixes[M_NUMERIC];
			prefixes[M_TEXT_UP].WrapNumeric().major<<900/* switch to Text Up */;
			
			DeriveTextsFromTextUp();			
		}
		if(Cost(M_NUMERIC).CodeWordCount()+1<Cost(M_BYTE).ZeroAsInfinity().CodeWordCount())
		{
			prefixes[M_BYTE]=prefixes[M_NUMERIC];
			prefixes[M_BYTE].WrapNumeric().major<<901/* switch to Byte */;						
		}
	}
	return *this;
}

Stage& Stage::operator= (Stage&& rhs)
{
	for(int i=0; i<6; ++i)
	{
		prefixes[i].major.Clear();
		prefixes[i].minor.Clear();
		Swap(prefixes[i].major,rhs.prefixes[i].major);
		Swap(prefixes[i].minor, rhs.prefixes[i].minor);
		p=rhs.p;
	}
	rhs.p=NULL;
	return *this;
}

Stage& Stage::InitialStage()
{
	prefixes[M_TEXT_LOW].minor<<(char)27;
	prefixes[M_TEXT_MIXED].minor<<(char)28;
	prefixes[M_TEXT_PUNC].minor<<(char)28<<(char)25;
	prefixes[M_BYTE].major<<901;
	// 901 switch to byte sequence that's not a multiple of 6
	// 924 switch to byte sequence that is a multiple of 6
	// we will need to review to see if this value need to be change to
	// 924 instead. Note the last major codeword before the byte sequence
	// is wrapped up is always 901, so this work is quite easy.
	prefixes[M_NUMERIC].major<<902/*switch to Numeric*/;
	return *this;
}


String Stage::ToString()const
{
	String s;
	s<<'['<<p;
	for(int i=0; i<6; ++i)
	{
		s<<"\nmajor=";
		for(int j=0; j<prefixes[i].major.GetCount(); ++j)
			s<<prefixes[i].major[j]<<' ';
		s<<"\nminor=";
		for(int j=0; j<prefixes[i].minor.GetCount(); ++j)
			s<<(int)prefixes[i].minor[j]<<' ';
		s<<'\n';
	}
	s<<"]";
	return s;
}

enum{BY_DATA_COL_COUNT, BY_ROW_COUNT, BY_BOTH, BY_ASPECT_RATIO};// shaping types


}//eons pdf417

NAMESPACE_UPP
// 3 phase
// 1. encoding to Code Words;
// 2. error correction and detection info (has to do with level)
// 3. draw the barcode
void PDF417::SetText(const char *c_, int len)
{
	if(len==-1)
		len=strlen(c_);
	if(len==0)
		return;
	codewords=encode(c_, len);
	if(codewords.GetCount()>925) // too long to be put in a PDF417 barcode
		codewords.Clear();
	MarkAsDirty();
}

PDF417& PDF417::ErrorCorrectionLevel(int level)
{
	if((unsigned)level>8)
		level=-1;
	if(eclevel!=level)
	{
		eclevel=level;
		MarkAsDirty();
	}	
	return *this;	
}

struct DrawInfo
{
	int x,y; // position to draw bar
	int h; // height of the bar
	int tableno; // table no for translation code word to barcode strokes
	int barwidths[8];// 4 bar and 4 spaces
	Color c;
	Draw& w;

	DrawInfo(Draw& w):w(w){}

	DrawInfo& DrawStart();
	DrawInfo& DrawStop();
	DrawInfo& DrawCodeWord(int cw);
	DrawInfo& Bar(int width)
	{
		ASSERT(width!=0 && (unsigned)width<=8);
		w.DrawRect(x,y,width, h, c);
		x+=width;
		return *this;
	}
	DrawInfo& Blank(int width)
	{
		ASSERT(width!=0 && (unsigned)width<=8);
		x+=width;
		return *this;
	}
	
};

void PDF417::Prepare()
{
	DrawingDraw w(1,1);
	Size sz=ActualDataRectangleSize();
	int realec=ActualEC();

	if(!codewords.IsEmpty() && realec>=0)
	{	
		if(sz.cx*sz.cy>=CodeWordCountBeforePadding())
		{
			ec=pdf417::ErrorCorrection(codewords, realec, sz.cx, sz.cy);
		}
		// fixed row height (in units of module width) for now 
		const int rowheight=5;
		w.Create(GetWidth(), rowheight*sz.cy);
		if(noQuietZone!=0)
		{
			w.Offset(2,0);
		}
		// draw start and stop
		DrawInfo di(w);
		di.x=di.y=0;
		di.c=color;
		di.h=rowheight*sz.cy;
		di.DrawStart();
		di.h=rowheight;
		di.tableno=0;
		
//		// dump code words
//		String s;
//		s<<"Code words: ";
//		for(int i=0; i<sz.cx*sz.cy; ++i)
//		{
//			s<<GetCodeWord(i, sz.cx*sz.cy)<<' ';
//		}
//		LOG(s);
		
		for(int r=0; r<sz.cy; ++r)
		{
			di.x=17;						
			di.DrawCodeWord(pdf417::LeftCW(r, sz.cy, sz.cx, realec));
			DUMP(r);
			for(int c=0; c<sz.cx; ++c)
			{
				DUMP(GetCodeWord(r*sz.cx+c, sz.cx*sz.cy));
				di.DrawCodeWord(GetCodeWord(r*sz.cx+c, sz.cx*sz.cy));
			}
			di.DrawCodeWord(pdf417::RightCW(r, sz.cy, sz.cx,realec));
			if(++di.tableno==3)
				di.tableno=0;
			di.y+=rowheight;			
		}
		di.h=rowheight*sz.cy;
		di.y=0;
		di.DrawStop();
		//w.DrawRect(0,0,w.GetSize().cx,w.GetSize().cy, Yellow);
		if(noQuietZone!=0)
		{
			w.End();
		}
	}
	draw.Create<Drawing>()=w;	
}

int PDF417::ActualEC()const
{
	int ecl=eclevel;
	if((unsigned)ecl>8)
	{
		int l=codewords.GetCount();
		ecl=l<40?2:l<160?3:l<320?4:5;
	}
	int maxec=MaximumCodeWordCount()-codewords.GetCount(); // maximum error correction code words count
	if(maxec<2)
		ecl=-1;
	else
		while( (2>>ecl) > maxec )
			--ecl;
	return ecl;	
}


int PDF417::MaximumCodeWordCount()const
{
	int c;
	switch(shapingType)
	{
	case pdf417::BY_DATA_COL_COUNT:
		c=shapingValue*90;
		break;
	case pdf417::BY_ROW_COUNT:
		c=shapingValue*30;
		break;
	case pdf417::BY_BOTH:
		c=((unsigned)shapingValue>>sizeof(int)*4) * (( (unsigned)-1>>sizeof(int)*4) & shapingValue);
		break;
	case pdf417::BY_ASPECT_RATIO:
		{
			unsigned r=(unsigned)shapingValue>>sizeof(int)*4;
			unsigned c=((unsigned)-1>>sizeof(int)*4) & shapingValue;
			ASSERT(r>0 && c>0);
			unsigned f=min(90/r,30/c);
			c=r*c*f*f; 
		}
		break;
	default:
		throw "Not yet implemented!";
	}
	if(c>928)
		c=928;
	
	return --c;
}

int PDF417::GetCodeWord(int i, int count)const
{
	if(i==0)
		return count-ec.GetCount();
	if(i<=codewords.GetCount())
		return codewords[--i];
	count-=i;
	return count<=ec.GetCount()?ec[ec.GetCount()-count]:900/* padding */;	
}

PDF417& PDF417::FixedColumnCount(int col)
{
	if(col<1 || col>30)
	{
		throw "Data column count should be between 1 and 30";
	}
	shapingType=pdf417::BY_DATA_COL_COUNT;
	shapingValue=col;
	return *this;
}

PDF417& PDF417::FixedRowCount(int row)
{
	if(row<3 || row>90)
		throw "Row count should not exceed 90 or be less than 3";
	shapingType=pdf417::BY_ROW_COUNT;
	shapingValue=row;
	return *this; 
}
PDF417& PDF417::FixedRowColCount(int row, int col)
{
	if(row<3 || row>90 || col<1 || col>30)
		throw "Range for column count [1,30], range for row count [3,90].";
	shapingType=pdf417::BY_BOTH;
	shapingValue=(row<<sizeof(int)*4) | col;
	return *this; 
}

static int gcd(int a, int b)
{
	int c;
	while(a!=0)
	{
		c=a; a=b%a;  b=c;
	}
	return b;
}

PDF417& PDF417::FixedAspectRatio(int row, int col)
{
	bool valid=false;
	if(row>0 && col>0){
		int n=gcd(row,col);
		row/=n; col/=n;
		int r=row, c=col;
		while(c<31 && r<91)
		{
			if(r>=3)
			{
				valid=true;
				break;
			}
			c+=col;
			r+=row;
		}		
	}
	if(!valid)
		throw "No valid barcode matrix can be created with given aspect ratio";
	shapingType=pdf417::BY_ASPECT_RATIO;
	shapingValue=(row<<sizeof(int)*4) | col;
	return *this;
}


Size PDF417::ActualDataRectangleSize()const
{
	Size sz;
	int cwcnt=codewords.GetCount()+(2<<ActualEC());
	++cwcnt;
	switch(shapingType)
	{
	case pdf417::BY_DATA_COL_COUNT:
		{
			ASSERT(shapingValue>0 && shapingValue<=30);
			sz.cx=shapingValue;
			sz.cy=(cwcnt+shapingValue-1)/shapingValue;
			if(sz.cy<3)
				sz.cy=3;
			else if(sz.cy>90)
				sz=Null;
		}
		break;
	case pdf417::BY_ROW_COUNT:
		{
			ASSERT(shapingValue>2 && shapingValue<=30);
			sz.cy=shapingValue;
			sz.cx=(cwcnt+shapingValue-1)/shapingValue;
			if(sz.cx>30)
				sz=Null;
		}
		break;
	case pdf417::BY_BOTH:
		{
			sz.cy=(unsigned)shapingValue>>sizeof(int)*4;
			sz.cx=shapingValue & (unsigned(-1)>>sizeof(int)*4);
			ASSERT(sz.cx>0 && sz.cx<31 && sz.cy>2 && sz.cy<91);
		}
		break;
	case pdf417::BY_ASPECT_RATIO:
		{
			unsigned r=(unsigned)shapingValue>>sizeof(int)*4;
			unsigned c=((unsigned)-1>>sizeof(int)*4) & shapingValue;
			ASSERT(r>0 && c>0);
			unsigned unitarea=r*c;
			unsigned scale=(unsigned)sqrt((cwcnt+unitarea-1)/unitarea);
			if( (unsigned)cwcnt > unitarea*scale*scale )
				++scale;
			sz.cx=c*scale;
			sz.cy=r*scale;
			if(sz.cx>30 || sz.cy>90)
				sz=Null;
			else	
				while(sz.cy<3)
					sz.cy+=r, sz.cx+=c;			
		}
		break;
	default:
		throw "Not implemented yet!";
	}
	return sz;
}

int PDF417::GetWidth()const
{
	Size sz=ActualDataRectangleSize();
	return (4/* start, left, right, stop*/+sz.cx)*17/*module per code word*/+1/* 1 extra for stop */
		+(noQuietZone?0:4);
}

void PDF417::Init()
{
	eclevel=-1;
	shapingType=pdf417::BY_DATA_COL_COUNT;
	shapingValue=5;
	Barcode::Init();
}

//String PDF417::ToString()const
//{
//	String s;
//	s<< "total codewords: "<<codewords.GetCount()<<'\n';
//
//	for(int i=0; i<codewords.GetCount(); ++i)
//	{
//		s<<codewords[i]<<',';
//		if((i+1)%10==0)
//		{
//			s<<'\n';
//		}
//	}
//	return s<<'\n';
//	
//}

DrawInfo& DrawInfo::DrawStart()
{
	Bar(8).Blank(1).Bar(1).Blank(1).Bar(1).Blank(1)
		.Bar(1); // not the trailing blank ignored so x is not correctly positioned	
	return *this;	
}
DrawInfo& DrawInfo::DrawStop()
{
	Bar(7).Blank(1).Bar(1).Blank(3)
	.Bar(1).Blank(1).Bar(1).Blank(2).Bar(1);
	return *this;
}

DrawInfo& DrawInfo::DrawCodeWord(int cw)
{
	ASSERT((unsigned)cw<929);
	pdf417::CodeWordBarWidths(cw, tableno, barwidths);
	    
	Bar(barwidths[0]).Blank(barwidths[1]).Bar(barwidths[2]).Blank(barwidths[3])
		.Bar(barwidths[4]).Blank(barwidths[5]).Bar(barwidths[6]).Blank(barwidths[7]);
	return *this;	
}


END_UPP_NAMESPACE