#include "Barcode.h"

using namespace Upp;

class Encoder128
{

public:
	Encoder128(const char* text)//:
	{
		input=Unescape(text);
		Encode();
	}
	
	Vector<uint8>& GetEncoded(){ return output; }
	
	static void Render(Draw& draw, const Rect& r, const Code128& c128);

	enum { CODE_A=1, CODE_B=2, CODE_C=4,
	     FNC1=256, FNC2, FNC3, FNC4 };
private:

	struct DrawInfo{
		Draw& draw;
		int   left;
		int   top;
		int   height;
		int   unitWidth;
		Color color;
		
		DrawInfo(Draw& _draw):draw(_draw){}
	};
	


private:
	void Encode(); // see comments here for description of the algorithum used

	int CheckDigit();
	//void Add(int value){ output<<(char)value; }
	static int  EncodeOne(unsigned ascii, int codeset);
	static Vector<uint16> Unescape(const char * text);
	
	static void DrawValue(DrawInfo& di, int code128value);
	static void DrawBar(DrawInfo& di, int cnt, bool blank=false);
	
	Vector<uint16> input;
	Vector<uint8> output;
};

// this function assumes the ascii can be represented in
// the requested codeset. codeset is used only 
// when a ascii has different representations in different
// code sets.
//
int Encoder128::EncodeOne(unsigned ascii, int codeset)
{
	//ASSERT(AvailableSet(ascii) & codeset);
	switch(ascii)
	{
	case FNC1:
		return 102;
	case FNC2:
		return 97;
	case FNC3:
		return 96;
//	case FNC4:
//		return codeset==CODE_A ? 101 : 100;
	}
	return ((int)ascii+ (ascii<=0x1fu?0x40:-0x20));
}

int Encoder128::CheckDigit()
{
	ASSERT(!output.IsEmpty());
	int check=output[0];
	for(int i=1; i<output.GetCount();++i)
	{
		check+=i*output[i];
		check%=103;
	}
	return check;
}

Vector<uint16> Encoder128::Unescape(const char * text)
{
	Vector<uint16> v;
	v.Reserve(2*strlen(text));
	int c;
	while( (c=(unsigned char)*text++)!=0)
	{
		if(c=='&') // escape sequence
		{
			switch(*text++)
			{
			case '0':
				v<<0;
				continue;
			case '1':
				v<<FNC1;  // FNC 1
				continue;
			case '2':
				v<<FNC2;   // FNC 2
				continue;
			case '3':
				v<<FNC3;  // FNC 3
				continue;
			case '&':
				v<<'&';
				continue;
			}
			v.Clear(); // invalid escape sequence
			break;
		}else
			v<<(unsigned char)c;
	}
	return v;
}

void Encoder128::Render(Draw& w, const Rect& r, const Code128& c128)
{
	class _:public Code128{
	public:
		using Code128::GetData;
		using Code128::BarCodeWidth;
		using Code128::color;
		using Code128::noQuietZone;
		using Code128::unitWidth;
		using Code128::text;
		using Code128::height;
		using Code128::font;
		using Code128::GetWidth;

	};
	const _&c=*reinterpret_cast<const _*>(&c128);
	int width=c.GetWidth();
	if(width==0)
		return;


	DrawInfo di(w);
	di.unitWidth=r.GetWidth()*c.unitWidth/width;
	if(di.unitWidth==0)
		return;
	Size textSize(0,0);
	if(!c.text.IsEmpty())
	{
		textSize=GetTextSize( c.text, c.font);
		textSize.cy+=2*di.unitWidth;
	}


	ASSERT(c.noQuietZone==0 || c.noQuietZone==1);
	di.left=(r.GetWidth()-c.BarCodeWidth())/2;
	if(c.noQuietZone==0)
		di.left+=10*di.unitWidth; // skip first quiet zone
	di.height=r.GetHeight()-textSize.cy;
	di.top=r.top;
	di.color=c.color;
	for(int i=0; i<c.GetData().GetCount(); ++i)
	{
		DrawValue(di, c.GetData()[i]);
	}
	DrawBar(di, 2); // last bar for stop symbol which we left out
	if(textSize.cy!=0)
	{
		di.draw.DrawText((r.GetWidth()-textSize.cx)/2, di.height+2*c.unitWidth,c.text,c.font,c.color);
	}
		
}

void Encoder128::DrawValue(DrawInfo& di, int v)
{
	static char stroke[][6]={
		{2,1,2,2,2,2}, {2,2,2,1,2,2}, {2,2,2,2,2,1}, {1,2,1,2,2,3},
		{1,2,1,3,2,2}, {1,3,1,2,2,2}, {1,2,2,2,1,3}, {1,2,2,3,1,2},
		{1,3,2,2,1,2}, {2,2,1,2,1,3}, {2,2,1,3,1,2}, {2,3,1,2,1,2},
		{1,1,2,2,3,2}, {1,2,2,1,3,2}, {1,2,2,2,3,1}, {1,1,3,2,2,2},
		{1,2,3,1,2,2}, {1,2,3,2,2,1}, {2,2,3,2,1,1}, {2,2,1,1,3,2},
		{2,2,1,2,3,1}, {2,1,3,2,1,2}, {2,2,3,1,1,2}, {3,1,2,1,3,1},
		{3,1,1,2,2,2}, {3,2,1,1,2,2}, {3,2,1,2,2,1},
		{3,1,2,2,1,2}, {3,2,2,1,1,2}, {3,2,2,2,1,1}, {2,1,2,1,2,3},
		{2,1,2,3,2,1}, {2,3,2,1,2,1}, {1,1,1,3,2,3}, {1,3,1,1,2,3},
		{1,3,1,3,2,1}, {1,1,2,3,1,3}, {1,3,2,1,1,3}, {1,3,2,3,1,1},
		{2,1,1,3,1,3}, {2,3,1,1,1,3}, {2,3,1,3,1,1}, {1,1,2,1,3,3},
		{1,1,2,3,3,1}, {1,3,2,1,3,1}, {1,1,3,1,2,3}, {1,1,3,3,2,1},
		{1,3,3,1,2,1}, {3,1,3,1,2,1}, {2,1,1,3,3,1}, {2,3,1,1,3,1},
		{2,1,3,1,1,3}, {2,1,3,3,1,1}, {2,1,3,1,3,1}, {3,1,1,1,2,3},
		{3,1,1,3,2,1}, {3,3,1,1,2,1}, {3,1,2,1,1,3}, {3,1,2,3,1,1},
		{3,3,2,1,1,1}, {3,1,4,1,1,1}, {2,2,1,4,1,1}, {4,3,1,1,1,1},
		{1,1,1,2,2,4}, {1,1,1,4,2,2}, {1,2,1,1,2,4}, {1,2,1,4,2,1},
		{1,4,1,1,2,2}, {1,4,1,2,2,1}, {1,1,2,2,1,4}, {1,1,2,4,1,2},
		{1,2,2,1,1,4}, {1,2,2,4,1,1}, {1,4,2,1,1,2}, {1,4,2,2,1,1},
		{2,4,1,2,1,1}, {2,2,1,1,1,4}, {4,1,3,1,1,1}, {2,4,1,1,1,2},
		{1,3,4,1,1,1}, {1,1,1,2,4,2}, {1,2,1,1,4,2}, {1,2,1,2,4,1},
		{1,1,4,2,1,2}, {1,2,4,1,1,2}, {1,2,4,2,1,1}, {4,1,1,2,1,2},
		{4,2,1,1,1,2}, {4,2,1,2,1,1}, {2,1,2,1,4,1}, {2,1,4,1,2,1},
		{4,1,2,1,2,1}, {1,1,1,1,4,3}, {1,1,1,3,4,1}, {1,3,1,1,4,1},
		{1,1,4,1,1,3}, {1,1,4,3,1,1}, {4,1,1,1,1,3}, {4,1,1,3,1,1},
		{1,1,3,1,4,1}, {1,1,4,1,3,1}, {3,1,1,1,4,1}, {4,1,1,1,3,1},
		{2,1,1,4,1,2}, {2,1,1,2,1,4}, {2,1,1,2,3,2}, {2,3,3,1,1,1}
	};
	ASSERT(sizeof(stroke)/sizeof(stroke[1])==107);
	ASSERT(v>=0 && v<107);
	DrawBar(di, stroke[v][0]);
	DrawBar(di, stroke[v][1], true);
	DrawBar(di, stroke[v][2]);
	DrawBar(di, stroke[v][3], true);
	DrawBar(di, stroke[v][4]);
	DrawBar(di, stroke[v][5], true);
}


void Encoder128::DrawBar(DrawInfo& di, int cnt, bool blank)
{
	int width=cnt*di.unitWidth;
	if(!blank)
	{
		di.draw.DrawRect(di.left, di.top, width, di.height, di.color);
	}
	di.left+=width;
}

void Code128::Prepare()
{
	DrawingDraw w(1,1);
	if(!data.IsEmpty())
	{
		int h=height;
		if(!text.IsEmpty())
		{
			h+=unitWidth*2+GetTextSize(text, font).cy;
		}
		w.Create(GetWidth(),h);
		Encoder128::Render(w, w.GetSize(), const_cast<const Code128&>(*this));
	}
	const_cast<Code128&>(*this).draw.Create<Drawing>()=w;	
}

bool AvailableInCodeA(unsigned v)
{
	return v<0x60 || v>0x80;
}

bool AvailableInCodeB(unsigned v)
{
	return v>=0x20;
}
class Stage
{
public:
	typedef uint16 IChar; // if FN1, FN2, FN3 are to be put in the input, char will not be able to hold them


	Stage& operator=(Stage&& rhs)
	{
		p=rhs.p;
		encoded[CODE_A]=rhs.encoded[CODE_A];
		encoded[CODE_AL]=rhs.encoded[CODE_AL];
		encoded[CODE_B]=rhs.encoded[CODE_B];
		encoded[CODE_BL]=rhs.encoded[CODE_BL];
		encoded[CODE_C]=rhs.encoded[CODE_C];
		rhs.encoded[CODE_A].Clear();
		rhs.encoded[CODE_B].Clear();
		rhs.encoded[CODE_AL].Clear();
		rhs.encoded[CODE_BL].Clear();
		rhs.encoded[CODE_C].Clear();
		return *this;
	}
	
	Stage& InitStage(const IChar * input)
	{
		p=input;
		for(int i=0; i<5; ++i)
			encoded[i].Clear();
		StartA(CODE_A).StartA(CODE_AL).Latin(CODE_AL).StartB(CODE_B).StartB(CODE_BL).Latin(CODE_BL).StartC(CODE_C);
		return *this;
	}
	Stage& Latin(int i)
	{
		ASSERT(i>=CODE_A && i<=CODE_BL);
		char c=i>=CODE_B?B_FN4:A_FN4;
		encoded[i]<<c<<c;
		return *this;
	}
	Stage& StartA(int i)
	{
		ASSERT(i>=CODE_A && i<=CODE_AL && encoded[i].IsEmpty());
		encoded[i]<<(char)START_A;
		return *this;
	}
	Stage& StartB(int i)
	{
		ASSERT(i>=CODE_B && i<=CODE_BL && encoded[i].IsEmpty());
		encoded[i]<<(char)START_B;
		return *this;
	}
	Stage& StartC(int i)
	{
		ASSERT(i==CODE_C && encoded[CODE_C].IsEmpty());
		encoded[i]<<(char)START_C;
		return *this;
	}
	Stage& ToA(int i)
	{
		ASSERT(i>=CODE_A && i<=CODE_AL);
		encoded[i]<<(char)TO_A;
		return *this;
	}
	Stage& ToB(int i)
	{
		ASSERT( i>=CODE_B && i<=CODE_BL);
		encoded[i]<<(char)TO_B;
		return *this;
	}
	Stage& ToC(int i)
	{
		ASSERT(i==CODE_C);
		encoded[i]<<(char)TO_C;
		return *this;
	}
	Stage& Shift(int i)
	{
		ASSERT(( i>=CODE_A && i<=CODE_BL));
		encoded[i]<<(char)SHIFT;
		return *this;
	}
	Stage& ShiftLatin(int i)
	{
		ASSERT( i>=CODE_A && i<=CODE_BL );
		encoded[i]<< (char)(i>CODE_AL?B_FN4:A_FN4);
		return *this;
	}
	
	void SetInput(const IChar * input)
	{
		this->p=input;
	}
	
	char EncodeOne(unsigned ascii)
	{
		switch(ascii)
		{
		case FNC1:
			return 102;
		case FNC2:
			return 97;
		case FNC3:
			return 96;
		}
		return ((int)ascii+ (ascii<=0x1fu?0x40:-0x20));
	}
	
	
	void AcceptInA(const String& prefix, int c, bool latin)
	{
		ASSERT( (unsigned)c<128 || (c>=Encoder128::FNC1 && c<Encoder128::FNC4));
		if(encoded[CODE_A].IsEmpty() || 
			encoded[CODE_A].GetLength()> prefix.GetLength()+
				(AvailableInCodeA(c)?1:2)+latin?1:0
		){
			encoded[CODE_A].Clear();
			encoded[CODE_A]<<prefix;
			
			// Question: in the case when both a shift to A/B and a shift to Latin is required,
			// which one should go first, shift A/B or FNC4????
			// Here we assume shift to Latin should go first.  
			if(latin)
				ShiftLatin(CODE_A);
			if(!AvailableInCodeA(c))
				Shift(CODE_A);
			encoded[CODE_A]<<EncodeOne(c);
			
			if(encoded[CODE_AL].IsEmpty() ||
			   encoded[CODE_AL].GetLength()>prefix.GetLength()+2
			){
				encoded[CODE_AL].Clear();
				encoded[CODE_AL]<<encoded[CODE_A];
				Latin(CODE_AL);
				
			}
			
			if(encoded[CODE_B].IsEmpty() ||
				encoded[CODE_B].GetLength()>encoded[CODE_A].GetLength()+1			
			)
			{
				encoded[CODE_B].Clear();
				encoded[CODE_B]<<encoded[CODE_A];
				ToB(CODE_B);
				if(encoded[CODE_BL].IsEmpty() ||
					encoded[CODE_BL].GetLength()>encoded[CODE_B].GetLength()+2			
				)
				{
					encoded[CODE_BL].Clear();
					encoded[CODE_BL]<<encoded[CODE_B];
					Latin(CODE_BL);
				}
			}
			if(encoded[CODE_C].IsEmpty() || encoded[CODE_C].GetLength()>encoded[CODE_A].GetLength()+1)
			{
				encoded[CODE_C].Clear();
				encoded[CODE_C]<<encoded[CODE_A];
				ToC(CODE_C);
			}
	
		}
	}

	void AcceptInAL(const String& prefix, int c, bool latin)
	{
		ASSERT( (unsigned)c<128 || (c>=Encoder128::FNC1 && c<Encoder128::FNC4));
		if(encoded[CODE_AL].IsEmpty() || 
			encoded[CODE_AL].GetLength()> prefix.GetLength()+
				(AvailableInCodeA(c)?1:2)+(latin?0:1)
		){
			encoded[CODE_AL].Clear();
			encoded[CODE_AL]<<prefix;
			if(!latin)
				ShiftLatin(CODE_AL);
			if(!AvailableInCodeA(c))
				Shift(CODE_AL);
			encoded[CODE_AL]<<	EncodeOne(c);
			
			if(encoded[CODE_A].IsEmpty() ||
			   encoded[CODE_A].GetLength()>encoded[CODE_AL].GetLength()+2
			){ // set the prefix for context A from that in context A latin if it gains.
				encoded[CODE_A].Clear();
				encoded[CODE_A]<<encoded[CODE_AL];
				Latin(CODE_A);
				
			}
			
			if(encoded[CODE_BL].IsEmpty() ||
				encoded[CODE_BL].GetLength()>encoded[CODE_AL].GetLength()+1			
			)
			{
				encoded[CODE_BL].Clear();
				encoded[CODE_BL]<<encoded[CODE_AL];
				ToB(CODE_BL);
				if(encoded[CODE_B].IsEmpty() ||
					encoded[CODE_B].GetLength()>encoded[CODE_BL].GetLength()+2			
				)
				{
					encoded[CODE_B].Clear();
					encoded[CODE_B]<<encoded[CODE_BL];
					Latin(CODE_B);
				}
			}
			if(encoded[CODE_C].IsEmpty() || encoded[CODE_C].GetLength()>encoded[CODE_BL].GetLength()+1)
			{
				encoded[CODE_C].Clear();
				encoded[CODE_C]<<encoded[CODE_BL];
				ToC(CODE_C);
			}
	
		}
	}


	void AcceptInB(const String& prefix, int c, bool latin)
	{
		ASSERT( (unsigned)c<128 || (c>=Encoder128::FNC1 && c<Encoder128::FNC4));
		if(encoded[CODE_B].IsEmpty() || 
			encoded[CODE_B].GetLength()> prefix.GetLength()+
				(AvailableInCodeB(c)?1:2)+latin?1:0
		){
			encoded[CODE_B].Clear();
			encoded[CODE_B]<<prefix;
			if(latin)
				ShiftLatin(CODE_B);
			if(!AvailableInCodeB(c))
				Shift(CODE_B);
			encoded[CODE_B]<< EncodeOne(c);
			
			if(encoded[CODE_BL].IsEmpty() ||
			   encoded[CODE_BL].GetLength()>prefix.GetLength()+2
			){
				encoded[CODE_BL].Clear();
				encoded[CODE_BL]<<encoded[CODE_B];
				Latin(CODE_BL);
				
			}
			
			
			if(encoded[CODE_A].IsEmpty() ||
				encoded[CODE_A].GetLength()>encoded[CODE_B].GetLength()+1			
			)
			{
				encoded[CODE_A].Clear();
				encoded[CODE_A]<<encoded[CODE_B];
				ToA(CODE_A);
				if(encoded[CODE_AL].IsEmpty() ||
					encoded[CODE_AL].GetLength()>encoded[CODE_A].GetLength()+2			
				)
				{
					encoded[CODE_AL].Clear();
					encoded[CODE_AL]<<encoded[CODE_A];
					Latin(CODE_AL);
				}
			}
			if(encoded[CODE_C].IsEmpty() || encoded[CODE_C].GetLength()>encoded[CODE_B].GetLength()+1)
			{
				encoded[CODE_C].Clear();
				encoded[CODE_C]<<encoded[CODE_B];
				ToC(CODE_C);
			}
	
		}
	}


	void AcceptInBL(const String& prefix, int c, bool latin)
	{
		ASSERT( (unsigned)c<128 || (c>=Encoder128::FNC1 && c<Encoder128::FNC4));
		if(encoded[CODE_BL].IsEmpty() || 
			encoded[CODE_BL].GetLength()> prefix.GetLength()+
				(AvailableInCodeB(c)?1:2)+(latin?0:1)
		){
			encoded[CODE_BL].Clear();
			encoded[CODE_BL]<<prefix;
			if(!latin)
				ShiftLatin(CODE_BL);
			if(!AvailableInCodeB(c))
				Shift(CODE_BL);
			encoded[CODE_BL]<<	EncodeOne(c);
			
			if(encoded[CODE_B].IsEmpty() ||
			   encoded[CODE_B].GetLength()>encoded[CODE_BL].GetLength()+2
			){ 
				encoded[CODE_B].Clear();
				encoded[CODE_B]<<encoded[CODE_BL];
				Latin(CODE_B);
				
			}
			
			if(encoded[CODE_AL].IsEmpty() ||
				encoded[CODE_AL].GetLength()>encoded[CODE_BL].GetLength()+1			
			)
			{
				encoded[CODE_AL].Clear();
				encoded[CODE_AL]<<encoded[CODE_BL];
				ToA(CODE_AL);
				if(encoded[CODE_A].IsEmpty() ||
					encoded[CODE_A].GetLength()>encoded[CODE_AL].GetLength()+2			
				)
				{
					encoded[CODE_A].Clear();
					encoded[CODE_A]<<encoded[CODE_AL];
					Latin(CODE_A);
				}
			}
			if(encoded[CODE_C].IsEmpty() || encoded[CODE_C].GetLength()>encoded[CODE_BL].GetLength()+1)
			{
				encoded[CODE_C].Clear();
				encoded[CODE_C]<<encoded[CODE_BL];
				ToC(CODE_C);
			}
	
		}
	}


	void AcceptInC(const String& prefix, int two_digits_value)
	{
		ASSERT( (two_digits_value>=0 && two_digits_value<100) || two_digits_value==Encoder128::FNC1);
		if(encoded[CODE_C].IsEmpty() || encoded[CODE_C].GetLength()>prefix.GetLength()+1)
		{
			encoded[CODE_C].Clear();
			encoded[CODE_C]<<prefix<<(char)two_digits_value;
			// context C always goes 1 character faster than other context, so we know for sure
			// all the rest prefixes are empty
			encoded[CODE_A]<<encoded[CODE_C];
			ToA(CODE_A);
			encoded[CODE_AL]<<encoded[CODE_A];
			Latin(CODE_AL);
			encoded[CODE_B]<<encoded[CODE_C];
			ToB(CODE_B);
			encoded[CODE_BL]<<encoded[CODE_C];
			Latin(CODE_BL);
		
		}
	}
	
	void Accept(IChar c, IChar c2, Stage& prev, Stage& next)
	{
		//Stage::IChar c=*cur.P();
		bool latin=false;
		if(c>127 && c<256)
		{
			c-=127;
			latin=true;
		}
				
		AcceptInA(prev.encoded[CODE_A],   c, latin);
		AcceptInAL(prev.encoded[CODE_AL], c, latin);
		AcceptInB(prev.encoded[CODE_B],   c, latin);
		AcceptInBL(prev.encoded[CODE_BL], c, latin);
		
		
		if( c>='0' && c<='9' && c2>='0' && c2<='9')
			next.AcceptInC(prev.encoded[CODE_C], (c-'0')*10+c2-'0');
		else if(c==Encoder128::FNC1)
			AcceptInC(prev.encoded[CODE_C], c);
	}
	
	const IChar * P(){ return p; }
	
	
	String ToString()const
	{
		String s;
		s<<'['<<p<<",\n";
		for(int i=0; i<5; ++i)
		{
			for(int j=0; j<encoded[i].GetLength(); ++j)
				s<<(int)encoded[i][j]<<' ';
			s<<"\n";
		}
		s<<"]\n";
		return s;
	}
	
//	static int CheckDigit(const String& encoded)
//	{
//		ASSERT(!encoded.IsEmpty());
//		int check=encoded[0];
//		for(int i=1; i<encoded.GetLength();++i)
//		{
//			check+=i*encoded[i];
//			check%=103;
//		}
//		return check;
//	}

	
//	static String Encode(const IChar* input, int len);
//private:
	const IChar * p; // points to next input word
	
	enum {CODE_A, CODE_AL, CODE_B, CODE_BL, CODE_C,
		SHIFT=98, TO_A=101, TO_B=100, TO_C=99, START_A=103, START_B, START_C, A_FN4=101, B_FN4=100,
		STOP=106,
		FNC1=256, FNC2, FNC3 // these value are used to allow passing FNC1, FNC2, FNC3 in input string to the encoder. 
	
	};
	String encoded[5];  // index: Context
						//     0: Code A
						//     1: Code A, latin
						//	   2: Code B
						//	   3: Code B, latin
						//     4: Code C
	// length of string is the cost of prefix, so no need extra variables for that purpose.
	// note, empty string (length==0) signals en undefined entry.
};


void Encoder128::Encode()
{
	Stage cur, next, next2;
	
	cur.InitStage(input.Begin());
    next.SetInput(input.Begin()+1);
    next2.SetInput(input.Begin()+2);
	while(cur.P()-input.Begin()<input.GetCount())
	{
		auto c=cur.p[0];
		auto c2=0;
		if(cur.p-input.Begin()<input.GetCount()-1)
			c2=cur.p[1];
		next.Accept(c,c2, cur, next2);
		
		cur=std::move(next);
		next=std::move(next2);
		next2.SetInput(cur.P()+2);
		
		ASSERT(cur.P()!=nullptr);
	}
	int best=0;
	for(int i=1; i<5; ++i)
	{
		if(cur.encoded[i]<cur.encoded[best])
			best=i;
	}
	String& s=cur.encoded[best];
	output.Clear();
	for(int i=0; i<s.GetLength(); ++i)
	{
		output<<s[i];
	}
	output<<(char)CheckDigit()<<(char)Stage::STOP;
}


void Code128::SetText(const char * text, int len)
{
	Encoder128 enc(text);	
	data=enc.GetEncoded();
	this->text=text; // will set display text as text by default
	draw.Clear(); // MarkAsDirty
}
