#ifndef _DragTest_DragTemplate_h_
#define _DragTest_DragTemplate_h_

#include <CtrlLib/CtrlLib.h>

using namespace Upp;

#define DRAG_DELAY 200

// Virtual class for storing/finding all possible drag-drop targets.
class DragTargetBase
{
public:
	static Ctrl *GetTargetFromMouse()
	{
		// Checks all registered targets against the mouse position.
		// First checks if the target is visible and contains the mouse pointer,
		// then if successful checks if it a child of the window that the OS thinks the mouse
		// is over (this is to eliminate overlapping windows)
		
		Ctrl *wnd = NULL, *c = NULL;
		Point p = GetMousePos();
		
		for (int i = 0; i < targets.GetCount(); i++) {
			c = targets[i];
			if (c->IsVisible() && c->GetScreenView().Contains(p)) {
				// If we don't have the target window yet get it now
				if (!wnd) {
					wnd = GetWindowFromPoint(p);
					if (!wnd) return NULL;
				}
				
				// Is the target control on this window?
				while (c != NULL) {
					if (c == wnd) return targets[i];
					c = c->GetParent();	
				}
			}
		}
		return NULL;
	}
	
protected:	
	// Target ctrl storage
	static Vector<Ctrl *> targets;
	
	static Ctrl *GetWindowFromPoint(Point p)
	{
		// Invokes native OS API call(s) to determine which native window is under p, then
		// searches the app windows to find a match
		
		Vector<Ctrl *> windows = Ctrl::GetTopWindows();
#ifdef flagWIN32
		POINT winpt;
		winpt.x = p.x;
		winpt.y = p.y;
		
		HWND hwnd = WindowFromPoint(winpt);
		
		for (int i = 0; i < windows.GetCount(); i++)
			if (windows[i]->GetHWND() == hwnd) {
				return windows[i];
			}
#elif flagLINUX
		return NULL;
		// Broken
		Window Xroot, Xchild;
		int root_x = 0, root_y = 0, child_x = 0, child_y = 0;
		unsigned int key_mask = 0;
		
		if (XQueryPointer(Xdisplay, windows[0]->GetWindow(), &Xroot, &Xchild, &root_x, &root_y, &child_x, &child_y, &key_mask)) {
	    	for (int i = 0; i < windows.GetCount(); i++) 		
				if (windows[i]->GetWindow() == Xroot) {
					return windows[i];
				}	                     
	    }
#endif		
		return NULL;
	}		
	
	// Storing/removing of possible drag-drop targets
	void RegisterDragTarget(Ctrl *c) {if (GetTargetIndex(c) < 0) targets.Add(c);}
	void DeregisterDragTarget(Ctrl *c) {int i; if ((i = GetTargetIndex(c)) >= 0) targets.Remove(i);}
	
	int GetTargetIndex(Ctrl *c) 
	{
		// Finds the index in the target vector of the given ctrl
		for (int i = 0; i <	targets.GetCount(); i++)
			if (targets[i] == c)
				return i;
		return -1;
	}
};

// Class to allow any Ctrl or derived class to accept (ONLY) data via drag drop. 
template <class T>
class DragTarget : public T, public DragTargetBase
{
public:
	DragTarget() : T()
		{DragTargetBase::RegisterDragTarget(this);}
	~DragTarget()
		{DragTargetBase::DeregisterDragTarget(this);}
};

// Class to allow any Ctrl or derived class to send (ONLY) data via drag drop. 
template <class T>
class DragSource : public T
{
public:
	typedef DragSource CLASSNAME;
	
	DragSource() {is_dragging = false;}

protected:
	bool is_dragging;
	
	virtual void LeftDown(Point p, dword keyflags)
	{
		T::LeftDown(p, keyflags);
		
		// Initiate drag timer and capture mouse
		T::SetTimeCallback(DRAG_DELAY, THISBACK(StartDrag), 1);
		T::SetCapture();
	}
	
	virtual void LeftUp(Point p, dword keyflags)
	{
		// Kill timer
		T::KillTimeCallback(1);
		if (!is_dragging)
			return T::LeftUp(p, keyflags);
			
		// Are we over a drag-drop target ctrl?
		Ctrl *c = DragTargetBase::GetTargetFromMouse();
		if (c) {
			// Copy data
			Value v = T::GetData();
			c->SetData(v);	
		}
		
		// End drag operation
		is_dragging = false;
		T::ReleaseCapture();
	}

	virtual void MouseMove(Point p, dword keyflags) {if (!is_dragging) return T::MouseMove(p, keyflags);}
	virtual void RightUp(Point p, dword keyflags) {if (!is_dragging) return T::RightUp(p, keyflags);}
	virtual void RightDown(Point p, dword keyflags) {if (!is_dragging) return T::RightDown(p, keyflags);}
	virtual Image CursorImage(Point p, dword keyflags) {
		// If dragging change the mouse pointer. Ideally this should be context dependant on 
		// what type of data is being dragged
		if (is_dragging)
			return CtrlImg::exclamation();
		else
			return Image::Arrow();
	}
	
	void StartDrag() {is_dragging = true;}
};

// Class to allow any Ctrl or derived class to send OR accept data via drag drop. 
template <class T>
class DragSourceTarget : public DragSource<T>, public DragTargetBase
{
public:
	DragSourceTarget()
		{DragTargetBase::RegisterDragTarget(this);}
	~DragSourceTarget()
		{DragTargetBase::DeregisterDragTarget(this);}	
};

#endif
