#ifndef _DragTest_DragTemplateHook_h_
#define _DragTest_DragTemplateHook_h_

#include <CtrlLib/CtrlLib.h>

using namespace Upp;

#define DRAG_DELAY_H 200

bool DragHook(Ctrl *ctrl, bool inframe, int event, Point p, int zdelta, dword keyflags);

// Virtual class for storing/finding all possible drag-drop targets.
class DragTargetBaseH
{
public:
	static bool IsTarget(Ctrl *c)
	{
		// Determines whether the passed ctrl is a registered target
		if (c) {
			for (int i = 0; i < targets.GetCount(); i++) {
				if (targets[i] == c) return true;
			}
		}		
		return false;
	}
	
protected:	
	// Target ctrl storage
	static Vector<Ctrl *> targets;
	
	static Ctrl *GetWindowFromPoint(Point p)
	{
		// Invokes native OS API call(s) to determine which native window is under p, then
		// searches the app windows to find a match
		
		Vector<Ctrl *> windows = Ctrl::GetTopWindows();
#ifdef flagWIN32
		POINT winpt;
		winpt.x = p.x;
		winpt.y = p.y;
		
		HWND hwnd = WindowFromPoint(winpt);
		
		for (int i = 0; i < windows.GetCount(); i++)
			if (windows[i]->GetHWND() == hwnd) {
				return windows[i];
			}
#elif flagLINUX
		return NULL;
		// Broken
		Window Xroot, Xchild;
		int root_x = 0, root_y = 0, child_x = 0, child_y = 0;
		unsigned int key_mask = 0;
		
		if (XQueryPointer(Xdisplay, windows[0]->GetWindow(), &Xroot, &Xchild, &root_x, &root_y, &child_x, &child_y, &key_mask)) {
	    	for (int i = 0; i < windows.GetCount(); i++) 		
				if (windows[i]->GetWindow() == Xroot) {
					return windows[i];
				}	                     
	    }
#endif		
		return NULL;
	}		
	
	// Storing/removing of possible drag-drop targets
	void RegisterDragTarget(Ctrl *c) {if (GetTargetIndex(c) < 0) targets.Add(c);}
	void DeregisterDragTarget(Ctrl *c) {int i; if ((i = GetTargetIndex(c)) >= 0) targets.Remove(i);}
	
	int GetTargetIndex(Ctrl *c) 
	{
		// Finds the index in the target vector of the given ctrl
		for (int i = 0; i <	targets.GetCount(); i++)
			if (targets[i] == c)
				return i;
		return -1;
	}
};

// Class to allow any Ctrl or derived class to accept (ONLY) data via drag drop. 
template <class T>
class DragTargetH : public T, public DragTargetBaseH
{
public:
	DragTargetH() : T()
		{DragTargetBaseH::RegisterDragTarget(this);}
	~DragTargetH()
		{DragTargetBaseH::DeregisterDragTarget(this);}
		
	virtual Image CursorImage(Point p, dword keyflags) {
		// If dragging change the mouse pointer. Ideally this should be context dependant on 
		// what type of data is being dragged
		if (DragSourceBaseH::source_ctrl)
			return CtrlImg::exclamation();
		else
			return Image::Arrow();
	}		
};

// Additional Base class is required for DragSourceH because we need a single static var across
//	template compilation
class DragSourceBaseH 
{
public:
	static void EndDrag(Ctrl *target)
	{
		if (target && DragSourceBaseH::source_ctrl && DragTargetBaseH::IsTarget(target)) {
			Value v = DragSourceBaseH::source_ctrl->GetData();
			target->SetData(v);
		}
		Ctrl::DeinstallMouseHook(&DragHook);
		DragSourceBaseH::source_ctrl = NULL;	
	}	
	static Ctrl *source_ctrl; // Stores the source control during drag drop operations
};

// Class to allow any Ctrl or derived class to send (ONLY) data via drag drop. 
template <class T>
class DragSourceH : public T, public DragSourceBaseH
{
public:
	typedef DragSourceH CLASSNAME;
	
	DragSourceH() { }

protected:
	virtual void LeftDown(Point p, dword keyflags)
	{
		T::LeftDown(p, keyflags);
		// Start drag timer and capture mouse
		T::SetTimeCallback(DRAG_DELAY_H, THISBACK(StartDrag), 1);
	}
	
	virtual void LeftUp(Point p, dword keyflags)
	{
		// Kill timer
		T::KillTimeCallback(1);
		return T::LeftUp(p, keyflags);
	}

	virtual Image CursorImage(Point p, dword keyflags) {
		// If dragging change the mouse pointer. Ideally this should be context dependant on 
		// what type of data is being dragged
		if (source_ctrl)
			return CtrlImg::exclamation();
		else
			return Image::Arrow();
	}
	
	void StartDrag() {
		T::ReleaseCapture(); // In case the template Ctrl (T) sets it
		source_ctrl = this;
		Ctrl::InstallMouseHook(&DragHook);
	}
};

// Class to allow any Ctrl or derived class to send OR accept data via drag drop. 
template <class T>
class DragSourceTargetH : public DragSourceH<T>, public DragTargetBaseH
{
public:
	DragSourceTargetH()
		{DragTargetBaseH::RegisterDragTarget(this);}
	~DragSourceTargetH()
		{DragTargetBaseH::DeregisterDragTarget(this);}	
};

#endif
