#ifndef _CORE_alt_MTalt_h_
#define _CORE_alt_MTalt_h_

#ifndef flagMT
	#error Please add 'MT' to build flags
#endif

#include <Core/Core.h>
using namespace Upp;

dword rdtsc();

#ifndef __PRETTY_FUNCTION__
	#define __PRETTY_FUNCTION__ __FUNCTION__
#endif

#ifdef _DEBUG
	struct SourcePosition : Moveable<SourcePosition>
	{
		String file;
		int    line;
		SourcePosition() {}
		SourcePosition(const char *_file, int _line) :file(_file), line(_line) {}
		unsigned GetHashValue() const {return CombineHash(file,line);}
		bool operator== (const SourcePosition &op) const {return (line == op.line) && (file == op.file);}
	};
	struct LatencyData : Moveable<LatencyData>
	{
		dword minV, maxV;
		qword meanSum;
		qword meanCount;
		BiVector<dword> latestV;
		LatencyData() :minV(-1),maxV(0),meanSum(0),meanCount(0) {}
	};

	#define DEBUG_REQUEST __PRETTY_FUNCTION__,__LINE__,
	#define DEBUG_THREAD
#else
	#define DEBUG_REQUEST
	#define DEBUG_THREAD
#endif

class CallbackQueue : private NoCopy
{
public:
	CallbackQueue(int _maxQueueLength = 100000) 
		:shuttingDown(false)
		,started(false)
		,maxQueueLength(_maxQueueLength)
	{
		#ifdef _DEBUG
		lastReported = GetTickCount();
		#endif
	}
	
	virtual ~CallbackQueue()
	{
		qMutex.Enter();
		queue.Clear();
		qMutex.Leave();
		
		#ifdef _DEBUG
		ReportStats(true);
		#endif
	}
	
	#define LOCK_ADD_UNLOCK \
		bool addSem = true; \
		qMutex.Enter(); \
		if (queue.GetCount() >= maxQueueLength) \
		{ \
			queue.Remove(0); \
			addSem = false; \
			LOG(Format("MtAlt: queue is too big (this:%s)", Format64Hex((uint64)this))); \
		} \
		if (queue.IsEmpty()) \
			queue.Add(e); \
		else \
		{ \
			if (queue[queue.GetCount()-1].priority >= e->priority) \
				queue.Add(e); \
			else \
			{ \
				if (queue[0].priority < e->priority) \
					queue.Insert(0, e); \
				else \
				{ \
					int i = FindUpperBound(queue, 0, queue.GetCount(), *e, order); \
					if (i < 0) \
						queue.Add(e); \
					else \
						queue.Insert(i,e); \
				} \
			} \
		} \
		qMutex.Leave(); \
		if (addSem) \
			qSemaphore.Release(); 

	void ClearQueue()
	{
		qMutex.Enter();
		queue.Clear();
		qMutex.Leave();
	}
	
	#ifdef _DEBUG
		#define __DEBUG_E \
			if (file && line >= 0) \
			{ \
				e->requestTime = GetTickCount(); \
				e->requestPosition = SourcePosition(file,line); \
			}
	#else
		#define __DEBUG_E
	#endif
	
	#ifdef _DEBUG
	template<class OBJECT>
	void Request(void (OBJECT::*m)(), OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT>(NULL,-1, m,obj,pr);
	}
	template<class OBJECT>
	void Request(const char *file, int line, void (OBJECT::*m)(), OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT>
	void Request(void (OBJECT::*m)(), OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element0<OBJECT> *e = new Element0<OBJECT>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1>
	void Request(void (OBJECT::*m)(pick_ P1 &), pick_ P1 &p1, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1>(NULL,-1,m,p1,obj,pr);
	}
	template<class OBJECT, class P1>
	void Request(const char *file, int line, void (OBJECT::*m)(pick_ P1 &), pick_ P1 &p1, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1>
	void Request(void (OBJECT::*m)(pick_ P1 &), pick_ P1 &p1, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element1<OBJECT, P1> *e = new Element1<OBJECT, P1>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1>
	void Request(void (OBJECT::*m)(P1), P1 p1, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1>(NULL,-1,m,p1,obj,pr);
	}
	template<class OBJECT, class P1>
	void Request(const char *file, int line, void (OBJECT::*m)(P1), P1 p1, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1>
	void Request(void (OBJECT::*m)(P1), P1 p1, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element1c<OBJECT, P1> *e = new Element1c<OBJECT, P1>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2>
	void Request(void (OBJECT::*m)(pick_ P1 &,pick_ P2 &), pick_ P1 &p1, pick_ P2 &p2, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2>(NULL,-1,m,p1,p2,obj,pr);
	}
	template<class OBJECT, class P1, class P2>
	void Request(const char *file, int line, void (OBJECT::*m)(pick_ P1 &,pick_ P2 &), pick_ P1 &p1, pick_ P2 &p2, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2>
	void Request(void (OBJECT::*m)(pick_ P1 &,pick_ P2 &), pick_ P1 &p1, pick_ P2 &p2, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element2<OBJECT, P1, P2> *e = new Element2<OBJECT, P1, P2>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2>
	void Request(void (OBJECT::*m)(P1,P2), P1 p1, P2 p2, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2>(NULL,-1,m,p1,p2,obj,pr);
	}
	template<class OBJECT, class P1, class P2>
	void Request(const char *file, int line, void (OBJECT::*m)(P1,P2), P1 p1, P2 p2, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2>
	void Request(void (OBJECT::*m)(P1,P2), P1 p1, P2 p2, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element2c<OBJECT, P1, P2> *e = new Element2c<OBJECT, P1, P2>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3>(NULL,-1,m,p1,p2,p3,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3>
	void Request(const char *file, int line, void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element3<OBJECT, P1, P2, P3> *e = new Element3<OBJECT, P1, P2, P3>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3>
	void Request(void (OBJECT::*m)(P1,P2,P3), P1 p1, P2 p2, P3 p3, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3>(NULL,-1,m,p1,p2,p3,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3>
	void Request(const char *file, int line, void (OBJECT::*m)(P1,P2,P3), P1 p1, P2 p2, P3 p3, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3>
	void Request(void (OBJECT::*m)(P1,P2,P3), P1 p1, P2 p2, P3 p3, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element3c<OBJECT, P1, P2, P3> *e = new Element3c<OBJECT, P1, P2, P3>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3,P4>(NULL,-1,m,p1,p2,p3,p4,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(const char *file, int line, void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element4<OBJECT, P1, P2, P3, P4> *e = new Element4<OBJECT, P1, P2, P3, P4>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->p4 = p4;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(void (OBJECT::*m)(P1,P2,P3,P4), P1 p1, P2 p2, P3 p3, P4 p4, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3,P4>(NULL,-1,m,p1,p2,p3,p4,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(const char *file, int line, void (OBJECT::*m)(P1,P2,P3,P4), P1 p1, P2 p2, P3 p3, P4 p4, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3, class P4>
	void Request(void (OBJECT::*m)(P1,P2,P3,P4), P1 p1, P2 p2, P3 p3, P4 p4, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element4c<OBJECT, P1, P2, P3, P4> *e = new Element4c<OBJECT, P1, P2, P3, P4>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->p4 = p4;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &, pick_ P5 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, pick_ P5 &p5, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3,P4,P5>(NULL,-1,m,p1,p2,p3,p4,p5,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(const char *file, int line, void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &, pick_ P5 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, pick_ P5 &p5, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(void (OBJECT::*m)(pick_ P1 &, pick_ P2 &, pick_ P3 &, pick_ P4 &, pick_ P5 &), pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, pick_ P5 &p5, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element5<OBJECT, P1, P2, P3, P4, P5> *e = new Element5<OBJECT, P1, P2, P3, P4, P5>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->p4 = p4;
		e->p5 = p5;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	#ifdef _DEBUG
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(void (OBJECT::*m)(P1,P2,P3,P4,P5), P1 p1, P2 p2, P3 p3, P4 p4, P5 p5, OBJECT *obj = NULL, int pr = 0)
	{
		Request<OBJECT,P1,P2,P3,P4,P5>(NULL,-1,m,p1,p2,p3,p4,p5,obj,pr);
	}
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(const char *file, int line, void (OBJECT::*m)(P1,P2,P3,P4,P5), P1 p1, P2 p2, P3 p3, P4 p4, P5 p5, OBJECT *obj = NULL, int pr = 0)
	#else
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	void Request(void (OBJECT::*m)(P1,P2,P3,P4,P5), P1 p1, P2 p2, P3 p3, P4 p4, P5 p5, OBJECT *obj = NULL, int pr = 0)
	#endif
	{
		Element5c<OBJECT, P1, P2, P3, P4, P5> *e = new Element5c<OBJECT, P1, P2, P3, P4, P5>;
		__DEBUG_E;
		e->caller = obj ? obj : (OBJECT *) this;
		e->m  = m;
		e->p1 = p1;
		e->p2 = p2;
		e->p3 = p3;
		e->p4 = p4;
		e->p5 = p5;
		e->priority = pr;
		LOCK_ADD_UNLOCK;
	}
	
	void DoTasks()
	{
		while (!IsShutdown())
			if (!Execute())
				return;
	}
	
	virtual void OnBeforeTask() {}
	virtual void OnAfterTask()  {}
	void WaitDoTasksInf()
	{
		while (!IsShutdown())
		{
			qSemaphore.Wait();
			OnBeforeTask();
			Execute();
			OnAfterTask();	
		}
	}

	void SetMaxQueueLength(int mql) {qMutex.Enter(); maxQueueLength = mql; while (queue.GetCount() >= mql) queue.Remove(0); qMutex.Leave();}
	int GetTasksCount() {qMutex.Enter(); int out=queue.GetCount(); qMutex.Leave(); return out;}

	bool IsShutdown() {return shuttingDown;}
	bool IsStarted() {return started;}
	
	virtual void Start() {started = true;}
	virtual void Shutdown()
	{
		if (!IsStarted() || IsShutdown())
			return;
		shuttingDown = true;
		Request(&CallbackQueue::DoNothingJustAwake);
	}

	void ClearShutdown() { shuttingDown = false; }
	
private:
	static VectorMap<int, Vector<CallbackQueue *> > queues;
	bool Execute()
	{
		if (queue.IsEmpty())
			return false;
		
		qMutex.Enter();
		Element *e = queue.Detach(0);
		qMutex.Leave();
		
		#ifdef _DEBUG
		dword rt = GetTickCount() - e->requestTime;
		if (rt && !e->requestPosition.file.IsEmpty())
		{
			LatencyData &ld = latency.GetAdd(e->requestPosition);
			ld.meanSum += (qword) rt;
			++ld.meanCount;
			if (rt > ld.maxV)
				ld.maxV = rt;
			if (rt < ld.minV)
				ld.minV = rt;
			
			ld.latestV.AddTail(rt);
			if (ld.latestV.GetCount() > 1000)
				ld.latestV.DropHead(ld.latestV.GetCount() - 1000);
		}

		ReportStats();
		#endif
		
		e->Execute();
		delete e;
		return true;
	}
	
	struct Element
	{
		#ifdef _DEBUG
		dword requestTime;
		SourcePosition requestPosition;
		#endif
		Element()
		#ifdef _DEBUG
			:requestTime(0)
		#endif
		{}
		
		virtual ~Element() {}
		virtual void Execute() = 0;
		void *caller;
		int   priority;
	};
	
	template<class OBJECT>
	struct Element0 : public Element
	{
		virtual ~Element0() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)();}
		void (OBJECT::*m)();
	};
	
	template<class OBJECT, class P1>
	struct Element1 : public Element
	{
		virtual ~Element1() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1);}
		void (OBJECT::*m)(pick_ P1 &p1);
		P1 p1;
	};
	
	template<class OBJECT, class P1>
	struct Element1c : public Element
	{
		virtual ~Element1c() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1);}
		void (OBJECT::*m)(P1 p1);
		P1 p1;
	};
	
	template<class OBJECT, class P1, class P2>
	struct Element2 : public Element
	{
		virtual ~Element2() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2);}
		void (OBJECT::*m)(pick_ P1 &p1, pick_ P2 &p2);
		P1 p1;
		P2 p2;
	};
	
	template<class OBJECT, class P1, class P2>
	struct Element2c : public Element
	{
		virtual ~Element2c() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2);}
		void (OBJECT::*m)(P1 p1, P2 p2);
		P1 p1;
		P2 p2;
	};
	
	template<class OBJECT, class P1, class P2, class P3>
	struct Element3 : public Element
	{
		virtual ~Element3() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3);}
		void (OBJECT::*m)(pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3);
		P1 p1;
		P2 p2;
		P3 p3;
	};
	
	template<class OBJECT, class P1, class P2, class P3>
	struct Element3c : public Element
	{
		virtual ~Element3c() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3);}
		void (OBJECT::*m)(P1 p1, P2 p2, P3 p3);
		P1 p1;
		P2 p2;
		P3 p3;
	};
	
	template<class OBJECT, class P1, class P2, class P3, class P4>
	struct Element4 : public Element
	{
		virtual ~Element4() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3,p4);}
		void (OBJECT::*m)(pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4);
		P1 p1;
		P2 p2;
		P3 p3;
		P4 p4;
	};
	
	template<class OBJECT, class P1, class P2, class P3, class P4>
	struct Element4c : public Element
	{
		virtual ~Element4c() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3,p4);}
		void (OBJECT::*m)(P1 p1, P2 p2, P3 p3, P4 p4);
		P1 p1;
		P2 p2;
		P3 p3;
		P4 p4;
	};
	
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	struct Element5 : public Element
	{
		virtual ~Element5() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3,p4,p5);}
		void (OBJECT::*m)(pick_ P1 &p1, pick_ P2 &p2, pick_ P3 &p3, pick_ P4 &p4, pick_ P5 &p5);
		P1 p1;
		P2 p2;
		P3 p3;
		P4 p4;
		P5 p5;
	};
	
	template<class OBJECT, class P1, class P2, class P3, class P4, class P5>
	struct Element5c : public Element
	{
		virtual ~Element5c() {}
		virtual void Execute() {((static_cast<OBJECT *>(caller))->*m)(p1,p2,p3,p4,p5);}
		void (OBJECT::*m)(P1 p1, P2 p2, P3 p3, P4 p4, P5 p5);
		P1 p1;
		P2 p2;
		P3 p3;
		P4 p4;
		P5 p5;
	};
	
	struct Order
	{
		bool operator()(const Element& v1, const Element& v2) const
		{
			return v1.priority > v2.priority;
		}
	};
	static Order order;
	
	void DoNothingJustAwake(){}

	Array<Element>   queue;
	Mutex            qMutex;
	Semaphore        qSemaphore;
	bool             shuttingDown;
	bool			 started;
	int              maxQueueLength;
	
	#ifdef _DEBUG
	void ReportStats(bool ignoreTimeout = false)
	{
		if (latency.IsEmpty())
			return;
		
		dword t = GetTickCount();
		if (!ignoreTimeout && t - lastReported < 1000*60*60*24)
			return;
		
		lastReported = t;
		
		for (int i=0; i<latency.GetCount(); ++i)
		{
			const SourcePosition &sp = latency.GetKey(i);
			const LatencyData    &ld = latency[i];
			
			String latestS;
			int count = ld.latestV.GetCount();
			if (count >= 20)
			{
				while (count > 1)
				{
					qword latestSum = 0;
					
					for (int latestI=ld.latestV.GetCount()-1-count; latestI<=ld.latestV.GetCount()-1; ++latestI)
						latestSum += ld.latestV[latestI];
					
					latestS += "..." + FormatUnsigned((unsigned long) (latestSum / ((qword) count)));
					
					count /= 10;
				}
			}
			
			LOG(Format("REQUEST: @ %40s(%-4d): [min=%-5d | mean=%-5d (called:%-6d) | max=%-5d] %s",
				sp.file,
				sp.line,
				(int) ld.minV,
				(int) (ld.meanSum / ld.meanCount),
				(int) ld.meanCount,
				(int) ld.maxV,
				latestS
			));
		}
	}
	
	dword lastReported;
	VectorMap<SourcePosition,LatencyData> latency;
	#endif
};

class CallbackThread : public CallbackQueue, protected Thread
{
public:
	CallbackThread(int maxQueueLength = 100000)  :CallbackQueue(maxQueueLength) {}
	virtual ~CallbackThread()
	{
		Shutdown();
	}

	virtual void Start()
	{
		CallbackQueue::Start();
		Thread::Run(callback(this, &CallbackThread::Run));
	}
	
	virtual void Shutdown()
	{
		if (!IsStarted() || IsShutdown())
			return;
		CallbackQueue::Shutdown();
		Wait();
	}

private:	
	void Run() {WaitDoTasksInf();}
};

#endif
