#ifdef flagPOSIX
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <string.h>
#include "crossserial.h"


typedef struct
{
    speed_t baudOut;
    int32_t baudIn;
} baudSelect_t;

static const baudSelect_t baudSelect[] = {
    {B50, 50}, {B75, 75}, {B110, 110}, {B134, 134}, {B150, 150}, {B200, 200},
    {B300, 300}, {B600, 600}, {B1200, 1200}, {B1800, 1800}, {B2400, 2400},
    {B4800, 4800}, {B9600, 9600}, {B19200, 19200}, {B38400, 38400},
    {B57600, 57600}, {B115200, 115200}};


serialError_t SerialOpen(serialPort_t *port, const char *name)
{
    int32_t fd;

    /* open port */
    fd = open(name, O_RDWR | O_NOCTTY);

    /* failed to open */
    if (fd == -1)
    {
        port->portHandle = 0;
        return SERIAL_FAIL;
    }

    /* get current options */
    tcgetattr(port->portHandle, &(port->optionsOld));
    
    /* return handle to port */
    port->portHandle = fd;

    return SERIAL_OK;
}


serialError_t SerialClose(serialPort_t *port)
{
    /* restore the options to their original state */
    tcsetattr(port->portHandle, TCSANOW, &(port->optionsOld));
    
    /* close the port */
    if (close(port->portHandle) == -1)
    {
        return SERIAL_FAIL;
    }

    return SERIAL_OK;
}


serialError_t SerialSettings(serialPort_t *port, uint32_t baud, char parity,
                             uint8_t dataBits, uint8_t stopBits,
                             int32_t timeout_tenths)
{
    struct termios &options = port->options;
    speed_t actual_baud;
    int32_t numBaud;
    int32_t i;

    /* clear termios struct */
    memset(&options, 0, sizeof(options));

    /* check baud is valid */
    numBaud = sizeof(baudSelect) / sizeof(baudSelect_t);
    for (i = 0; i < numBaud; i++)
    {
        if (baud == baudSelect[i].baudIn)
        {
            actual_baud = baudSelect[i].baudOut;
            break; 
        }
    }
    if (i == numBaud)
    {
        return SERIAL_BAD_BAUD;
    }

    /* set port speed */
    cfsetispeed(&options, baud);
    cfsetospeed(&options, baud);

    /* check and set parity */
    switch (parity)
    {
        case 'n':
            options.c_cflag &= ~PARENB;
            break;

        case 'o':
            options.c_cflag |= PARENB;
            options.c_cflag |= PARODD;
            break;

        case 'e':
            options.c_cflag |= PARENB;
            options.c_cflag &= ~PARODD;
            break;

        default:
            return SERIAL_BAD_PARITY;
    }

    /* check and set data bits */
    options.c_cflag &= ~CSIZE; 
    switch (dataBits)
    {
        case 5:
            options.c_cflag |= CS5;
            break;

        case 6:
            options.c_cflag |= CS6;
            break;

        case 7:
            options.c_cflag |= CS7;
            break;

        case 8:
            options.c_cflag |= CS8;
            break;

        default:
            return SERIAL_BAD_DATABITS;
    }

    /* check and set stop bits */
    switch (stopBits)
    {
        case 1:
            options.c_cflag &= ~CSTOPB;
            break;

        case 2:
            options.c_cflag |= CSTOPB;
            break;

        default:
            return SERIAL_BAD_STOPBITS;
    }

    /* set timeouts */
    options.c_cc[VMIN] = 0;
    options.c_cc[VTIME] = timeout_tenths;
    
    options.c_cflag |= CLOCAL | CREAD;

    /* set options for port */
    if (tcsetattr(port->portHandle, TCSAFLUSH, &options) != 0)
    {
        return SERIAL_FAIL;
    }

    return SERIAL_OK;
}

serialError_t SerialTimeouts(serialPort_t *port, int32_t timeout_tenths)
{
	options.c_cc[VMIN] = 0;
	options.c_cc[VTIME] = timeout_tenths;
	
	if (tcsetattr(port->portHandle, TCSAFLUSH, &options) != 0)
		return SERIAL_FAIL;
	
	return SERIAL_OK;
}

serialError_t SerialRead(serialPort_t *port, uint8_t *data, int32_t size, int32_t *bytesRead)
{
    int32_t readCount = 0;

    *bytesRead = 0;

    /* loop reading bytes from port until all bytes are read or there is a timeout*/
    while (*bytesRead != size)
    {
        /* get bytes from port */
        readCount = read(port->portHandle, data + *bytesRead, size - *bytesRead);
        *bytesRead += readCount;

        /* if no bytes read timeout has occured */
        if (readCount == 0)
        {
            return SERIAL_TIMEOUT;
        }
    }
    
    return SERIAL_OK;
}


serialError_t SerialWrite(serialPort_t *port, const uint8_t *data, int32_t size, int32_t *bytesWritten)
{
    /* write bytes */
    *bytesWritten = write(port->portHandle, data, size);

    /* check all bytes were written */
    if (*bytesWritten != size)
    {
        return SERIAL_WRITE_INCOMPLETE;
    }
    
    return SERIAL_OK;
}


serialError_t SerialEnumerate(void)
{
    return SERIAL_NOT_IMPLEMENTED;
}
#endif