#ifndef _BNF_BNF_h_
#define _BNF_BNF_h_

#include <Core/Core.h>
using namespace Upp;

#ifndef BNF_DUMP
#define BNF_DUMP(a) //LOG(a) //Cout() << (a)
#endif

class BNFelement;
class BNFarticle;

enum {BNF_TERMINAL, BNF_NONTERMINAL, BNF_CODE, BNF_LINK, BNF_UNITOR, BNF_SPLITTER, BNF_BEGIN, BNF_END, BNF_BEND};
enum {DIR_RIGHT, DIR_LEFT, DIR_UP, DIR_DOWN};
struct ParseState
{
	String inData;
	bool   stop;        

	struct State : Moveable<State>
	{
		BNFelement    *ptr;
		int            inOffset;
		Vector<Value>  values;

		enum {REGISTER_COUNT = 16};
		Vector<int>          ri[REGISTER_COUNT];
		Vector<String>       rs[REGISTER_COUNT];
		Vector<Value>        rv[REGISTER_COUNT];
		Vector<BNFelement *> re[REGISTER_COUNT];
		
		State()
			:ptr        (NULL)
			,inOffset   (0)
		{}
		State(const State &op)
			:ptr        (op.ptr)
			,inOffset   (op.inOffset)
			,values     (op.values ,1)
		{
			for (int i=0; i<REGISTER_COUNT; ++i)
			{
				ri[i] <<= op.ri[i];
				rs[i] <<= op.rs[i];
				re[i] <<= op.re[i];
				rv[i] <<= op.rv[i];
			}
		}
		State & operator=(const State &op)
		{
			ptr        = op.ptr;
			inOffset   = op.inOffset;
			values   <<= op.values;
			
			for (int i=0; i<REGISTER_COUNT; ++i)
			{
				ri[i] <<= op.ri[i];
				rs[i] <<= op.rs[i];
				re[i] <<= op.re[i];
				rv[i] <<= op.rv[i];
			}
			
			return *this;
		}
		void Clear()
		{
			ptr = NULL;
			inOffset = 0;
			values.Clear();
		}
	};
	
	State state;
	ParseState() :stop(false) {}
};

struct GenState
{
	Vector<Value> inValues;

	struct State : Moveable<State>
	{
		bool           firstEnter;
		BNFelement    *ptr;
		int            inOffset;
		String         outData;

		enum {REGISTER_COUNT = 16};
		Vector<int>          ri[REGISTER_COUNT];
		Vector<String>       rs[REGISTER_COUNT];
		Vector<Value>        rv[REGISTER_COUNT];
		Vector<BNFelement *> re[REGISTER_COUNT];
		
		State()
			:firstEnter (true)
			,ptr        (NULL)
			,inOffset   (0)
		{}
		State(const State &op)
			:firstEnter (op.firstEnter)
			,ptr        (op.ptr)
			,inOffset   (op.inOffset)
			,outData    (op.outData)
		{
			for (int i=0; i<REGISTER_COUNT; ++i)
			{
				ri[i] <<= op.ri[i];
				rs[i] <<= op.rs[i];
				re[i] <<= op.re[i];
				rv[i] <<= op.rv[i];
			}
		}
		State & operator=(const State &op)
		{
			firstEnter = op.firstEnter;
			ptr        = op.ptr;
			inOffset   = op.inOffset;
			outData    = op.outData;
			
			for (int i=0; i<REGISTER_COUNT; ++i)
			{
				ri[i] <<= op.ri[i];
				rs[i] <<= op.rs[i];
				re[i] <<= op.re[i];
				rv[i] <<= op.rv[i];
			}
			
			return *this;
		}
	};

	State state;
	Vector<State> stack;
};

inline WString DumpifyString(const String &s)
{
	WString ws = s.ToWString();
	for (int i=0; i<ws.GetLength(); ++i)
		if (((word)ws[i]) < 0x20)
			ws.Set(i, '.');
	
	return "\""+ws+"\"";
}

inline String DumpifyValue(const Value &v)
{
	if (v.Is<int>())
		return FormatInt(static_cast<int>(v));
	else if (v.Is<String>())
		return DumpifyString(static_cast<String>(v)).ToString();
	else if (v.Is<double>())
		return FormatDouble(static_cast<double>(v));
	else
		return "???";
}

namespace Upp
{
template<> void   XmlAttrLoad(Uuid& var, const String& text);
template<> String XmlAttrStore(const Uuid& var);
}
class BNFelement
{
public:
	BNFelement()
		:debugID(++debugIDcounter)
		,type(-1), x(-1), y(-1), l(-1), dir(-1), dir2(-1)
		{links[0]=NULL; links[1]=NULL; inputs[0]=NULL; inputs[1]=NULL;}
	virtual ~BNFelement() {}
	
	virtual bool Parse(ParseState *state) = 0;
	virtual bool Gen  (GenState   *state) = 0;

	virtual String GetName() const {return "";}
	virtual void   SetName(const String &s) {}
	virtual int FirstLinkIndex() {return 0;}
	
	virtual BNFelement *GetParentSplitter() {return NULL;}
	
	void UpdateInput(BNFelement *oldE, BNFelement* newE) {if (inputs[0]==oldE) inputs[0]=newE; else if (inputs[1]==oldE) inputs[1]=newE; else ASSERT(false);}
	void UpdateLink (BNFelement *oldE, BNFelement* newE) {if (links [0]==oldE) links [0]=newE; else if (links [1]==oldE) links [1]=newE; else ASSERT(false);}
	
	virtual void Dump()
		{/*Cout()<<Format("%2d in[%2d %2d] out[%2d %2d] ", 
			(int)debugID, inputs[0]?inputs[0]->debugID:0, inputs[1]?inputs[1]->debugID:0, links[0]?links[0]->debugID:0, links[1]?links[1]->debugID:0
			,(int)inputs[0], (int)inputs[1], (int)links[0], (int)links[1]);*/}
	
	void XmlizeLink(XmlIO &xml, const char *tag, BNFelement * &link)
	{
		if (xml.IsStoring())
		{
			int i = link ? link->debugID : 0;
			xml.Attr(tag, i);
		}
		else
		{
			int i = (int) NULL;
			xml.Attr(tag, i);
			link = (BNFelement *) i;
		}
	}
	
	virtual void Xmlize(XmlIO &xml)
	{
		xml.Attr("type",type).Attr("i",debugID).Attr("x",x,-1).Attr("y",y,-1).Attr("l",l,-1).Attr("dir",dir,-1).Attr("dir2",dir2,-1);
		XmlizeLink(xml, "input0", inputs[0]);
		XmlizeLink(xml, "input1", inputs[1]);
		XmlizeLink(xml, "link0",  links [0]);
		XmlizeLink(xml, "link1",  links [1]);
		
		if (xml.IsLoading() && debugIDcounter <= debugID)
			debugIDcounter = debugID + 1;
	}
	
	BNFelement *RealizeLink(BNFelement *id, const ArrayMap<dword, BNFelement> &elements)
	{
		if (!id)
			return NULL;
		
		for (int i=0; i<elements.GetCount(); ++i)
			if (elements[i].debugID == (int)id)
				return const_cast<BNFelement *>(&elements[i]);
			
		return NULL;
	}
	
	virtual void RealizeLinksOnLoad(const ArrayMap<dword, BNFelement> &elements)
	{
		inputs[0] = RealizeLink(inputs[0], elements);
		inputs[1] = RealizeLink(inputs[1], elements);
		links[0]  = RealizeLink(links [0], elements);
		links[1]  = RealizeLink(links [1], elements);
	}
	
	int type;
	int x,y,l,dir,dir2;
	BNFelement *links[2], *inputs[2];
	int debugID;
	
private:
	static int debugIDcounter;
};

class BNFterminal : public BNFelement
{
public:
	BNFterminal(const String &value);
	virtual String GetName() const {return value;}
	virtual void   SetName(const String &s) {value = s;}
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump()
	{
		BNFelement::Dump(); 
		BNF_DUMP(DumpifyString(value));
	}
	virtual void Xmlize(XmlIO &xml)
	{
		BNFelement::Xmlize(xml);
		xml.Attr("value", value);
	}

private:
	String value;
};

class BNFnonterminal : public BNFelement
{
public:
	BNFnonterminal(const String &nspace, const Uuid &id);
	virtual String GetName() const;
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump();
	virtual void Xmlize(XmlIO &xml)
	{
		BNFelement::Xmlize(xml);
		xml.Attr("nspace", nspace).Attr("uid", aid);
		if (xml.IsLoading()) SetName(GetName());
	}
	
	Uuid    aid;
	String nspace;
private:
};

class BNFunitor;
class BNFsplitter : public BNFelement
{
public:
	BNFsplitter(bool up);
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() {BNFelement::Dump(); String s=up?"^":"v"; BNF_DUMP(s+"["+FormatInt(links[1] ? links[1]->debugID : 0)+"]");}
	virtual int FirstLinkIndex() {return up ? 1 : 0;}
	
	bool InvertUp() {return up;}
	virtual void Xmlize(XmlIO &xml)
	{
		BNFelement::Xmlize(xml);
		xml.Attr("up", up);
	}
	
private:
	bool up;
};

class BNFunitor : public BNFelement
{
public:
	BNFunitor(BNFelement *splitter);
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() {BNFelement::Dump(); BNF_DUMP(String("+")+"["+FormatInt(inputs[1] ? inputs[1]->debugID : 0)+"]");}
	virtual BNFelement *GetParentSplitter() {return splitter;}
	virtual void Xmlize(XmlIO &xml)
	{
		BNFelement::Xmlize(xml);
		XmlizeLink(xml,"splitter",splitter);
	}
	virtual void RealizeLinksOnLoad(const ArrayMap<dword, BNFelement> &elements)
	{
		BNFelement::RealizeLinksOnLoad(elements);
		splitter = RealizeLink(splitter, elements);
	}
private:
	BNFelement *splitter;
};

class BNFcode : public BNFelement
{
public:
	BNFcode(const String &id, Value v = 0);
	void SetupID(const String &name) {id = name;}
	virtual String GetName() const {return id;}
	virtual void   SetName(const String &s) {id=s;}
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() 
	{
		BNFelement::Dump(); 
		BNF_DUMP(id);
		if (v.Is<int>() && ((int)v) != 0)
			{ BNF_DUMP(":"+(int)v);}
		else if (v.Is<String>())
			{ BNF_DUMP(":"+DumpifyString((String)v)); }
		else if (v.Is<double>())
			{ BNF_DUMP("f"); }
	}
	virtual void Xmlize(XmlIO &xml)
	{
		BNFelement::Xmlize(xml);
		xml
			.Attr("id", id)
			("v", v)
		;
	}
	
	void  SetArg(Value _v) {v=_v;}
	Value GetArg() const   {return v;}
private:
	String id;
	Value v;
};

class BNFbend : public BNFelement
{
public:
	BNFbend();
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual int FirstLinkIndex() {return 0;}
	virtual void Dump() {BNFelement::Dump();}
private:
};

class BNFlink : public BNFelement
{
public:
	BNFlink();
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() {BNFelement::Dump(); BNF_DUMP(Format("-%d-",debugID));}
private:
};

class BNFbegin : public BNFelement
{
public:
	BNFbegin();
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() {BNFelement::Dump(); BNF_DUMP("o-");}
private:
};

class BNFend : public BNFelement
{
public:
	BNFend();
	virtual bool Parse(ParseState *state);
	virtual bool Gen  (GenState   *state);
	virtual void Dump() {BNFelement::Dump(); BNF_DUMP("-o");}
private:
};

class BNFarticle
{
public:
	BNFarticle(bool init = true);
	virtual ~BNFarticle();

	bool Parse(ParseState *state);
	bool Gen  (GenState   *state);
	void Dump();
	
	BNFarticle &operator>=(int);             //добавить пустую связь
	BNFarticle &operator> (const char *);    //добавить нетерминал
	BNFarticle &operator> (const BNFcode &); //добавить процедуру
	BNFarticle &operator>=(const char *);    //добавить терминал
	BNFarticle &operator>=(bool);            //точка начала развилки
	BNFarticle &operator<=(bool);            //начало развилки
	BNFarticle &operator<=(int);             //окончание развилки
	BNFarticle &operator>=(char);            //сдвинуть курсор на конец последней ветки
	BNFarticle &operator<=(char);            //сдвинуть курсор на начало последней ветки
	
	BNFelement *AppendLink   (BNFelement *topEl);
	BNFelement *AppendElement(BNFelement *topEl, BNFelement *newEl);
	BNFelement *AppendBranch (bool up, BNFelement *startEl, BNFelement *endEl, BNFelement **newStartEl, BNFelement **newEndEl);
	
	String GetName() {return name;}
	
	Uuid    uid;
	String  name;
	WString comment, desc;

	void Xmlize(XmlIO &xml);

protected:
	BNFelement & AddElement(BNFelement *);
	ArrayMap<dword, BNFelement> elements;
	
private:
	BNFelement *begin;
	
	BNFelement *curBranchStart, *curBranchEnd;
	BNFelement *outer;
	Vector<BNFelement *> branchOuters;
};
typedef bool (* BNFParseProc)(ParseState *, Value v);
typedef bool (* BNFGenProc  )(GenState   *, Value v);

BNFarticle *BNFAddArticle(const String &nspace, const String &s, const String &desc = "");
BNFarticle *BNFAddArticle(const String &nspace, const String &s, const String &desc, const Uuid &uid);
void CreateDefaultBNFFunctions();

typedef ArrayMap<Uuid, BNFarticle>     BNFarticles;

extern ArrayMap<String, BNFarticles>  bnfNamespaces;
extern ArrayMap<String, BNFParseProc> bnfParseProcs;
extern ArrayMap<String, BNFGenProc>   bnfGenProcs;

#define NAMESPACE_BNF "BNF"

extern BNFarticle *startArticle;
extern BNFelement *startElement;

#endif
