#include "BNF.h"

#ifdef _DEBUG
	#define LOG_(a) //LOG(a) //Cout() << a
	#define out(a) //LOG(a) //Cout() << a
#else
	#define LOG_(a)
	#define out(a)
#endif

ArrayMap<String, BNFarticles>  bnfNamespaces;
ArrayMap<String, BNFParseProc> bnfParseProcs;
ArrayMap<String, BNFGenProc>   bnfGenProcs;

//---------------------------------------------------------------------------------------------- TERMINAL
BNFterminal::BNFterminal(const String &v)
	:value(v)
{
	type = BNF_TERMINAL;
}

bool BNFterminal::Parse(ParseState *state)
{
	out(" "+DumpifyString(value));
	if (state->state.inOffset+value.GetLength() > state->inData.GetLength())
	{
		out("-");
		return false;
	}
	
	if (state->inData.Mid(state->state.inOffset, value.GetLength()) == value)
	{
		state->state.inOffset += value.GetLength();
		out("+");
		return true;
	}
	
	out("-");
	return false;
}

bool BNFterminal::Gen(GenState *state)
{
	state->state.outData << value;
	return true;
}

//---------------------------------------------------------------------------------------------- NONTERMINAL
BNFnonterminal::BNFnonterminal(const String &n, const Uuid &_id)
	:aid(_id)
	,nspace(n)
{
	type = BNF_NONTERMINAL;
}

String BNFnonterminal::GetName() const
{
	int i = bnfNamespaces.Find(nspace);
	if (i<0)
		return String();
	int i2 = bnfNamespaces[i].Find(aid);
	if (i2<0)
		return String();
	return bnfNamespaces[i][i2].name; 
}

bool BNFnonterminal::Parse(ParseState *state)
{
	int i = bnfNamespaces.Find(nspace);
	if (i<0)
	{
		LOG_("Не найдено пространство имён "+nspace);
		out("!");
		return true;
	}
	int i2 = bnfNamespaces[i].Find(aid);
	if (i2<0)
	{
		LOG_("Не найдена словарная статья "+aid.ToStringWithDashes()+" в пространстве имён "+nspace);
		out("!");
		return true;
	}

	out(Format(" <%s>", bnfNamespaces[i][i2].name));
	
	BNFarticle &article = bnfNamespaces[i][i2];
	
	//создаём новое состояние с пустым стеком
	ParseState articleState;
	articleState.inData = state->inData;
	articleState.state = state->state;
	
	if (!article.Parse(&articleState))
		return false;
	
	//прячем внутренний стек вложенной статьи
	state->state = articleState.state;

	return true;
}

bool BNFnonterminal::Gen(GenState *state)
{
	int i = bnfNamespaces.Find(nspace);
	if (i<0)
	{
		LOG_("Не найдено пространство имён "+nspace);
		out("!");
		return true;
	}
	int i2 = bnfNamespaces[i].Find(aid);
	if (i2<0)
	{
		LOG_("Не найдена словарная статья "+aid.ToStringWithDashes()+" в пространстве имён "+nspace);
		out("!");
		return true;
	}

	out(Format(" <%s>", bnfNamespaces[i][i2].name));
	
	BNFarticle &article = bnfNamespaces[i][i2];
	
	//создаём новое состояние с пустым стеком
	GenState articleState;
	articleState.inValues <<= state->inValues;
	articleState.state = state->state;
	
	if (!article.Gen(&articleState))
	{
		out("-");
		return false;
	}
	
	//прячем внутренний стек вложенной статьи
	state->state = articleState.state;

	out("+");
	return true;
}

void BNFnonterminal::Dump()
{
	BNFelement::Dump();
	int i=bnfNamespaces.Find(nspace);
	int i2=-1;
	if (i>=0)
		i2=bnfNamespaces[i].Find(aid);
	BNF_DUMP(Format("<%s::%s=%s>", nspace, aid.ToStringWithDashes(), i2<0?"?":bnfNamespaces[i][i2].name));
}

//---------------------------------------------------------------------------------------------- SPLITTER
BNFsplitter::BNFsplitter(bool _up)
	:up(_up)
{
	type = BNF_SPLITTER;
}

bool BNFsplitter::Parse(ParseState *state)
{
	out(" [split]");
	return true;
}

bool BNFsplitter::Gen(GenState *state)
{
	out(" [split]");
	return true;
}

//---------------------------------------------------------------------------------------------- UNITOR
BNFunitor::BNFunitor(BNFelement *_splitter)
	:splitter(_splitter)
{
	type = BNF_UNITOR;
}

bool BNFunitor::Parse(ParseState *state)
{
	out(" [unite]");
	return true;
}

bool BNFunitor::Gen(GenState *state)
{
	out(" [unite]");
	return true;
}

//---------------------------------------------------------------------------------------------- CODE
BNFcode::BNFcode(const String &_id, Value _v)
	:id(_id)
	,v(_v)
{
	type = BNF_CODE;
}

bool BNFcode::Parse(ParseState *state)
{
	out(Format(" $%s:", id));
	out(DumpifyValue(v));
	int procI = bnfParseProcs.Find(id);
	if (procI < 0)
	{
		LOG_("Не найдена процедура "+id);
		out("!");
		return false;
	}
	
	BNFParseProc proc = bnfParseProcs[procI];
	ParseState oldPS;
	oldPS.inData = state->inData;
	//oldPS.stack <<= state->stack;
	oldPS.state = state->state;	

	if (!proc(state,v))
	{
		state->inData  = oldPS.inData;
		//state->stack <<= oldPS.stack;
		state->state   = oldPS.state;	

		out("-");
		return false;
	}
	
	out("+");
	return true;
}

bool BNFcode::Gen(GenState *state)
{
	out(Format(" $%s", id));
	int procI = bnfGenProcs.Find(id);
	if (procI < 0)
	{
		LOG_("Не найдена процедура "+id);
		out("!");
		return false;
	}
	
	BNFGenProc proc = bnfGenProcs[procI];
	GenState oldPS;
	oldPS.inValues <<= state->inValues;
	oldPS.stack    <<= state->stack;
	oldPS.state      = state->state;	
	
	if (!proc(state,v))
	{
		state->inValues <<= oldPS.inValues;
		state->stack    <<= oldPS.stack;
		state->state      = oldPS.state;	

		out("-");
		return false;
	}
	
	out("+");
	return true;
}

//---------------------------------------------------------------------------------------------- BEND
BNFbend::BNFbend()
{
	type = BNF_BEND;
}

bool BNFbend::Parse(ParseState *state)
{
	out(" ,");
	return true;
}

bool BNFbend::Gen(GenState *state)
{
	out(" ,");
	return true;
}

//---------------------------------------------------------------------------------------------- LINK
BNFlink::BNFlink()
{
	type = BNF_LINK;
}

bool BNFlink::Parse(ParseState *state)
{
	out(" .");
	return true;
}

bool BNFlink::Gen(GenState *state)
{
	out(" .");
	return true;
}

//---------------------------------------------------------------------------------------------- BEGIN
BNFbegin::BNFbegin()
{
	type = BNF_BEGIN;
}

bool BNFbegin::Parse(ParseState *state)
{
	out(" |>");
	return true;
}

bool BNFbegin::Gen(GenState *state)
{
	out(" |>");
	return true;
}

//---------------------------------------------------------------------------------------------- END
BNFend::BNFend()
{
	type = BNF_END;
}

bool BNFend::Parse(ParseState *state)
{
	out(" <|");
	return true;
}

bool BNFend::Gen(GenState *state)
{
	out(" <|");
	return true;
}

//---------------------------------------------------------------------------------------------- ARTICLE
BNFarticle::BNFarticle(bool init /* = true*/)
	:outer(NULL)
	,curBranchStart(NULL)
	,curBranchEnd  (NULL)
{
	uid.New();
	
	if (init)
	{
		//добавляем минимальный набор элементов
		BNFelement *elBegin = new BNFbegin;
		BNFelement *elLink  = new BNFlink ;
		BNFelement *elEnd   = new BNFend  ;
		
		elBegin->links [0] = elLink;
		elLink ->inputs[0] = elBegin;
		elEnd  ->inputs[0] = elLink;
		elLink ->links [0] = elEnd;
		
		elements.Add((dword)elBegin,elBegin);
		elements.Add((dword)elLink ,elLink );
		elements.Add((dword)elEnd  ,elEnd  );
	
		begin          = elBegin;
		outer          = elLink;
		curBranchStart = elLink;
		curBranchEnd   = elLink;
		
		startArticle = this;
		startElement = elLink;
	}
}

BNFarticle::~BNFarticle()
{
}

static int paragraph = -1;

bool BNFarticle::Parse(ParseState *state)
{
	++paragraph;
	String para(' ', 4*paragraph);
	para.Insert(0,'\n');
	out(para+"***************** PARSE: "+this->name+ ": " + DumpifyString(state->inData.Right(state->inData.GetLength() - state->state.inOffset)).ToString() );
	int startOffset = state->state.inOffset;
	
	if (!begin || elements.IsEmpty())
	{
		out(para+"***************** + NULL + ("+this->name+") ");
		--paragraph;
		return true;
	}
	
	Vector<ParseState::State> stack;
	ParseState::State curState = state->state;
	curState.ptr = begin;
	
	while (!state->stop && curState.ptr)
	{
		bool parseResult = curState.ptr->Parse(state);
		if (state->stop)
			break;

		if (parseResult)
		{
			if (curState.ptr->links[1])
			{
				out(" +>"+Format("%d", curState.ptr->debugID)+" total:" + DumpifyString(state->inData.Left(curState.inOffset)).ToString());
				stack.Add(curState);
				stack.Top().inOffset = state->state.inOffset;
				stack.Top().values <<= state->state.values;
			}

			curState.ptr = curState.ptr->links[ curState.ptr->FirstLinkIndex() ];
		}
		else
		{
			bool success = false;
			if (!stack.IsEmpty())
			{
				success = true;
				curState = stack.Pop();
				curState.ptr = curState.ptr->links[ 1 - curState.ptr->FirstLinkIndex() ];
				state->state.inOffset = curState.inOffset;
				state->state.values <<= curState.values;
				
				#ifdef _DEBUG
				String __dbg(para+"+<[");
				for (int i=stack.GetCount()-1; i>=0; --i)
				{
					__dbg += FormatInt(stack[i].ptr->debugID);
					if (i) __dbg += ",";
				}
				out((__dbg + "] " + "total:" + DumpifyString(state->inData.Left(curState.inOffset)).ToString()));
				#endif
			}
			if (!success)
			{
				out("----------------- ("+this->name+") ");
				--paragraph;
				return false;
			}
		}
	}
	
	//state->state.inOffset = curState.inOffset;
	//state->state.values   = curState.values;
	
	out(para+"***************** "+this->name+" +++++++ ");
	out(para.ToWString() + DumpifyString(state->inData.Mid(startOffset, state->state.inOffset-startOffset)) + ", total:" + DumpifyString(state->inData.Left(state->state.inOffset)));
	--paragraph;
	return true;
}

bool BNFarticle::Gen(GenState *state)
{
	++paragraph;
	String para(' ', 4*paragraph);
	para.Insert(0,'\n');
	out(para+"***************** GEN: "+this->name+para);

	int startOffset = state->state.inOffset;
	
	if (!begin || elements.IsEmpty())
		return true;
	
	state->state.ptr = begin;
	while (state->state.ptr)
	{
		GenState::State curState = state->state;

		bool parseResult = state->state.ptr->Gen(state);
		if (!state->stack.IsEmpty())
			out(Format("/%d",state->stack.GetCount()));
		if (parseResult)
		{
			BNFelement *stackTopPtr = state->stack.IsEmpty() ? NULL : state->stack.Top().ptr;
			if (curState.ptr->links[1] &&  curState.ptr != stackTopPtr)
			{
				out(Format("*%d", curState.ptr->debugID));
				curState.firstEnter = true;
				state->stack.Add(curState);
			}

			if (stackTopPtr && curState.ptr->GetParentSplitter() == stackTopPtr)
				state->stack.Pop();
			
			stackTopPtr = state->stack.IsEmpty() ? NULL : state->stack.Top().ptr;
			int firstI = curState.ptr->FirstLinkIndex();
			int linkN = curState.ptr != stackTopPtr ? firstI : (curState.firstEnter ? firstI : 1-firstI);
			BNFelement *next = curState.ptr->links[linkN];
			
			state->state.ptr = next;
		}
		else
		{
			bool success = false;
			while (!state->stack.IsEmpty())
			{
				if (!state->stack.Top().firstEnter)
				{
					state->stack.Pop();
					out(para+"^");
					continue;
				}

				state->stack.Top().firstEnter = false;
				state->state = state->stack.Top();
				success = true;

				#ifdef _DEBUG
				//out(para+"+<"/*Format(" #%d",curState.ptr->debugID)*/);
				out(para+"+<[");
				for (int i=state->stack.GetCount()-1; i>=0; --i)
				{
					out(FormatInt(state->stack[i].ptr->debugID));
					if (i) out(",");
				}
				out("]");
				#endif
				
				break;
			}
			if (!success)
			{
				out("----------------- ("+this->name+") ");
				--paragraph;
				return false;
			}
		}
	}
	
	out(Format("\n+ (%d values)", state->state.inOffset));
	return true;
}

void DumpBranchRecursive(BNFelement *start, int l)
{
	BNF_DUMP("\n\n");

	BNFelement *cur = start;
	String prefix;
	for (int i=0; i<l; ++i)
		BNF_DUMP("    ");
	
	Vector<BNFelement *> branches;
	while (cur)
	{
		out(FormatInt(cur->debugID)+":");
		cur->Dump();
		if (cur->links[1])
			branches << cur->links[1];
		if (cur->links[0] && cur == cur->links[0]->inputs[1])
			break;
		
		cur = cur->links[0];
	}
	
	for (int i=0; i<branches.GetCount(); ++i)
	{
		DumpBranchRecursive(branches[i], l+1);
	}
	
	BNF_DUMP("\n");
}

void BNFarticle::Dump()
{
	DumpBranchRecursive(&elements[0], 1);
}

BNFarticle & BNFarticle::operator>=(int)
{
	ASSERT(outer);
	BNFelement *elLink    = new BNFlink;
	
	elLink->inputs[0] = outer;
	elLink->links[0] = outer->links[0];
	outer->links[0]->inputs[0] = elLink;
	outer->links[0] = elLink;
	
	outer = elLink ;
	
	elements.Add((dword)elLink   , elLink   );
	
	return *this;
}

BNFarticle & BNFarticle::operator>(const char *id)
{
	ASSERT(outer);

	int nspaceI = bnfNamespaces.Find(NAMESPACE_BNF);
	ASSERT(nspaceI>=0);
	if (nspaceI < 0)
		return *this;
	
	Uuid aid;
	aid.SetNull();
	for (int i=0; i<bnfNamespaces[nspaceI].GetCount(); ++i)
	{
		if (bnfNamespaces[nspaceI][i].name == id)
		{
			aid = bnfNamespaces[nspaceI][i].uid;
			break;
		}
	}
	if (aid.IsNullInstance())
		return *this;
	
	BNFelement *elNonterm = new BNFnonterminal(NAMESPACE_BNF, aid);
	BNFelement *elLink    = new BNFlink;
	
	elLink->links[0] = outer->links[0];
	outer->links[0]->UpdateInput(outer,elLink);
	outer->links[0] = elNonterm;
	elNonterm->inputs[0] = outer;
	elNonterm->links[0] = elLink;
	elLink->inputs[0] = elNonterm;
	
	outer = elLink ;
	
	elements.Add((dword)elNonterm, elNonterm);
	elements.Add((dword)elLink   , elLink   );
	
	return *this;
}

BNFarticle & BNFarticle::operator> (const BNFcode &op)
{
	ASSERT(outer);
	BNFelement *elNonterm = new BNFcode(op);
	BNFelement *elLink    = new BNFlink;
	
	elLink->links[0] = outer->links[0];
	outer->links[0]->UpdateInput(outer,elLink);
	outer->links[0] = elNonterm;
	elNonterm->inputs[0] = outer;
	elNonterm->links[0] = elLink;
	elLink->inputs[0] = elNonterm;
	
	outer = elLink ;
	
	elements.Add((dword)elNonterm, elNonterm);
	elements.Add((dword)elLink   , elLink   );
	
	return *this;
}

BNFarticle & BNFarticle::operator>=(const char *v)
{
	ASSERT(outer);
	BNFelement *elNonterm = new BNFterminal(v);
	BNFelement *elLink    = new BNFlink;
	
	elLink->links[0] = outer->links[0];
	outer->links[0]->UpdateInput(outer,elLink);
	outer->links[0] = elNonterm;
	elNonterm->inputs[0] = outer;
	elNonterm->links[0] = elLink;
	elLink->inputs[0] = elNonterm;
	
	outer = elLink ;
	
	elements.Add((dword)elNonterm, elNonterm);
	elements.Add((dword)elLink   , elLink   );
	
	return *this;
}

BNFarticle & BNFarticle::operator>=(bool start)
{
	if (start)
		curBranchStart = outer;
	else
		curBranchEnd = outer;
	return *this;
}

BNFarticle & BNFarticle::operator<=(bool up)
{
	ASSERT(curBranchStart && curBranchEnd && outer);
	branchOuters << outer;
	
	if (up)
	{
		//обратная связь:
		//пристыковать развилку справа
		BNFelement *elSplitter = new BNFsplitter(up);
		curBranchStart->links[0]->inputs[0] = elSplitter;
		elSplitter->links[0] = curBranchStart->links[0];
		elSplitter->inputs[0] = curBranchStart;
		curBranchStart->links[0] = elSplitter;
		elements.Add((dword)elSplitter, elSplitter);
		
		//пристыковать соединитель слева
		BNFelement *elUnitor = new BNFunitor(elSplitter);
		curBranchEnd->inputs[0]->links[0] = elUnitor;
		elUnitor->inputs[0] = curBranchEnd->inputs[0];
		elUnitor->links[0] = curBranchEnd;
		curBranchEnd->inputs[0] = elUnitor;
		elements.Add((dword)elUnitor, elUnitor);
	
		//добавить между ними связь, установить внешний (активный) элемент
		BNFelement *elLink    = new BNFlink;
		elSplitter->links[1] = elLink;
		elLink->inputs[0] = elSplitter;
		elUnitor->inputs[1] = elLink;
		elLink->links[0] = elUnitor;
		elements.Add((dword)elLink, elLink);
		
		outer = elLink;
	}
	else
	{
		//прямая параллельная ветка:
		//пристыковать развилку слева
		BNFelement *elSplitter = new BNFsplitter(up);
		elSplitter->inputs[0] = curBranchStart->inputs[0];
		elSplitter->inputs[0]->links[0] = elSplitter;
		curBranchStart->inputs[0] = elSplitter;
		elSplitter->links[0] = curBranchStart;
		elements.Add((dword)elSplitter, elSplitter);
		
		//пристыковать соединитель справа
		BNFelement *elUnitor = new BNFunitor(elSplitter);
		elUnitor->inputs[0] = curBranchEnd;
		elUnitor->links[0] = curBranchEnd->links[0];
		curBranchEnd->links[0] = elUnitor;
		elUnitor->links[0]->inputs[0] = elUnitor;
		elements.Add((dword)elUnitor, elUnitor);
	
		//добавить между ними связь, установить внешний (активный) элемент
		BNFelement *elLink    = new BNFlink;
		elSplitter->links[1] = elLink;
		elLink->inputs[0] = elSplitter;
		elUnitor->inputs[1] = elLink;
		elLink->links[0] = elUnitor;
		elements.Add((dword)elLink, elLink);
		
		outer = elLink;
	}

	return *this;
}

BNFarticle & BNFarticle::operator<=(int)
{
	outer = branchOuters.Pop();
	return *this;
}

BNFarticle & BNFarticle::operator>=(char s)
{
	if (s == '<')
		outer = curBranchEnd->inputs[0];
	else
		outer = curBranchEnd->links[0];
	return *this;
}

BNFarticle & BNFarticle::operator<=(char s)
{
	if (s == '<')
		outer = curBranchStart->inputs[0];
	else
		outer = curBranchStart->links[0];
	return *this;
}

BNFelement * BNFarticle::AppendLink(BNFelement *topEl)
{
	int topElI = elements.Find((dword) topEl);
	if (topElI < 0)
		return NULL;
	
	BNFelement *elLink    = new BNFlink;

	topEl->links[0]->UpdateInput(topEl, elLink);
	elLink->links[0] = topEl->links[0];
	topEl->links[0] = elLink;
	elLink->inputs[0] = topEl;
	
	elements.Add((dword)elLink, elLink);
	
	return elLink;
}

BNFelement * BNFarticle::AppendElement(BNFelement *topEl, BNFelement *newEl)
{
	int topElI = elements.Find((dword) topEl);
	if (topElI < 0)
		return NULL;
	
	BNFelement *elLink    = new BNFlink;

	topEl->links[0]->UpdateInput(topEl, elLink);
	elLink->links[0] = topEl->links[0];
	elLink->inputs[0] = newEl;
	newEl->links[0] = elLink;
	newEl->inputs[0] = topEl;
	topEl->links[0] = newEl;
	
	elements.Add((dword)newEl , newEl );
	elements.Add((dword)elLink, elLink);
	
	return elLink;
}

BNFelement * BNFarticle::AppendBranch(bool up, BNFelement *startEl, BNFelement *endEl, BNFelement **newStartEl, BNFelement **newEndEl)
{
	if (up)
	{
		//обратная связь:
		BNFelement *elSplitter = new BNFsplitter(up);
		elements.Add((dword)elSplitter, elSplitter);
		BNFelement *elUnitor = new BNFunitor(elSplitter);
		elements.Add((dword)elUnitor, elUnitor);
		BNFelement *elLink    = new BNFlink;
		elements.Add((dword)elLink, elLink);
		BNFelement *elLinkStart    = new BNFlink;
		elements.Add((dword)elLinkStart, elLinkStart);
		BNFelement *elLinkEnd    = new BNFlink;
		elements.Add((dword)elLinkEnd, elLinkEnd);

		startEl->links[0]->UpdateInput(startEl,elLinkStart);
		elLinkStart->links[0] = startEl->links[0];
		elLinkStart->inputs[0] = elSplitter;
		elSplitter->links[0] = elLinkStart;
		elSplitter->inputs[0] = startEl;
		startEl->links[0] = elSplitter;
		
		endEl->inputs[0]->UpdateLink(endEl, elLinkEnd);
		elLinkEnd->inputs[0] = endEl->inputs[0];
		elLinkEnd->links[0] = elUnitor;
		elUnitor->inputs[0] = elLinkEnd;
		elUnitor->links[0] = endEl;
		endEl->inputs[0] = elUnitor;
		
		elSplitter->links[1] = elLink;
		elLink->inputs[0] = elSplitter;
		elLink->links[0] = elUnitor;
		elUnitor->inputs[1] = elLink;
		
		*newStartEl = elLinkEnd;
		*newEndEl = elLinkStart;
		return elLink;
	}
	else
	{
		//прямая параллельная ветка:
		BNFelement *elSplitter = new BNFsplitter(up);
		elements.Add((dword)elSplitter, elSplitter);
		BNFelement *elUnitor = new BNFunitor(elSplitter);
		elements.Add((dword)elUnitor, elUnitor);
		BNFelement *elLink    = new BNFlink;
		elements.Add((dword)elLink, elLink);
		BNFelement *elLinkStart    = new BNFlink;
		elements.Add((dword)elLinkStart, elLinkStart);
		BNFelement *elLinkEnd    = new BNFlink;
		elements.Add((dword)elLinkEnd, elLinkEnd);

		startEl->inputs[0]->UpdateLink(startEl,elLinkStart);
		elLinkStart->inputs[0] = startEl->inputs[0];
		elLinkStart->links[0] = elSplitter;
		elSplitter->inputs[0] = elLinkStart;
		
		elSplitter->links[0] = startEl;
		startEl->inputs[0] = elSplitter;
		
		endEl->links[0]->UpdateInput(endEl, elLinkEnd);
		elLinkEnd->links[0] = endEl->links[0];
		elLinkEnd->inputs[0] = elUnitor;
		elUnitor->links[0] = elLinkEnd;
		
		elUnitor->inputs[0] = endEl;
		endEl->links[0] = elUnitor;
		
		elSplitter->links[1] = elLink;
		elLink->inputs[0] = elSplitter;
		elLink->links[0] = elUnitor;
		elUnitor->inputs[1] = elLink;
		
		*newStartEl = elLinkStart;
		*newEndEl = elLinkEnd;
		return elLink;
	}
}

void BNFarticle::Xmlize(XmlIO &xml)
{
	xml.Attr("name", name).Attr("comment", comment).Attr("uid", uid);
	
	if (xml.IsStoring())
	{
		for (int i=0; i<elements.GetCount(); ++i)
		{
			XmlIO xmlElement(xml->Add("element"), false);
			(&elements[i])->Xmlize(xmlElement);
		}
	}
	else
	{
		elements.Clear();
		
		XmlNode &node = *(xml.operator->());
		for (int i=0; i<node.GetCount(); ++i)
		{
			if (node[i].GetTag() != "element")
				continue;
			XmlNode &xmlNodeTag = const_cast<XmlNode &>(node[i]);
			XmlIO xmlTag(xmlNodeTag, true);
			
			int type = -1;
			BNFelement *element = NULL;
			xmlTag.Attr("type", type);
			switch (type)
			{
			case BNF_TERMINAL:    element = new BNFterminal("");    break;
			case BNF_NONTERMINAL: element = new BNFnonterminal("", Uuid()); break;
			case BNF_SPLITTER:    element = new BNFsplitter(false); break;
			case BNF_UNITOR:      element = new BNFunitor(NULL);    break;
			case BNF_CODE:        element = new BNFcode("");        break;
			case BNF_BEND:        element = new BNFbend();          break;
			case BNF_LINK:        element = new BNFlink();          break;
			case BNF_BEGIN:       element = new BNFbegin();         break;
			case BNF_END:         element = new BNFend();           break;
			}
			
			element->Xmlize(xmlTag);
			elements.Add((dword) element, element);
		}
		
		for (int i=0; i<elements.GetCount(); ++i)
			(&elements[i])->RealizeLinksOnLoad(elements);
	}
}

BNFelement & BNFarticle::AddElement(BNFelement *em)
{
	return elements.Add((dword)em, em);
}

template<> void Upp::XmlAttrLoad(Uuid &uid, const String &text)
{
	uid = ScanUuid(~text);
}

template<> String Upp::XmlAttrStore(const Uuid &uid)
{
	return FormatWithDashes(uid);
}



int BNFelement::debugIDcounter = 0;

BNFarticle *startArticle = NULL;
BNFelement *startElement = NULL;
