#include <Core/Core.h>
#include "SerialPort.h"
using namespace Upp;

SerialPort::SerialPort(const String &device)
{
	#ifdef _DEBUG
	__t0 = GetTickCount();
	#endif
	
	if (SerialOpen(&port,const_cast<char *>(~device)) != SERIAL_OK)
		throw Exc("Не удаётся открыть последовательный порт  "+device);
	CallbackThread::Start();
}

SerialPort::~SerialPort()
{
	CallbackThread::Shutdown();
	CallbackThread::Wait();
	SerialClose(&port);
}

int SerialPort::GetProtoICached(BNFarticles &bnfArticles, const String &name)
{
	int protoI = protocolCache.Find(name);
	if (protoI >= 0)
		return bnfArticles.Find(protocolCache[protoI]);

	for (int i=0; i<bnfArticles.GetCount(); ++i)
	{
		if (bnfArticles[i].name == name)
		{
			protocolCache.Add(name, bnfArticles[i].uid);
			return bnfArticles.Find(bnfArticles[i].uid);
		}
	}
	return -1;
}

void SerialPort::Init(uint32_t baud, char parity, uint8_t dataBits, uint8_t stopBits)
{
	SerialSettings(&port,baud,parity,dataBits,stopBits,1);
}

/*static*/ void SerialPort::BNFInit()
{
	FileIn f;
	if (!f.Open("proto"))
	{
		RLOG("Файл \"proto\" не найден. Для работы по протоколам, создайте его в директории программы и добавьте туда описания протоколов.");
		return;
	}
	
	String s = f.Get(static_cast<dword>(f.GetSize()));
	
	BNFarticle *protoParser = BNFAddArticle
	(
		"PROTOPARSER",
		"PROTOparser", 
		"<BNFwhite> {\"***\" <BNFwhite> (BNFstr10:\""+NAMESPACE_IO+"\" BNFstr:\" \n\r\t\" BNFnewScheme | BNFstr:\" \n\r\t:\" \"::\" BNFstr:\" \n\r\t\" BNFnewScheme) <BNFwhite> <BNFscheme> BNFresume}"
	);

	ParseState state;
	state.inData = s;
	protoParser->Parse(&state);
}

/*static*/ void SerialPort::OpenPorts()
{
	FileIn f;
	if (!f.Open("ports"))
	{
		RLOG("Не найден файл \"ports\". Создайте его в директории программы и добавьте туда описания портов.");
		return;
	}
	
	String s = f.Get(static_cast<int>(f.GetSize()));
	f.Close();

	XmlNode xmlRoot = ParseXML(~s);
	for (int j=0; j<xmlRoot.GetCount(); ++j)
	{
		if (xmlRoot[j].GetTag() != "ports")
			continue;
		const XmlNode &xml = xmlRoot[j];
		
		//порты
		for (int i=0; i<xml.GetCount(); ++i)
		{
			if (xml[i].GetTag() == "port")
			{
				String name = xml[i].Attr("name");
				String desc = AsXML(xml[i]);
				String type = xml[i].Attr("type");
				if (type != "serial")
					continue;
				
				String device   = xml[i].Attr("com");
				int    baud     = xml[i].AttrInt("baud",9600);
				int    parity   = xml[i].AttrInt("parity",0);
				int    wordlen  = xml[i].AttrInt("wordlen",8);
				int    stopbits = xml[i].AttrInt("stopbits",1);

				SerialPort *newPort = new SerialPort(device);
				newPort->Init(
					baud,
					(!parity ? 'n' : parity == 1 ? 'o' : 'e'),
					wordlen,
					stopbits
				);
				portsMap.Add(name,newPort);
			}
		}
		break;
	}
}

/*static*/ void SerialPort::ClosePorts()
{
	portsMap.Clear();
}

/*static*/ SerialPort * SerialPort::GetPort(const String &name)
{
	int portI = portsMap.Find(name);
	if (portI < 0)
		return NULL;
	return &portsMap[portI];
}

ArrayMap<String, SerialPort> SerialPort::portsMap;
