#include "Utils.h"
#include <AESStream/AESStream.h>
#include <unicode/ucsdet.h>
#include <iconv.h>
#include "html/ParserDom.h"
#include "css/parser.h"

#define LOG_HTML     LOG
#define LOG_ANALYZE  LOG
#define LOG_TEXTBODY LOG
#define LOG_GETDATA  LOG

const String ATTR_STYLE = "style";

String Encrypt(const String &src, const String &pass)
{
	try
	{
		AESEncoderStream conv(src.GetLength(), pass);
		conv << src;
		return conv.GetEncryptedData();
	}
	catch (...)
	{
		return String();
	}
}

String Decrypt(const String &src, const String &pass)
{
	try
	{
		AESDecoderStream conv(pass);
		conv << src;
		return conv.GetDecryptedData();
	}
	catch (...)
	{
		return String();
	}
}

using namespace Upp;
using namespace std;
using namespace htmlcxx;

String ConvertCharsetIconv(const String &src, const char *charsetFrom, const char *charsetTo)
{
	iconv_t icv = iconv_open(charsetTo, charsetFrom);
	if (icv == (iconv_t) -1)
		return src;
	
	size_t        srcLeft  = src.GetLength();
	size_t        destLeft = 16 + srcLeft * 4;
	StringBuffer  dest(destLeft);
	const char   *srcPtr   = ~src;
	char         *destPtr  = ~dest;
	size_t ret = iconv(icv, &srcPtr, &srcLeft, &destPtr, &destLeft);
	iconv_close(icv);
	if (ret == (size_t) -1)
		return src;
	dest.SetLength(dest.GetLength() - destLeft);
	return dest;
}

String ConvertTextToUtf8(const String &_text, bool filterTags, bool isConvertAnyway /*= false*/)
{
	String text(_text);
	UErrorCode        status = U_ZERO_ERROR;
	UCharsetDetector *detector = ucsdet_open(&status);
	
	if (filterTags)
		ucsdet_enableInputFilter(detector, TRUE);
	
	if (detector)
	{
		ucsdet_setText(detector, ~text, text.GetLength(), &status);
		const UCharsetMatch *match = ucsdet_detect(detector, &status);
		if (match)
		{
			int confidence = ucsdet_getConfidence(match, &status);
			const char *detectedCharset = ucsdet_getName(match, &status);
			LOG(String("{charset ") + detectedCharset + " [" + FormatInt(confidence) + "]}");
			if (isConvertAnyway || (text.GetLength() > 16 && confidence > 16))
				text = ConvertCharsetIconv(text, detectedCharset, "UTF-8");
		}

		ucsdet_close(detector);
	}
	return text;
}

const HTMLFilter & GetHTMLFilter(bool init/*= false*/)
{
	static HTMLFilter htmlFilter;
	if (init)
	{
		htmlFilter.tags.Clear();
		htmlFilter.tags.GetAdd('\0').Add("*",HTMLFilter::ATTR_CONVERT_NO_CAPTION);
		htmlFilter.tags.GetAdd("BODY").Add("*",HTMLFilter::ATTR_CONVERT_AS_SPAN);
		htmlFilter.tags.GetAdd("HTML").Add("*",HTMLFilter::ATTR_CONVERT_AS_SPAN);
		htmlFilter.tags.GetAdd("A")   .Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("A")   .Add("href",HTMLFilter::ATTR_CONVERT_LINK);
		htmlFilter.tags.GetAdd("ABBR").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("ACRONYM").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("ADDRESS").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("B").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("BDO").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("BIG").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("BLOCKQUOTE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("BR").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("CAPTION").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("CENTER").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("CITE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("CODE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COL").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COL").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COL").Add("span",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COL").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COL").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COLGROUP").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COLGROUP").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COLGROUP").Add("span",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COLGROUP").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("COLGROUP").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("DD").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("DFN").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("DIV").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("DL").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("DT").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("EM").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("FONT").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H1").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H2").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H3").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H4").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H5").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("H6").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("HR").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("I").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("IMG").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("IMG").Add("alt",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("IMG").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("IMG").Add("height",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("IMG").Add("src",HTMLFilter::ATTR_CONVERT_IMG);
		htmlFilter.tags.GetAdd("KBD").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("LI").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("LI").Add("TYPE",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("UL").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("UL").Add("TYPE",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("OL").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("OL").Add("TYPE",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("P").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("PRE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("Q").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("VAR").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("SAMP").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("SPAN").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("STRIKE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("S").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("STRONG").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("SUB").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("SUP").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TT").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("U").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("bgcolor",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("border",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("cellpadding",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("frame",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("rules",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("summary",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TABLE").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TR").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TR").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TR").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TFOOT").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TFOOT").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TFOOT").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TBODY").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TBODY").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TBODY").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("THEAD").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("THEAD").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("THEAD").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("abbr",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("axis",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("bgcolor",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("colspan",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("headers",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("header",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("nowrap",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("rowspan",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("scope",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TH").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("*",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("abbr",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("align",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("axis",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("bgcolor",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("colspan",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("headers",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("header",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("nowrap",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("rowspan",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("scope",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("valign",HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.tags.GetAdd("TD").Add("width",HTMLFilter::ATTR_CONVERT_AS_IS);
		
		htmlFilter.styles.Clear();
		htmlFilter.styles.Add("direction", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font-family", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font-style", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font-variant", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font-size", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("font-weight", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("letter-spacing", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("line-height", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-align", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-decoration", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-indent", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-overflow", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-shadow", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("text-transform", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("white-space", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("word-spacing", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("vertical-align", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("color", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("background-color", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("border", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("height", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("margin", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("width", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("clear", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("display", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("float", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("overflow", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("list-style-position", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("list-style-type", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("border-collapse", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("border-spacing", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("caption-side", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("empty-cells", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("table-layout", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("margin-bottom", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("margin-left", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("margin-right", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("margin-top", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding-bottom", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding-left", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding-right", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("padding-top", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("bottom", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("left", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("right", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("top", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("visibility", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("orphans", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("page-break-after", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("page-break-before", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("page-break-inside", HTMLFilter::ATTR_CONVERT_AS_IS);
		htmlFilter.styles.Add("widows", HTMLFilter::ATTR_CONVERT_AS_IS);
	}
	return htmlFilter;
}

String FilterTag(String s)
{
	int i=0;
	while (i < s.GetLength())
	{
		if (!IsAlNum(s[i]))
			s.Remove(i);
		else
			++i;
	}

	return s;
}

String FilterProperty(String s)
{
	int i=0;
	while (i < s.GetLength())
	{
		if (IsAlNum(s[i]) || s[i] == '-')
			++i;
		else
			s.Remove(i);
	}

	return s;
}

String FilterText(String s)
{
	s.Replace("<","&lt;");
	s.Replace(">","&gt;");
	s.Replace("\"","&quot;");
	s.Replace("\'","&#39;");
	s.Replace("{","&#123;");
	s.Replace("}","&#125;");
	//s.Replace("#","&#35;");
	//s.Replace("&","&#38;");

	s.Replace("%3c","&lt;");
	s.Replace("%3e","&gt;");
	s.Replace("%22","&quot;");
	s.Replace("%60","&#39;");
	s.Replace("%27","&#39;");
	s.Replace("%7b","&#123;");
	s.Replace("%7d","&#125;");
	//s.Replace("%23","&#35;");
	//s.Replace("%26","&#38;");

	s.Replace("%3C","&lt;");
	s.Replace("%3E","&gt;");
	s.Replace("%27","&#39;");
	s.Replace("%7B","&#123;");
	s.Replace("%7D","&#125;");

	s.Replace("\\x3c","&lt;");
	s.Replace("\\x3e","&gt;");
	s.Replace("\\x22","&quot;");
	s.Replace("\\x60","&#39;");
	s.Replace("\\x27","&#39;");
	s.Replace("\\x7b","&#123;");
	s.Replace("\\x7d","&#125;");
	//s.Replace("\\x23","&#35;");
	//s.Replace("\\x26","&#38;");

	s.Replace("\\x3C","&lt;");
	s.Replace("\\x3E","&gt;");
	s.Replace("\\x27","&#39;");
	s.Replace("\\x7B","&#123;");
	s.Replace("\\x7D","&#125;");
	
	return s;
}

String SafeguardHTML(const String &s, bool convertUtf8/*= true*/)
{
	String _out;
	HTML::ParserDom parser;
	tree<HTML::Node> dom = parser.parseTree(s);
	
	tree<HTML::Node>::iterator domBeginIt = dom.begin();
	tree<HTML::Node>::iterator domEndIt   = dom.end  ();
	
	const HTMLFilter &htmlFilter = GetHTMLFilter();
	
	Vector<Tuple2<bool,int> > offsetStack;
	String __out=" ";
	if (s.GetLength() > __out.GetLength())
		__out.Reserve(s.GetLength());
	offsetStack.Add(MakeTuple(false, 0));

	tree<HTML::Node>::iterator domCurIt = domBeginIt;
	int level = 0;
	for (; domCurIt != domEndIt; ++domCurIt)
	{
		int curLevel = dom.depth(domCurIt);
		if (curLevel < level)
			offsetStack.Trim(offsetStack.GetCount()-1);

		int &topOffsetStack = offsetStack[offsetStack.GetCount()-1].b;
		if (offsetStack[offsetStack.GetCount()-1].a)
			continue;
		
		if ((*domCurIt).isTag())
		{
			String tagName = ToUpper(FilterTag((*domCurIt).tagName()));
			bool isIgnoredCaption = false;
			bool isIgnoredTag     = false;

			int htmlFilterTagI = tagName.IsEmpty() ? 0 : htmlFilter.tags.Find(tagName);
			if (htmlFilterTagI < 0)
				isIgnoredTag = true;
			
			if (!isIgnoredTag)
			{
				int htmlFilterTagGeneralI = htmlFilter.tags[htmlFilterTagI].Find("*");
				if (htmlFilterTagGeneralI >= 0)
				{
					bool doContinue = false;
					switch (htmlFilter.tags[htmlFilterTagI][htmlFilterTagGeneralI])
					{
					case HTMLFilter::ATTR_CONVERT_AS_IS     :                            break;
					case HTMLFilter::ATTR_CONVERT_AS_SPAN   : tagName          = "SPAN"; break;
					case HTMLFilter::ATTR_CONVERT_REMOVE    : doContinue       = true;   break;
					case HTMLFilter::ATTR_CONVERT_NO_CAPTION: isIgnoredCaption = true;   break;
					}
					if (doContinue)
						isIgnoredTag = true;
				}
			
				LOG_HTML(String(' ',2*curLevel) + "TAG :" + tagName);
			}
			else
				LOG_HTML(String(' ',2*curLevel) + "TAG :" + tagName+" -- IGNORED!");

			String attrsS;
			if (!isIgnoredTag && !isIgnoredCaption)
			{
				(*domCurIt).parseAttributes();
				const std::map<std::string, std::string> &attrs = (*domCurIt).attributes();
				for (std::map<std::string, std::string>::const_iterator attrsIt=attrs.begin(); attrsIt != attrs.end(); ++attrsIt)
				{
					String attrName  = ToLower(FilterTag((*attrsIt).first));
					if (attrName == ATTR_STYLE)
					{
						String propertyS;
						String styleString = tagName + " {" + (*attrsIt).second + '}';
						LOG_HTML("CSS: styleString: " + styleString);
						selector_list_t *selectors = css_parse(~styleString, styleString.GetLength());
						if (selectors)
						{
							selector_list_t *curSelectors = selectors;
							while (curSelectors)
							{
								selector_t *selector = curSelectors->selector;
								if (selector && selector->element_name && (tagName == selector->element_name))
								{
									property_t *property = selector->property;
									while (property)
									{
										String propertyName = ToLower(FilterTag(property->name));
										LOG_HTML("CSS: \tpropertyName:" + propertyName);
										int htmlFilterStyleI = htmlFilter.styles.Find(propertyName);
										if (htmlFilterStyleI < 0)
										{
											property = property->next;
											continue;
										}

										bool doContinue = false;
										bool isIMG      = false;
										bool isLink     = false;
										switch (htmlFilter.styles[htmlFilterStyleI])
										{
										case HTMLFilter::ATTR_CONVERT_AS_IS  :                      break;
										case HTMLFilter::ATTR_CONVERT_REMOVE : doContinue = true;   break;
										case HTMLFilter::ATTR_CONVERT_IMG    : isIMG      = true;   break;
										case HTMLFilter::ATTR_CONVERT_LINK   : isLink     = true;   break;
										}
										
										if (doContinue)
										{
											property = property->next;
											continue;
										}
										
										propertyS << propertyName << ':' << property->val << ';';
										property = property->next;
									}
								}
								curSelectors = curSelectors->next;
							}
							free_rulesets(selectors);
						}
						
						if (!propertyS.IsEmpty())
							attrsS << " style=\'"<< propertyS << '\'';
					}
					else
					{
						int htmlFilterTagAttrI = htmlFilter.tags[htmlFilterTagI].Find(attrName);
						if (htmlFilterTagAttrI < 0)
							continue;
						
						bool doContinue = false;
						bool isIMG      = false;
						bool isLink     = false;
						switch (htmlFilter.tags[htmlFilterTagI][htmlFilterTagAttrI])
						{
						case HTMLFilter::ATTR_CONVERT_AS_IS  :                      break;
						case HTMLFilter::ATTR_CONVERT_REMOVE : doContinue = true;   break;
						case HTMLFilter::ATTR_CONVERT_IMG    : isIMG      = true;   break;
						case HTMLFilter::ATTR_CONVERT_LINK   : isLink     = true;   break;
						}
						if (doContinue)
							continue;
	
						attrsS << (" " + attrName + "=\"" + (*attrsIt).second + "\"");
					}
				}
			}

			_out.Clear();
			if (dom.begin(domCurIt) == dom.end(domCurIt))
			{
				if (!isIgnoredTag && !isIgnoredCaption)
				{
					_out = Format("<%s%s />", tagName, attrsS);
					__out.Insert(topOffsetStack, _out);
					for (int i=0; i<offsetStack.GetCount(); ++i)
						offsetStack[i].b += tagName.GetLength() + attrsS.GetLength() + 4;
				}
				level = curLevel;
			}
			else
			{
				if (!isIgnoredTag && !isIgnoredCaption)
				{
					_out = Format("<%s%s></%s>", tagName, attrsS, tagName);
					__out.Insert(topOffsetStack, _out);
				}
				
				offsetStack << MakeTuple<bool,int>(isIgnoredTag, offsetStack[offsetStack.GetCount()-1].b);
				
				if (!isIgnoredTag && !isIgnoredCaption)
				{
					offsetStack[offsetStack.GetCount()-1].b += tagName.GetLength() + attrsS.GetLength() + 2;
					for (int i=0; i<offsetStack.GetCount()-1; ++i)
						offsetStack[i].b += 2*tagName.GetLength() + attrsS.GetLength() + 5;
				}
			
				level = curLevel+1;
			}
		}
		else
		if (!(*domCurIt).isComment())
		{
			String tagText = FilterText((*domCurIt).text());
			__out.Insert(topOffsetStack, tagText);
			LOG_HTML(String(' ',2*curLevel) + "TEXT:" + tagText);

			for (int i=0; i<offsetStack.GetCount(); ++i)
				offsetStack[i].b += tagText.GetLength();
			
			level = curLevel;
		}
	}
	
	return __out;
}

String SafeguardPlain(const String &s, bool convertUtf8 /*= true*/)
{
	String out = s;//convertUtf8 ? ConvertTextToUtf8(s, false) : s;

	out.Replace("<","&lt;");
	out.Replace(">","&gt;");
	out.Replace("\"","&quot;");
	out.Replace("\'","&#39;");
	out.Replace("{","&#123;");
	out.Replace("}","&#125;");
	//out.Replace("#","&#35;");
	//out.Replace("&","&#38;");

	out.Replace("%3c","&lt;");
	out.Replace("%3e","&gt;");
	out.Replace("%22","&quot;");
	out.Replace("%60","&#39;");
	out.Replace("%27","&#39;");
	out.Replace("%7b","&#123;");
	out.Replace("%7d","&#125;");
	//out.Replace("%23","&#35;");
	//out.Replace("%26","&#38;");

	out.Replace("%3C","&lt;");
	out.Replace("%3E","&gt;");
	out.Replace("%27","&#39;");
	out.Replace("%7B","&#123;");
	out.Replace("%7D","&#125;");

	out.Replace("\\x3c","&lt;");
	out.Replace("\\x3e","&gt;");
	out.Replace("\\x22","&quot;");
	out.Replace("\\x60","&#39;");
	out.Replace("\\x27","&#39;");
	out.Replace("\\x7b","&#123;");
	out.Replace("\\x7d","&#125;");
	//out.Replace("\\x23","&#35;");
	//out.Replace("\\x26","&#38;");

	out.Replace("\\x3C","&lt;");
	out.Replace("\\x3E","&gt;");
	out.Replace("\\x27","&#39;");
	out.Replace("\\x7B","&#123;");
	out.Replace("\\x7D","&#125;");

	out.Replace("\t","&nbsp;");
	out.Replace("\n","<br/>");
	return out;
}

String NormalizeLocale(const String &_loc)
{
	String loc = ToLower(_loc);
	if (loc.GetLength() != 2)
		return loc;
	
	return loc+'-'+loc;
}

int64 ToJSTime(const Time &t)
{
	static const int64 t0 = Time(1970,1,1).Get();
	return (1000L * (t.Get() - t0)); //по стандарту javascript, время хранится в миллисекундах от 0:00 1 января 1970
}

Time FromJSTime(int64 t)
{
	static const Time t0(1970,1,1);
	return (t0+t/1000);
}

bool CheckContentTypeGetBoundary(const String &contentType, const String &multipartType, String &boundary)
{
	const static String BOUNDARY = "boundary=";

	int multipartI;
	if ((multipartI = contentType.Find(multipartType)) < 0)
		return false; //!multipart/*****
	
	int boundaryI = contentType.Find(BOUNDARY, multipartI+multipartType.GetLength());
	if (boundaryI < 0)
		return false; //!boundary=
	
	boundaryI += BOUNDARY.GetLength();
	int l = 0;
	for (int i=boundaryI; i<contentType.GetLength(); ++i, ++l)
	{
		char c = contentType[i];
		if (c == ';' || c == '\r' || c == '\n')
			break;
	}
	
	boundary = contentType.Mid(boundaryI, l);
	if (boundary.IsEmpty())
		return false; //boundary == ""
	
	return true;
}

bool CheckDisposition(const String &contentDisposition, bool &isFile, String &name, String &filename)
{
	static const String CONTENT_DISPOSITION_HEADER    = "Content-Disposition: ";
	static const String NAME_HEADER                   = " name=\"";
	static const String FILENAME_HEADER               = " filename=\"";
	
	isFile = false;
	if (!contentDisposition.StartsWith(CONTENT_DISPOSITION_HEADER))
		return false;
	int cur = CONTENT_DISPOSITION_HEADER.GetLength();
	
	int curName = contentDisposition.Find(NAME_HEADER, cur);
	if (curName >= 0)
	{
		curName += NAME_HEADER.GetLength();
		int l = 0;
		for (int i=curName; i<contentDisposition.GetLength(); ++i,++l)
		{
			if (contentDisposition[i] == '\"')
				break;
		}
		name = UrlDecode(contentDisposition.Mid(curName, l));
	}

	int curFilename = contentDisposition.Find(FILENAME_HEADER, cur);
	if (curFilename >= 0)
	{
		curFilename += FILENAME_HEADER.GetLength();
		int l = 0;
		for (int i=curFilename; i<contentDisposition.GetLength(); ++i,++l)
		{
			if (contentDisposition[i] == '\"')
				break;
		}
		filename = UrlDecode(contentDisposition.Mid(curFilename, l));
		isFile = true;
	}
	return true;
}

String ReadUntilTerm(Socket &socket, char term, int timeout, int maxlen)
{
	ASSERT(socket.IsOpen() && maxlen != 0);
	int ticks = GetTickCount(), end_ticks = IsNull(timeout) ? int(Null) : ticks + timeout, seek = 0;
	String out = socket.Read(timeout, maxlen);
	if(out.IsVoid())
		return out;

	for(;;) {
		int f = out.Find((byte)term, seek);
		if(f >= 0) {
			socket.UnRead(String(out.Begin() + f + 1, out.GetLength() - f - 1));
			return out.Left(f+1);
		}
		seek = out.GetLength();
		ticks = GetTickCount();
		if(!IsNull(timeout)) timeout = end_ticks - ticks;
		if(!IsNull(timeout) && timeout <= 0 || out.GetLength() >= maxlen)
			return out;
		String part = socket.Read(timeout, maxlen - out.GetLength());
		if(part.IsVoid()) {
			return out;
		}
		out.Cat(part);
	}
}

bool ParseFormData(
	Socket                   &socket, 
	dword                     timeout, 
	const String             &contentType, 
	const String             &dir,
	bool                      isFieldsListOnly, 
	VectorMap<String,String> &fields, 
	VectorMap<String,String> &files,
	const int                 maxFileSize/*=100*1024*1024*/
	)
{
	const static String MULTIPART_FORMDATA = "multipart/form-data";
	const static String MULTIPART_MIXED    = "multipart/mixed";
	const static String DOUBLE_DASH        = "--";
	
	///////////////////////////////
	String boundary;
	if (!CheckContentTypeGetBoundary(contentType, MULTIPART_FORMDATA, boundary))
		return false;
	boundary.Insert(0, DOUBLE_DASH);

	//////////////////////////////	
	static const int CHUNK = 8192;
	String buf;
	
	enum STATE {BOUNDARY, HEADER, DATA} state = BOUNDARY;
	String stateTitle;
	bool   stateFile;
	String boundary2;
	int    fileCounter = 0;
	FileOut fOut;
	
	dword t0 = GetTickCount();
	dword dt;
	while ((dt = GetTickCount() - t0) < timeout)
	{
		dword to = timeout-dt;
		switch (state)
		{
		case BOUNDARY:
			buf = socket.ReadUntil('\n',to,CHUNK);
			if (!boundary2.IsEmpty())
			{
				if (!buf.StartsWith(boundary2))
					return false; //!boundary2
			}
			else
			{
				if (!buf.StartsWith(boundary))
					return false; //!boundary
				else
				{
					if (buf.GetLength() >= boundary.GetLength()+2 
					&& buf[boundary.GetLength()] == '-'
					&& buf[boundary.GetLength()+1] == '-'
					)
						return true;
				}
			}
			state = HEADER;
			break;
		case HEADER:
			buf = socket.ReadUntil('\n',to,CHUNK);
			{
				++fileCounter;
				String name;
				String filename;
				if (!CheckDisposition(buf, stateFile, name, filename))
					return false; //bad content disposition
				if (stateFile)
					filename = GetFileName(UnixPath(filename));
				stateTitle = stateFile ? (filename.IsEmpty() ? (name + FormatInt(fileCounter)) : filename) : name;
				if (stateTitle.IsEmpty())
					return false; //!name
				
				bool allHeaders = false;
				bool b2 = false;
				while ((dt = GetTickCount() - t0) < timeout)
				{
					to = timeout-dt;
					buf = socket.ReadUntil('\n',to,CHUNK);
					if (buf.GetLength() <= 1)
					{
						allHeaders = true;
						break;
					}
					if (!b2 && CheckContentTypeGetBoundary(buf, MULTIPART_MIXED, boundary2))
					{
						boundary2.Insert(0, DOUBLE_DASH);
						b2 = true;
					}
				}
				if (!allHeaders)
					return false;
				state = b2 ? BOUNDARY : DATA;
			}
			break;
		case DATA:
			{
				int fileSize = 0;
				bool ignoreData = false;
				if (isFieldsListOnly)
				{
					if (stateFile && files.Find(stateTitle) < 0)
						ignoreData = true;
					else
					if (!stateFile && fields.Find(stateTitle) < 0)
						ignoreData = true;
				}

				if (!ignoreData)
				{
					if (stateFile)
					{
						String fn = stateTitle;
						if (!dir.IsEmpty())
						{
							char finalChar = dir[dir.GetLength()-1];
							if (finalChar != '/' && finalChar != '\\')
								fn.Insert(0,'/');
							fn.Insert(0,dir);
						}
						fOut.Open(fn);
						files.GetAdd(stateTitle,fn);
					}
					else
					{
						fields.GetAdd(stateTitle).Clear();
					}
				}
				
				buf.Clear();
				String b = boundary2.IsEmpty() ? boundary : boundary2;
				while ((dt = GetTickCount() - t0) < timeout)
				{
					to = timeout-dt;
					buf.Cat(ReadUntilTerm(socket,'\n',to,CHUNK));
					int bI = -1;
					if (buf[buf.GetLength()-1] != '\n'
					|| buf.GetLength() < b.GetLength()+4
					|| (bI = buf.Find(b,buf.GetLength()-b.GetLength()-4 < 0 ? 0 : buf.GetLength()-b.GetLength()-4)) < 0
					)
					{
						int dl = buf.GetLength()-b.GetLength()-4;
						if (dl > 0)
						{
							if (!ignoreData)
							{
								if (stateFile)
								{
									fileSize += dl;
									if (fileSize >= maxFileSize)
										ignoreData = true;
									fOut << buf.Left(dl);
								}
								else
									fields.GetAdd(stateTitle) << buf.Left(dl);
							}
							buf.Remove(0,dl);
						}
					}
					else
					{
						int dl = bI - 2;
						if (dl > 0)
						{
							if (!ignoreData)
							{
								if (stateFile)
								{
									fOut << buf.Left(dl);
									fileSize += dl;
									if (fileSize >= maxFileSize)
										ignoreData = true;
								}
								else
									fields.GetAdd(stateTitle) << buf.Left(dl);
							}
						}
						if (buf.GetLength() < bI+b.GetLength()+2 || buf[bI+b.GetLength()] != '-' || buf[bI+b.GetLength()+1] != '-')
						{
							state = HEADER;
							if (fOut.IsOpen())
								fOut.Close();
							break;
						}
						else
						{
							if (boundary2.IsEmpty())
							{
								if (fOut.IsOpen())
									fOut.Close();
								return true;
							}
							else
							{
								state = BOUNDARY;
								boundary2.Clear();
								if (fOut.IsOpen())
									fOut.Close();
								break;
							}
						}
					}
				}
				
				if (fOut.IsOpen())
					fOut.Close();
			}
			break;
		}
	}
	
	return false;
}

