#include "Esc.h"


NAMESPACE_UPP

#define LTIMING(x)  // RTIMING(x)

void ESC_count(EscEscape& e)
{
	e = e[0].GetCount();
}

void ESC_is_void(EscEscape& e)
{
	e = e[0].IsVoid();
}

void ESC_is_number(EscEscape& e)
{
	e = e[0].IsNumber();
}

void ESC_is_array(EscEscape& e)
{
	e = e[0].IsArray();
}

void ESC_is_map(EscEscape& e)
{
	e = e[0].IsMap();
}

void ESC_int(EscEscape& e)
{
	if(e[0].IsNumber())
		e = (int)e[0].GetNumber();
}

void ESC_to_string(EscEscape& e)
{
	e = e[0].ToString();
}

void ESC_to_number(EscEscape& e)
{
	if(!e[0].IsArray()) return;

	const String& s= (String)e[0];
	CParser parser(~s);
	int sign = parser.Sgn();

	// similar to number parsing in Esc::Term(SRVal& r) code but here we deal with sign too

	//1234567890123456789
	//9223372036854775807 (INT64_MAX)
	//18446744073709551615 (UINT64_MAX)

	static const int INT64_MAXDIGITS= 19; // of signed int64
	const wchar* p= parser.GetPtr();
	int cnt= 0;
	while (IsDigit(*p)) {
		++cnt;
		++p;
		if (cnt > INT64_MAXDIGITS)
			break;
	}
	// if length < max uint64 length and no char of floating point format after digits parse as unsigned int
	// note: sign is handled outside this function
	if (cnt <= INT64_MAXDIGITS && *p != '.' && *p != 'e'  && *p != 'E' ) {
		uint64 n= parser.ReadNumber64();  // read as uint64, so even if outside int64 range it will work
		if (n <= (uint64)INT64_MAX)
			e = (sign * (int64) n);
		else if (sign == -1 && n == 1 + (uint64)INT64_MAX)
			e = INT64_MIN;
		else
			e = (sign * (double) n);  // this is possible for negative values
	}
	else {
		parser.Set(~s);  // start over, and let ReadDouble handle the sign
		e = (parser.ReadDouble());
	}
}

void ESC_rand(EscEscape& e)
{
	e = rand();
}

void ESC_keys(EscEscape& e)
{
	e.CheckMap(0);
	EscValue v;
	const VectorMap<EscValue, EscValue>& m = e[0].GetMap();
	for(int i = 0; i < m.GetCount(); i++)
		if(!m.IsUnlinked(i))
			v.ArrayAdd(m.GetKey(i));
	e = v;
}

void ESC_values(EscEscape& e)
{
	e.CheckMap(0);
	EscValue v;
	const VectorMap<EscValue, EscValue>& m = e[0].GetMap();
	for(int i = 0; i < m.GetCount(); i++)
		if(!m.IsUnlinked(i))
			v.ArrayAdd(m[i]);
	e = v;
}

void ESC_reverse(EscEscape& e)
{
	e.CheckArray(0);
	const Vector<EscValue>& a = e[0].GetArray();
	EscValue r;
	for(int i = a.GetCount() - 1; i >= 0; i--)
		r.ArrayAdd(a[i]);
	e = r;
}

struct EscCmp {
	Esc *esc;
	bool operator()(const EscValue& a, const EscValue& b) const {
		return esc->DoCompare(a, b, L"< (sort)") < 0;
	}
};

Vector<int> EscGetSortOrder(EscEscape& e)
{
	e.CheckArray(0);
	const Vector<EscValue>& va = e[0].GetArray();
	EscCmp cmp;
	cmp.esc = &e.esc;
	return GetSortOrder(va, cmp);
}

void ESC_sort(EscEscape& e)
{
	Vector<int> so = EscGetSortOrder(e);
	EscValue r;
	for(int i = 0; i < so.GetCount(); i++)
		r.ArrayAdd(e[0].ArrayGet(so[i]));
	e = r;
}

void ESC_order(EscEscape& e)
{
	Vector<int> so = EscGetSortOrder(e);
	EscValue r;
	for(int i = 0; i < so.GetCount(); i++)
		r.ArrayAdd(so[i]);
	e = r;
}

// ---------------------------

void ESC_mid(EscEscape& e)
{
	e.CheckArray(0);
	int pos = e.Int(1);
	int count = e.Int(2);
	if(pos < 0 || pos + count > e[0].GetCount())
		e.ThrowError(L"out of bounds in call to 'mid'");
	e.ret_val = e[0].ArrayGet(pos, count);
}

void ESC_exists(EscEscape& e)
{
	e.CheckMap(0);
	e = !e[0].MapGet(e[1]).IsVoid();
}

struct ESC_FileOut : public EscHandle {
	FileStream file;
	void Put(EscEscape& e)         { if(file) file.Put(e.Int(0)); }
	void PutLine(EscEscape& e)     { if(file) file.PutLine(String(e[0])); }
	void Close(EscEscape& e)       { if(file) file.Close(); }

	typedef ESC_FileOut CLASSNAME;

	ESC_FileOut(EscEscape& e, EscValue& v, int style) {
		file.Open(String(e[0]), style);
		v.Escape(L"Put(a)", this, THISBACK(Put));
		v.Escape(L"PutLine(a)", this, THISBACK(PutLine));
		v.Escape(L"Close()", this, THISBACK(Close));
	}
};

void ESC_OpenFileOut(EscEscape& e)
{
	EscValue v;
	ESC_FileOut *f = new ESC_FileOut(e, v, FileStream::CREATE);
	if(f->file)
		e = v;
}

void ESC_OpenFileAppend(EscEscape& e)
{
	EscValue v;
	ESC_FileOut *f = new ESC_FileOut(e, v, FileStream::APPEND);
	if(f->file)
		e = v;
}

struct ESC_FileIn : public EscHandle {
	FileIn file;
	void IsEof(EscEscape& e)       { e = file.IsEof(); }
	void Get(EscEscape& e)         { e = file.Get(); }
	void GetLine(EscEscape& e)     { e = file.GetLine(); }
	void Close(EscEscape& e)       { if(file) file.Close(); }

	typedef ESC_FileIn CLASSNAME;

	ESC_FileIn(EscEscape& e, EscValue& v) {
		file.Open(String(e[0]));
		v.Escape(L"IsEof()", this, THISBACK(IsEof));
		v.Escape(L"Get()", this, THISBACK(Get));
		v.Escape(L"GetLine()", this, THISBACK(GetLine));
		v.Escape(L"Close()", this, THISBACK(Close));
	}
};

void ESC_OpenFileIn(EscEscape& e)
{
	EscValue v;
	ESC_FileIn *f = new ESC_FileIn(e, v);
	if(f->file)
		e = v;
}

// ---------------------------

void ESC_sin(EscEscape& e)
{
	e = sin(e[0].GetNumber());
}

void ESC_cos(EscEscape& e)
{
	e = cos(e[0].GetNumber());
}


bool IsDate(const EscValue& v)
{
	return v.HasNumberField(L"year") && v.HasNumberField(L"month") && v.HasNumberField(L"day");
}

bool IsTime(const EscValue& v)
{
	return IsDate(v) && v.HasNumberField(L"hour") && v.HasNumberField(L"minute") && v.HasNumberField(L"second");
}

void SIC_IsDate(EscEscape& e)
{
	e = IsDate(e[0]);
}

void SIC_IsTime(EscEscape& e)
{
	e = IsTime(e[0]);
}

void SIC_GetSysTime(EscEscape& e)
{
	Time tm = GetSysTime();
	EscValue v;
	v.MapSet(L"year", (int)tm.year);
	v.MapSet(L"month", (int)tm.month);
	v.MapSet(L"day", (int)tm.day);
	v.MapSet(L"hour", (int)tm.hour);
	v.MapSet(L"minute", (int)tm.minute);
	v.MapSet(L"second", (int)tm.second);
	e = v;
}

void ESC_ToLower(EscEscape& e)
{
	String s = e[0];
	e = ToLower(s);
}

void ESC_ToUpper(EscEscape& e)
{
	String s = e[0];
	e = ToUpper(s);
}

void StdLib(ArrayMap<String, EscValue>& global)
{
	Escape(global, L"is_number(value)", ESC_is_number);
	Escape(global, L"is_array(value)", ESC_is_array);
	Escape(global, L"is_map(value)", ESC_is_map);
	Escape(global, L"is_void(value)", ESC_is_void);
	Escape(global, L"int(value)", ESC_int);
	Escape(global, L"to_string(value)", ESC_to_string);
	Escape(global, L"to_number(value)", ESC_to_number);
	Escape(global, L"count(value)", ESC_count);
	Escape(global, L"keys(map)", ESC_keys);
	Escape(global, L"values(map)", ESC_values);
	Escape(global, L"rand()", ESC_rand);
	Escape(global, L"reverse(array)", ESC_reverse);
	Escape(global, L"sort(array)", ESC_sort);
	Escape(global, L"order(array)", ESC_order);

	Escape(global, L"ToUpper(value)", ESC_ToUpper);
	Escape(global, L"ToLower(value)", ESC_ToLower);

	Escape(global, L"len(x)", ESC_count);
	Escape(global, L"mid(array, pos, count)", ESC_mid);
	Escape(global, L"exists(map, key)", ESC_exists);

	Escape(global, L"OpenFileOut(x)", ESC_OpenFileOut);
	Escape(global, L"OpenFileAppend(x)", ESC_OpenFileOut);
	Escape(global, L"OpenFileIn(x)", ESC_OpenFileIn);

	Escape(global, L"GetSysTime()", SIC_GetSysTime);
	Escape(global, L"IsDate(x)", SIC_IsDate);
	Escape(global, L"IsTime(x)", SIC_IsTime);

	Escape(global, L"sin(value)", ESC_sin);
	Escape(global, L"cos(value)", ESC_cos);
}

END_UPP_NAMESPACE
