// does execl() in a fork returning control to a terminal immediately.

#include <stdio.h> // stderr
#include <stdlib.h> // EXIT_FAILURE
#include <unistd.h> // fork(), execl(), _exit()
#include <sys/types.h> // pid_t
#include "usage.str"
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include "version.h"

struct
{
  char warn;
}flg;

int usage(int err);
int unparse_args(char* cmdline, int limit, int argc, char** argv);
int print_version(int err);
// if error, returns argc of option to list in error display
int parse_options(int argc, char** argv);


int main(int argc, char** argv)
{
  int err = 0;
  
  // help and version switches
  if(argc == 1)
    return usage(0);
  
  if(argc == 2)
  {
    if(strcmp(argv[1], "--help") == 0)
      return usage(0);
    if(strcmp(argv[1], "-v") == 0)
      return print_version(0);
  }

  err = parse_options(argc, argv);
  if(err) 
  {
    printf("Unknown launch option '%s'\n", argv[err]);
  }
  
  // reassemble commandline
  char args[1024];
  err = unparse_args(args, 1000, argc, argv);
  if(err)
  {
    if(err == ENOMEM)
      fprintf(stderr, "Out of memory or args list too long\n");
    else if (err == E2BIG)
      fprintf(stderr, "Argument list too long\n");
    return 1;
  }

  // if -q suppress all feedback from the app
  if(!flg.warn)
  {
    fclose(stdout);
    fclose(stderr);
    strcat(args, " > /dev/null 2>&1");
  }

  // do the fork and execute the commandline
  pid_t pid;

  pid = fork ();
  if (pid == 0)
  {
    

    // command path, command name, arg1, arg2, ...
    execl ("/bin/sh", "sh", "-c", args, NULL);

    // _exit(code) for threads and forks.
    _exit (EXIT_FAILURE);
  }
  else if (pid < 0)
    return -1;
  else
    return 0;
}


int usage(int err)
{
  if(err)
    fprintf(stderr, "%s", usage_str);
  else
    fprintf(stdout, "%s", usage_str);
  return err;
}

// returns 0 on success, or one of the following
// #define ENOMEM    12  /* Out of memory */
// #define E2BIG    7  /* Argument list too long */
// defined in /usr/include/asm-generic/errno-base.h

int unparse_args(char* cmdline, int limit, int argc, char** argv)
{
  cmdline[0] = 0;
  if(argc < 2)
    return 0;
  
  char* buf = (char*) malloc(limit+256);
  if(!buf)
    return errno = ENOMEM;
  
  char* p = buf;
  char* stopat = buf+limit -1;
  int len;

  // point to args list
  argc --;
  argv ++;

  // skip option switches
  if(flg.warn)
  {
    argc --;
    argv ++;
  }
  
  //////////////////////////////////////
  // recreate command and commandline
  //////////////////////////////////////
  
  for(int i = 0; i < argc; i++)
  {
    p += sprintf(p, "%s ", argv[i]);
    if(p >= stopat)
    {
      free(buf);
      return errno = E2BIG;
    }
  }
  p[-1] = 0; // terminate
  memcpy(cmdline, buf, p-buf);
  free(buf);
  return 0;
}

int print_version(int err)
{
  printf("Launch version %s\n", VERSION);
  return err;
}

int parse_options(int argc, char** argv)
{
  // if no options, return
  if(argv[1][0] != '-')
    return 0;
  
  // init opts
  flg.warn = 0; // default

  // check args (only 1 so far)
  int arg = 1;

  // assume error each iteration
  int err = 1;

  // check and break if match
  if(strcmp(argv[arg], "-a") == 0)    
  { 
    flg.warn = 1;
    err = 0;
  }
  
  // return argN where error occurred or 0
  if(err) 
    return arg;
  else 
    return 0;
}
