#include <Core/Core.h>
#include <sqrat/sqrat.h>

using namespace Upp;

struct Oper {
	virtual double Execute() = 0;
	virtual ~Oper() {}
};

struct BinOper : Oper {
	One<Oper> a;
	One<Oper> b;
};

struct Add : BinOper {
	virtual double Execute() { return a->Execute() + b->Execute(); }
};

struct Sub : BinOper {
	virtual double Execute() { return a->Execute() - b->Execute(); }
};

struct Mul : BinOper {
	virtual double Execute() { return a->Execute() * b->Execute(); }
};

struct Div : BinOper {
	virtual double Execute() { return a->Execute() / b->Execute(); }
};

struct Const : Oper {
	double value;
	virtual double Execute() { return value; }
};

struct Var : Oper {
	double *var;
	virtual double Execute() { return *var; }
};

struct Compiler {
	VectorMap<String, double *> var;
	One<Oper> Term(CParser& p);
	One<Oper> Exp(CParser& p);
	One<Oper> Factor(CParser& p);
};

One<Oper> Compiler::Term(CParser& p)
{
	One<Oper> result;
	if(p.IsId()) {
		double *v = var.Get(p.ReadId(), NULL);
		if(!v)
			p.ThrowError("unknown variable");
		result.Create<Var>().var = v;
	}
	else
	if(p.Char('(')) {
		result = Exp(p);
		p.PassChar(')');
	}
	else
		result.Create<Const>().value = p.ReadDouble();
	return result;
}

One<Oper> Compiler::Factor(CParser& p)
{
	One<Oper> result = Term(p);
	for(;;)
		if(p.Char('*')) {
			One<Oper> rr;
			Mul& m = rr.Create<Mul>();
			m.a = result;
			m.b = Term(p);
			result = rr;
		}
		else
		if(p.Char('/')) {
			One<Oper> rr;
			Div& m = rr.Create<Div>();
			m.a = result;
			m.b = Term(p);
			result = rr;
		}
		else
			return result;
}

One<Oper> Compiler::Exp(CParser& p)
{
	One<Oper> result = Factor(p);
	for(;;)
		if(p.Char('+')) {
			One<Oper> rr;
			Add& m = rr.Create<Add>();
			m.a = result;
			m.b = Factor(p);
			result = rr;
		}
		else
		if(p.Char('-')) {
			One<Oper> rr;
			Sub& m = rr.Create<Sub>();
			m.a = result;
			m.b = Factor(p);
			result = rr;
		}
		else
			return result;
}

VectorMap<String, double> var;

double Exp(CParser& p);

double Term(CParser& p)
{
	if(p.Id("abs")) {
		p.PassChar('(');
		double x = Exp(p);
		p.PassChar(')');
		return fabs(x);
	}
	if(p.IsId())
		return var.Get(p.ReadId(), 0);
	if(p.Char('(')) {
		double x = Exp(p);
		p.PassChar(')');
		return x;
	}
	return p.ReadDouble();
}

double Mul(CParser& p)
{
	double x = Term(p);
	for(;;)
		if(p.Char('*'))
			x = x * Term(p);
		else
		if(p.Char('/')) {
			double y = Term(p);
			if(y == 0)
				p.ThrowError("Divide by zero");
			x = x / y;
		}
		else
			return x;
}

double Exp(CParser& p)
{
	double x = Mul(p);
	for(;;)
		if(p.Char('+'))
			x = x + Mul(p);
		else
		if(p.Char('-'))
			x = x - Mul(p);
		else
			return x;
}

void OutError(const std::string &error) {
	Cerr() << "Exception: " << error.c_str() << '\n';
	SetExitCode(1);
}

CONSOLE_APP_MAIN
{
	double x, y;
	
	Compiler c;
	c.var.Add("x", &x);
	c.var.Add("y", &y);
	
	CParser p("1 / (1 - x * y + x - y)");
	One<Oper> fn = c.Exp(p);
	
	x = 5;
	y = 10;
	
	RDUMP(1 / (1 - x * y + x - y));
	RDUMP(fn->Execute());

	{
		RTIMING("Interpreted");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001) {
				var.GetAdd("x") = x;
				var.GetAdd("y") = y;
				CParser c = CParser("1 / (1 - x * y + x - y)");
				sum += Exp(c);
			}
		RDUMP(sum);
	}
	{
		RTIMING("Compiled");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001)
				sum += fn->Execute();
		RDUMP(sum);
	}
	{
		RTIMING("Direct");
		double sum = 0;
		for(x = 0; x < 1; x += 0.001)
			for(y = 0; y < 1; y += 0.001)
				sum += 1 / (1 - x * y + x - y);
		RDUMP(sum);
	}

	using namespace Sqrat;

	double sum = 0;
	const String text =
	"function compute() {\n"
	"	local x, y, sum = 0.0;\n"
	"	for (x = 0.0; x < 1; x += 0.001)\n"
	"		for (y = 0.0; y < 1; y += 0.001)\n"
	"			sum += 1 / (1 - x * y + x - y);\n"
	"	return sum;\n"
	"}\n";

	// creates a VM with initial stack size 1024
	HSQUIRRELVM vm = sq_open(1024);

	DefaultVM::Set(vm);

	Script script;
	try {
		script.CompileString(~text);
	}
	catch (Error ex) {
		OutError(ex.Message(vm));
		return;
	}

	{
		RTIMING("Squirrel (fully interpreted)")
		try {
			script.Run();
		}
		catch(Error ex) {
			OutError(ex.Message(vm));
			return;
		}

		Function compute = RootTable().GetFunction("compute");
		if (!compute.IsNull()) {
			try {
				SharedPtr<double> val = compute.Evaluate<double>();
				sum = *val;
			}
			catch (Error ex) {
				OutError(ex.Message(vm));
				return;
			}
		}

		//sq_close(vm);
		RDUMP(sum);
	}
}
