#ifndef _treegrid_h_
#define _treegrid_h_

#include <CtrlLib/CtrlLib.h>

using namespace Upp;

#include <vector>
#include <map>


//===============================================================================
template<typename L, typename R> struct BIndex
{
	std::map<L, R> LMap;
	std::map<R, L> RMap;
	BIndex() {}
	virtual ~BIndex() {}
	void Add(const L &l, const R &r) { LMap[l]=r; RMap[r]=l; }
	void LDel(const L &l)
	{
		auto itl=LMap.find(l);
		if (itl!=LMap.end())
		{
			auto itr=RMap.find((*itl).second);
			if (itr!=RMap.end()) RMap.erase(itr);
			LMap.erase(itl);
		}
	}
	void RDel(const R &r)
	{
		auto itr=RMap.find(r);
		if (itr!=RMap.end())
		{
			auto itl=LMap.find((*itr).second);
			if (itl!=LMap.end()) LMap.erase(itl);
			RMap.erase(itr);
		}
	}
	bool LFind(const L &l, R &r)
	{
		auto it=LMap.find(l);
		if (it!=LMap.end()) { r=(*it).second; return true; }
		return false;
	}
	bool RFind(const R &r, L &l)
	{
		auto it=RMap.find(r);
		if (it!=RMap.end()) { l=(*it).second; return true; }
		return false;
	}
	size_t size() { return LMap.size(); }
	void Clear() { LMap.clear(); RMap.clear(); }
};


//-----------------------------------------------------------------------------
//todo: make these variables (if needed (customizing))...
///todo: G_ROW_CY - should be tossed & the individual cy of rows calculted from max cell-heights (+ 1 or 2 pixel spacing between rows)
#define G_ROW_CY (int)24
#define G_SPACE (int)16
#define G_GAP (int)8
///todo: G_PIC_WIDTH...larger pics? ... height?
#define G_PIC_WIDTH (int)16
///todo: G_LINE_HEIGHT should depend on font-size...which will effect cell-height(=>G_ROW_CY)...
#define G_LINE_HEIGHT (int)16

//using different (light/faint) pastels for contained expanded nodes ...
#define G_PASTEL_ONE Color(255,255,235)
#define G_PASTEL_ONE_ALT Color(255,255,210)
#define G_PASTEL_TWO White()
#define G_PASTEL_TWO_ALT Color(235,235,235)
#define G_PASTEL_THREE Color(235,255,255)
#define G_PASTEL_THREE_ALT Color(210,255,255)

#define G_COL_FG_DEFAULT Black()
#define G_COL_BG_DEFAULT White()

#define G_COL_FG_FOCUS White()
#define G_COL_BG_FOCUS Color(75, 105, 131)

#define G_COL_FG_SELECTED Black()
#define G_COL_BG_SELECTED Color(255, 120, 100)



//-------------------------------------------------------------------------------
class Tree;
class TreeGrid;
//--------------------------------------
class Cell
{
	Value val;
	const Display *dsp;
	bool bstddsp; //false if user-defined dsp
	bool has_display() const; //true == non-std-display/user-defined
	
	Callback when_resync;
	
public:
	Cell();
	Cell(const Value &v);
	Cell(const Cell &C);
	virtual ~Cell();

	Cell& operator=(const Cell &C);

	const Display&	GetDisplay() const;
	void			SetDisplay(const Display &d);
	const Value	GetData() const;
	void		SetData(const Value &val);
	
	Callback WhenCellChanged; //val-changed
	
	friend class Tree;
	friend class TreeGrid;
};
//---------------------------------------
typedef std::vector<Cell> Cells;


//-------------------------------------------------------------------------------
struct NodeData;
typedef NodeData* Node; //defined here for Column::Sorting/Sorter-callback


//-------------------------------------------------------------------------------
class Column : Moveable<Column>
{
	String title;
	int w, align;
	bool b_sort, b_asc;
	const Display *dsp;

public:
	Column();
	Column(const Column &C);
	Column(const String &title, int width);
	virtual ~Column();

	Column& Sorting(Gate2<const Node&, const Node&> sortfunc=nullptr);
	Gate2<const Node&, const Node&> Sorter;

	bool IsSorting() const;
	bool IsASC() const;
	void ToggleSort();

	Column&	Align(int align);
	int		Align() const;

	Column&			SetDisplay(const Display &d);
	const Display&	GetDisplay() const;

	const String&	Title() const;
	void			Title(const String &title);

	int		Width() const;
	void	Width(int width);

};
//---------------------------------------
//typedef std::vector<Column> Columns;
typedef Vector<Column> Columns;


//-------------------------------------------------------------------------------
struct Nodes : public std::vector<Node> , NoCopy
{
	Nodes();
	virtual ~Nodes();
	void clear(); //overloading vector::clear
	Node Add(String key);
	void Drop(Node&);
};


//-------------------------------------------------------------------------------
class NodeData
{
	Node ParentNode; //nullptr at root of tree
	Cells cells;
	Nodes nodes;
	String key;
	int y_pos, e_x_pos, p_x_pos, l_x_pos;
	int picid;
	bool b_expanded, b_selected, b_locked;
	
	bool has_nodes()			{ return (nodes.size()>0); }
	int get_indent();
	void expand();
	void contract();
	void select(bool b=true)	{ b_selected=b; }

	void add_cell(){}
	template<typename H, typename...T> void add_cell(H h, T...t)
	{
		AddCell(h);
		add_cell(t...);
	}

public:
	NodeData(String key="");
	virtual ~NodeData();

	void	Clear();

	size_t	NodeCount()			{ return nodes.size(); }
	void	ClearCells()		{ cells.clear(); }
	bool	IsEmpty()			{ return (nodes.size()==0); }

	Node	NodeAt(size_t idx); //in nodes --- why this?
	
	Node GetParent()			{ return ParentNode; }
	String Key()				{ return key; }
	bool CanExpand()			{ return (has_nodes()||IsLocked()); }
	bool IsExpanded()			{ return b_expanded; }
	bool IsVisible()			{ return (!ParentNode||ParentNode->IsExpanded()); } //true=>is at root or parent is expanded
	bool IsSelected()			{ return b_selected; }
	bool IsLocked()				{ return b_locked; } //lock is set by container

	void AddCell(const Value &v); //can add extra cells; displayed only up to column-count; cannot remove cells (must drop node)
	size_t	CellCount() { return cells.size(); } //may be > column-count
	Cell*	CellAt(size_t idx); //0==tree-label
	bool GetPic(Image &img);
	void SetPic(const Image &img);

	friend class Tree;
	friend class TreeGrid;
};


//-------------------------------------------------------------------------------
class Header : public Ctrl
{
	TreeGrid *owner;
	Columns cols;
	size_t CurCol, size_col;
	int left_down;
	bool b_is_sizing;

	void scroll_x(int xa); //xa == -horz-S/Bar-pos
	bool is_on_sizer(int p) const;
	int x_pos(size_t idx);
	bool col_at_x(int p, size_t &idx);
	bool set_cur_col(int p, size_t &idx); //p==x-coordinate of mouse-CLICK(&not IsSizer), idx set to col; True: curcol set
	int sort_pic_width(); //used for column-display adjustment in tree::paint()

	Callback1<size_t> WhenSorting;
	Callback WhenColumnAdded;

public:
	typedef Header CLASSNAME;
	
	Header();
	virtual ~Header();

	void Clear();
	
	virtual void Paint(Draw &drw);

	Image CursorImage(Point p, dword keyflags);

	virtual void LeftDown(Point p, dword keyflags);
	virtual void MouseMove(Point p, dword keyflags);
	virtual void LeftUp(Point p, dword keyflags);

	Column& Add(const String &title, int width);
	
	Column* ColumnAt(size_t idx);

	int Width();
	bool IsSizer(int p, size_t &idx) const; //p==x-coordinate of mouse-pointer, idx set to col-on-left
	
	friend class Tree;
	friend class TreeGrid;
};


//-------------------------------------------------------------------------------
enum EXDisp //EXpandDisposition
{
	EXD_NONE=0, //node-carrying nodes (node::nodes not empty) - no special treatment
	EXD_FIRST, //appear first in tree
	EXD_LAST, //last in tree
};
class Tree : public Ctrl
{
	TreeGrid *owner;
	Nodes roots;
	Node CurNode, sel_pivot;
	int lines_max_width, lines_total_height;
	bool bShowTreeLines, b_has_selected;
	EXDisp exd;
	VScrollBar SBV;
	HScrollBar SBH;
	size_t cur_column;
	
	int show_nodes(Draw &drw, Nodes &NS, int Y=0);
	void show_tree_lines(Draw &drw, Nodes &NS);
	void show_vert_grid(Draw &drw);
	int set_tree_positions(Nodes &NS, int Y);
	Node get_node(Nodes &NS, const String &key);
	Node get_last_node();
	bool check_node_key(String &key);
	Node get_y_node(int Y, Node N);
	Node get_node_at_y(int Y, Nodes &NS);
	bool check_expand(Node p, Point pt);
	void check_cur_node_contract(Node A);
	void sort_nodes(Header *pH, Nodes &NS, size_t c, bool b);
	void unselect_nodes(Nodes &NS);
	size_t get_selected_nodes(Nodes &NS, std::vector<String> &V); //recursive
	bool contains_locked_nodes(Node N);
	Rect get_abs_tree_rect();
	size_t set_click_column(int x);
	bool point_in_tree(Point p);
	Node prev_node(Node N);
	void sync_nodes();

	//AddNode() can fail (nullptr) if: no-mem, or duplicate key
	Node AddNode(Node Parent, const String &treetext, const String &key);
	Node AddNode(Node Parent, Image pic, const String &treetext, const String &key);

	Node GetNode(const String &key);
	void SelectNode(Node N, bool bSelect);
	void SelectRange(Node N1, Node N2);
	void Lock(Node N, bool b);
	void DropNode(Node N);
	void DropNodes(Nodes &NS);
	void Expand(Node N, bool b=true);
	void FocusNode(Node pn, bool bView=true);
	void BringToView(Node pn);
	void SortNodes(Header *pH, size_t cellidx, bool bASC);

	Callback2<Node, dword> WhenLeftDown;
	Callback2<Node, size_t> WhenLeftDouble;
	Callback1<Node> WhenRightDown;

	Callback1<Node> WhenFocus;
	Callback1<Node> WhenBeforeNodeExpand; //before expanding
	Callback1<Node> WhenAfterNodeContract; //before contracting

	void Clear();
	void Layout();
	void Paint(Draw &drw);
	void DoVScroll();
	void DoHScroll();
	virtual bool Key(dword key, int);
	virtual void MouseWheel(Point p, int zdelta, dword keyflags);
	virtual void LeftDown(Point p, dword flags);
	virtual void LeftUp(Point p, dword flags);
	virtual void RightDown(Point p, dword flags);
	virtual void LeftDouble(Point p, dword flags);
	virtual void MouseMove(Point p, dword flags);
	
	Image CursorImage(Point p, dword keyflags);

public:
	typedef Tree CLASSNAME;

	Tree();
	virtual ~Tree();

	friend class TreeGrid;
};


//-------------------------------------------------------------------------------
class TreeGrid : public Ctrl
{
	typedef TreeGrid CLASSNAME;

	Header header;
	Tree tree;
	size_t sort_col;
	bool b_vert_grid, b_internal; //b_internal: pad cells to column::count - see AddNode() and <>AddNode()
		
	void check_cells(Node N);
	void FillMenuBar(MenuBar &menubar);
	void OnTGHelp();
	void ShowPopMenu(Point p);
	void OnBNE(Node N);
	void OnANC(Node N);
	void OnLeftDown(Node N, dword flags);
	void OnLeftDouble(Node N, size_t cellidx);
	void OnRightDown(Node N);

public:
	TreeGrid();
	virtual ~TreeGrid();
	
	void Clear();
	void ClearTree();

	Column& AddColumn(const String title, int width);

	Node AddNode(Node Parent, const String &name, const String &key="");
	template<typename...CellData> Node AddNode(Node Parent, const String &name, const String &key, CellData...celldata)
	{
		Node N=tree.AddNode(Parent, name, key);
		if (N) { N->add_cell(celldata...); check_cells(N); }
		return N;
	}
	template<typename...CellData> Node AddNode(Node Parent, Image pic, const String &name, const String &key, CellData...celldata)
	{
		Node N=tree.AddNode(Parent, pic, name, key);
		if (N) { N->add_cell(celldata...); check_cells(N); }
		return N;
	}

	void RefreshTreeGrid();
	void DeleteNode(Node N);
	Node GetNode(const String &key);
	size_t RootCount();
	Node RootAt(size_t idx);
	void LockNode(Node N, bool bExpand=true);
	void UnlockNode(Node N);
	void Expand(Node N, bool b=true);
	void ClearNodes(Node N);

	void SetFocusNode(Node N, bool b=true);
	Node GetFocusNode();
	size_t GetFocusCell();
	Node GetPrevNode(Node N);

	void Select(Node N, bool bSelect=true);
	void SelectRange(Node N1, Node N2);
	void ClearSelection();
	size_t GetSelection(std::vector<String> &vSel);

	virtual bool Key(dword key, int r);

	void OnSorting(size_t idx);
	size_t GetSortColumnID();
	bool IsSortASC();

	void SyncNodes();

	bool NodeHasCell(Node N, size_t cellidx);
	Cell* GetCell(Node N, size_t cellidx);

	//----------------------------------customizing:
	void ListExpandingNodes(EXDisp exd); //EXD_FIRST, .._NONE, .._LAST
	EXDisp ListExpandingNodes();
	void ShowTreeLines(bool bShow);
	bool ShowTreeLines(); //i know, i know ..
	void ShowHeader(bool bShow);
	bool ShowHeader(); //told u i know!

	//----------------------------------
	Callback1<Node> WhenFocus;
	Callback2<Node, size_t> WhenActivate;
	Callback1<MenuBar&> WhenMenuBar;
	Callback WhenHelp;
	Callback1<Node> WhenBeforeNodeExpand; //before expanding
	Callback1<Node> WhenAfterNodeContract; //after contracting

	//----------------------------------
	friend class Header;
	friend class Tree;
};



#endif //#ifndef _treegrid_h_

