
#include "dircontent.h"
#include "util.h"
#include "linuxsys.h"
#include "custom.h"

#include <sstream>
#include <fstream>
#include <dirent.h>
#include <sys/types.h>
#include <pwd.h>
#include <sys/stat.h>
#include <ctime>
#include <cstdlib>
#include <cstring>
#include <thread>

//-----------------------------------------------------------------------------
#define  IMAGEFILE  <sfm/sfm.iml>
#define  IMAGECLASS CNTImg
#include <Draw/iml.h>


//-----------------------------------------------------------------------------
bool isint_s(std::string s) { for (auto c:s) { if ((c<'0')||(c>'9')) return false; } return true; }
#define SFAC(c) (c=='b')?1:(c=='K')?1000:(c=='M')?1000000:(c=='G')?1000000000:(c=='T')?1000000000000:0
size_t getsize_v(const std::string sv)
{
	std::string s;
	size_t n;
	s=sv.substr(0,sv.length()-3);
	TRIM(s);
	n=(isint_s(s))?stot<size_t>(s):0;
	n*=SFAC(sv[sv.length()-2]);
	return n;
}
int MySizeCompare(const std::string &a, const std::string &b)
{
	if (a.empty()||b.empty()) return (!a.empty())?1:((!b.empty())?(-1):0);
	size_t na, nb;
	na=getsize_v(a);
	nb=getsize_v(b);
	return (na>nb)?1:(na<nb)?(-1):0;
}
int MySizeCompare(const Value& a, const Value& b)
{
	bool baN=IsNull(a), bbN=IsNull(b);
	if (baN || bbN) return !baN ? 1 : !bbN ? -1 : 0;
	std::string sa=a.ToString().ToStd(), sb=b.ToString().ToStd();
	return MySizeCompare(sa, sb);
}

//-----------------------------------------------------------------------------
const Image add_mask(Image tgt, Image mask)
{
	ImageDraw idraw(tgt.GetSize());
	idraw.DrawRect(tgt.GetSize(), White());
	idraw.DrawImage(0, 0, tgt);
	idraw.DrawImage(0, 0, mask);
	return idraw.GetStraight();
}

//-----------------------------------------------------------------------------
void to_time_str(const time_t *t, std::string &sDate)
{
	struct tm *ptm;
	std::stringstream ss;
	ptm = localtime(t);
	ss << (ptm->tm_year+1900) //a #define or variable somewhere?
		<< "-" << (((ptm->tm_mon+1)<=9)?"0":"") << (ptm->tm_mon+1)
		<< "-" << ((ptm->tm_mday<=9)?"0":"") << ptm->tm_mday
		<< " " << ((ptm->tm_hour<=9)?"0":"") << ptm->tm_hour
		<< ":" << ((ptm->tm_min<=9)?"0":"") << ptm->tm_min
		<< ":" << ((ptm->tm_sec<=9)?"0":"") << ptm->tm_sec;
	sDate=ss.str(); ss.str(""); ss.flush();
}

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
DlgGetRunFile::DlgGetRunFile()
{
	CtrlLayout(*this, "Executable");
	Sizeable();
	b=false;
	d.Hide();
	a.NoHeader().NoGrid().AddColumn();
	a.WhenSel << [&]{ e.SetData(a.GetKey()); p.SetData(mex[(const char*)a.GetKey().ToString()].c_str()); };
	a.WhenLeftDouble << THISBACK(ok);
	k.WhenPush << THISBACK(ok);
	c.WhenPush << THISBACK(quit);
	x.WhenPush << THISBACK(browse);
}

void DlgGetRunFile::setmex(std::map<std::string, std::string> m)
{
	mex=m;
	for (auto p:mex) a.Add(p.first.c_str());
}

void DlgGetRunFile::browse()
{
	FileSel fs;
	fs.Type("All", "*");
	fs.ExecuteOpen();
	std::string s=fs.Get().ToStd();
	if (isexecutable(s)) e.SetData(s.c_str());
}


//-----------------------------------------------------------------------------
const std::string GetNewName(const std::string &title, const std::string &sdefault="")
{
	TopWindow dlg;
	std::string s="";
	dlg.SetRect(Size(350, 70));
	dlg.Title(title.c_str());
	EditString e;
	Button k;
	Button c;
	dlg.Add(e.HSizePosZ().TopPos(0,20));
	e.SetData(sdefault.c_str());
	dlg.Add(k.SetLabel(t_("OK")).RightPosZ(140, 60).TopPosZ(30, 20));
	dlg.Add(c.SetLabel(t_("Cancel")).RightPosZ(70, 60).TopPosZ(30, 20));
	k.WhenPush << [&]{ s=e.GetData().ToString().ToStd(); dlg.Close(); };
	c.WhenPush << [&]{ dlg.Close(); };
	dlg.Execute();
	TRIM(s);
	if (!sdefault.empty()&&(s==sdefault)) s="";
	return s;
}

//-----------------------------------------------------------------------------
inline const std::string k2p(const std::string &key) //Key-to-Path -chops off leading R/N
{
	std::string sd="";
	if (!key.empty()&&(key[0]!='/')) sd=key.substr(1);
	return sd;
}
inline const std::string k2p(String key) { return k2p(key.ToStd()); }

//-----------------------------------------------------------------------------
bool IsDir(Node N) { return (isextant(k2p(N->Key()), DE_DIR)); }
	

//===================================================================================================================================================
//===============================================================================PropsDlg
PropsDlg::PropsDlg() { N=nullptr; bdirty=false; }
PropsDlg::PropsDlg(Node P) { N=P; sDE=k2p(N->Key()); bdirty=false; ShowInfo(); }
PropsDlg::~PropsDlg() { if (bdirty&&PromptYesNo("Do you want to apply changes?")) OnApply(); }
void PropsDlg::OnClose() { Close(); }

void PropsDlg::ShowInfo()
{
	CtrlLayout(*this, "Properties");
	Sizeable();
	
	btnClose.WhenPush << THISBACK(OnClose);
	btnReset.WhenPush << THISBACK(OnReset);
	btnApply.WhenPush << THISBACK(OnApply);
	
	if (!sDE.empty())
	{
		std::string s="", st;
		std::stringstream ss;
		struct stat STAT;
		struct passwd *pw;
	
		lstat(sDE.c_str(), &STAT); //---links
		ebName.SetData(sDE.c_str());
		ebSize.SetData(ttos<size_t>(STAT.st_size).c_str());
		pw=getpwuid(STAT.st_uid);
		ebOwner.SetData(pw->pw_name);
		ebType.SetData(getfiletypeinfo(sDE).c_str());
		if ((STAT.st_mode&S_IFMT)==S_IFLNK)
		{
			char buf[1024];
			int n=readlink(sDE.c_str(), (char*)&buf, 1023);
			buf[n]=0;
			ebLink.SetData((char*)buf);
		}
		else ebLink.Clear();
		
		OnReset();
	}
}

void PropsDlg::OnChange() { bdirty=true; btnReset.Enable(true); btnApply.Enable(true); }

void PropsDlg::OnReset()
{
	struct stat STAT;
	lstat(sDE.c_str(), &STAT);
	bool bchg;
	std::string sdt;

	bdirty=false;

	to_time_str(&STAT.st_atime, sdt); ebAccessed.SetData(sdt.c_str());
	to_time_str(&STAT.st_ctime, sdt); ebChanged.SetData(sdt.c_str());
	to_time_str(&STAT.st_mtime, sdt); ebModified.SetData(sdt.c_str());
	
	bchg=(this_userid()==STAT.st_uid); //owner
	
	chkOwnR=((STAT.st_mode&S_IRUSR)==S_IRUSR); chkOwnR.Enable(bchg); chkOwnR.WhenPush << THISBACK(OnChange);
	chkOwnW=((STAT.st_mode&S_IWUSR)==S_IWUSR); chkOwnW.Enable(bchg); chkOwnW.WhenPush << THISBACK(OnChange);
	chkOwnX=((STAT.st_mode&S_IXUSR)==S_IXUSR); chkOwnX.Enable(bchg); chkOwnX.WhenPush << THISBACK(OnChange);

	chkGrpR=((STAT.st_mode&S_IRGRP)==S_IRGRP); chkGrpR.Enable(bchg); chkGrpR.WhenPush << THISBACK(OnChange);
	chkGrpW=((STAT.st_mode&S_IWGRP)==S_IWGRP); chkGrpW.Enable(bchg); chkGrpW.WhenPush << THISBACK(OnChange);
	chkGrpX=((STAT.st_mode&S_IXGRP)==S_IXGRP); chkGrpX.Enable(bchg); chkGrpX.WhenPush << THISBACK(OnChange);

	chkOthR=((STAT.st_mode&S_IROTH)==S_IROTH); chkOthR.Enable(bchg); chkOthR.WhenPush << THISBACK(OnChange);
	chkOthW=((STAT.st_mode&S_IWOTH)==S_IWOTH); chkOthW.Enable(bchg); chkOthW.WhenPush << THISBACK(OnChange);
	chkOthX=((STAT.st_mode&S_IXOTH)==S_IXOTH); chkOthX.Enable(bchg); chkOthX.WhenPush << THISBACK(OnChange);
	
	btnReset.Enable(bdirty);
	btnApply.Enable(bdirty);
}

void PropsDlg::OnApply()
{
	if (bdirty)
	{
		std::string sc, sm="", sr="";
		int n=0;
		
		if (chkOwnR) { n+=4; sr="r"; } else sr="_";
		if (chkOwnW) { n+=2; sr+="w"; } else sr+="_";
		if (chkOwnX) { n+=1; sr+="x"; } else sr+="_";
		sm+=ttos<int>(n);
		
		n=0;
		if (chkGrpR) { n=4; sr+="r"; } else sr+="_";
		if (chkGrpW) { n+=2; sr+="w"; } else sr+="_";
		if (chkGrpX) { n+=1; sr+="x"; } else sr+="_";
		sm+=ttos<int>(n);
		
		n=0;
		if (chkOthR) { n=4; sr+="r"; } else sr+="_";
		if (chkOthW) { n+=2; sr+="w"; } else sr+="_";
		if (chkOthX) { n+=1; sr+="x"; } else sr+="_";
		sm+=ttos<int>(n);
		
		spf(sc, "chmod %s \"%s\"", sm.c_str(), sDE.c_str());
		SysCmd(sc);

		N->CellAt(3)->SetData(sr.c_str());
		
		OnReset();
	}
}


//===================================================================================================================================================
//===============================================================================DirContent
DirContent::DirContent()
{
	std::string thisuser=username();
	int y=33;

	sRNDir="";
	custom.exd=EXD_FIRST; //directories first

	Add(Pan.HSizePosZ().TopPos(0, y));
	Pan.btnFind.Tip("Find Entries");		Pan.btnFind.WhenPush << THISBACK(OnPanelFind);
	Pan.btnHome.Tip("Home Directory");		Pan.btnHome.WhenPush << THISBACK(OnPanelHome);
	Pan.btnRefresh.Tip("Reload this tab");	Pan.btnRefresh.WhenPush << THISBACK(OnPanelRefresh);
	Pan.btnNodesTop.Data("T");	Pan.btnNodesTop.Tip("Arrange Directories first");	Pan.btnNodesTop.WhenPush << THISBACK(OnPanelNodeTBN);
	Pan.btnNodesBot.Data("B");	Pan.btnNodesBot.Tip("Arrange Directories last");	Pan.btnNodesBot.WhenPush << THISBACK(OnPanelNodeTBN);
	Pan.btnNodesNone.Data("N");	Pan.btnNodesNone.Tip("Don't arrange Directories");	Pan.btnNodesNone.WhenPush << THISBACK(OnPanelNodeTBN);
	Pan.btnNodesTop.Select();
	
	if (thisuser.compare("root")==0)
	{
		Add(IAmROOT.SetFrame(ThinOutsetFrame()).HSizePosZ(0, 0).TopPosZ(y, 20));
		y+=24;
	}
	
	Add(TG.HSizePosZ().VSizePosZ(y, 0));

	TG.AddColumn("Tree", 400).Sorting();
	TG.AddColumn("Size", 80).Align(ALIGN_RIGHT).Sorting(THISBACK(sort_size_col)); //idx==1
	TG.AddColumn("Type", 80).Align(ALIGN_CENTER).Sorting();
	TG.AddColumn("Rights", 100).Align(ALIGN_CENTER); //idx=3 --- use consts for these indices?
	TG.AddColumn("Owner", 80).Align(ALIGN_CENTER).Sorting();
	TG.AddColumn("Modified", 150).Sorting();

	TG.WhenFocus << THISBACK(OnFocus);
	TG.WhenBeforeNodeExpand	<< THISBACK(OnBExpand);
	TG.WhenAfterNodeContract << THISBACK(OnAContract);
	TG.WhenActivate << THISBACK(OnActivate);
	TG.WhenMenuBar << THISBACK(OnMenuBar);
	TG.WhenHelp << THISBACK(OnMenuHelp);

	TG.ListExpandingNodes(custom.exd);
	TG.Refresh();
}

DirContent::~DirContent() { TG.Clear(); }

void DirContent::OnPanelFind(PicButton *pb)
{
	if (dlgfind.IsOpen()) dlgfind.SetFocus();
	else { dlgfind.set_pdc(this); dlgfind.Open(nullptr); dlgfind.ebDir.SetData(GetDCDir().c_str()); }
}

void DirContent::OnPanelHome(PicButton *pb)
{
	std::string sk="N", sn;
	sk+=homedir();
	sn=getdename(homedir());
	WhenOpenTab(sn, sk, true, "");
}

void DirContent::OnPanelRefresh(PicButton *pb)
{
	Node R=TG.RootAt(0), F=TG.GetFocusNode();
	String fk=F->Key();
	TG.UnlockNode(R);
	TG.Expand(R, false);
	TG.LockNode(R, true);
	ExpandFocusKey(fk.ToStd());
}

void DirContent::OnPanelNodeTBN(PicButton *pb)
{
	Pan.btnNodesTop.Select(false);
	Pan.btnNodesBot.Select(false);
	Pan.btnNodesNone.Select(false);
	if (pb) pb->Select(true);
	custom.exd=(pb->Data()[0]=='T')?EXD_FIRST:(pb->Data()[0]=='B')?EXD_LAST:EXD_NONE;
	TG.ListExpandingNodes(custom.exd);
}
bool DirContent::sort_size_col(const Node &l, const Node &r)
{
	size_t idxSizeCol=1;
	bool b=TG.IsSortASC();
	int n=0;
	if ((IsDir(l)&&!IsDir(r))||!l->CellCount()) return b;
	if ((!IsDir(l)&&IsDir(r))||!r->CellCount()) return !b;
	n=MySizeCompare(l->CellAt(idxSizeCol)->GetData(), r->CellAt(idxSizeCol)->GetData());
	if (b) return (n<0);
	return (n>0);
}

const std::string DirContent::GetDCDir()
{
	std::string sd="";
	if (!sRNDir.empty()) sd=sRNDir.substr(1);
	return sd;
}

void DirContent::SetLock(const std::string &key, bool bLock)
{
	Node N=TG.GetNode(key);
	if (N)
	{
		if (bLock) { if (IsDir(N)) TG.LockNode(N, false); }
		else { if (IsDir(N)) { TG.AddNode(N, "*"); TG.UnlockNode(N); }}
	}
}

void DirContent::FocusNodeKey(const std::string &key)
{
	ExpandFocusKey(key);
}

void DirContent::ExpandFocusKey(const std::string &key)
{
	std::string p,d="",k;
	std::vector<std::string> v;
	Node F;
	
	p=k2p(key);
	v=splitpath(p);
	for (auto s:v)
	{
		d+=s;
		k="N"; k+=d;
		if (d[d.size()-1]!='/') d+="/";
		F=TG.GetNode(k);
		TG.Expand(F);
		TG.SetFocusNode(F);
	}
}

bool DirContent::Key(dword key, int c) { return TG.Key(key, c); }

bool DirContent::HotKey(dword key)
{
	Node N=TG.GetFocusNode();
	dword k=key;
	bool shift=k&K_SHIFT;
	bool ctrl=k&K_CTRL;
	bool alt=k&K_ALT;
	U_RIGHTS R=R_NONE, rr;

	if (shift) k&=~K_SHIFT;
	if (ctrl) k&=~K_CTRL;
	if (alt) k&=~K_ALT;
	if (N) { R=getrights(k2p(N->Key().ToStd())); }
	switch(k)
	{
		case K_R:	if (ctrl&&CAN_RUN(R))		{ OnMenuRun(N);		return true; } break;
		case K_O:	if (ctrl&&CAN_OPEN(R))		{ OnMenuOpen(N);	return true; } break;
		case K_W:	if (ctrl&&CAN_OPEN(R))		{ OnOpenWith(N);	return true; } break;
		case K_C:	if (ctrl&&CAN_OPEN(R))		{ OnMenuCopy();		return true; } break;
		case K_V:	if (ctrl&&CAN_MODIFY(R))	{ OnMenuPaste(N);	return true; } break;
		case K_Z:	if (ctrl&&CAN_MODIFY(R))	{ OnMenuMove(N);	return true; } break;
		case K_F2:	if (CAN_MODIFY(R))			{ OnMenuRename(N);	return true; } break;
		case K_F:	if (ctrl&&CAN_MODIFY(R))	{ OnMenuNewDir(N);	return true; } break;
		case K_T:	if (ctrl&&CAN_MODIFY(R))	{ OnMenuNewTextFile(N);	return true; } break;
		case K_DELETE: { if (CAN_MODIFY(R)) { if (ctrl&&shift) OnMenuErase(); else OnMenuDelete(); return true; }} break;
		case K_F3:	{ OnMenuProperties(N);	return true; } break;
		case K_F1:	{ OnMenuHelp();			return true; } break;
	}
	return false;
}

void DirContent::Load() //NB: called from tabxctrl::AddTab() to fill tree (see SFM-ctor)
{
	std::string sDir, sFN="";
	Image pic;
	Node Root;
	U_RIGHTS R;

	if (sRNDir.empty()) return;
	sDir=sRNDir.substr(1);
	if ((R=getrights(sDir))!=R_NONE)
	{
		get_root_stats(sDir);
		pic=(CAN_OPEN_DIR(R))?CNTImg::PicFolder():CNTImg::PicNoEntry();
		sFN=(sDir=="/")?"Root":getdename(sDir);
		Root=TG.AddNode(nullptr, pic, sFN.c_str(), sRNDir.c_str(), EI.sSize.c_str(), "<dir>",
						EI.sRights.c_str(), EI.sOwner.c_str(), EI.sDate.c_str());
		if (Root)
		{
			TG.AddNode(Root, "*"); //dummy entry to show expand
			TG.LockNode(Root, true); //true=>lock in expanded state
			TG.SetFocusNode(Root);
		}
	}
}

void DirContent::get_root_stats(std::string &sDir)
{
	struct stat STAT;
	std::stringstream ss;

	EI.clear();
	lstat(sDir.c_str(), &STAT);
	EI.sSize=to_kmgt(STAT.st_size);
	EI.sOwner=username(STAT.st_uid);
	ss << ((STAT.st_mode & S_IRUSR)?"r":"_") << ((STAT.st_mode & S_IWUSR)?"w":"_") << ((STAT.st_mode & S_IXUSR)?"x":"_"); //speed
	ss << ((STAT.st_mode & S_IRGRP)?"r":"_") << ((STAT.st_mode & S_IWGRP)?"w":"_") << ((STAT.st_mode & S_IXGRP)?"x":"_");
	ss << ((STAT.st_mode & S_IROTH)?"r":"_") << ((STAT.st_mode & S_IWOTH)?"w":"_") << ((STAT.st_mode & S_IXOTH)?"x":"_");
	EI.sRights=ss.str(); ss.str(""); ss.flush();
	to_time_str(&STAT.st_mtime, EI.sDate);
}

const std::string DirContent::to_kmgt(long long n)
{
	std::string s("");
	std::stringstream ss, st;
	int p;
	ss << n;
	p = ss.str().length();
	if (p<=3) st << n << " b "; //byte < 999
	else if (p<=6) st << ss.str().substr(0,p-3) << " Kb"; //kilo-byte 10^3
	else if (p<=9) st << ss.str().substr(0,p-6) << " Mb"; //mega-byte 10^6
	else if (p<=12) st << ss.str().substr(0,p-9) << " Gb"; //giga-byte 10^9
	else if (p<=15) st << ss.str().substr(0,p-12) << " Tb"; //tera-byte 10^12
	else st << "(lots) Pb"; //peta-byte 10^15 //exa-byte 10^18 //zetta-byte 10^21 //yotta-byte 10^24
	s=st.str();
	return s;
}

void DirContent::load_dir_entries(Node parent, std::string sDir)
{
	struct dirent *pd;
	DIR *dir;
	std::string sw, skey;
	int r;
	Node N=nullptr;

	WaitCursor WC; WC.Show();
	dir = opendir(sDir.c_str());
	if (dir)
	{
		while ((pd = readdir(dir)) != NULL)
		{
			sw=sDir; sw+=((sw[sw.length()-1]=='/')?"":"/"); sw+=pd->d_name;
			skey="N"; skey+=sw;
			if ((r=get_stats(pd, sw))!=DO_DISCARD)
			{
				if (r==DO_DIR)
				{
					U_RIGHTS R=getrights(sw);
					Image pic=(CAN_OPEN_DIR(R))?CNTImg::PicFolder():CNTImg::PicNoEntry();
					N = TG.AddNode(parent, pic, EI.sName.c_str(), skey.c_str(),
									EI.sSize.c_str(), EI.sType.c_str(), EI.sRights.c_str(),
									EI.sOwner.c_str(), EI.sDate.c_str());
					if (N) { if (CheckIsTabOpen(skey, false)) TG.LockNode(N); else TG.AddNode(N, "*"); }
				}
				else TG.AddNode(parent, EI.pic, EI.sName.c_str(), skey.c_str(),
								EI.sSize.c_str(), EI.sType.c_str(), EI.sRights.c_str(),
								EI.sOwner.c_str(), EI.sDate.c_str());
			}
		}
		closedir(dir);
	}
}

int DirContent::get_stats(struct dirent *pd, const std::string &sDE)
{
	struct stat STAT;
	std::stringstream ss;
	int Retval=DO_FILE;

	EI.sName=pd->d_name;
	if ((EI.sName==".")||(EI.sName=="..")) return DO_DISCARD;
	if (lstat(sDE.c_str(), &STAT)!=0) return DO_DISCARD; ///todo:...show errno etc...
	EI.sOwner=username(STAT.st_uid);
	EI.sSize=to_kmgt(STAT.st_size);
	ss << ((STAT.st_mode & S_IRUSR)?"r":"_") << ((STAT.st_mode & S_IWUSR)?"w":"_") << ((STAT.st_mode & S_IXUSR)?"x":"_"); //speed
	ss << ((STAT.st_mode & S_IRGRP)?"r":"_") << ((STAT.st_mode & S_IWGRP)?"w":"_") << ((STAT.st_mode & S_IXGRP)?"x":"_");
	ss << ((STAT.st_mode & S_IROTH)?"r":"_") << ((STAT.st_mode & S_IWOTH)?"w":"_") << ((STAT.st_mode & S_IXOTH)?"x":"_");
	EI.sRights=ss.str(); ss.str(""); ss.flush();
	to_time_str(&STAT.st_mtime, EI.sDate);

	if (((pd->d_type&DT_DIR)==DT_DIR)&&!((pd->d_type&DT_LNK)==DT_LNK))
	{
		EI.pic=CNTImg::PicFolder();
		EI.sType="<dir>";
		Retval=DO_DIR;
	}
	else
	{
		switch (STAT.st_mode&S_IFMT)
		{
			case S_IFBLK: //fall-thru
			case S_IFCHR: { EI.pic=CNTImg::PicDevice(); EI.sType="device"; } break;
			case S_IFIFO: { EI.pic=CNTImg::PicPipe(); EI.sType="pipe"; } break;
			case S_IFLNK:
				{
					char buf[1024]; //should be max-path?
					int n=readlink(sDE.c_str(), (char*)&buf, 1023);
					if (n>=0) buf[n]=0; else return DO_DISCARD; ///todo:...show errno etc...
					EI.sName+=" -> "; EI.sName+=(char*)buf;
					EI.pic=CNTImg::PicLink();
					EI.sType="link";
				} break;
			case S_IFREG: { EI.pic = CNTImg::PicFile(); EI.sType="file"; } break;
			case S_IFSOCK: { EI.pic = CNTImg::PicSocket(); EI.sType="socket"; } break;
			default: { EI.pic=CNTImg::PicUnknown(); EI.sType="(unknown)"; } break;
		}
	}
	return Retval;
}

void DirContent::reload_focus(Node N, const std::string sKey)
{
	if (N)
	{
		String key=sKey.c_str(); if (key.IsEmpty()) key=N->Key();
		Node T=((N->GetParent())?N->GetParent():N);
		if (T->IsLocked()) { TG.UnlockNode(T); TG.Expand(T, false); TG.LockNode(T, true); }
		else { TG.Expand(T, false); TG.Expand(T); }
		T=TG.GetNode(key); //key will be attached to a new node
		if (T) { TG.Expand(T); TG.SetFocusNode(T); }
	}
	else TG.RefreshTreeGrid();
}

void DirContent::OnFocus(Node N)		{ if (N) Pan.CurPath(k2p(N->Key())); else Pan.CurPath(""); }

void DirContent::OnBExpand(Node N)		{ TG.ClearNodes(N); if (IsDir(N)) { load_dir_entries(N, k2p(N->Key().ToStd())); }}
void DirContent::OnAContract(Node N)	{ TG.ClearNodes(N); if (IsDir(N)) { if (!N->IsLocked()) TG.AddNode(N, "*"); }}

void DirContent::OnActivate(Node N, size_t cellidx)
{
	ClearStatusText();
	if (N)
	{
		std::string sn;
		if (IsDir(N))
		{
			sn=getdename(k2p(N->Key().ToStd()));
			WhenOpenTab(sn, N->Key().ToStd(), true, "");
		}
		else
		{
			sn=k2p(N->Key().ToStd());
			if (isexec(sn)) RunApp(sn);
			else if (isshell(sn)) SysTerm(sn);
			//else if (isperl(sn)) ...
			//..
			else OnOpenWith(N);
		}
	}
}

void DirContent::OnMenuBar(MenuBar &bar)
{
	Node N=TG.GetFocusNode();
	bar.Clear();
	ClearStatusText();
	if (N->IsLocked()&&N->GetParent()) { bar.Add("Open", THISBACK1(OnMenuOpen,N)).Key(K_CTRL|K_O); return; }
	if (N)
	{
		U_RIGHTS rr, R=getrights(k2p(N->Key().ToStd()));
		if (!IsDir(N)) rr=getrights(k2p(N->GetParent()->Key().ToStd())); else rr=R;
		if (CAN_OPEN(R)||CAN_RUN(R))
		{
			if (CAN_RUN(R)) bar.Add("Run", THISBACK1(OnMenuRun,N)).Key(K_CTRL|K_R);
			if (CAN_OPEN_DIR(R)) bar.Add("Open", THISBACK1(OnMenuOpen,N)).Key(K_CTRL|K_O);
			if (CAN_OPEN_FILE(R)) bar.Add("Open with ..", THISBACK1(OnOpenWith,N)).Key(K_CTRL|K_W);
			bar.Separator();
		}
		if (CAN_MODIFY(R))
		{
			if (!N->IsLocked()) { bar.Add("Rename", THISBACK1(OnMenuRename,N)).Key(K_F2); bar.Separator(); }
			if (CAN_MODIFY_DIR(rr))
			{
				bar.Add("New Folder", THISBACK1(OnMenuNewDir,N)).Key(K_CTRL|K_F);
				bar.Add("New Textfile", THISBACK1(OnMenuNewTextFile,N)).Key(K_CTRL|K_T);
				bar.Separator();
			}
		}
		if (CAN_OPEN(R)) //is contained in CAN_MODIFY
		{
			bar.Add("Copy (Select)", THISBACK(OnMenuCopy)).Key(K_CTRL|K_C);
			if (CAN_MODIFY(R))
			{
				if (CAN_MODIFY_DIR(rr)) if (cb_has_files())
				{
					bar.Add("Paste", THISBACK1(OnMenuPaste,N)).Key(K_CTRL|K_V);
					if (!N->IsLocked()) bar.Add("Move here", THISBACK1(OnMenuMove,N)).Key(K_CTRL|K_H);
					
					//if (system_has_tool("rsync")) -- "Copy" (clipboard) then "Backup" - src-dir/files to tgt-dir==FocusNode(or ->GetParent())...
					//bar.Add("Backup", THISBACK1(OnMenuBackup,N)).Key(K_CTRL|K_B);
					//if (cb_has_files()) - ... synchronize files --- backup must call rsync ... later
					
				}
			}
			bar.Separator();
		}
		if (CAN_MODIFY(R))
		{
			if (!N->IsLocked())
			{
				bar.Add("Delete", THISBACK(OnMenuDelete)).Key(K_DELETE); //moved to dumpster
				bar.Add("Erase", THISBACK(OnMenuErase)).Key(K_CTRL|K_SHIFT|K_DELETE);
				bar.Separator();
			}
		}
		bar.Add("Properties", THISBACK1(OnMenuProperties,N)).Key(K_F3);
	}
	bar.Separator();
	bar.Add("Help", THISBACK(OnMenuHelp)).Key(K_F1);
	bar.Add("Customize", THISBACK(OnCustomize));
}

void DirContent::OnCustomize(Bar &bar)
{
	bar.Add("Expandable nodes first", Callback(lambda([&]{ custom.exd=EXD_FIRST; TG.ListExpandingNodes(custom.exd); })));
	bar.Add("Expandable nodes last", Callback(lambda([&]{ custom.exd=EXD_LAST; TG.ListExpandingNodes(custom.exd); })));
	bar.Add("Expandable nodes default", Callback(lambda([&]{ custom.exd=EXD_NONE; TG.ListExpandingNodes(custom.exd); })));
	bar.Separator();
	bar.Add((TG.ShowTreeLines())?"Hide Treelines":"Show Treelines", Callback(lambda([&]{ TG.ShowTreeLines(!TG.ShowTreeLines()); })));
}

void DirContent::OnMenuOpen(Node N) { ClearStatusText(); OnActivate(N, TG.GetFocusCell()); }

void DirContent::OnOpenWith(Node N)
{
	ClearStatusText();
	if (N)
	{
		std::string sa="", sn, sde=k2p(N->Key().ToStd());
		if (!sde.empty())
		{
			sn=getdename(sde);
			DlgGetRunFile dlg;
			std::map<std::string, std::string> mex;

			spf(sn, "Open %s", sn.c_str());
			dlg.Title(sn.c_str());
			size_t nvx=getexecs(sde, mex);
			if (nvx==INIT_PREP_BUSY) { PromptOK("&OYou're too quick!&&sfm is still initializing (see Statusbar)&&Try again in a few moments&"); return; }
			if (nvx==0)
			{
				if (PromptYesNo("&No directly associated applications were found.&&Do you want a list of possibly usable applications?&"))
					nvx=getexecs_possible(sde, mex);
			}
			if (nvx>0) dlg.setmex(mex);
			else
			{
				spf(sa, "No application found.\nThe following information may be helpful:\nFile-type: %s", getfiletypeinfo(sde).c_str());
				dlg.a.Hide(); dlg.o.Hide();
				dlg.d.Show();
				dlg.d.SetData(sa.c_str());
			}
			dlg.Execute();
			if (dlg.b)
			{
				sa=dlg.e.GetData().ToString().ToStd();
				if (!sa.empty())
				{
					if (mex.find(sa)!=mex.end()) RunAppFile(mex[sa], sde);
					else if (isexec(sa)) RunAppFile(sa, sde);
					//else if (isshell(sa) ...
					//else if (isperl(sa) ...
					else
					{
						spf(sn, "&&I'm sorry, %s. I'm afraid I can't do that.&&", username().c_str());
						PromptOK(sn.c_str());
					}
				}
			}
		}
	}
}

void DirContent::OnMenuRun(Node N) { if (N) OnActivate(N, TG.GetFocusCell()); }

void DirContent::OnMenuRename(Node N)
{
	ClearStatusText();
	if (N&&!N->IsLocked()&&N->GetParent())
	{
		Node P=N->GetParent();
		std::string sOldName, oldkey, sNewName, newkey;
		char prefix;

		prefix=N->Key().ToStd()[0];
		oldkey=k2p(N->Key()); if (oldkey.size()<=1) return; //root or empty
		sOldName=getdename(oldkey);

		newkey=k2p(P->Key()); if (newkey.empty()) return;
		newkey+=(newkey[newkey.size()-1]=='/')?"":"/";
		
		sNewName=GetNewName("Rename", sOldName);
		if (isvalidname(sNewName)&&(sNewName!=sOldName))
		{
			newkey+=sNewName;
			if (!isextant(newkey))
			{
				movede(oldkey, newkey);
				spf(newkey, "%c%s", prefix, newkey.c_str());
				reload_focus(P, newkey);
				ShowStatusText("Renamed");
			}
			else { spfa(newkey, " - already exists."); PromptOK(newkey.c_str()); }
		}
	}
}

void DirContent::OnMenuNewDir(Node N)
{
	ClearStatusText();
	if (!N) return;
	Node T=((IsDir(N))?N:N->GetParent()); //in target-dir
	if (!T) return;

	std::string tgt, sP=k2p(T->Key()), sn=GetNewName("Directory Name", "New Directory");

	if (!isextant(sP, DE_DIR)) return; //?
	sP+=(sP[sP.size()-1]=='/')?"":"/";
	tgt=sP; tgt+=sn;
	if (isvalidname(sn)&&(!isextant(tgt)))
	{
		if (createdir(tgt))
		{
			spf(tgt, "N%s", tgt.c_str());
			reload_focus(T, tgt);
		}
		else { spf(tgt, "Failed to create directory:&%s", tgt.c_str()); ShowStatusText(tgt); } ///todo:...errno/reason...
	}
	else ShowStatusText("Name is invalid or already exists.");
}

void DirContent::OnMenuNewTextFile(Node N)
{
	ClearStatusText();
	if (!N) return;
	Node T=(IsDir(N))?N:N->GetParent();
	if (!T) return;

	std::string tgt, sP=k2p(T->Key()), sn=GetNewName("Textfile Name", "New Textfile.txt");
	
	if (!isextant(sP, DE_DIR)) return; //?
	sP+=(sP[sP.size()-1]=='/')?"":"/";
	tgt=sP; tgt+=sn;
	if (isvalidname(sn)&&(!isextant(tgt)))
	{
		if (createfile(tgt))
		{
			spf(tgt, "N%s", tgt.c_str());
			reload_focus(T, tgt);
		}
		else { spf(sn, "Failed to create textfile:&%s", tgt.c_str()); ShowStatusText(sn); } ///todo:...errno/reason...
	}
	else ShowStatusText("Name is invalid or already exists.");
}

void normalize_selection(std::vector<String> &v)
{
	//remove subs if parent is in selection
	if (v.size()<=1) return;
	size_t i=0;
	volatile size_t n;
	std::sort(v.begin(), v.end());
	n=v.size();
	while (i<(n-1))
	{
		if (issub(v[i+1].ToStd(), v[i].ToStd()))
		{
			auto it=(v.begin()+i+1);
			v.erase(it);
			n=v.size();
		}
		else i++;
	}
}

void DirContent::OnMenuCopy()
{
	std::string sfiles="", sformat="text/uri-list";
	std::vector<String> v;
	
	ClearStatusText();
	if (TG.GetSelection(v)==0) { Node N=TG.GetFocusNode(); if (N) v.push_back(N->Key()); }
	normalize_selection(v);
	if (v.size()!=0)
	{
		ClearClipboard();
		for (auto S:v) { spfa(sfiles, "file://%s\n", k2p(S).c_str()); }
		WriteClipboard(sformat.c_str(), sfiles.c_str());
	}
}

bool DirContent::cb_has_files() { return (IsAvailableFiles(Clipboard())); }

bool verify_pasteparms(const std::string &tgtdir, const Vector<String> &Vf)
{
	//not to self
	std::string st, den;
	bool b=true;
	for (auto S:Vf)
	{
		den=getdename(S.ToStd());
		spf(st, "%s%s%s", tgtdir.c_str(), (tgtdir[tgtdir.size()-1]=='/')?"":"/", den.c_str());
		if (st==S.ToStd()) { b=false; break; }
	}
	return b;
}

void DirContent::OnMenuPaste(Node N)
{
	if (!N) return;

	std::string tdir, k=N->Key().ToStd();
	int ncopies=0;
	Node T=((IsDir(N))?N:N->GetParent());
	Vector<String> Vf;
	
	ClearStatusText();
	Vf=GetFiles(Clipboard()); if (Vf.IsEmpty()) return;
	tdir=k2p(T->Key().ToStd()); ///...rights...shouldhavebeenchecked by menu... ?
	if (verify_pasteparms(tdir, Vf))
	{
		WaitCursor WC; WC.Show();
		if (tdir[tdir.size()-1]!='/') tdir+="/";
		for (auto S:Vf)
		{
			if (isextant(S.ToStd(), DE_DIR)) { copydir(S.ToStd(), tdir); ncopies++; }
			else { copyfile(S.ToStd(), tdir); ncopies++; }
		}
		reload_focus(T);
	}
	else ShowStatusText("Cannot copy to self");
	spf(tdir, "Copied %d files/directories", ncopies);
	ShowStatusText(tdir);
}

void DirContent::OnMenuMove(Node N)
{
	if (!N) return;

	std::string tdir, k=N->Key().ToStd();
	int ncopies=0;
	Node T=((IsDir(N))?N:N->GetParent());
	Vector<String> Vf;
	
	ClearStatusText();
	Vf=GetFiles(Clipboard()); if (Vf.IsEmpty()) return;
	tdir=k2p(T->Key().ToStd()); ///...rights...shouldhavebeenchecked by menu... ?
	if (verify_pasteparms(tdir, Vf))
	{
		WaitCursor WC; WC.Show();
		if (tdir[tdir.size()-1]!='/') tdir+="/";
		for (auto S:Vf) { if (isextant(S.ToStd())) { movede(S.ToStd(), tdir); ncopies++; }}
		reload_focus(T);
	}
	else ShowStatusText("Cannot move to self");
	spf(tdir, "Moved %d files/directories", ncopies);
	ShowStatusText(tdir);
}

bool valid_for_delete(std::vector<String> &v)
{
	if (v.size())
	{
		for (auto S:v)
		{
			if (S.Find(".sfm")>=0)
			{
				PromptOK("&Use 'Erase' to remove entries from '.sfm'");
				return false;
			}
		}
		return true;
	}
	return false;
}

Node get_toggle_node(Node F, std::vector<String> &v)
{
	Node R=nullptr, T=F;
	if (!v.size()) R=F;
	else
	{
		bool b;
		std::string s;
		while (!R&&T)
		{
			b=false;
			s=k2p(T->Key());
			for (auto S:v) { if (issub(s, k2p(S))) { b=true; break; }}
			if (!b) { R=T; break; } else T=T->GetParent();
		}
	}
	return R;
}

void DirContent::OnMenuDelete()
{
	std::vector<String> v;
	bool b=true;
	Node F=TG.GetFocusNode();

	ClearStatusText();
	if (!TG.GetSelection(v)) { if (F) v.push_back(F->Key()); }
	if ((v.size()!=0)&&valid_for_delete(v))
	{
		WaitCursor WC; WC.Show();
		F=get_toggle_node(F, v);
		for (auto S:v) do_delete_node(TG.GetNode(S));
		reload_focus(F);
		ShowStatusText("Selection deleted");
	}
}

void DirContent::OnMenuErase()
{
	std::vector<String> v;
	bool b=true;
	Node F=TG.GetFocusNode();

	ClearStatusText();
	if (TG.GetSelection(v)==0) { if (F) v.push_back(F->Key()); }
	if (v.size()>0)
	{
		if (PromptOKCancel("&Consider when last you made backups ...&This will permanently remove the selection.&&Are you sure?&"))
		{
			WaitCursor WC; WC.Show();
			F=get_toggle_node(F, v);
			for (auto S:v) deletede(k2p(S));
			reload_focus(F);
			ShowStatusText("Selection erased");
		}
	}
}

void DirContent::OnMenuHelp()
{
	std::string s = "\nStarted off as a test of TreeGrid and then got out of control ...\n"
					"\nMouse:\n"
					"     o  (normal stuff)\n"
					"     o  Right-click : context popup menu\n"
					"     o  Dbl-click : new tab or run or open with..\n"
					"     o  Ctrl+Left-click : multiple row select\n"
					"\nKeyboard:\n"
					"     o  Ctrl+M : context popup menu\n"
					"     o  (menu-hotkeys)\n"
					"     o  Enter : (same as Dbl-click)\n"
					"     o  Home/End : first/last cell in row\n"
					"     o  Ctrl+Home/End : top and bottom of tree\n"
					"     o  Up/Down, PageUp/PageDown : (as expected)\n"
					"     o  Shift+Up/Down arrow-keys : multiple row select\n"
					"     o  Left/Right arrow-keys : selects along the cells in a row\n"
					"     o  Ctrl+Left/Right arrow-keys : expand/contract current node (if it can)\n"
					"\nOther info:\n"
					"     o  Delete : 'mv's the selected files&dirs to ~/.sfm/dumpster/\n"
					"     o  Paste to existing target : backup to ~/.sfm/backup/<target> and numbered in target\n"
					"\nTodo's: (Still under construction - feedback appreciated)\n"
					" - ...still far from smooth and clean...\n"
					" - Cut / Move ?\n"
					" - undelete (dumpster-diving)\n"
					"(- TreeGrid To Do:\n"
					"    - make rows variable height and text-content wrapable\n"
					"    - adding controls to cells/columns)\n"
					" - and and and ...\n";
					
	PromptOK(DeQtf(s.c_str()));
}

void DirContent::OnMenuProperties(Node N)
{
	if (N) { PropsDlg dlg(N); dlg.Execute(); }
	else PromptOK("No current selection");
}

bool DirContent::create_dot_sfm()
{
	std::string sD="", s="";
	spf(sD, "%s/.sfm", homedir().c_str()); if (!createdir(sD)) return false;
	spf(s, "%s/dumpster", sD.c_str()); if (!createdir(s)) return false;
	spf(s, "%s/backup", sD.c_str()); if (!createdir(s)) return false;
	return true;
}

bool DirContent::do_delete_node(Node N)
{
	bool b=false;
	if (N && create_dot_sfm())
	{
		WaitCursor WC; WC.Show();
		std::string sdump="", k=k2p(N->Key().ToStd());
		spf(sdump, "%s/.sfm/dumpster/", homedir().c_str());
		movede(k, sdump);
		b=!isextant(k);
	}
	return b;
}

void DirContent::do_backup(const std::string sS)
{
	if (!isextant(sS)) return;
	std::string sbup="";

	WaitCursor WC; WC.Show();
	if (create_dot_sfm())
	{
		spf(sbup, "%s/.sfm/backup/", homedir().c_str());
		backup(sS, sbup);
	}
}

