
#include "linuxsys.h"
#include "util.h"

#include <string>
#include <cstdlib>
#include <map>
#include <fstream>
#include <thread>
#include <sstream>
#include <ctime>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <unistd.h>
//#include <cerrno> ... later


std::string ls_err_text;

//==========================================================================================================================================================
//-------------------------------------------------------------------------------user & process
int this_userid() { return getuid(); }
int this_appid() { return getpid(); }

const std::string this_appname()
{
	std::string s;
	std::ifstream inf("/proc/self/cmdline");
	inf >> s;
	return s;
}

const std::string username(int uid)
{
	std::string s;
	struct passwd *pw=getpwuid(uid);
	s=pw->pw_name;
	return s;
}

const std::string homedir(int uid)
{
	std::string s;
	struct passwd *pw=getpwuid(uid);
	s=pw->pw_dir;
	return s;
}


//==========================================================================================================================================================
//-----------------------------------------------------------------------------dirs & files
bool isextant(const std::string &sDE, DE_TYPE det)
{
	struct stat st;
	if (stat(sDE.c_str(), &st)>=0)
	{
		int t=(st.st_mode&S_IFMT);
		switch(det)
		{
			//case DE_EXIST: return true;
			case DE_FILE:	return (t==S_IFREG);
			case DE_DIR:	return (t==S_IFDIR);
			case DE_DEVICE:	return ((t==S_IFBLK)||(t==S_IFCHR));
			case DE_PIPE:	return (t==S_IFIFO);
			case DE_LINK:	return (t==S_IFLNK);
			case DE_SOCKET:	return (t==S_IFSOCK);
			default: return true;
		}
	}
	return false;
}

DE_TYPE getdetype(const std::string &sDE)
{
	struct stat st;
	if (stat(sDE.c_str(), &st)>=0)
	{
		switch(st.st_mode&S_IFMT)
		{
			case S_IFREG:	return DE_FILE;
			case S_IFDIR:	return DE_DIR;
			case S_IFBLK: //..
			case S_IFCHR:	return DE_DEVICE;
			case S_IFIFO:	return DE_PIPE;
			case S_IFLNK:	return DE_LINK;
			case S_IFSOCK:	return DE_SOCKET;
		}
	}
	return DE_ERROR;
}

bool existdir(const std::string &sD) { return (isextant(sD, DE_DIR)); }

bool createdir(const std::string &sD)
{
	if (isextant(sD)&&!existdir(sD)) return false; //name exists & is not a dir
	if (!existdir(sD))
	{
		std::string s;
		spf(s, "mkdir \"%s\"", sD.c_str());
		system(s.c_str());
	}
	return (existdir(sD));
}

bool deletedir(const std::string &sD)
{
	if (existdir(sD))
	{
		std::string s;
		spf(s, "rm -r \"%s\"", sD.c_str());
		system(s.c_str());
	}
	return (!existdir(sD));
}

bool existfile(const std::string &sF) { return (isextant(sF, DE_FILE)); }

bool createfile(const std::string &sF)
{
	std::string sc;
	spf(sc, "touch \"%s\"", sF.c_str());
	SysCmd(sc);
	return (existfile(sF));
}

bool deletefile(const std::string &sF)
{
	if (existfile(sF)) deletede(sF); //unlink(sF.c_str());
	return (!existfile(sF));
}

const std::vector<std::string> splitpath(const std::string &sP)
{
	std::vector<std::string> vp;
	vp.push_back("/");
	splitslist(sP, '/', vp, false);
	return vp;
}

void movede(const std::string &sDE, const std::string &sTgt)
{
	std::string sc;
	spf(sc, "mv -f --strip-trailing-slashes --backup=numbered \"%s\" \"%s\"", sDE.c_str(), sTgt.c_str());
	SysCmd(sc);
}

void deletede(const std::string &sDE)
{
	std::string sc;
	spf(sc, "rm -r \"%s\"", sDE.c_str());
	SysCmd(sc);
}

const std::string getdename(const std::string &sDE)
{
	std::string s="";
	if (isextant(sDE))
	{
		std::vector<std::string> v=splitpath(sDE);
		s=v[v.size()-1];
	}
	return s;
}

const std::string getdepath(const std::string &sDE)
{
	std::string s="";
	if (isextant(sDE))
	{
		std::vector<std::string> v=splitpath(sDE);
		s=v[0];
		for (size_t i=1;i<v.size()-1;i++) { s+=v[i]; s+="/"; }
		RTRIM(s,"/");
	}
	return s;
}

void backup(const std::string &sDE, const std::string &sTgt)
{
	std::string sc;
	spf(sc, "cp -r -P --preserve=all --strip-trailing-slashes --backup=numbered \"%s\" \"%s\"", sDE.c_str(), sTgt.c_str());
	SysCmd(sc);
}

void copydir(const std::string &sDF, const std::string &sDT)
{
	if (existdir(sDF)&&existdir(sDT)) backup(sDF, sDT);
}

bool copyfile(const std::string &sF, const std::string &sT)
{
	if (existfile(sF)) backup(sF, sT);
	return (existfile(sT));
}

bool issub(const std::string &sSub, const std::string &sParent)
{
	if (sParent==sSub) return true;
	if (sSub.size()<=sParent.size()) return false;
	size_t i=0;
	while ((i<sParent.size())&&(sParent[i]==sSub[i])) i++;
	return ((i==sParent.size())&&(sSub[i]=='/'));
}

void findfiles(const std::string &sD, bool bS, const std::string &sN, bool bN, std::vector<std::string> &vFound)
{
	std::string sc;
	spf(sc, "find -P \"%s\"%s-%s \"%s\"", sD.c_str(), (!bS)?" -maxdepth 0 ":" ", (bN)?"name":"iname", sN.c_str());
	sc=SysCmd(sc);
	vFound.clear();
	splitslist(sc, '\n', vFound, false);
}

void grepfiles(const std::string &sD, bool bS, const std::string &sC, bool bC, std::vector<std::string> &vFound)
{
	std::string sc;
	spf(sc, "cd \"%s\" ; grep -ls -m 1 %s%s\"%s\"", sD.c_str(), (bS)?"-r ":"", (bC)?" ":"-i ", sC.c_str());
	sc=SysCmd(sc);
	vFound.clear();
	splitslist(sc, '\n', vFound, false);
	sc=sD; sc+=(sc[sc.size()-1]=='/')?"":"/";
	for (auto &f:vFound) spf(f, "%s%s", sc.c_str(), f.c_str());
}


//==========================================================================================================================================================
//-------------------------------------------------------------------------------file-info
//--------------------------------------
//check if needed utilities are available...
bool system_has_tool(const std::string &t)
{
	std::string sc="which ", sr;
	sc+=t;
	sr=SysCmd(sc);
	TRIM(sr);
	return (!sr.empty());
}
bool is_a_useable_system(bool bredo=false)
{
	ls_err_text="";
	static int is_ok=(-1);
	if (bredo) is_ok=(-1); //if needed
	if (is_ok==1) return true;
	if (is_ok==0) return false;
	bool b=true;
	std::string sc, sr, sneed="";
	if (!system_has_tool("file")) { b=false; sneed+="file\n"; }
	if (!system_has_tool("grep")) { b=false; sneed+="grep\n"; }
	if (!system_has_tool("find")) { b=false; sneed+="find\n"; }
	if (!system_has_tool("touch")) { b=false; sneed+="touch\n"; }
	//if (!system_has_tool("shred")) { b=false; sneed+="shred\n"; }
	//.. awk sed ..
	is_ok=(b)?1:0;
	if (!b) { ls_err_text="Missing needed tools:\n"; ls_err_text+=sneed; }
	return b;
}
//--------------------------------------

bool isfiletype(const std::string &sF, FT_TYPE ftt)
{
	if (!is_a_useable_system()) return false;
	
	std::string sc, st="";
	std::string sl="if [ -x \"$1\" ] && file \"$1\" | grep -qi ";
	std::string sr=" ; then echo \"1\" ; else echo \"0\" ; fi";
	
	switch(ftt)
	{
		case FT_EXECUTABLE: st="\" elf \""; break;
		case FT_SHELL_SCRIPT: st="\"shell script\""; break;
		case FT_PERL_SCRIPT: st="\"perl script\""; break;
		//..
	}
	if (st.empty()) return false;
	
	sc=sl;  sc+=st;  sc+=sr;  ReplacePhrase(sc, "$1", sF);
	//eg.: if [ -x "/usr/bin/leafpad" ] && file "/usr/bin/leafpad" | grep -qi " elf " ; then echo "1" ; else echo "0" ; fi
	st=SysCmd(sc);
	TRIM(st); //lf/cr/?
	return (st=="1");
}

const std::string getfiletypeinfo(const std::string &sF)
{
	std::string st="", sc="file \"";
	if (is_a_useable_system())
	{
		sc+=sF;  sc+="\"";
		st=SysCmd(sc.c_str());
		int p=st.find(':');
		if (p!=std::string::npos) st=st.substr(p+1);
		TRIM(st);
	}
	return st;
}

bool isvalidname(const std::string &sN)
{
	std::string s=sN;
	TRIM(s);
	if (s.empty()) return false;
	if (s.find('/')!=std::string::npos) return false;
	//?
	return true;
}

bool isexec(const std::string &sF)			{ if (!existfile(sF)) return false; return (isfiletype(sF, FT_EXECUTABLE)); }
bool isshell(const std::string &sF)			{ if (!existfile(sF)) return false; return (isfiletype(sF, FT_SHELL_SCRIPT)); }
bool isperl(const std::string &sF)			{ if (!existfile(sF)) return false; return (isfiletype(sF, FT_PERL_SCRIPT)); }
//..
bool isexecutable(const std::string &sF)	{ if (!existfile(sF)) return false; return (isexec(sF)||isshell(sF)||isperl(sF)); }

const std::string mimetype(const std::string &sF)
{
	std::string st="", sc="file --mime-type \"";
	if (is_a_useable_system())
	{
		sc+=sF;  sc+="\"";
		st=SysCmd(sc.c_str());
		TRIM(st);
		if (st.size())
		{
			int p;
			if ((p=st.find(":"))!=std::string::npos) { st=st.substr(p+1); TRIM(st); }
		}
	}
	return st;
}

//--------------------------------------
//search for "*.desktop" in "$HOME/.local/share", /usr/local/share/, /usr/share/
typedef std::map<std::string, std::map<std::string, std::string> > MTX; //..[mimetype]->>[name]->executable-full-path
MTX mtx; //re-used
bool b_1st_time_mtx=false;//true; --- rather use prepexecs()

void process_dtf(const std::string &sf, MTX &mtx)
{
	char buf[4096];
	std::string s, sl, sc, srcN="name=", srcM="mimetype=", srcE="exec=", sn="", sm="", se="";
	std::ifstream inf(sf);
	while (inf.good()&&(sm.empty()||se.empty()))
	{
		inf.getline(buf, 4096, '\n');
		sl=s=buf;
		lcase(sl);
		if (sl.substr(0, srcN.size())==srcN) { sn=s.substr(srcN.size()); TRIM(sn); }
		if (sl.substr(0, srcM.size())==srcM) { sm=s.substr(srcM.size()); TRIM(sm); }
		if (sl.substr(0, srcE.size())==srcE) { se=s.substr(srcE.size()); TRIM(se); }
	}
	if (!se.empty())
	{
		TRIM(se);
		std::string sw=se;
		int p; if ((p=se.find(' '))!=std::string::npos) sw=se.substr(0, p);
		sc="which \""; sc+=sw; sc+="\""; sw=SysCmd(sc);
		if (!sw.empty()) se=sw;
		TRIM(se);
		if (!isextant(se)) se.clear();
	}
	if (!sn.empty()) TRIM(sn); else sn=se;
	if (sm=="application;") { TRIM(sm); TRIM(sm, ";"); }
	if (!sm.empty()&&!se.empty())
	{
		std::vector<std::string> vs;
		lcase(sm); //easier to find later...
		splitslist(sm, ';', vs, false);
		for (auto s:vs) { if (!s.empty()) mtx[s][sn]=se; }
	}
}
void get_process_dtf(const std::string &SD, MTX &mtx)
{
	std::string sc, sdtf, sf, src=".desktop";
	int f=0,t=0;
	
	spf(sc, "find \"%s\" -iname \"*.desktop\"", SD.c_str());
	sdtf=SysCmd(sc);
	TRIM(sdtf);
	sdtf+="\n";
	
	t=sdtf.find(src.c_str(), f);
	while (t!=std::string::npos)
	{
		t+=src.size();
		sf=sdtf.substr(f, t-f);
		TRIM(sf);
		process_dtf(sf, mtx);
		f=t;
		t=sdtf.find(src.c_str(), f);
	}
}
void fill_mtx(MTX &mtx)
{
	std::string sd=homedir();
	mtx.clear();
	sd+="/.local/share";
	get_process_dtf(sd, mtx); //search homedir
	get_process_dtf("/usr/local/share", mtx);
	get_process_dtf("/usr/share", mtx);
}

//--------------------------------------
bool b_prepexecs_busy;
void prepexecs() { mtx.clear(); fill_mtx(mtx); b_1st_time_mtx=false; b_prepexecs_busy=false; }

//--------------------------------------

size_t getexecs(const std::string &sFile, std::map<std::string, std::string> &ms, bool bSearchAgain)
{
	if (b_prepexecs_busy) return INIT_PREP_BUSY; //mutex not needed
	ms.clear();
	if (!existfile(sFile)) return 0;
	if (b_1st_time_mtx||bSearchAgain||mtx.empty()) prepexecs(); //NB this may take some time...
	std::string smt=mimetype(sFile);
	if (!smt.empty()&&!mtx.empty())
	{
		lcase(smt);
		std::vector<std::string> vm;
		if (splitslist(smt, ';', vm))
		{
			for (auto s:vm)
			{
				auto it=mtx.find(s);
				if (it!=mtx.end()) { for (auto m:(*it).second) ms[m.first]=m.second; }
			}
		}
	}
	return ms.size();
}

size_t getexecs_possible(const std::string &sFile, std::map<std::string, std::string> &ms)
{
	if (b_prepexecs_busy) return INIT_PREP_BUSY; //mutex not needed
	ms.clear();
	if (!existfile(sFile)||mtx.empty()) return 0;
	std::string sb, se, smt=mimetype(sFile);
	if (!smt.empty())
	{
		lcase(smt);
		int p;
		if ((p=smt.find('/'))!=std::string::npos)
		{
			sb=smt.substr(0, p);
			//application audio example image message model multipart text video
			if (IsInStrSet(sb, "[audio][image][message][text][video]"))
			{
				auto it=mtx.begin();
				while (it!=mtx.end())
				{
					if ((p=(*it).first.find('/'))!=std::string::npos)
					{
						se=(*it).first.substr(0, p);
						if (sb==se) for (auto m:(*it).second) ms[m.first]=m.second;
					}
					it++;
				}
			}
		}
	}
	return ms.size();
}

U_RIGHTS getrights(const std::string &sDE)
{
	if (!isextant(sDE)) return R_NONE;
	struct stat st;
	int uid=getuid();
	struct passwd *pw=getpwuid(uid);
	U_RIGHTS R=R_NONE;
	DE_TYPE det=getdetype(sDE);
	std::string sc;
	std::vector<std::string> vg;
	bool bgok=false;

	if (stat(sDE.c_str(), &st)<0) return R;
	
	if (!uid||(uid==st.st_uid)) //root/owner
	{
		if (isexecutable(sDE)) R = R_FMOD | R_FRUN;
		else if (det==DE_DIR) R = R_DMOD;
		else R = R_FMOD;
		return R;
	}

	spf(sc, "id -G %s", pw->pw_name);
	sc=SysCmd(sc);
	if (!sc.empty()) //check group-access
	{
		int n, degid=(int)st.st_gid;
		if ((n=splitslist(sc, ' ', vg))>0) { for (size_t i=0;(!bgok&&(i<n));i++) bgok=(stot<int>(vg[i])==degid); }
	}
	
	if (det==DE_FILE)
	{
		if (isexecutable(sDE)&&((st.st_mode&S_IXOTH)||(bgok&&(st.st_mode&S_IXGRP)))) R = R_FRUN;
		if ((st.st_mode&S_IWOTH)||(bgok&&(st.st_mode&S_IWGRP))) R += R_FMOD;
		else if ((st.st_mode&S_IROTH)||(bgok&&(st.st_mode&S_IRGRP))) R += R_FOPEN;
		return R;
	}

	if (det==DE_DIR)
	{
		if ((st.st_mode&S_IWOTH)||(bgok&&(st.st_mode&S_IWGRP))) R = R_DMOD;
		else if ((st.st_mode&S_IROTH)||(bgok&&(st.st_mode&S_IRGRP))) R = R_DOPEN;
		return R;
	}
			
	return R;
}


//==========================================================================================================================================================
//-----------------------------------------------------------------------------system()-calls
struct clean_sys_terms
{
	int dummy;
	clean_sys_terms() { dummy=0; }
	~clean_sys_terms();
};
clean_sys_terms c_s_t;

const std::string make_tmp_file_name()
{
	std::string tfn;
	spf(tfn, "/tmp/%s", get_unique_name().c_str());
	return tfn;
}

const std::string SysCmd(const std::string &cmd)
{
	std::string st, s=cmd, fn=make_tmp_file_name();
	s+=" >"; s+=fn;
	if (system(s.c_str())>=0)
	{
		std::stringstream ss;
		std::ifstream inf(fn.c_str());
		ss << inf.rdbuf();
		s=ss.str();
	}
	st="rm \""; st+=fn; st+="\"";
	system(st.c_str());
	return s;
}

std::map<std::string,std::string> m_shell_calls;
void remove_shell_id(const std::string &id)
{
	auto it=m_shell_calls.find(id);
	if (it!=m_shell_calls.end()) m_shell_calls.erase(it);
}

void do_sys_blocking_shell(const std::string sc, const std::string id) 
{
	system(sc.c_str());
	if (existfile(id.c_str())) deletefile(id.c_str());
	remove_shell_id(id);
}

void SysTerm(const std::string &command)
{
	std::string id=make_tmp_file_name(), sc, sh=SysCmd("echo \"$SHELL\"");
	std::string waitenter="echo \"Press Enter to close...\" ; key=1 ; while [ \"$key\" != \"\" ]; do read -n1 -s key ; done";
	std::ofstream ofs(id);
	if (ofs.good()) ofs << "#!" << sh << command << "\n" << waitenter << "\nrm \"" << id << "\""; ofs.close();
	sc="chmod 755 \""; sc+=id; sc+="\"";
	SysCmd(sc); //create script to be called

	///todo: ... determine default terminal emulator to use ...
	///... using xterm for now ...
	
	sc="xterm -e \""; sc+=id; sc+="\"";
	std::thread (do_sys_blocking_shell, sc, id).detach(); //don't block sfm
	m_shell_calls[id]=sc;
	c_s_t.dummy++;
}

void kill_sys_shell(const std::string &id)
{
	std::string sc;
	spf(sc, "ps ax |grep \"%s\" | grep tty", m_shell_calls[id].c_str());
	sc=SysCmd(sc);
	if (!sc.empty())
	{
		spf(sc, "kill %d", stot<int>(sc.substr(0, sc.find(' ')))); //soft-kill
		SysCmd(sc);
	}
}

void CleanupSysTerms()
{
	auto it=m_shell_calls.begin();
	while (it!=m_shell_calls.end())
	{
		if (existfile((*it).first.c_str())) kill_sys_shell((*it).first);
		it++;
	}
	m_shell_calls.clear();
}

clean_sys_terms::~clean_sys_terms()
{
	if (dummy>0) CleanupSysTerms();
}


//==============================================================================
#define MAX_PARMS 100 //crazy
#define MAX_PARM_SIZE 5120 //make it 5K ///max is command-line-max i think too lazy to check
#define SPACE_CHAR 32
bool is_valid_space(const std::string s, int pos)
{
	if (s[pos]!=SPACE_CHAR) return false;
	if (!pos) return true;
	if (s[pos-1]=='\\') return false;
	return true;
}
void RunApp(const std::string sApp, const std::string sParm)
{
	std::string sTEMP;
	sTEMP = getenv("PATH");
	if (!isexec(sApp)) return;
	int n=0,i=0;
	while (i < (int)sParm.length()) { if (is_valid_space(sParm, i)) n++; i++; }
	if ((n>=(MAX_PARMS-1)) || ((sApp.length() + sParm.length() + 1) > MAX_PARM_SIZE)) return;
	pid_t pID = fork();
	if (pID == 0)
	{ //child..
		if (sParm.length()!=0)
		{
			static char *arsz[MAX_PARMS]; //pointers to offsets in sz[]-buffer
			static char sz[MAX_PARM_SIZE];
			int p=1,t=0,i=0,n;
			n = sApp.length(); //copy app into sz[] as first parm
			while (i<n) { sz[i]=sApp.at(i); i++; }
			sz[i] = 0;
			t = i+1;
			arsz[0] = sz;
			n = sParm.length();
			i=0;
			while (i<n)
			{
				if (is_valid_space(sParm, i)) i++;
				if ((i<n) && (!is_valid_space(sParm, i)))
				{
					arsz[p] = sz+t; p++;
					while ((i<n) && (!is_valid_space(sParm, i)))
					{
						sz[t]=sParm.at(i);
						t++;
						i++;
					}
					sz[t]=0; t++;
				}
			}
			sz[t]=0; //double null at end
			execvp(sApp.c_str(), arsz);
		} //NB: here-be-zombies: must waitpid in caller
		else execlp(sApp.c_str(), sApp.c_str(), (char*)0);
		exit(0); //only if exec..-error
	}
}

void RunAppFile(const std::string sApp, const std::string sFile)
{
	if (!isexec(sApp)) return;
	std::string sTEMP;
	sTEMP = getenv("PATH");
	int n=0,i=0;
	pid_t pID = fork();
	if (pID == 0)
	{ //child..
		setsid(); //if (setsid()<0) ...
		if (sFile.length()!=0)
		{
			static char *arsz[2]; //[0]=app-name and [1]=file
			static char sz[MAX_PARM_SIZE];
			int t=0,i=0,n;
			n = sApp.length(); //copy app into sz[] as first parm
			i=0;
			while (i<n) { sz[i]=sApp.at(i); i++; }
			sz[i] = 0; //zero-term first parm
			t = i+1; //next pos
			arsz[0]=sz; //first parm
			arsz[1]=sz+t; //second parm
			n = sFile.length();
			i=0;
			while (i<n) { sz[t+i]=sFile.at(i); i++; }
			sz[t+i]=0;
			sz[t+i+1]=0; //double null at end
			execvp(sApp.c_str(), arsz);
		} //NB: here-be-zombies: must waitpid in caller
		else execlp(sApp.c_str(), sApp.c_str(), (char*)0);
		exit(0);
	}
}



