
#include "treegrid.h"

#include <cstring> //sprintf
#include <map>

#define IMAGEFILE <TreeGrid/treegrid.iml>
#define IMAGECLASS TreeGridPics
#include <Draw/iml.h>


//==============================================================================================================================================================
//===============================================================================
const Color inv_col(const Color C) { return Color(~C.GetR(), ~C.GetG(), ~C.GetB()); } //invert color


//==============================================================================================================================================================
//===============================================================================Cell
Cell::Cell() : val(""), dsp(&StdDisplay()), bstddsp(true) {}
Cell::Cell(const Value &v) : val(v), dsp(&StdDisplay()), bstddsp(true) {}
Cell::Cell(const Cell &C) : val(C.val), dsp(C.dsp), bstddsp(C.bstddsp) {}
Cell::~Cell() {}

bool Cell::has_display() const				{ return !bstddsp; }

Cell& Cell::operator=(const Cell &C)		{ val=C.val; dsp=C.dsp; }

void Cell::SetDisplay(const Display &d)		{ dsp=&d; bstddsp=false; when_resync(); }
const Display& Cell::GetDisplay() const		{ return *dsp; }
void Cell::SetData(const Value &v)			{ val=v; when_resync(); WhenCellChanged(); }
const Value Cell::GetData() const			{ return val; }


//==============================================================================================================================================================
//===============================================================================Column
Column::Column() : title(""), w(0), align(ALIGN_LEFT), b_asc(true), b_sort(false), dsp(&StdDisplay()) {}
Column::Column(const Column &C) : title(C.title), w(C.w), align(C.align), b_asc(C.b_asc), b_sort(C.b_sort), dsp(C.dsp), Sorter(C.Sorter) {}
Column::Column(const String &title, int width) : title(title), w(width), align(ALIGN_LEFT), b_asc(true), b_sort(false), dsp(&StdDisplay()) {}
Column::~Column() {}

Column& Column::Sorting(Gate2<const Node&, const Node&> sortfunc)
{
	b_sort=true;
	Sorter << sortfunc;
	return *this;
}

bool Column::IsSorting() const { return b_sort; }
bool Column::IsASC() const { return b_asc; }
void Column::ToggleSort() { b_asc=!b_asc; }

Column& Column::Align(int align)
{
	this->align=align;
	if (align==ALIGN_RIGHT) dsp=&StdRightDisplay();
	else if (align==ALIGN_CENTER) dsp=&StdCenterDisplay();
	else dsp=&StdDisplay();
	return *this;
}

int Column::Align() const						{ return align; }
Column& Column::SetDisplay(const Display &d)	{ dsp=&d; return *this; }
const Display& Column::GetDisplay() const		{ return *dsp; }
const String& Column::Title() const				{ return title; }
void Column::Title(const String &title)			{ this->title=title; }
int Column::Width() const						{ return w; }
void Column::Width(int width)					{ w=width; }


//==============================================================================================================================================================
//===============================================================================Nodes
Nodes::Nodes() {}
Nodes::~Nodes() { clear(); }
BIndex<Node, String> node_keys;
void Nodes::clear()
{
	///for (auto &p:(*this)) { node_keys.LDel(p); p->Clear(); delete p; }
	///std::vector<Node>::clear();
	while (size()) Drop((*begin()));
}

Node Nodes::Add(String key)
{
	Node R = new NodeData(key);
	if (R)
	{
		node_keys.Add(R, R->Key());
		push_back(R);
	}
	return R;
}

void Nodes::Drop(Node &p)
{
	auto it=begin();
	while (it!=end()) { if ((*it)==p) { node_keys.LDel(p); delete p; p=nullptr; erase(it); break; } it++; }
}

//==============================================================================================================================================================
//===============================================================================NodeData
//--------------------------------------- (reduces NodeData(Node) size)
struct node_pic_list : std::map<int, std::pair<Image, int> > //[id]=[image, usecount]
{
	node_pic_list() { clear(); }
	~node_pic_list() { clear(); }
	
	int Add(Image img)
	{
		int i=1;
		for (auto &p:(*this)) { if (p.second.first==img) { p.second.second++; return p.first; } else if (i==p.first) i++; }
		(*this)[i]=std::pair<Image, int>(img, 1);
		return i;
	}
	
	void Drop(int pid)
	{
		auto it=find(pid);
		if (it!=end()) { if ((*it).second.second<=1) erase(it); else (*it).second.second--; }
	}
	
	bool Get(int pid, Image &img) { if (find(pid)!=end()) { img=(*this)[pid].first; return true; } return false; }
};

node_pic_list node_pics;

//---------------------------------------
NodeData::NodeData(String skey)
{
	ParentNode=nullptr;
	Clear();
	if (skey.IsEmpty())
	{
		int i=0, n=(sizeof(uintptr_t)+2);
		char s[n]; while (i<n) s[i++]=0;
		sprintf(s, "K%llu", this); //use 'this' for default key-value
		key=s;
	}
	else key=skey;
}

NodeData::~NodeData() { Clear(); }

void NodeData::Clear()
{
	cells.clear();
	nodes.clear();
	key.Clear();
	y_pos=e_x_pos=p_x_pos=l_x_pos=0;
	node_pics.Drop(picid); picid=0;
	b_expanded=b_selected=b_locked=false;
}

int NodeData::get_indent()				{ int n=0; Node p=ParentNode; while (p) { n++; p=p->ParentNode; } return n; }
void NodeData::expand()					{ b_expanded=((!ParentNode||ParentNode->IsExpanded())&&(nodes.size()>0)); }
void NodeData::contract()				{ b_expanded=false; }
Node NodeData::NodeAt(size_t idx)		{ if (idx<nodes.size()) return nodes[idx]; else return nullptr; }
void NodeData::AddCell(const Value &v)	{ cells.push_back(Cell(v)); }
Cell* NodeData::CellAt(size_t idx)		{ if (idx<cells.size()) return &cells[idx]; else return nullptr; }
bool NodeData::GetPic(Image &img)		{ return node_pics.Get(picid, img); }
void NodeData::SetPic(const Image &img)	{ node_pics.Drop(picid); picid=node_pics.Add(img); }


//==============================================================================================================================================================
//===============================================================================Header
Header::Header()	{ Clear(); owner=nullptr; }
Header::~Header()	{ Clear(); }

void Header::Clear() { cols.clear(); CurCol=size_col=0; left_down=0; b_is_sizing=false; }
	
void Header::Paint(Draw &drw)
{
	Size sz=GetSize();
	drw.DrawRect(sz, SColorFace());
	Image picsort;
	Color fg=Black(), bg=SColorFace();

	if (cols.size()>0)
	{
		size_t i;
		int x, w, h=G_LINE_HEIGHT, xd, yd, px, py, pw, align;
	
		for (i=0;i<cols.size();i++)
		{
			if (cols[i].Width()>1)
			{
				//text=cols[i].title.c_str();
				align=cols[i].Align();
				x=x_pos(i);
				w=cols[i].Width();
				px=py=pw=0;
				
				drw.DrawRect(x, 0, w, h, bg);
				drw.DrawLine(x, 0, x, h, 1, White());
				drw.DrawLine(x+w-1, 0, x+w-1, h, 1, Black()); //show "sizer"
	
				if (i==owner->sort_col)
				{
					picsort=(cols[i].IsASC())?TreeGridPics::PicASC():TreeGridPics::PicDSC();
					py=(G_LINE_HEIGHT-picsort.GetHeight())/2; //vert-center
					px=x+w-picsort.GetWidth()-3; //3-pixels away from R side
					pw=picsort.GetWidth();
				}
	
				if (!cols[i].Title().IsEmpty())
				{
					const Display &dsp=cols[i].GetDisplay();
					Rect r(x+3, 0, x+w-pw-3, h); //3-pixels away from L & R sides
					dsp.Paint(drw, r, cols[i].Title(), fg, bg, 0);
				}
				if (px&&py) drw.DrawImage(px, py, picsort);
			}
		}
	}
}

void Header::scroll_x(int xa)
{
	if (xa>0) xa=0;
	SetRectX(xa, GetSize().cx-xa);
	Refresh();
}

bool Header::is_on_sizer(int p) const
{
	size_t t;
	return IsSizer(p, t);
}

Image Header::CursorImage(Point p, dword keyflags)
{
	if (b_is_sizing) return TreeGridPics::PicSizer();
	if (is_on_sizer(p.x)) return TreeGridPics::PicSizer();
	return Image::Arrow();
}

void Header::LeftDown(Point p, dword keyflags)
{
	size_t idx;
	if (b_is_sizing)
	{
		left_down=0;
		b_is_sizing=false;
		size_col=0;
		if (HasCapture()) ReleaseCapture();
	}
	else if (IsSizer(p.x, idx))
	{
		size_col=idx;
		left_down=p.x;
		b_is_sizing=true;
		SetCapture();
	}
	else
	{
		if (set_cur_col(p.x, idx)) WhenSorting(idx);
		Refresh();
	}
}

void Header::MouseMove(Point p, dword keyflags)
{
	if (b_is_sizing)
	{
		if (p.x>x_pos(size_col))
		{
			int w=cols[size_col].Width();
			cols[size_col].Width(w+p.x-left_down);
			Refresh();
			owner->Refresh();
		}
		left_down=p.x;
	}
}

void Header::LeftUp(Point p, dword keyflags)
{
	if (b_is_sizing)
	{
		left_down=0;
		b_is_sizing=false;
		size_col=0;
		ReleaseCapture();
	}
}

Column& Header::Add(const String &title, int width)
{
	size_t i=cols.size();
	cols.push_back(Column(title, width));
	return cols[i];
}

 Column* Header::ColumnAt(size_t idx) { if (idx<cols.size()) return &cols[idx]; else return nullptr; }

int Header::x_pos(size_t idx)
{
	int x=0;
	size_t i=0;
	while (i<cols.size()) { if (i==idx) break; x+=cols[i++].Width(); }
	return x;
}

int Header::Width()
{
	int n=0;
	if (cols.size()>0) { for (auto c:cols) n+=c.Width(); }
	return n;
}

bool Header::IsSizer(int p, size_t &idx) const
{
	int w=0;
	for (idx=0;idx<cols.size();idx++)
	{
		w+=cols[idx].Width();
		if ((p>=w-2)&&(p<=w+2)) return true;
	}
	return false;
}

bool Header::col_at_x(int p, size_t &idx)
{
	int w=0;
	for (idx=0;idx<cols.size();idx++)
	{
		w+=cols[idx].Width();
		if (p<w) return true;
	}
	return false;
}

bool Header::set_cur_col(int p, size_t &idx)
{
	if (IsSizer(p, idx)) return false;
	if (col_at_x(p, idx))
	{
		if (CurCol==idx) cols[idx].ToggleSort();
		CurCol=idx;
		if (cols[idx].IsSorting()) return true; else return false;
	}
	return false;
}


//==============================================================================================================================================================
//===============================================================================Tree
Tree::Tree()
{
	Clear();
	//AddFrame(ThinInsetFrame()); - using blackframe in tgw..
	
	AddFrame(SBH);
	SBH.WhenScroll = THISBACK(DoHScroll);
	SBH.SetLine(1);
	SBH.AutoHide();

	AddFrame(SBV);
	SBV.WhenScroll = THISBACK(DoVScroll);
	SBV.SetLine(G_ROW_CY);
	SBV.AutoHide();
}

Tree::~Tree() { Clear(); }

void Tree::Clear()
{
	roots.clear();
	CurNode=sel_pivot=nullptr;
	lines_max_width=lines_total_height=0;
	bShowTreeLines=true;
	exd=EXD_NONE;
	cur_column=0;
}

void Tree::Layout()
{
	Size sz=GetSize();
	SBV.SetPage(sz.cy);
	SBH.SetPage(sz.cx);
}

void Tree::Paint(Draw &drw)
{
	drw.DrawRect(GetSize(), G_COL_BG_DEFAULT);
	if (owner->header.cols.size()>0)
	{
		lines_max_width=owner->header.Width();
		lines_total_height=show_nodes(drw, roots, 0);
		if (bShowTreeLines) { for (auto p:roots) { if (p->IsExpanded()&&!p->nodes.empty()) show_tree_lines(drw, p->nodes); }}
		if (owner->b_vert_grid) { show_vert_grid(drw); }
		SBV.SetTotal(lines_total_height);
		SBH.SetTotal(lines_max_width+5);
	}
}

int Tree::show_nodes(Draw &drw, Nodes &NS, int Y)
{
	int	YY=Y,LC=((G_ROW_CY>G_LINE_HEIGHT)?((G_ROW_CY-G_LINE_HEIGHT)/2):0);
	Color bg, fg;
	Size sz=GetSize();
	auto focus_box=[&](const Rect &r, Color col, int lwidth=2)
					{
						drw.DrawLine(r.left, r.top, r.right, r.top, lwidth, col);
						drw.DrawLine(r.left, r.top, r.left, r.bottom-lwidth, lwidth, col);
						drw.DrawLine(r.right-lwidth, r.top, r.right-lwidth, r.bottom-lwidth, lwidth, col);
						drw.DrawLine(r.left, r.bottom-lwidth, r.right, r.bottom-lwidth, lwidth, col);
					};

	for (auto &p:NS)
	{
		if (p->IsVisible() && ((YY-SBV)>(-G_ROW_CY)) && ((YY-SBV)<(sz.cy+G_ROW_CY))) //only if seen in view
		{
			int x, y=p->y_pos, xc, w, xd, lrpad=3; //lrpad keeps text at least 3 pixels away from L & R edges
			int YC=((y+LC)-SBV), YS=y-SBV; //YC==v-center-top, YS==top-edge (both of current row)
	
			if (p->IsLocked()) drw.DrawImage(p->e_x_pos-SBH, YC, TreeGridPics::PicNodeLock());
			else if (p->nodes.empty()) drw.DrawImage(p->e_x_pos-SBH, YC, TreeGridPics::PicNodeLeaf());
			else if (p->IsExpanded()) drw.DrawImage(p->e_x_pos-SBH, YC, TreeGridPics::PicNodeContract());
			else drw.DrawImage(p->e_x_pos-SBH, YC, TreeGridPics::PicNodeExpand());
			
			//node-picture...
			Image np;
			if (p->GetPic(np)) drw.DrawImage(p->p_x_pos-SBH, YC, np);
	
			//foreground & background colors
			fg=G_COL_FG_DEFAULT;
			bg=G_COL_BG_DEFAULT;
			if (p==owner->tree.CurNode) { bg=G_COL_BG_FOCUS; fg=G_COL_FG_FOCUS; }
			else if (p->IsSelected()) { bg=G_COL_BG_SELECTED; fg=G_COL_FG_SELECTED; }
			else
			{
				int indent=p->get_indent();
				switch (indent%3)
				{
					case 0: bg=(((YY/G_ROW_CY)%2)==0)?G_PASTEL_ONE:G_PASTEL_ONE_ALT; break;
					case 1: bg=(((YY/G_ROW_CY)%2)==0)?G_PASTEL_TWO:G_PASTEL_TWO_ALT; break;
					case 2: bg=(((YY/G_ROW_CY)%2)==0)?G_PASTEL_THREE:G_PASTEL_THREE_ALT; break;
				}
			}
			drw.DrawRect(p->l_x_pos-SBH, YS, sz.cx+SBH, G_ROW_CY, bg); //NB: using YS

			//tree-label display
			const Display &dsp=(p->cells[0].has_display())?p->cells[0].GetDisplay():owner->header.cols[0].GetDisplay();
			Rect r(p->l_x_pos-SBH, YS, (p->l_x_pos+owner->header.cols[0].Width()-SBH-lrpad), YS+G_ROW_CY);
			dsp.Paint(drw, r, p->cells[0].GetData(), fg, bg, 0);

			//cell-focus-box
			if ((cur_column==0)&&(p==owner->tree.CurNode)) focus_box(Rect(p->l_x_pos-SBH-2, YS, owner->header.cols[0].Width()-SBH, YS+G_ROW_CY), inv_col(bg), 3);
			
			//cell-contents...
			for (size_t i=1;i<p->cells.size();i++)
			{
				if (i>=owner->header.cols.size()) continue;
				x=owner->header.x_pos(i);
				w=owner->header.cols[i].Width();
				drw.DrawRect(x, YS, w, G_ROW_CY, bg); //NB: YS
				
				//content-display
				const Display &dsp=(p->cells[i].has_display())?p->cells[i].GetDisplay():owner->header.cols[i].GetDisplay();
				Rect r(x-SBH+lrpad, YS, x+w-SBH-lrpad, YS+G_ROW_CY);
				dsp.Paint(drw, r, p->cells[i].GetData(), fg, bg, 0);

				if ((i==cur_column)&&(p==owner->tree.CurNode)) focus_box(Rect(owner->header.x_pos(i)-SBH+1, YS, owner->header.x_pos(i)+owner->header.cols[i].Width()-SBH, YS+G_ROW_CY), inv_col(bg), 3);
			}
		}
		YY+=G_ROW_CY; //need total for vscrollbar
		if (p->IsExpanded()) YY=show_nodes(drw, p->nodes, YY);
	}
	return YY;
}

void Tree::show_tree_lines(Draw &drw, Nodes &NS)
{
	if (NS.empty()) return;
	Node A=(*NS.begin());
	Node Z=(*NS.rbegin());
	int x,yt,yb,indent=A->get_indent();
	Color col=(indent%2)?Gray():LtGray();

	x=(((indent-1)*G_SPACE)+G_GAP+1);
	if ((x+5)<owner->header.ColumnAt(0)->Width())
	{
		x-=SBH;
		yt=A->y_pos-SBV;
		yb=Z->y_pos+G_ROW_CY/2-1-SBV;
		if (A->ParentNode)
		{
			Color bg;
			if (A->ParentNode==CurNode) bg=G_COL_BG_FOCUS;
			else
			{
				switch(indent%3)
				{
					case 0: bg=G_PASTEL_ONE_ALT; break;
					case 1: bg=G_PASTEL_TWO_ALT; break;
					case 2: bg=G_PASTEL_THREE_ALT; break;
				}
			}
			drw.DrawRect(x-G_GAP, yt, G_SPACE, (yb-yt), bg);
		}
		drw.DrawLine(x, yt, x, yb, 1, col);
		drw.DrawLine(x, yb, (x+5), yb, 1, col);
		for (auto p:NS) { if (p->IsExpanded()) show_tree_lines(drw, p->nodes); }
	}
}

void Tree::show_vert_grid(Draw &drw)
{
	Node N=owner->tree.get_last_node();
	if (N)
	{
		size_t i=0;
		int X=0, B=(N->y_pos+G_ROW_CY-1);
		while (i<owner->header.cols.size())
		{
			X+=owner->header.cols[i].Width();
			drw.DrawLine(X-SBH, 0, X-SBH, B, 1, SColorFace());
			i++;
		}
	}
}

int Tree::set_tree_positions(Nodes &NS, int Y)
{
	int	y=Y, i;

	for (auto p:NS)
	{
		if (p->IsVisible())
		{
			int x=1; //using x=0 to find nodes, else it will toggle expand/contract
			p->y_pos=y;
			for (i=0;i<p->get_indent();i++) x+=G_SPACE;
			p->e_x_pos=x; //pos of empty, dot or expand/contract picture
			x+=G_SPACE+G_GAP;
			//if (!p->pic.IsEmpty()) { p->p_x_pos=x; x+=G_PIC_WIDTH+G_SPACE; } //pic-pos
			if (p->picid) { p->p_x_pos=x; x+=G_PIC_WIDTH+G_SPACE; } //pic-pos
			p->l_x_pos=x; //text-position
			y+=G_ROW_CY;
			if (p->IsExpanded()) y=set_tree_positions(p->nodes, y);
		}
	}
	return y;
}

Node Tree::get_node(Nodes &NS, const String &key)
{
	if (key.IsEmpty()) return nullptr;
	Node N=nullptr;
	
	auto it=NS.begin();
	while (!N&&(it!=NS.end()))
	{
		if ((*it)->key==key) N=(*it);
		else N=get_node((*it)->nodes, key);
		it++;
	}
	return N;
}

Node Tree::get_last_node()
{
	Node p=nullptr;
	if (roots.size()>0)
	{
		p=roots.at(roots.size()-1);
		while (p->IsExpanded()&&(p->nodes.size()>0)) { p=p->nodes.at(p->nodes.size()-1); }
	}
	return p;
}

void Tree::sync_nodes()	{ set_tree_positions(roots, 0); Refresh(); }

Node Tree::AddNode(Node Parent, const String &treetext, const String &key)
{
	Node N=nullptr;
	if (!key.IsEmpty()&&node_keys.RFind(key, N)) return nullptr;
	Nodes &NS=(Parent)?Parent->nodes:roots;
	N=NS.Add(key);
	if (N)
	{
		N->ParentNode=Parent;
		N->AddCell(treetext);
	}
	return N;
}

Node Tree::AddNode(Node Parent, Image pic, const String &treetext, const String &key)
{
	Node N=AddNode(Parent, treetext, key);
	if (N) N->picid=node_pics.Add(pic);
	return N;
}

Node Tree::GetNode(const String &key)
{
	if (roots.size()>0) return get_node(roots, key);
	return nullptr;
}

void Tree::SelectNode(Node N, bool bSelect)
{
	if (N&&!N->IsLocked()&&!contains_locked_nodes(N))
	{
		N->select(bSelect);
		if (bSelect) b_has_selected=true;
	}
}

void Tree::SelectRange(Node N1, Node N2)
{
	if (N1&&N2)
	{
		unselect_nodes(roots);
		Node t=N1, b=N2, n;
		if (b->y_pos<t->y_pos) { n=t; t=b; b=n; }
		n=t;
		while (n&&(n!=b))
		{
			SelectNode(n, true);
			n=get_node_at_y((n->y_pos+G_ROW_CY+1), roots);
		}
		b->select();
		b_has_selected=true;
	}
}

void Tree::Lock(Node N, bool b)
{
	if (N)
	{
		if (b&&N->IsLocked()) return;
		N->b_locked=b;
		Refresh();
	}
}

void Tree::DropNodes(Nodes &NS) { NS.clear(); }
void Tree::DropNode(Node N)
{
	Nodes &NS=(N->ParentNode)?N->ParentNode->nodes:roots;
	NS.Drop(N);
}

void Tree::DoVScroll()	{ Refresh(); }
void Tree::DoHScroll()	{ Refresh(); owner->header.scroll_x((SBH>0)?(-SBH):0); }

Node Tree::get_y_node(int Y, Node N)
{
	Node pn=nullptr;
	auto Is_Y = [&](Node n, int y){ Node r=nullptr; if (((n->y_pos)<=y)&&((n->y_pos+G_ROW_CY)>=y)) r=n; return r; };
	if (N&&!(pn=Is_Y(N, Y))&&N->IsExpanded()) { for (auto p:N->nodes) { pn=get_y_node(Y, p); if (pn) break; }}
	return pn;
}

Node Tree::get_node_at_y(int Y, Nodes &NS)
{
	Node R=nullptr;
	for (auto p:NS) { if ((R=get_y_node(Y, p))) break; }
	return R;
}

bool Tree::check_expand(Node N, Point p) //toggle expand/contract if clicked on "button"
{
	if (!N) return false;
	if (((p.x+SBH)>=(N->e_x_pos-G_SPACE))&&((p.x+SBH)<(N->e_x_pos+(G_SPACE+G_GAP))))
	{
		Expand(N, !N->IsExpanded());
		return true;
	}
	return false;
}

void Tree::check_cur_node_contract(Node A)
{
	if (CurNode)
	{
		Node p=CurNode->ParentNode;
		while (p && (p!=A)) p=p->ParentNode;
		if (p==A) FocusNode(A); //if ancestor - change focus
	}
}

void Tree::Expand(Node N, bool b)
{
	if (N) //&&!N->IsLocked())
	{
		if (b)
		{
			owner->OnBNE(N); //before expanding
			N->expand();
			sort_nodes(&owner->header, N->nodes, owner->sort_col, owner->header.cols[owner->header.CurCol].IsASC());
		}
		else
		{
			check_cur_node_contract(N);
			for (auto &n:N->nodes) Expand(n, false);
			N->contract();
			owner->OnANC(N); //after contracting
		}
		sync_nodes();
	}
}

void Tree::FocusNode(Node pn, bool bView)
{
	if (pn)
	{
		CurNode=pn;
		if (CurNode->ParentNode&&!CurNode->ParentNode->IsExpanded()) Expand(CurNode->ParentNode, true);
		if (bView) BringToView(CurNode);
		Refresh();
		WhenFocus(CurNode);
	}
}

void Tree::BringToView(Node N)
{
	if (N)
	{
		Size sz=GetSize();
		int XL=owner->header.x_pos(cur_column), XR=(XL+owner->header.ColumnAt(cur_column)->Width());
		SBV.ScrollInto(N->y_pos);
		if ((XL-SBH)<0) SBH.ScrollInto(XL); else if (XR>sz.cx) SBH.ScrollInto(XR);
		Refresh();
	}
}

void Tree::sort_nodes(Header *pH, Nodes &NS, size_t c, bool b)
{
	if (pH->cols[c].Sorter) std::sort(NS.begin(), NS.end(), pH->cols[c].Sorter); //user's sorting func
	else std::sort(NS.begin(),NS.end(), [&](const Node &l, const Node &r) // default = lcase-compare string-sort
										{
											bool br=false;
											if (exd!=EXD_NONE)
											{
												if (l->CanExpand()&&!r->CanExpand()) return ((exd==EXD_FIRST)?true:false); //expandable nodes ..
												if (!l->CanExpand()&&r->CanExpand()) return ((exd==EXD_FIRST)?false:true); //.. appear first:last
											}
											//else fall-thru for EXD_NONE
											if ((c<l->cells.size())&&(c<r->cells.size()))
											{
												int n=0,i=0,m=max(l->cells[c].GetData().ToString().GetLength(), r->cells[c].GetData().ToString().GetLength());
												while (!n&&(i<m)) { n=(ToLower(l->cells[c].GetData().ToString()[i])-ToLower(r->cells[c].GetData().ToString()[i])); i++; }
												//int n=CompareNoCase(l->cells[c].GetData().ToString(), r->cells[c].GetData().ToString());
												if (b) br=(n<0); else br=(n>0);
											}
											return br;
										});
	for (auto N:NS) sort_nodes(pH, N->nodes, c, b);
}

void Tree::SortNodes(Header *pH, size_t cellidx, bool bASC)
{
	 sort_nodes(pH, roots, cellidx, bASC);
}

void Tree::unselect_nodes(Nodes &NS)
{
	for (auto &p:NS)
	{
		if (p->IsSelected()) p->select(false);
		unselect_nodes(p->nodes);
		b_has_selected=false;
	}
}

size_t Tree::get_selected_nodes(Nodes &NS, std::vector<String> &V)
{
	if (!b_has_selected) return 0;
	for (auto p:NS)
	{
		if (p->IsSelected()) V.push_back(p->Key());
		get_selected_nodes(p->nodes, V);
	}
	return V.size();
}

bool Tree::contains_locked_nodes(Node N)
{
	bool b=false;
	for (auto p:N->nodes)
	{
		if (p->IsLocked()) b=true; else b=contains_locked_nodes(p);
		if (b) break;
	}
	return b;
}

bool Tree::Key(dword key, int)
{
	bool shift=key&K_SHIFT;
	bool ctrl=key&K_CTRL;
	bool alt=key&K_ALT;

	if (shift) key&=~K_SHIFT;
	if (ctrl) key&=~K_CTRL;
	if (alt) key&=~K_ALT;

	switch(key)
	{
		case K_F1: { if (owner->WhenHelp) owner->WhenHelp(); else owner->OnTGHelp(); return true; } break;
		case K_ENTER: { if (CurNode) { WhenLeftDouble(CurNode, cur_column); return true; }} break;
		case K_UP: case K_DOWN:
			{
				if (CurNode)
				{
					Node pn=get_node_at_y((key==K_UP)?(CurNode->y_pos-G_ROW_CY+1):(CurNode->y_pos+G_ROW_CY+1), roots);
					if (pn)
					{
						if (shift)
						{
							if (!sel_pivot) sel_pivot=CurNode;
							owner->SelectRange(sel_pivot, pn);
						}
						else
						{
							unselect_nodes(roots);
							sel_pivot=nullptr;
						}
						FocusNode(pn);
					}
					return true;
				}
				else return SBV.VertKey(key);
			} break;
		case K_PAGEUP: case K_PAGEDOWN:
			{
				owner->ClearSelection();
				sel_pivot=nullptr;
				if (CurNode)
				{
					Size sz=GetSize();
					int y=CurNode->y_pos-SBV+(sz.cy%G_ROW_CY);
					if (key==K_PAGEUP) SBV=SBV-sz.cy; else SBV=SBV+sz.cy;
					Node pn=get_node_at_y(y+SBV, roots);
					FocusNode(pn);
					return true;
				}
				else return SBV.VertKey(key);
			} break;
		case K_HOME: case K_END:
			{
				if (roots.size()>0)
				{
					if (ctrl)
					{
						owner->ClearSelection();
						sel_pivot=nullptr;
						Node N=(key==K_HOME)?roots.at(0):get_last_node();
						FocusNode(N);
						return true;
					}
					else if (CurNode) { cur_column=(key==K_HOME)?0:(owner->header.cols.size()-1); BringToView(CurNode); return true; }
				}
				return SBH.VertKey(key);
			} break;
		case K_LEFT:
			{
				if (CurNode)
				{
					if (ctrl&&CurNode->IsExpanded())
					{
						owner->ClearSelection();
						sel_pivot=nullptr;
						Expand(CurNode, false);
					}
					else { cur_column-=(cur_column>0)?1:0; BringToView(CurNode); }
					return true;
				}
				return SBH.VertKey(key);
			} break;
		case K_RIGHT:
			{
				if (CurNode)
				{
					if (ctrl&&!CurNode->IsExpanded())
					{
						owner->ClearSelection();
						sel_pivot=nullptr;
						Expand(CurNode, true);
					}
					else { cur_column+=(cur_column<(owner->header.cols.size()-1))?1:0; BringToView(CurNode); }
					return true;
				}
				else return SBH.VertKey(key);
			} break;
		case K_M: //popupmenu.. Ctrl+M
			{
				if (ctrl)
				{
					Rect r=get_abs_tree_rect(); ///works - chg to GetScreenRect() / GetScreenView() ... later maybe
					Point p;
					if (CurNode)
					{
						p.x=r.left+(owner->header.x_pos(cur_column)+owner->header.ColumnAt(cur_column)->Width()/2)-SBH;
						p.y=r.top+(CurNode->y_pos+G_ROW_CY/2)-SBV;
					}
					else
					{
						Size sz=GetSize(); //how to get dimensions of menu-popup? - for centering
						p.x=r.left+sz.cx/3;
						p.y=r.top+sz.cy/3; //default to top-left 1/3-pos
					}
					owner->ShowPopMenu(p);
				}
			} break;
		default: return SBV.VertKey(key); break;
	}
	return false;
}

Rect Tree::get_abs_tree_rect()
{
	Ctrl *p=GetParent();
	Rect w, r=GetRect();
	while (p) { w=p->GetRect(); r.left+=w.left; r.top+=w.top; p=p->GetParent(); } //top parent (p==nullptr) should have absolute coord's?
	return r;
}

size_t Tree::set_click_column(int x) { owner->header.col_at_x(x-SBH, cur_column); }

bool Tree::point_in_tree(Point p)
{
	if (roots.size()==0) return false;
	if ((p.x<0)||(p.x>owner->header.Width())) return false;
	if ((p.y<0)||(p.y>(get_last_node()->y_pos+G_ROW_CY))) return false;
	return true;
}

Node Tree::prev_node(Node N)
{
	if (N)
	{
		Node pn=get_node_at_y((N->y_pos-G_ROW_CY+1), roots);
		return pn;
	}
	return nullptr;
}

void Tree::MouseWheel(Point p, int zdelta, dword keyflags) { SBV.Wheel(zdelta); }

void Tree::LeftDown(Point p, dword flags)
{
	size_t idx;
	if (owner->header.IsSizer(p.x, idx))
	{
		owner->header.LeftDown(p, flags);
	}
	else
	{
		if (!point_in_tree(p)) return;
		Node N=get_node_at_y(p.y+SBV, roots);
		if (N)
		{
			if (check_expand(N, p)) Refresh();
			else
			{
				set_click_column(p.x);
				WhenLeftDown(N, flags);
			}
		}
	}
}

void Tree::LeftUp(Point p, dword flags)
{
	if (owner->header.b_is_sizing) owner->header.LeftUp(p, flags);
}

void Tree::RightDown(Point p, dword flags)
{
	if (point_in_tree(p))
	{
		Node pn=get_node_at_y((p.y+SBV), roots);
		if (!pn->IsSelected()) unselect_nodes(roots);
		FocusNode(pn);
		set_click_column(p.x);
	}
	WhenRightDown(CurNode); //CurNode may be nullptr..
}

void Tree::LeftDouble(Point p, dword flags)
{
	if (!point_in_tree(p)) return;
	Node N=get_node_at_y(p.y+SBV, roots);
	unselect_nodes(roots);
	if (check_expand(N, p)) Refresh();
	else
	{
		set_click_column(p.x);
		WhenLeftDouble(N, cur_column);
	}
	//FocusNode(N);
}

void Tree::MouseMove(Point p, dword flags)
{
	if (owner->header.b_is_sizing) owner->header.MouseMove(p, flags);
}

Image Tree::CursorImage(Point p, dword flags)
{
	return owner->header.CursorImage(p, flags);
	//if (b_is_sizing) return TreeGridPics::PicSizer();
	//if (is_on_sizer(p.x)) return TreeGridPics::PicSizer();
	//return Image::Arrow();
}


//==============================================================================================================================================================
//===============================================================================TreeGrid
TreeGrid::TreeGrid()
{
	Clear(); //init's
	
	
	Add(header.HSizePosZ().TopPos(0, G_LINE_HEIGHT).AddFrame(ThinInsetFrame()));
	header.owner=this;
	header.WhenSorting = THISBACK(OnSorting);
	
	Add(tree.HSizePosZ().VSizePosZ(G_LINE_HEIGHT));
	tree.owner=this;
	tree.WhenFocus << Proxy(WhenFocus);
	tree.WhenLeftDown << THISBACK(OnLeftDown);
	tree.WhenLeftDouble << THISBACK(OnLeftDouble);
	tree.WhenRightDown << THISBACK(OnRightDown);
	tree.WhenBeforeNodeExpand << THISBACK(OnBNE); //Proxy(WhenBeforeNodeExpand); ???
	tree.WhenAfterNodeContract << THISBACK(OnANC); //Proxy(WhenAfterNodeContract);
	
	AddFrame(BlackFrame());
}

TreeGrid::~TreeGrid() { Clear(); }

void TreeGrid::OnBNE(Node N) { WhenBeforeNodeExpand(N); }
void TreeGrid::OnANC(Node N) { WhenAfterNodeContract(N); }

void TreeGrid::Clear()
{
	header.Clear();
	tree.Clear();
	sort_col=0;
	b_vert_grid=true;
	b_internal=false;
}

void TreeGrid::Select(Node N, bool bSelect) { if (N) { tree.SelectNode(N, bSelect); Refresh(); }}

void TreeGrid::SelectRange(Node N1, Node N2) { tree.SelectRange(N1, N2); Refresh(); }

void TreeGrid::ClearSelection()
{
	tree.unselect_nodes(tree.roots);
	Refresh();
}

size_t TreeGrid::GetSelection(std::vector<String> &v)
{
	v.clear();
	return tree.get_selected_nodes(tree.roots, v);
}

bool TreeGrid::Key(dword key, int r) { return tree.Key(key, r); }

void TreeGrid::OnSorting(size_t idx)	{ sort_col=idx; SyncNodes(); tree.BringToView(GetFocusNode()); }
size_t TreeGrid::GetSortColumnID()		{ return sort_col; }
bool TreeGrid::IsSortASC()				{ return header.cols[sort_col].IsASC(); }

Column& TreeGrid::AddColumn(const String title, int width) { return header.Add(title, width); }

void TreeGrid::check_cells(Node N)
{
	if (N)
	{
		size_t n=N->CellCount();
		while (n<header.cols.size()) { N->AddCell(""); n++; }
		for (size_t i=0;i<header.cols.size();i++) N->CellAt(i)->when_resync = THISBACK(SyncNodes); //sorting etc.
	}
}

Node TreeGrid::AddNode(Node Parent, const String &name, const String &key) { Node N=tree.AddNode(Parent, name, key); check_cells(N); return N; }

void TreeGrid::SyncNodes()
{
	if (header.cols.size()>0)
	{
		if (header.cols[sort_col].IsSorting()) tree.SortNodes(&header, sort_col, header.cols[sort_col].IsASC());
		tree.sync_nodes();
	}
}

void TreeGrid::ClearTree()								{ tree.Clear(); }
void TreeGrid::RefreshTreeGrid()						{ SyncNodes(); }
void TreeGrid::DeleteNode(Node N)						{ tree.DropNode(N); }
Node TreeGrid::GetNode(const String &key)				{ return tree.GetNode(key); }
Node TreeGrid::RootAt(size_t idx)						{ return ((idx<tree.roots.size())?tree.roots[idx]:nullptr); }
size_t TreeGrid::RootCount()							{ return tree.roots.size(); }

void TreeGrid::LockNode(Node N, bool bExpand)
{
	if (bExpand) { Expand(N, bExpand); tree.Lock(N, true); }
	else { tree.Lock(N, true); Expand(N, bExpand); }
}

void TreeGrid::UnlockNode(Node N)						{ tree.Lock(N, false); }
void TreeGrid::Expand(Node N, bool b)					{ tree.Expand(N, b); }
void TreeGrid::ClearNodes(Node N)						{ N->nodes.clear(); }
void TreeGrid::SetFocusNode(Node N, bool b)				{ tree.FocusNode(N, b); }
Node TreeGrid::GetFocusNode()							{ return tree.CurNode; }
size_t TreeGrid::GetFocusCell()							{ return tree.cur_column; }
Node TreeGrid::GetPrevNode(Node N)						{ return tree.prev_node(N); }
bool TreeGrid::NodeHasCell(Node N, size_t cellidx)			{ return (cellidx<N->CellCount()); }
Cell* TreeGrid::GetCell(Node N, size_t cellidx)			{ return N->CellAt(cellidx); }

void TreeGrid::OnLeftDown(Node N, dword flags)
{
	if (N)
	{
		if (flags&K_SHIFT) { tree.sel_pivot=tree.CurNode; SelectRange(tree.CurNode, N); }
		else if (flags&K_CTRL) { tree.sel_pivot=nullptr; Select(N, !N->IsSelected()); }
		else ClearSelection();
		SetFocusNode(N);
	}
}

void TreeGrid::OnLeftDouble(Node N, size_t cellidx)
{
	if (N)
	{
		ClearSelection();
		tree.sel_pivot=nullptr;
		SetFocusNode(N);
		WhenActivate(N, cellidx);
	}
}

void TreeGrid::FillMenuBar(MenuBar &menubar)
{
	menubar.Add("TreeGrid Help", THISBACK(OnTGHelp));
	menubar.Separator();
	WhenMenuBar(menubar);
}

void TreeGrid::OnTGHelp()
{
	std::string s = "\nMouse:\n"
					"     o  (normal stuff)\n"
					"     o  Right-click : context popup menu\n"
					"     o  Ctrl+Left-click : multiple row select\n"
					"\nKeyboard:\n"
					"     o  Left/Right arrow-keys : selects along the cells in a row\n"
					"     o  Ctrl+Left/Right arrow-keys : expand/contract current node (if it can)\n"
					"     o  Ctrl+M : context popup menu\n"
					"     o  Up/Down, PageUp/PageDown : (as expected)\n"
					"     o  Shift+Up/Down arrow-keys : multiple row select\n"
					"     o  Home/End : first/last cell in row\n"
					"     o  Ctrl+Home/End : top and bottom of tree\n"
					"\nUnder construction (feedback appreciated)\n";
					//"     - To Do:\n"
					//"          o  make rows variable height and text-content wrapable\n"
					//"          o  adding controls to cells/columns\n"
					//"          o  ...\n";

	PromptOK(DeQtf(s.c_str()));
}

void TreeGrid::OnRightDown(Node N)
{
	MenuBar menubar;
	FillMenuBar(menubar);
	menubar.Execute();
}

void TreeGrid::ShowPopMenu(Point p)
{
	MenuBar menubar;
	FillMenuBar(menubar);
	if (!menubar.IsEmpty())	menubar.Execute(this, p);
}

void TreeGrid::ListExpandingNodes(EXDisp exd) { tree.exd=exd; SyncNodes(); }
EXDisp TreeGrid::ListExpandingNodes() { return tree.exd; }

void TreeGrid::ShowTreeLines(bool bShow) { tree.bShowTreeLines=bShow; Refresh(); }
bool TreeGrid::ShowTreeLines() { return tree.bShowTreeLines; }

void TreeGrid::ShowHeader(bool bShow)
{
	header.Show(bShow);
	if (!bShow) tree.SetRectY(0, tree.GetSize().cy+G_LINE_HEIGHT);
	else tree.SetRectY(G_LINE_HEIGHT+1, tree.GetSize().cy-G_LINE_HEIGHT-1);
}

bool TreeGrid::ShowHeader() { return header.IsVisible(); }

