#ifndef _Barcode_Barcode_h_
#define _Barcode_Barcode_h_

#include <Draw/Draw.h>

NAMESPACE_UPP
 
// define some common interfaces for the linear&2-D barcodes that will be implemented
//
//
class Barcode{
public:
	const Drawing& AsDrawing()const
	{
		if(draw.IsEmpty())
		{
			const_cast<Barcode&>(*this).Prepare();
		}
		return *draw;
	}
	operator const Drawing&()const{
		return AsDrawing();
	}
	
	Size GetSize()const{
		return AsDrawing().GetSize();
	}
	
	Barcode& Text(const char * text)
	{
		SetText(text);
		return *this;
	}
	
	Barcode& Text(char * text){
		return Text(const_cast<const char*>(text));
	}
	
	Barcode& DisplayText(const char * text)
	{
		this->text=text;
		return MarkAsDirty();
	}
	Barcode& DisplayText(char* text){
		return DisplayText(const_cast<const char*>(text));
	}
	
	Barcode& Font(Upp::Font font){
		this->font=font;
		return MarkAsDirty();
	}
	
	Barcode& Color(Upp::Color color){
		this->color=color;
		return MarkAsDirty();
	}
	
	Barcode& BarRatio(unsigned ratio){
		ASSERT(ratio>0 && ratio<128); // for debug time
		if(ratio>127)
			ratio=127; // for runtime
		else if(ratio==0)
			++ratio;
		
		barRatio=ratio;
		return MarkAsDirty();
	}
	
	Barcode& BarHeight(unsigned height)
	{
		return BarRatio(height);
	}
	
	Barcode& NoQuietZone()
	{
		if(!noQuietZone)
		{
			ClearDraw();
			noQuietZone=true;
		}
		return *this;
	}
	
	Barcode& WithQuietZone()
	{
		if(noQuietZone)
		{
			ClearDraw();
			noQuietZone=false;
		}
		return *this;
	}

	bool IsEmpty()const{ return GetSize().cx==1; }
	virtual ~Barcode(){}
	virtual void Prepare()=0;
	virtual void SetText(const char * text, int len=-1)=0;
	
protected:
	One<Drawing>   draw;
	Upp::String    text; // displayed text;
	Upp::Font	   font; // for text display
	Upp::Color	  color; // the bar color & display text
	unsigned barRatio:7; //
	bool  noQuietZone:1; //
	bool  truncated:1;   // for PDF417; for I2Of5, width of wide bar/that of narrow bar, 0 for 2, 1 for 3.


	Barcode& MarkAsDirty(){
		ClearDraw();
		return *this;
	}

	void Init()
	{
		font=StdFont().Height(10);
		color=Black();
		noQuietZone=false;
		barRatio=1;
	}
	
	void ClearDraw()
	{
		draw.Clear();
	}
};

class Code128 :public Moveable<Code128>, public Barcode
{
public:
	Code128(){Init();  }
	Code128(const char* text)
	{ Init();Text(text);}
	Code128(char * text)
	{ Init();Text(text);}

	virtual void Prepare();
	virtual void SetText(const char * text, int len=-1);
protected:
	Vector<uint8> data;

	const Vector<uint8>& GetData()const{ return data; }

	// implementation detail, to be moved.

	int BarCodeWidth()const{
		return (data.GetCount()*11+2+(noQuietZone?0:20));
	}
	
	int  GetWidth()const
	{
		return  max(BarCodeWidth(),
					GetTextSize(text,font).cx
				);
	}
};


// 2013-3-6: Added EAN8 and UPCA
//
class EAN : public Moveable<EAN>,public Barcode
{
public:
	EAN()
	{Init(); ClearData();}
	EAN(const char* text)
	{Init(); Text(text);}
	EAN(char * text)
	{Init(); Text(text);}
	
	EAN& Type(int upcvariant);
	int    Type()const;

	virtual void Prepare();
	virtual void SetText(const char *, int len=-1);

	enum{EAN13, DEFAULT=EAN13, EAN8, UPCA,
		// virtual types, automatic choice
			EAN8UPCAEAN13, // if can be represented as EAN8
						   // do it; otherwise if can be
						   // represented as UPCA, do it;
						   // otherwise represented as EAN13
			EAN8EAN13		// similar to above
		};
protected:
	char ___[14];
	int		type; // one of EAN, UPCA, EAN8
	
	int BarcodeWidth()const{
		const int guard=2*3+5; //guard area width(stard,end and middle)
		int digits=	ActualType()==EAN8?8:12;
		// each digit has a total width of 7 x-dimensions
		return guard+(noQuietZone?12:18)+7*digits;
	}

	/*virtual */int  GetWidth()const{
		return max(BarcodeWidth(), GetTextSize(text,font).cx);
	}
	
	void ClearData()
	{
		*reinterpret_cast<int*>(___)=0;
	}

	// convert virtual type to actual type
	int ActualType()const;
	
	void Init()
	{
		Barcode::Init();
		type=EAN13;
		barRatio=83;
	}
};

class ITF : public Moveable<ITF>,public Barcode
{
public:
	ITF()
	{Init();}
	ITF(const char* text)
	{Init(); Text(text);}
	ITF(char * text)
	{Init(); Text(text);}
	
	ITF& WideBarWidth(int width);
	int  WideBarWidth()const{ return 2+truncated; }
	
	ITF& AddCheckSum();
	
protected:

	virtual void Prepare();
	virtual void SetText(const char *, int len=-1);

private:
	//void Init(){}
	String data;
	void Init()
	{
		Barcode::Init();
		barRatio=30;
	}

	int BarcodeWidth()const;
	
	void DoDraw(Draw& w, int width);
};

class PDF417 : public Moveable<PDF417>, public Barcode
{
public:
	PDF417(){Init();}
	PDF417(const char * begin, const char *end){Init(); SetText(begin, end-begin); }
	PDF417(const char * text){Init(); SetText(text, -1); }
	virtual void SetText(const char * c, int len=-1);
	virtual void Prepare();
	
	PDF417& Truncated(){ if(!truncated){ ClearDraw(); truncated=true; } return *this; }
	PDF417& NotTruncated(){ if(truncated){ ClearDraw(); truncated=false;} return *this; }
	
	PDF417& ErrorCorrectionLevel(int level);
	
	PDF417& FixedColumnCount(int col);
	PDF417& FixedRowCount(int row);
	PDF417& FixedRowColCount(int row, int col);
	PDF417& FixedAspectRatio(int row, int col);
	
	//String ToString()const; // dump codewords for debugging

	typedef uint16 CodeWord;
		
protected:
	Vector<CodeWord> codewords;
	Vector<CodeWord> ec; // error correction code words
	int				 eclevel; // error correction level;
	int				 shapingType;
	int				 shapingValue;
	
	
	int GetWidth()const;

	int ActualEC()const; // the effective error correction level
	Size ActualDataRectangleSize()const;
	int MaximumCodeWordCount()const;
	int CodeWordCountBeforePadding()const{ return codewords.GetCount()+(2<<ActualEC())+1; }
	
	int GetCodeWord(int i, int count)const;
	
	void Init();
};


char UpcCheckDigit(const char * numbers);


END_UPP_NAMESPACE

#endif
