#ifndef _JobQueue_JobQueue_h
#define _JobQueue_JobQueue_h

#include <Core/Core.h>

NAMESPACE_UPP;

typedef Function<bool(void)> Job;

class JobQueue : Moveable<JobQueue> {
    
    Vector<Job> queue;
    bool        halted = false;
    int         start_time;
    int         timeout;
    int         error_code;
    String      error_desc;
 
    void                        Pick(JobQueue&& jq);            
                
protected:
    Job&                        AddJob()                         { return queue.Add(); }
    JobQueue&                   AddJob(const Job& jb)             { queue.Add(jb); return *this; }
    Job&                        InsertJob(int i)                 { return queue.Insert(i); }
    JobQueue&                   InsertJob(int i,const Job& jb)   { queue.Insert(i, jb); return *this; }
    Job&                        GetJob(int i = 0)                { return queue.At(i); }
    void                        RemoveJob(int i)                 { queue.Remove(i); }
    void                        NextJob()                        { if(!queue.IsEmpty()) queue.Remove(0); }
    void                        ClearQueue()                     { if(!queue.IsEmpty()) queue.Clear(); halted = false; start_time = 0; error_code = 0; }
    bool                        QueueHasJobs() const             { return queue.GetCount() > 1; }
    bool                        QueueIsHalted() const            { return halted; }
    bool                        QueueIsEmpty() const             { return queue.IsEmpty(); }
    int                         GetJobCount() const              { return queue.GetCount(); }
    bool                        Halt(const char* reason = NULL, int code = -1);  
    bool                        IsTimeout() const                { return timeout > 0 && msecs(start_time) >= timeout; }   

public:
    virtual bool                Do(); 
    
    bool                        InProgress() const                { return !queue.IsEmpty();  }
    bool                        IsSuccess() const                 { return queue.IsEmpty() && !halted; }
    bool                        IsFailure() const                 { return halted;  }   
    
    void                        Timeout(int ms)                   { timeout = Nvl(ms, 0); }
    Event<>                     WhenDo;

    int                         GetError() const                  { return error_code; }                
    const String&               GetErrorDesc() const              { return error_desc; }
    
    JobQueue() : start_time(0), timeout(0), error_code(0) {}
    virtual ~JobQueue() {};
  
    JobQueue(JobQueue&& jq)                                       { Pick(pick(jq)); }
    void operator=(JobQueue&& jq)                                 { Pick(pick(jq)); }    
};
END_UPP_NAMESPACE;

#endif
